const isPlainObject = require('lodash/isPlainObject');
const url = require('url');

const Controller = require('./index.js');
/**
 * The object responsible for interactions of the registration process
 *
 * Api requests should be proxied here, request data should be fetched here, response should be sent here
 *
 * @type {RegistrationController}
 */

class RegistrationController extends Controller {
    /**
     * Common method to register an account
     *
     * @param {object} normalizedFormData       Registration data to be passed to api
     * @param {string} apiRegistrationMethod    Api method to use
     * @returns {Promise}
     * @private
     * @throws
     */
    _commonRegistrationProxy(normalizedFormData, apiRegistrationMethod) {
        const logger = this._logger;

        return this._getApiPromise()
            .then(function(api) {
                logger.info('Creating an account using', apiRegistrationMethod);
                logger.verbose('Creating an account with form data:', normalizedFormData);

                return api[apiRegistrationMethod](normalizedFormData);
            })
            .then(function(response) {
                if (isPlainObject(response.body)) {
                    if (response.body.status === 'ok') {
                        logger.info('Registration OK using', apiRegistrationMethod);
                        return Promise.resolve(response);
                    }

                    if (response.body.errors) {
                        return Promise.reject(response.body.errors);
                    }
                }

                return Promise.reject(response);
            });
    }

    /**
     * Register using accountRegisterWithEmail
     * @see _commonRegistrationProxy
     *
     * @param {object} normalizedFormData
     * @returns {Promise}
     * @throws
     */
    withEmailRegistration(normalizedFormData) {
        return this._commonRegistrationProxy(normalizedFormData, 'accountRegisterWithEmail');
    }

    /**
     * Register using accountRegisterAlternative
     * @see _commonRegistrationProxy
     *
     * @param {object} normalizedFormData
     * @returns {Promise}
     * @throws
     */
    alternativeRegistration(normalizedFormData) {
        return this._commonRegistrationProxy(normalizedFormData, 'accountRegisterAlternative');
    }

    /**
     * Register using accountRegister
     * @see _commonRegistrationProxy
     *
     * @param {object} normalizedFormData
     * @returns {Promise}
     * @throws
     */

    simpleRegistration(normalizedFormData) {
        return this._commonRegistrationProxy(normalizedFormData, 'accountRegister');
    }

    /**
     * Register using accountRegisterWithPhoneAlias
     * @see _commonRegistrationProxy
     *
     * @param {object} normalizedFormData
     * @returns {Promise}
     * @throws
     */
    digitAliasRegistration(normalizedFormData) {
        return this._commonRegistrationProxy(normalizedFormData, 'accountRegisterWithPhoneAlias');
    }

    /**
     * Register using workspaceRegistration
     * @see _commonRegistrationProxy
     *
     * @param {object} normalizedFormData
     * @returns {Promise}
     * @throws
     */
    workspaceRegistration(normalizedFormData) {
        return this._commonRegistrationProxy(normalizedFormData, 'workspaceRegistration');
    }

    /**
     * Fetch the api and pass it to the form
     * @param {Form} form
     * @returns {Promise}
     * @throws
     */
    setFormApi(form) {
        return this._getApiPromise().then(function(api) {
            return form.setApi(api);
        });
    }

    /**
     * Create a session for the user
     * @param {object} options
     * @returns {Promise}
     */
    createSession(options = {}) {
        const {retpath} = options;

        return this._getApiPromise()
            .then(function(api) {
                return api.bundleSession({
                    track_id: api.track(),
                    retpath
                });
            })
            .then((result) => {
                const cookies = result.body && result.body.cookies;

                if (cookies) {
                    this.augmentResponse(result.body);
                }

                return result;
            });
    }

    /**
     * Subscribe to galatasaray.net SID
     *
     * @returns {Promise}
     */
    subscribeToGalatasaray() {
        const self = this;

        if (self.getRequestParam('from') === 'galatasaray') {
            return self.readTrack().then(function(trackContents) {
                if (trackContents.uid) {
                    return self.accountSubscribe({
                        uid: trackContents.uid,
                        service_slug: 'galatasaray'
                    });
                }
                return Promise.resolve();
            });
        }
        return Promise.resolve();
    }

    /**
     * Check the session cookie
     * @param {string} sessionCookieContents    The session to check
     * @returns {Promise}
     * @throws
     */
    checkSession(sessionCookieContents) {
        const logger = this._logger;

        return this._getApiPromise()
            .then(function(api) {
                return api.sessionCheck({session: sessionCookieContents});
            })
            .then(function(result) {
                if (result.body && typeof result.body.session_is_correct !== 'undefined') {
                    // Response seems valid, act on its result
                    if (result.body.session_is_correct) {
                        return Promise.resolve();
                    } else {
                        logger
                            .warn()
                            .type('checkSession', 'sessionCheckFailed')
                            .write('Session is incorrect');

                        return Promise.reject();
                    }
                }

                logger
                    .warn()
                    .type('checkSession', 'sessionCheckFailed')
                    .write('Session check failed with %s', result);

                // Looks invalid, reject with it as though it is an error
                return Promise.reject(result);
            });
    }

    /**
     * Redirect the user to registration finishing route
     *
     * Redirect makes the browser send newly set cookies, so that hey can be checked
     *
     * @param {string} trackId
     * @returns undefined
     * @throws
     */
    redirectToFinish(trackId) {
        const currentUrl = this.getUrl();
        const redirectUrl = url.format({
            protocol: currentUrl.protocol,
            hostname: currentUrl.hostname,
            pathname: '/registration/finish',
            query: {
                track_id: trackId
            }
        });

        this._logger.debug('redirectToFinish — Redirecting to', redirectUrl);
        this.redirect(redirectUrl);
    }

    /**
     * Subscribe user to SID
     * @returns {Promise}
     */
    accountSubscribe(data) {
        return this._getApiPromise().then(function(api) {
            return api.accountSubscribe(data);
        });
    }

    setRawErrors(form) {
        const rawErrors = this.getLocalsField('rawErrors');

        if (Array.isArray(rawErrors) && rawErrors.length) {
            rawErrors.forEach(function(error) {
                form.getField(error.field).setErrorsActive([error.code]);
            });
        }

        return Promise.resolve(form);
    }
}

module.exports = RegistrationController;
