'use strict';
var config = require('../../configs/current');
const isLocalDev = config.version === 'local';

var _ = require('lodash');
var url = require('url');
var util = require('util');
var PLog = require('plog');
var putils = require('putils');
var prequest = require('prequest');
var when = require('when');
var PController = require('pcontroller');
var expressLangDetect = isLocalDev
    ? require('express-http-langdetect')
    : require('@yandex-int/express-langdetect').default;
var expressUaTraits = isLocalDev ? require('express-http-uatraits') : require('@yandex-int/express-uatraits');
var PassportApi = require('../passport-api');
var crypto = require('crypto');
var punycode = require('punycode');
var UATRAITS_LIB = 'uatraits';
var langs = config.langs;
const isNewNode = process.versions && process.versions.modules > 48;
const sessguardDomains = config.sessguardDomains || [];

// Change path to binding for Nodejs 0.12 and higher
if (process.versions && process.versions.modules > 11) {
    UATRAITS_LIB = '/usr/lib/node_modules/uatraits';
}

PController.setUtils(putils);

class Controller extends PController {
    /**
     * Create a controller from express request and response
     *
     * @param {express.request}  req
     * @param {express.response} res
     * @param {string}           logId
     * @param {object} [tvmServiceAliases]  tvm service aliases
     * @constructs RegistrationController
     * @throws
     */
    constructor(req, res, logId, tvmServiceAliases) {
        /* jshint unused:false */

        super(req, res, logId, tvmServiceAliases, 'passport');
        this._apiPromise = null;

        this._logger = new PLog(logId, 'passport', 'controller');
    }

    /**
     * Returns a promise that resolves with api
     *
     * @returns {Promise}
     * @private
     */
    _getApiPromise() {
        if (this._apiPromise) {
            return this._apiPromise;
        }

        this._apiPromise = PassportApi.client({
            ...this._request,
            track_id: this.getRequestParam('track_id'),
            language: this.getRequestParam('language'),
            country: this.getRequestParam('country'),
            logID: this.getLogId()
        });

        return this._getApiPromise();
    }

    getHeaders({withPassportServiceTicket = false} = {}) {
        if (!withPassportServiceTicket) {
            return Object.assign({}, this._request.headers);
        }

        return Object.assign(
            {
                'X-Ya-Service-Ticket': (this._request.serviceTickets || {})['passport-api']
            },
            this._request.headers
        );
    }

    /**
     * Get a current config object
     *
     * @returns {object}
     */
    getConfig() {
        return config;
    }

    getRequest() {
        return this._request;
    }

    getResponse() {
        return this._response;
    }

    getEnv() {
        return Controller.getEnv();
    }

    isIntranet() {
        return Controller.isIntranet();
    }

    dateToChunks(date, pad) {
        var addPadding = pad
            ? function(val) {
                  val = Number(val);
                  if (val < 10) {
                      return `0${val}`;
                  }

                  return val;
              }
            : function(val) {
                  return val;
              };

        return {
            year: date.getFullYear(),
            month: addPadding(date.getMonth() + 1),
            day: addPadding(date.getDate())
        };
    }

    setIsMimino(isMimino) {
        this._isMimino = isMimino;
    }

    getAuth() {
        const auth = super
            .getAuth()
            .setBlackbox(this._isMimino ? 'blackbox-mimino.yandex.net' : this.getConfig().paths.blackbox);

        if (typeof auth.togglePinning === 'function') {
            auth.togglePinning(this.getConfig().multiauthPinning);
        }
        return auth;
    }

    /**
     * Validates and follows the retpath from the request params
     *
     * Does not handles the retpath stored in a track
     *
     * @returns {Promise}
     */
    followRetpathOrYandexTld() {
        // TODO: move to ExpressController after passport-api is in a separate package
        const self = this;
        const retpath = this.getRequestParam('retpath');

        if (!retpath) {
            self._logger.debug('Following retpath. Missing, redirecting to Yandex frontpage');
            self.redirectToYandexTld();
        }

        return this.validateRetpath(retpath).then(function(validRetpath) {
            if (validRetpath) {
                self._logger.debug('Following retpath. Valid:', validRetpath);
                self.redirect(validRetpath);
            } else {
                self._logger.debug('Following retpath. Invalid, redirecting to Yandex frontpage');
                self.redirectToYandexTld();
            }
        });
    }

    /**
     * Validates and follows the retpath from the request params
     *
     * Does not handles the retpath stored in a track
     *
     * @returns {Promise}
     */
    followRetpath() {
        // TODO: move to ExpressController after passport-api is in a separate package
        const self = this;
        const retpath = this.getRequestParam('retpath');

        if (!retpath) {
            self._logger.debug('Following retpath. Missing, redirecting to frontpage');
            return self.redirectToFrontpage();
        }

        return this.validateRetpath(retpath).then(function(validRetpath) {
            if (validRetpath) {
                self._logger.debug('Following retpath. Valid:', validRetpath);
                self.redirect(validRetpath);
            } else {
                self._logger.debug('Following retpath. Invalid, redirecting to frontpage');
                self.redirectToFrontpage();
            }
        });
    }

    /**
     * Validates the retpath.
     *
     * Resolves with validated and decoded retpath if valid,
     * or resolves with null, if retpath was invalid
     *
     * @param {string} retpath
     * @returns {Promise}
     */
    validateRetpath(retpath) {
        var logger = this._logger;

        return this._getApiPromise()
            .then(function(api) {
                return api.validateRetpath({retpath});
            })
            .then(function(response) {
                if (response.body.validation_errors) {
                    logger.debug('Validate retpath: invalid');
                    logger.verbose('Validate retpath. Api response:', response.body);
                    return null;
                }

                if (response.body.retpath) {
                    logger.debug('Validate retpath. Valid:', response.body.retpath);
                    return response.body.retpath;
                }

                logger.warn('Validate retpath. Unexpected api response format:', response);
                return null;
            });
    }

    /**
     * Redirect to the passport frontpage
     */
    redirectToFrontpage() {
        this.redirectToLocalUrl({
            pathname: 'profile'
        });
    }

    redirectToAuthVerify() {
        var currentUrl = this.getUrl();
        var authVerifyUrl = _.extend({}, currentUrl, {
            pathname: 'auth/verify',
            search: null,
            query: {
                retpath: url.format(currentUrl)
            }
        });

        this.redirect(url.format(authVerifyUrl));
    }

    redirectToYandexTld() {
        const req = this._request;
        const resultUrl = {
            protocol: 'https',
            hostname: req.hostname
                .split('.')
                .slice(1)
                .join('.'),
            search: null,
            query: null
        };

        this.redirect(url.format(resultUrl));
    }

    redirectToLocalUrl(localUrl) {
        var req = this._request;
        var resultUrl = {
            protocol: 'https',
            hostname: req.hostname,
            search: null,
            query: null
        };

        this.redirect(url.format(Object.assign({}, resultUrl, localUrl)));
    }

    replaceToAuth() {
        const currentUrl = this.getUrl();
        const authUrl = Object.assign({}, currentUrl, {
            pathname: 'auth'
        });

        this.redirect(url.format(authUrl));
    }

    redirectToAuth() {
        var currentUrl = this.getUrl();
        var authUrl = _.extend({}, currentUrl, {
            pathname: 'auth',
            search: null,
            query: {
                retpath: url.format(currentUrl),
                noreturn: '1'
            }
        });

        this.redirect(url.format(authUrl));
    }

    redirectToAuthWithOrigin(origin) {
        const currentUrl = this.getUrl();
        const authUrl = _.extend({}, currentUrl, {
            pathname: 'auth',
            search: null,
            query: {
                retpath: url.format(currentUrl),
                origin,
                noreturn: '1'
            }
        });

        this.redirect(url.format(authUrl));
    }

    redirectToAuthWithUid(uid, retpath = '/profile') {
        this.redirect(
            url.format(
                Object.assign({}, this.getAuthUrl(), {
                    query: {
                        uid,
                        retpath
                    }
                })
            )
        );
    }

    redirectToFinish(trackId) {
        return this.redirectToLocalUrl({
            pathname: 'auth/finish',
            query: {
                track_id: trackId
            }
        });
    }

    redirectToRegistration() {
        const currentUrl = this.getUrl();
        const registrationUrl = Object.assign({}, currentUrl, {
            pathname: 'registration'
        });

        this.redirect(url.format(registrationUrl));
    }

    decodePunycodeEmail(email) {
        var emailParts = email.split('@');
        var decodedDomain;
        var decodedEmail;

        if (emailParts[0] && emailParts[1]) {
            try {
                decodedDomain = punycode.toUnicode(emailParts[1]);
            } catch (e) {
                decodedDomain = emailParts[1];
            }

            decodedEmail = `${emailParts[0]}@${decodedDomain}`;
        }

        return decodedEmail;
    }

    decodePunycodeEmails(emails) {
        var punycodedEmails = {};

        Object.keys(emails).forEach(function(email) {
            var emailParts = email.split('@');
            var decodedDomain;
            var decodedEmail;

            if (emailParts[0] && emailParts[1]) {
                try {
                    decodedDomain = punycode.toUnicode(emailParts[1]);
                } catch (e) {
                    decodedDomain = emailParts[1];
                }
                decodedEmail = `${emailParts[0]}@${decodedDomain}`;
                punycodedEmails[decodedEmail] = emails[email];
            }
        });

        return punycodedEmails;
    }

    encodeEmailToPunycode(email) {
        var encodedEmail = email;
        // eslint-disable-next-line no-unused-vars
        var decodedEmail;
        var encodedDomain;
        var emailParts = email.split('@');

        if (emailParts[0] && emailParts[1]) {
            encodedDomain = punycode.toASCII(emailParts[1]);
            encodedEmail = `${emailParts[0]}@${encodedDomain}`;
            decodedEmail = punycode.toUnicode(encodedDomain);
        }

        return encodedEmail;
    }

    /**
     * Write the data into statbox
     *
     * @param {object}           data            Data to write into statbox
     * @param {boolean=false}    requireTrack    Should the track be reuqired
     * @returns {Promise}
     */
    writeStatbox(data, requireTrack) {
        data = data || {};

        data.ignoreMissedTrack = !requireTrack;

        return this._getApiPromise().then(function(api) {
            return api.statboxLogger(data);
        });
    }

    isYandexUidValid() {
        const yandexuid = this._response._yandexuid;

        return /^\d+$/.test(yandexuid);
    }

    prepareUserAgent(userAgent = '') {
        return userAgent.replace(/\n/g, '');
    }

    /**
     * Get experiment info
     *
     * @returns {Promise}
     */
    getYaExperimentInfoPromise() {
        const TIMEOUT = 160;

        const logger = this._logger;
        const headers = this.getHeaders();
        const isIntranet = this.isIntranet();
        const userAgent = this.prepareUserAgent(headers['user-agent']);
        const host = headers['host'];
        const regionId = this._response.locals && this._response.locals.regionId;
        const xForwardedFor = headers['x-real-ip'];
        const cookieString = util.format('yandexuid=%s', this._response._yandexuid);
        const experiments = {
            flags: [],
            boxes: []
        };

        return new Promise((resolve) => {
            const requestHeaders = {
                Cookie: cookieString,
                Host: host,
                'X-Region-City-Id': regionId,
                'User-Agent': userAgent,
                'X-Forwarded-For-Y': xForwardedFor
            };
            const testId = this.getCookie('test-id') || this.getRequestParam('test-id');
            const flagIds = this.getCookie('flag-ids') || this.getRequestParam('flag-ids');

            if (flagIds && (isLocalDev || this.getEnv() === 'development')) {
                experiments.flags = flagIds.split(';');
            }

            let uaasUrl = config.paths.experiments;

            if (isIntranet) {
                return resolve(experiments);
            }

            if (!this.isYandexUidValid()) {
                logger.info('Failed to request exps. Invalid yandexuid: %s', JSON.stringify(requestHeaders));
                return resolve(experiments);
            }

            if (testId) {
                const isYandexInternalNetwork = this.getLocalsField('isYandexInternalNetwork');
                const isProduction = this.getEnv() === 'production';

                if (!isProduction || isYandexInternalNetwork) {
                    requestHeaders['X-Yandex-UAAS'] = 'testing';
                    uaasUrl = url.format(
                        Object.assign({}, url.parse(uaasUrl), {
                            // pathname: null,
                            search: null,
                            query: {
                                'test-id': testId
                            }
                        })
                    );
                }
            }

            prequest({
                url: uaasUrl,
                params: {
                    timeout: TIMEOUT,
                    headers: requestHeaders,
                    responseType: 'text',
                    retry: {
                        calculateDelay: ({attemptCount, retryOptions: {limit = 2}}) =>
                            attemptCount <= limit ? TIMEOUT : 0
                    }
                },
                logID: this.getLogId()
            })
                .then((response = {}) => {
                    const {statusCode, headers = {}} = response;

                    if (statusCode === 200 && headers['x-yandex-expflags']) {
                        logger.info('Experiment status header was: ', headers['x-yandex-expflags']);

                        const experimentBoxes = headers['x-yandex-expboxes'] || '';
                        const flagsHeader = headers['x-yandex-expflags'].split(',');

                        flagsHeader.forEach((header) => {
                            try {
                                const buffer = Buffer.from(header, 'base64');
                                const parsed = JSON.parse(buffer.toString());

                                if (Array.isArray(parsed) && parsed[0] && parsed[0].HANDLER === 'PASSPORT') {
                                    const flags =
                                        (parsed[0].CONTEXT &&
                                            parsed[0].CONTEXT.PASSPORT &&
                                            parsed[0].CONTEXT.PASSPORT.flags) ||
                                        [];

                                    experiments.flags = experiments.flags.concat(flags);
                                }
                            } catch (e) {
                                logger.warn('Reading experiment status was not "ok": ', e);
                            }
                        });

                        experiments.boxes = experimentBoxes.split(';');
                    }

                    return resolve(experiments);
                })
                .catch((error) => {
                    logger.error('Failed to request exps. Headers %s, error %s', JSON.stringify(requestHeaders), error);

                    return resolve(experiments);
                });
        });
    }

    /**
     * Read track contents
     *
     * @param {string} [track]  Track to read, if none provided, would read track stored by api
     * @returns {Promise}
     * @throws
     */
    readTrack(track) {
        var logger = this._logger;

        return this._getApiPromise()
            .then(function(api) {
                if (track) {
                    api.track(track);
                }

                return api.readTrack();
            })
            .then(function(result) {
                if (result.body.status === 'ok') {
                    logger.verbose('Track contents:', result.body);
                    return when.resolve(result.body);
                }

                return when.reject(new Error('Reading track status was not "ok"'));
            });
    }

    /**
     * Delete track record
     *
     * @param {string} [track]  Track to delete, if non provided, would delete track stored by api
     * @returns {Promise}
     * @throws
     */
    deleteTrack(track) {
        return this._getApiPromise().then(function(api) {
            if (track) {
                api.track(track);
            }

            return api.delTrack();
        });
    }

    /**
     * Write the data into track
     *
     * @param {object} data     Plain object with data to write into the track
     * @returns {Promise}
     * @throws
     */
    writeTrack(data) {
        if (_.size(data) === 0 || (_.size(data) === 1 && data.track_id)) {
            return when.resolve();
        }

        var self = this;

        return this._getApiPromise()
            .then(function(api) {
                return api.writeTrack(data);
            })
            .then(null, function(error) {
                if (error[0].field === 'retpath') {
                    // If there was an error with retpath, repeat without it
                    return self.writeTrack(_.omit(data, 'retpath'));
                }

                return when.reject(error);
            });
    }

    getLangdetect() {
        if (this._request._langdetect) {
            return when.resolve(this._request._langdetect);
        }

        var self = this;
        var passLang = '';
        var sessionInfo = self._request.blackbox;
        var users = sessionInfo && sessionInfo.users;
        var defaultUid = sessionInfo && sessionInfo.default_uid;
        var defaultUser = null;
        var i = users && users.length;

        if (users) {
            while (i--) {
                if (users[i].id === defaultUid) {
                    defaultUser = users[i];
                    break;
                }
            }

            if (defaultUser) {
                passLang = defaultUser.attributes && defaultUser.attributes['34'];
            }
        }

        return Controller.getLangdetect(
            self.getHeaders(),
            self.getIp(),
            self.getCookie('my') || '',
            passLang,
            self.getTld(),
            self.getHeader('host')
        ).then(function(langdetect) {
            self._request._langdetect = langdetect;
            return langdetect;
        });
    }

    setExplicitLanguage(language) {
        if (typeof language === 'string' && language.length === 2 && langs.includes(language)) {
            this.explicitLanguage = language;
        }
    }

    /**
     * Get the suggested language
     *
     * Resolves with the suggested language
     * Resolves with 'ru' if suggestion failed
     *
     * @returns {Promise}
     */
    getLanguage() {
        const self = this;
        const isYandexMusic = Boolean(self.getHeader('x-yandex-music-client'));

        let explicitLanguage = self.explicitLanguage || self.getHeader('x-yandex-required-language');

        let detectorPromise = self.getLangdetect();

        if (isYandexMusic) {
            explicitLanguage = self.getHeader('accept-language');
        }

        if (
            typeof explicitLanguage === 'string' &&
            explicitLanguage.length === 2 &&
            langs.indexOf(explicitLanguage) !== -1
        ) {
            detectorPromise = when.resolve({
                id: explicitLanguage
            });
        }

        return detectorPromise.then(function(langdetect) {
            return self
                ._getApiPromise()
                .then(function(api) {
                    return api.language(langdetect.id || 'ru');
                })
                .catch(function() {
                    return langdetect.id || 'ru';
                });
        });
    }

    getLanglist() {
        return this.getLangdetect().then(function(langdetect) {
            return langdetect.list || [];
        });
    }

    getUatraits() {
        if (this._request.uatraits) {
            return when.resolve(this._request.uatraits);
        }

        var req = this._request;
        var res = this._response;

        return when.promise((resolve, reject) => {
            this._request.uatraitsDetector(req, res, (error) => {
                if (error) {
                    reject(error);
                } else {
                    if (req.uatraits) {
                        if (req.uatraits.BrowserName === 'MSIE' && req.uatraits.BrowserVersion <= 8) {
                            req.uatraits.isIE = true;
                        }

                        this._request.uatraits = req.uatraits;
                    }

                    resolve(req.uatraits);
                }
            });
        });
    }

    getHost() {
        return url.format({
            protocol: this._request.headers['x-real-scheme'],
            hostname: this._request.hostname
        });
    }

    setLocalsField(field, data) {
        this._response.locals[field] = data;

        return this;
    }

    getLocalsField(field) {
        return this._response.locals[field];
    }

    prepareStatData(data) {
        var statData = this.getLocalsField('statData');

        this.setLocalsField('statData', _.extend({}, statData, data));

        return this;
    }

    /**
     * Suggest gender from person's name
     *
     * Resolves with suggested gender or 'unknown'
     * Api resolves with 'unknown' if it could not determine the gender
     * And this method resolves with 'unknown' itself, if api
     *
     * @param {string} firstname    Firstname
     * @param {string} lastname     Lastname
     * @returns {Promise}
     * @throws
     */
    suggestGender(firstname, lastname) {
        var unknownGender = 'unknown';
        var logger = this._logger;

        return this._getApiPromise()
            .then(function(api) {
                return api.suggestGender({
                    firstname,
                    lastname
                });
            })
            .then(function(response) {
                return response.body.gender || unknownGender;
            })
            .then(null, function() {
                logger.verbose('Failed determining gender');
                return unknownGender;
            });
    }

    suggestLanguage() {
        var logger = this._logger;

        return this._getApiPromise()
            .then(function(api) {
                return api.suggestLanguage({});
            })
            .then(function(response) {
                return response.body.language;
            })
            .then(null, function() {
                logger.verbose('Failed determining language');
                return null;
            });
    }

    fetchNativeAmExperiments(deviceId) {
        return this._getApiPromise().then(function(api) {
            return api.fetchNativeAmExperiments(deviceId);
        });
    }

    /**
     * Get the request body
     * @returns {object}
     */
    getFormData() {
        return this._request.body || {};
    }

    /**
     * Render a page
     * Uses express.response.render
     *
     * @param {string} skinName         Template file name
     * @param {object} templateData     Plain object for templating
     * @returns undefined
     * @throws
     */
    render(skinName, templateData) {
        this._response.render(skinName, templateData);
    }

    /**
     * Redirect to the given url
     *
     * Beware of open redirects when using this method directly,
     * check the retpath is predefined or is validated by the backend
     *
     * @see followRetpath
     * @see redirectToFrontpage
     * @see redirectToFinish
     *
     * @param {string} url  An url to redirect to
     */
    redirect(url) {
        this._response.redirect(url);
    }

    /**
     * Hackety-hack to cache the parsed url between controllers for the same request
     * @returns {*}
     */
    getUrl() {
        if (this._request._controllerParsedUrl) {
            return _.clone(this._request._controllerParsedUrl);
        }

        this._request._controllerParsedUrl = super.getUrl();
        return this.getUrl();
    }

    getModePassportUrl() {
        return _.extend(this.getUrl(), {
            pathname: '/passport',
            search: null,
            query: {
                mode: 'passport'
            }
        });
    }

    getAuthUrl() {
        return _.extend(this.getUrl(), {
            pathname: '/auth',
            search: null,
            query: null
        });
    }

    getProfileUpgradeUrl() {
        return _.extend(this.getUrl(), {
            pathname: '/profile/upgrade',
            search: null,
            query: null
        });
    }

    /**
     * Returns secret key for requests to yandex.money API
     *
     * @param {number} uid      user id
     * @param {number} time     UNIX timestamp
     * @returns string
     */
    getYaMoneyRequestHash(uid, time) {
        return crypto
            .createHash('sha1')
            .update(`${uid}.${time}.${config.secrets.yamoney}`)
            .digest('hex');
    }

    appendToResponse(field, value) {
        const res = this.getResponse();

        res.append(field, value);

        return;
    }

    augmentResponse(apiResponse) {
        const {
            cookies,
            default_uid: defaultUid,
            logged_out_uids: loggedOutUids,
            service_guard_container: sessguardContainer,
            retpath
        } = apiResponse;

        if (Array.isArray(cookies)) {
            cookies.forEach((cookie) => {
                this.appendToResponse('Set-Cookie', cookie);
            });
        }

        if (defaultUid) {
            this.appendToResponse('X-Default-UID', defaultUid);
        }

        if (Array.isArray(loggedOutUids)) {
            this.appendToResponse('X-Logout-UID', loggedOutUids.join());
        }

        if (sessguardContainer) {
            this.setLocalsField('sessguardContainer', sessguardContainer);

            if (typeof retpath === 'string') {
                if (this.checkSessguard(retpath)) {
                    const parsedRetpath = url.parse(retpath, true);
                    const domain = this.getServiceDomain(retpath);

                    this.setLocalsField(
                        'guardURL',
                        url.format(
                            Object.assign(parsedRetpath, {
                                host: `guard.${domain}`,
                                pathname: 'set',
                                query: null,
                                search: null
                            })
                        )
                    );
                }
            }
        }
    }

    hasExp(expName = '') {
        const experiments = this.getLocalsField('experiments');

        if (!experiments) {
            return false;
        }

        const {flags = []} = experiments;

        return flags.includes(expName);
    }

    getServiceDomain(input = '') {
        const parsedRetpath = url.parse(input, true);

        let domain = '';

        if (parsedRetpath && parsedRetpath.hostname) {
            domain = parsedRetpath.hostname
                .split('.')
                .slice(-3)
                .join('.');
        }

        return domain;
    }

    checkSessguard(retpath = '') {
        const domain = this.getServiceDomain(retpath);

        let result = false;

        sessguardDomains.forEach((pattern) => {
            if (domain.startsWith(pattern)) {
                result = true;
            }
        });

        return result;
    }
}

Controller.getLangdetect = function(headers, ip, myCookie, passLang, tld, hostname) {
    var req = {
        ip,
        headers,
        cookies: {
            my: myCookie
        },
        tld
    };

    if (passLang) {
        req.blackbox = {
            dbfields: {
                'userinfo.lang.uid': passLang
            }
        };
    }

    if (
        [
            'passport.yango.delivery',
            'passport.toloka.ai',
            'passport-test.toloka-test.ai',
            'passport.practicum.com'
        ].includes(hostname)
    ) {
        req.hostname = 'passport.yandex.com';
        req.tld = 'com';
    }

    return when.promise((resolve, reject) => {
        this.expressLangDetector(req, {}, function(error) {
            if (error) {
                reject(error);
            } else {
                resolve(Object.assign({}, req.langdetect, {langlist: req.langdetect.list}));
            }
        });
    });
};

Controller.expressLangDetector = expressLangDetect({
    geobase: {
        version: (isNewNode && 6) || 5
    },
    defaultLanguage: 'ru',
    availableLanguages: {
        az: ['ru', 'en', 'az'],
        'com.am': ['ru', 'en'],
        'com.ge': ['ru', 'en'],
        'co.il': ['he', 'en'],
        kg: ['ru', 'en'],
        lv: ['ru', 'en'],
        lt: ['ru', 'en'],
        md: ['ru', 'en'],
        tj: ['ru', 'en'],
        tm: ['ru', 'en'],
        uz: ['ru', 'en', 'uz'],
        fr: ['en', 'ru', 'ua'],
        ee: ['ru', 'en'],
        eu: ['ru', 'en'],
        fi: ['ru', 'en'],
        pl: ['ru', 'en'],
        he: ['he', 'en'],
        ru: ['ru', 'uk', 'en'],
        ua: ['ru', 'uk', 'en'],
        by: ['ru', 'uk', 'en'],
        kz: ['ru', 'en', 'kk'],
        com: ['en', 'id', 'fr', 'fi', 'he', 'ky', 'pt'],
        'com.tr': ['tr', 'en'],
        default: ['en', 'ru', 'uk', 'tr', 'fr', 'fi', 'kk', 'uz', 'az', 'he', 'ky', 'pt']
    },
    list: true,
    geobaseVersion: 6
});

Controller.expressUatraitsDetector = isLocalDev
    ? expressUaTraits()
    : expressUaTraits({
          browser: '/usr/share/uatraits/browser.xml',
          profiles: '/usr/share/uatraits/profiles.xml',
          uatraits: UATRAITS_LIB
      });

Controller.getEnv = function() {
    if (this.isIntranet()) {
        return 'intranet';
    }

    return process.env.NODE_ENV || 'development';
};

Controller.isIntranet = function() {
    return process.env.INTRANET === 'intranet';
};

module.exports = Controller;
