'use strict';

var url = require('url');
var fs = require('fs');
var _ = require('lodash');
var util = require('util');
var when = require('when');
var inherit = require('inherit');
var config = require('../../configs/current');
var putils = require('putils');
var langs = require('../../configs/current').langs;
var debug = require('debug')('passport:fetch');
var assert = require('assert');
var NewerPassportApi = require('../api/passport');
var PLog = require('plog');
const prepareSocialRetpath = require('../../routes/common/prepareSocialRetpath');
const prequest = require('prequest');
const tools = require('prequest/tools');
// var TSKVLog = require('passport-tskv-log')(config.paths.logs.graphite);

var Controller; // Required dynamically once to avoid a circular dependency

function encodedFormat(format) {
    var params = Array.prototype.slice.call(arguments, 1).map(function(param) {
        return encodeURIComponent(param);
    });

    return util.format.apply(util, [format].concat(params));
}

function getDefaults() {
    return JSON.parse(JSON.stringify(config.defaults));
}

var filters = {
    base(handleResponse) {
        var handleBody = handleResponse.body;
        var data = {
            field: handleResponse.field,
            body: {
                status: handleBody.status
            }
        };

        if (handleBody.status === 'error') {
            data.body.errors = handleBody.errors;
        }

        return data;
    }
};

function API(opts) {
    this.logID = opts.logID;
    this._rawHeaders = {
        ...(opts.headers || {}),
        'X-Ya-Service-Ticket': opts.headers['X-Ya-Service-Ticket'] || (opts.serviceTickets || {})['passport-api']
    };
    this._headers = NewerPassportApi.transformHeaders(this._rawHeaders, this.logID);
    var _trackID = null;
    var _Language = null;

    // A check for calls to api.client, that don't pass the whole request object
    assert(this.logID && typeof this.logID === 'string', 'Log id should be defined');
    this._logger = new PLog(this.logID, 'passport', 'api');
    // this._tskvlogger = new TSKVLog(this.logID, 'passport', 'api');
    this.input = opts;

    if (opts.track_id !== undefined && /^\w{16,34}$/.test(opts.track_id)) {
        _trackID = opts.track_id;
    }

    if (typeof opts.language === 'string' && opts.language.length === 2 && langs.indexOf(opts.language) !== -1) {
        _Language = opts.language;
    }

    this._locals = {};
    this._data = {};

    this.track = function(id) {
        if (id && !_trackID) {
            _trackID = id;
        }

        return _trackID;
    };

    this.language = function(lang) {
        if (lang && !_Language) {
            _Language = lang;
        }

        return _Language;
    };
}

API.prototype.params = function(field, rejectOnEmpty) {
    var self = this;
    var data = self._data;
    var param = self.input;
    var fields = {
        language() {
            self._logger.debug('Requesting language');

            var language;
            var input = self.language();

            if (input) {
                self._logger.verbose('Resolving with cached language %s', input);
                language = when.resolve({field: 'language', body: {status: 'ok', language: input}});
            } else {
                self._logger.verbose('Using langdetect to determine the language');

                // Parse the cookie 'my'
                var myCookie = '';

                if (self._rawHeaders.cookie) {
                    // Don't want to parse all the cookies, hence regexp
                    myCookie = self._rawHeaders.cookie.match(/(^|;)\s*my=\w*/);
                    myCookie = ((myCookie && myCookie[0]) || '').split('=', 2)[1] || '';
                }

                language = Controller.getLangdetect(self._rawHeaders, self._rawHeaders['x-real-ip'] || '', myCookie)
                    .then(function(langdetect) {
                        return langdetect.id || 'ru';
                    })
                    .then(function(lang) {
                        return {
                            // Mimic api response
                            field: 'language',
                            body: {
                                status: 'ok',
                                language: lang
                            }
                        };
                    });
            }

            return language;
        },
        country() {
            self._logger.debug('Requesting country');

            var country;
            var input = param.country;

            if (input !== undefined && input.length > 0) {
                self._logger.verbose('Resolving with cached country %s', input);
                country = when.resolve({field: 'country', body: {status: 'ok', country: [input]}});
            } else {
                self._logger.verbose('Going to api to suggest country');
                country = self.suggestCountry();
            }

            return country;
        },
        track_id() {
            self._logger.debug('Requesting track');

            var track_id;
            var input = self.track();

            if (input) {
                self._logger.verbose('Resolving with cached track %s', input);
                track_id = when.resolve({field: 'track_id', body: {status: 'ok', id: input}});
            } else {
                if (!rejectOnEmpty) {
                    self._logger.verbose('Going to api to get track');
                    track_id = self.getTrack();
                } else {
                    self._logger.verbose('No cached track, reject on empty is set, rejecting');
                    track_id = when.reject();
                }
            }

            return track_id;
        }
    };

    if (typeof data[field] === 'undefined') {
        data[field] = fields[field]();
    }

    return data[field];
};

API.prototype.locals = function(data) {
    if (data) {
        this._locals = _.merge({}, this._locals, data);
    }

    return this._locals;
};

// API.prototype.get = function(field) {
// var deferred = when.defer();

// deferred.resolve(this._data[field]);

// return deferred.promise;
// };

/**
 *
 * @param {Object}  options
 * @param {Boolean} [options.passErrors] Whether API should return errors
 *                                       for the field instead of rejecting the deferred
 *
 * @param {String} fieldname
 * @returns {Function}
 */
API.prototype.fetch = function(options = {}, fieldname) {
    const {filter, passErrors = false, ignoreMissedTrack, retries, ...requestParams} = options;
    const _retries = Object.assign({}, config.retries, retries);
    const deferred = when.defer();
    const defaults = getDefaults();
    const params = _.merge(
        {},
        {
            headers: this._headers,
            retry: {limit: _retries.timeout}
        },
        defaults,
        requestParams
    );
    const {errorsToRetry = []} = _retries;

    const {url: urlObj, qs: query, ...settings} = params;
    const urlParams = Object.assign({}, urlObj, {query: _.pickBy(query, (item) => typeof item !== 'undefined')});

    const URI = url.format(urlParams);
    const logger = this._logger;

    let {error: repeatOnError} = _retries;

    let result = {
        field: fieldname
    };

    const fetch = () => {
        const logBodyNonstringified = tools.clearLog(settings.form || settings.body);

        if (settings.method === 'GET') {
            delete settings.form;
            delete settings.body;
        }

        debug(URI);

        const hooks = {
            afterResponse: [
                (response = {}) => {
                    const {body: {status, errors} = {}, statusCode} = response;
                    const backendErrorRegexp = /^backend\..*_failed$/i;

                    if (statusCode === 200 && status === 'error' && Array.isArray(errors)) {
                        const errorCode = errors[0];
                        const hasBackendError = backendErrorRegexp.test(errorCode);
                        const hasErrorFromErrorsToRetry = errorsToRetry.length > 0 && errorsToRetry.includes(errorCode);

                        if (typeof errorCode === 'string' && (hasBackendError || hasErrorFromErrorsToRetry)) {
                            response.statusCode = 504;
                            response.statusMessage = 'FORCED Gateway Timeout';
                        }
                    }

                    return response;
                }
            ]
        };

        prequest({url: URI, params: settings, logID: this.logID, hooks})
            .then((response = {}) => {
                const {body = {}, statusCode} = response;
                const {track_id, status, errors} = body;

                if (track_id) {
                    this.track(track_id);
                }

                if (!passErrors && status === 'error') {
                    if (statusCode < 500) {
                        repeatOnError = 0;
                        logger.info(
                            'API: %s %s %j => [input error] %j',
                            settings.method,
                            urlParams.pathname,
                            logBodyNonstringified,
                            errors
                        );
                    } else {
                        logger.warn(
                            'API: %s %s %j => [api error] %j',
                            settings.method,
                            urlParams.pathname,
                            logBodyNonstringified,
                            errors
                        );
                    }

                    if (repeatOnError > 0) {
                        repeatOnError--;
                        fetch();
                    } else {
                        logger.info(
                            'API: %s %s %j => [reject] %j',
                            settings.method,
                            urlParams.pathname,
                            logBodyNonstringified,
                            errors
                        );
                        deferred.reject(errors);
                    }
                }

                if (passErrors || status !== 'error') {
                    if (settings.extend) {
                        _(body).extend(settings.extend);
                    }

                    result.body = body;
                    if (settings.form && settings.form.track_id) {
                        if (typeof result.body === 'object') {
                            result.body.id = settings.form.track_id;
                        }
                    }

                    if (typeof filter === 'function') {
                        result = filter(result);
                    }

                    deferred.resolve(result);
                }
            })
            .catch(deferred.reject);
    };

    if (
        fieldname === 'track_id' ||
        (settings.form && typeof settings.form.track_id !== 'undefined') ||
        (settings.body &&
            (typeof settings.body.track_id !== 'undefined' || tools.isInstanceOfFormData(settings.body))) ||
        ignoreMissedTrack ||
        settings.method === 'GET'
    ) {
        let skipTrackReason = '';

        if (fieldname === 'track_id') {
            skipTrackReason = 'explicitly requesting a new track';
        } else if (ignoreMissedTrack) {
            skipTrackReason = 'ignoring missed track';
        } else if (settings.method === 'GET') {
            skipTrackReason = 'method GET';
        } else {
            skipTrackReason = 'track is already known';
        }

        this._logger.verbose('No track requested since %s', skipTrackReason);

        fetch();
    } else {
        when(this.params('track_id')).then(
            function(results) {
                if (!settings.form) {
                    settings.form = {};
                }

                settings.form.track_id = results.body.id;

                fetch();
            },
            function(errors) {
                deferred.reject(errors);
            }
        );
    }

    return deferred.promise;
};

/**
 * Calls fetch with the given options and unwraps the result
 * @param {object} options  Fetch options
 * @return {Promise} fetch method promise
 */
API.prototype.prettyFetch = function(options) {
    return this.fetch(options, 'prettyfetch').then(function(response) {
        return response.body || {};
    });
};

API.prototype.dispatchTrack = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/track/dispatch/'
        },
        form: {
            track_id: data.track_id
        }
    };

    return this.fetch(options, 'track_id.dispatch');
};

API.prototype.additionalDataAsk = function() {
    var options = {
        ignoreMissedTrack: true,
        method: 'GET',
        url: {
            pathname: '/1/bundle/auth/additional_data/ask/'
        },
        passErrors: true
    };

    return this.fetch(options, 'additionalData.ask');
};

API.prototype.authPasswordGetState = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/auth/password/get_state/'
            },
            form: {
                track_id: data.track_id
            },
            passErrors: true
        },
        'authPasswordGetState'
    );
};

API.prototype.passwordConfirmSubmit = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/confirm/submit/'
        },
        form: {
            track_id: data.track_id
        },
        passErrors: true
    };

    if (data && data.retpath) {
        options.form.retpath = data.retpath;
    }

    if (data && data.origin) {
        options.form.origin = data.origin;
    }

    return this.fetch(options, 'password_confirm.submit');
};

API.prototype.passwordConfirmCommit = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/confirm/commit_password/'
        },
        form: {
            track_id: data.track_id,
            password: data.password
        },
        passErrors: true // Whatever the error, pass it to the frontend
    };

    return this.fetch(options, 'password_confirm.commit');
};

API.prototype.bundleSecuritySession = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/security/session/'
        },
        form: {
            track_id: data.track_id
        }
    };

    if (data.yandexuid) {
        options.form.yandexuid = data.yandexuid;
    }

    return this.fetch(options, 'session.security.bundle');
};

API.prototype.bundleSession = function(data = {}) {
    const {track_id, retpath} = data;
    const options = {
        url: {
            pathname: '/1/bundle/session/'
        },
        form: {
            track_id
        }
    };

    return this.readTrack(track_id)
        .then((trackInfo = {}) => {
            const {body = {}} = trackInfo;
            const {social_track_id: socialTrackId, social_task_id: socialTaskId, uid} = body;

            // идем за сессией если это не процесс из социализма
            if (!socialTrackId) {
                return this.fetch(options, 'session.bundle');
            }

            // включаем вход по соц. сети
            return this.socialAllowAuth({track_id, set_auth: 1})
                .then(() => {
                    // получаем сессию и параметры для формирования ретпаса
                    return this.socialCallback({
                        track_id: socialTrackId,
                        task_id: socialTaskId,
                        status: 'ok',
                        uid
                    })
                        .then((response = {}) => {
                            const {body = {}} = response;
                            const {status, errors} = body;

                            if (status !== 'ok') {
                                throw errors;
                            }

                            try {
                                const newRetpath = prepareSocialRetpath(retpath, body);

                                if (status === 'ok') {
                                    response.body.retpath = newRetpath;
                                    response.body.track_id = socialTrackId;
                                    return when.resolve(response);
                                }
                            } catch (error) {
                                throw error;
                            }
                        })
                        .catch((errors) => {
                            throw errors;
                        });
                })
                .catch((errors) => {
                    throw errors;
                });
        })
        .catch(() => {
            // пытаемся выписать сессию если что-то пошло не так
            return this.fetch(options, 'session.bundle');
        });
};

API.prototype.bundleSessionSessGuardContainerUse = function({container}) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/session/sessguard_container/use/'
        },
        form: {
            container
        }
    };

    return this.fetch(options, 'bundle.session.sessguard_container.use');
};

API.prototype.authKeyLinkSubmit = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/auth/key_link/submit/'
        },
        form: {
            secret_key: data.secret_key
        }
    };

    return this.fetch(options, 'auth.key_link');
};

API.prototype.getTrack = function(data, isInit) {
    var options = {
        url: {
            pathname: '/1/track/'
        }
    };

    if (isInit) {
        options.url.pathname = '/1/bundle/track/init/';
    }

    if (data && _.isObjectLike(data)) {
        options.form = {};

        if (data.type) {
            options.form.track_type = data.type;
        }

        if (data.process) {
            options.form.process_name = data.process;
        }

        if (data.scenario) {
            options.form.scenario = data.scenario;
        }

        if (data.app_id) {
            options.form.app_id = data.app_id;
        }

        if (data && data.uid) {
            options.form.uid = data.uid;
        }
    }

    return this.fetch(options, 'track_id');
};

API.prototype.initTrack = function(data) {
    return this.getTrack(data, true);
};

API.prototype.getTrackWithUid = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/track/init/'
        }
    };

    if (data && data.type) {
        options.form = {
            track_type: data.type
        };
    }

    return this.fetch(options, 'track_id');
};

API.prototype.delTrack = function() {
    var self = this;

    var options = {
        method: 'DELETE',
        form: {}
    };

    return when.join(this.params('track_id', true), options).then(
        function(values) {
            var res = values[values.length - 1];

            res.url = {
                pathname: encodedFormat('/1/track/%s/', values[0].body.id)
            };
            res.form.track_id = values[0].body.id;
            return self.fetch(res, 'track');
        },
        function() {
            // Все уже хорошо, трека нет
            return when.resolve();
        }
    );
};

API.prototype.writeTrack = function(data) {
    var self = this;

    var options = {
        form: {
            retpath: data.retpath
            // service: data.service,
        }
    };

    [
        'origin',
        'app_id',
        'service',
        'key',
        'device_os_id',
        'page_loading_info',
        'check_js_load',
        'check_css_load',
        'language',
        'frontend_state',
        'js_fingerprint',
        'device_name'
    ].forEach(function(key) {
        if (data[key]) {
            options.form[key] = data[key];
        }
    });

    var trackPromise = data.track_id ? when.resolve({body: {id: data.track_id}}) : this.params('track_id');

    return when.join(trackPromise, options).then(function(values) {
        var res = values[values.length - 1];

        res.url = {
            pathname: encodedFormat('/1/track/%s/', values[0].body.id)
        };
        res.form.track_id = values[0].body.id;
        return self.fetch(res, 'track');
    });
};

API.prototype.accountSubscribe = function(data) {
    var self = this;

    var options = {
        url: {
            pathname: encodedFormat('/1/account/%s/subscription/%s/', data.uid, data.service_slug)
        },
        passErrors: true
    };

    return self.fetch(options, 'subscribe');
};

API.prototype.readTrack = function(trackId) {
    var self = this;

    var options = {
        method: 'GET',
        form: {}
    };

    var trackPromise = trackId ? when.resolve({body: {id: trackId}}) : this.params('track_id');

    return when.join(trackPromise, options).then(function(values) {
        var res = values[values.length - 1];

        res.url = {
            pathname: encodedFormat('/1/track/%s/', values[0].body.id)
        };
        res.form.track_id = values[0].body.id;
        return self.fetch(res, 'track');
    });
};

API.prototype.validateLogin = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/validate/login/'
        },
        form: {
            login: data.login,
            is_pdd: data.isPdd,
            is_lite: data.is_lite
        }
    };

    return this.fetch(options, 'login');
};

/**
 * Method to transform a err to legacy form
 * @param {string} err
 * @private
 */
var _errToLegacy = function(err) {
    assert(err && typeof err === 'string', 'Error should be a string code');

    return {
        field: 'login',
        code: err.indexOf('.') > -1 ? err.split('.')[1] : err
    };
};

/**
 * Login validator handle
 *
 * @param {object} data
 * @param {string}  data.login              Login to validate
 * @param {boolean} data.isPdd              Flag to check login as a domain login
 * @param {boolean} data.strict             Strict check flag for pdd logins
 * @param {boolean} data.ignoreStoplist     Skip checking the stoplist flag
 *
 * @returns {Function}
 */
API.prototype.validateLogin2 = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/validate/login/'
        },
        form: {
            login: data.login,
            is_pdd: data.isPdd,
            strict: data.strict,
            ignore_stoplist: data.ignoreStoplist,
            require_domain_existence: data.requireDomainExistence
        },

        passErrors: true // Whatever the error, pass it to the frontend
    };

    return this.fetch(options, 'login').then(function(response) {
        // Transforming the response to legacy form
        var res = {status: 'ok'};

        if (response.body.errors) {
            res.validation_errors = Array.isArray(response.body.errors)
                ? response.body.errors.map(_errToLegacy)
                : _errToLegacy(response.body.errors);
        }

        return {
            body: res
        };
    });
};

API.prototype.suggestLogin = function(data) {
    var options = {
        url: {
            pathname: '/1/suggest/login/'
        },
        form: {}
    };

    if (data.login) {
        options.form.login = data.login;
    }

    if (data.firstname) {
        options.form.firstname = data.firstname;
    }

    if (data.lastname) {
        options.form.lastname = data.lastname;
    }

    return this.fetch(options, 'login');
};

API.prototype.suggestLoginV2 = function(data = {}) {
    const options = {
        url: {
            pathname: '/1/bundle/suggest/login/'
        },
        method: 'GET',
        qs: {}
    };

    if (data.timeout) {
        options.timeout = Number(data.timeout);
    }

    delete data.timeout;

    if (data.login && data.login.trim()) {
        options.qs.login = data.login;
    }

    if (data.firstname) {
        options.qs.first_name = data.firstname;
    }

    if (data.lastname) {
        options.qs.last_name = data.lastname;
    }

    return this.fetch(options, 'loginV2');
};

API.prototype.suggestGender = function(data) {
    var options = {
        url: {
            pathname: '/1/suggest/gender/'
        },
        form: {
            lastname: data.lastname,
            firstname: data.firstname
        }
    };

    return this.fetch(options, 'gender');
};

API.prototype.validatePassword = function(data) {
    var self = this;
    var options = {
        url: {
            pathname: '/1/validation/password/'
        },
        form: {
            login: data.login,
            password: data.password
        }
    };

    if (data.passErrors) {
        options.passErrors = true;
    }

    if (data.phone_number) {
        options.form.phone_number = data.phone_number;
    }

    if (data.policy) {
        options.form.policy = data.policy;
    }

    return this.params('country').then(function(result) {
        options.form.country = (result.body && result.body.country[0]) || '';
        return self.fetch(options, 'password');
    });
};

API.prototype.validatePhone = function(data) {
    var self = this;

    if (!data.phone_number) {
        return when.resolve({field: 'phone_number', body: {status: 'ok'}});
    }

    var options = {
        url: {
            pathname: '/1/validation/phone_number/'
        },
        form: {
            phone_number: data.phone_number,
            ignore_phone_compare: 'yes'
        }
    };

    if (data.validate_for_call) {
        options.form.validate_for_call = 'yes';
    }

    return when.join(self.params('country'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.country = values[0].body.country[0];

        return self.fetch(res, 'phone_number');
    });
};

API.prototype.validatePhoneV2 = function(data) {
    var self = this;

    if (!data.phone_number) {
        return when.resolve({field: 'phone_number', body: {status: 'ok'}});
    }

    var options = {
        url: {
            pathname: '/1/bundle/validate/phone_number/'
        },
        form: {
            phone_number: data.phone_number,
            track_id: data.track_id
        }
    };

    if (data.validate_for_call) {
        options.form.validate_for_call = 'yes';
    }

    if (data.check_speech) {
        options.form.check_speech = 'yes';
    }

    return when.join(self.params('country'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.country = values[0].body.country.pop();

        return self.fetch(res, 'phone_number');
    });
};

API.prototype.validatePhoneById = function(data) {
    if (!data.phoneId) {
        return when.reject(['phone_id.missing']);
    }

    const options = {
        url: {
            pathname: '/1/bundle/validate/phone_id/'
        },
        form: {
            phone_id: data.phoneId,
            track_id: data.trackId
        }
    };

    return this.fetch(options, 'phone_id');
};

API.prototype.validateDisplayName = function(data) {
    var self = this;

    if (!data.display_name) {
        return when.resolve({field: 'display_name', body: {status: 'ok'}});
    }

    var options = {
        url: {
            pathname: '/1/bundle/validate/display_name/'
        },
        form: {
            display_name: data.display_name,
            track_id: data.track_id
        }
    };

    return self.fetch(options, 'display_name');
};

API.prototype.validateFirstName = function(data) {
    var self = this;

    if (!data.firstname) {
        return when.resolve({field: 'firstname', body: {status: 'ok'}});
    }

    var options = {
        url: {
            pathname: '/1/bundle/validate/firstname/'
        },
        form: {
            firstname: data.firstname,
            track_id: data.track_id
        }
    };

    return self.fetch(options, 'firstname');
};

API.prototype.validateLastName = function(data) {
    var self = this;

    if (!data.lastname) {
        return when.resolve({field: 'lastname', body: {status: 'ok'}});
    }

    var options = {
        url: {
            pathname: '/1/bundle/validate/lastname/'
        },
        form: {
            lastname: data.lastname,
            track_id: data.track_id
        }
    };

    return self.fetch(options, 'lastname');
};

API.prototype._fetchForm = function(pathname, form, options) {
    return this.prettyFetch(
        _.merge(
            {},
            {
                url: {
                    pathname
                },
                form,

                passErrors: true // Whatever the error, pass it to the frontend
            },
            options
        )
    );
};

API.prototype._genericPhoneConfirmSubmit = function(pathname, form) {
    assert(pathname && typeof pathname === 'string', 'Pathname should be a string');
    form = form || {};

    var self = this;

    return when
        .join(self.params('country'), self.params('language'))
        .then(function(values) {
            form.country = values[0].body.country.pop();
            form.display_language = form.display_language || values[1].body.language;

            return self._fetchForm(pathname, form);
        })
        .then(function(result) {
            // Can't trust client time
            result.now = String(new Date().getTime());
            return result;
        });
};

API.prototype.confirmPhoneSubmit = function(data) {
    const {
        number,
        phone_id,
        confirm_method: confirmMethod, // by_flash_call | by_call | by_sms
        gps_package_name: gpsPackageName,
        display_language
    } = data;
    const params = {
        number,
        phone_id,
        confirm_method: confirmMethod || 'by_sms',
        gps_package_name: gpsPackageName,
        display_language
    };

    if (data.code_format) {
        params.code_format = data.code_format;
    }

    return this._genericPhoneConfirmSubmit('/1/bundle/phone/confirm/submit/', params);
};

API.prototype.confirmPhoneCommit = function(code) {
    return this._fetchForm('/1/bundle/phone/confirm/commit/', {
        code
    });
};

API.prototype.confirmPhoneCheckStatus = function(data) {
    var options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/phone/confirm/check_status/'
        },
        qs: {
            track_id: data.track_id
        }
    };

    return this.fetch(options, 'confirmPhoneCheckStatus');
};

var _phoneCleanupCallback = function(result = {}) {
    if (result.number) {
        result.number.international =
            result.number.international.slice(0, 2) +
            result.number.international.slice(2, -2).replace(/\d/g, '*') +
            result.number.international.slice(-2);

        delete result.number.e164;
        delete result.number.original;
    }

    return result;
};

API.prototype.confirmTrackedPhoneSubmit = function(form) {
    const params = {
        gps_package_name: form.gps_package_name
    };

    if (form.code_format) {
        params.code_format = form.code_format;
    }
    return this._genericPhoneConfirmSubmit('/1/bundle/phone/confirm_tracked_secure/submit/', params).then(
        _phoneCleanupCallback
    );
};

API.prototype.confirmTrackedPhoneCommit = function(code) {
    return this._fetchForm('/1/bundle/phone/confirm_tracked_secure/commit/', {
        code
    }).then(_phoneCleanupCallback);
};

API.prototype.confirmTrackedPhoneWithUpdateSubmit = function() {
    return this._genericPhoneConfirmSubmit('/1/bundle/phone/confirm_and_update_tracked_secure/submit/').then(
        _phoneCleanupCallback
    );
};

API.prototype.confirmTrackedPhoneWithUpdateCommit = function(code) {
    return this._fetchForm('/1/bundle/phone/confirm_and_update_tracked_secure/commit/', {
        code
    }).then(_phoneCleanupCallback);
};

API.prototype.checkSecurePhoneRestore = function(number) {
    var self = this;

    return when.join(self.params('country')).then(function(values) {
        return self._fetchForm('/1/bundle/restore/otp/check_secure_phone/', {
            country: values[0].body.country[0],
            phone_number: number
        });
    });
};

var omitEmpty = function(obj) {
    return _.pickBy(obj, function(val) {
        return Boolean(val);
    });
};

/**
 * Submit to confirm and bind a secure phone number to the account
 * @see https://beta.wiki.yandex-team.ru/passport/python/api/bundle/phone_v2/#/submit
 *
 * @param {string}  number      Phone number to confirm and bind
 * @returns {When.Promise}
 */
API.prototype.confirmAndBindSecurePhoneSubmit = function(number) {
    return this._genericPhoneConfirmSubmit('/2/bundle/phone/confirm_and_bind_secure/submit/', {
        number
    });
};

API.prototype.confirmAndBindSecureWithAliasPhoneSubmit = function(data) {
    const {number, enableAliasAsEmail} = data;

    return this._genericPhoneConfirmSubmit('/2/bundle/phone/confirm_and_bind_secure_and_aliasify/submit/', {
        number,
        enable_alias_as_email: enableAliasAsEmail
    });
};

/**
 * Commit to confirm and bind a secure phone number to the account
 * @see https://beta.wiki.yandex-team.ru/passport/python/api/bundle/phone_v2/#/commit
 *
 * @param {string}  code        Confirmation code as entered by the user
 * @param {string} [password]   Account password, if it was required
 * @returns {When.Promise}
 */
API.prototype.confirmAndBindSecurePhoneCommit = function(code, password) {
    return this._fetchForm(
        '/2/bundle/phone/confirm_and_bind_secure/commit/',
        omitEmpty({
            code,
            password
        })
    );
};

API.prototype.confirmAndBindSecureWithAliasPhoneCommit = function(code, password) {
    return this._fetchForm(
        '/2/bundle/phone/confirm_and_bind_secure_and_aliasify/commit/',
        omitEmpty({
            code,
            password
        })
    );
};

API.prototype.validateRetpath = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/validation/retpath/'
        },
        form: {
            retpath: data.retpath
        }
    };

    return this.fetch(options, 'retpath');
};

API.prototype.validateBackpath = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/validation/retpath/'
        },
        form: {
            retpath: data.backpath
        }
    };

    return this.fetch(options, 'backpath');
};

API.prototype.statboxLogger = function(data) {
    data = data || {};
    assert(_.isObjectLike(data), 'Data to log should be a plain hash');

    var options = {
        ignoreMissedTrack: data.ignoreMissedTrack || false,
        url: {
            pathname: '/1/statbox/'
        },
        form: {}
    };

    delete data.ignoreMissedTrack;

    if (Array.isArray(data.experiment_boxes)) {
        data.experiment_boxes = data.experiment_boxes.join(';');
    }

    Object.keys(data).forEach(function(key) {
        if (!data[key]) {
            delete data[key];
        }
    });
    options.form = putils.flattenObject(data);
    return this.fetch(options, 'statbox');
};

API.prototype.getQuestions = function() {
    var self = this;

    return when.join(this.params('language'), {url: {pathname: '/1/questions/'}}).then(function(values) {
        var options = values[values.length - 1];

        options.ignoreMissedTrack = true;

        options.form = {
            display_language: values[0].body.language
        };
        return self.fetch(options, 'hint_questions');
    });
};

function contrQuestionParams(data) {
    var form = {};

    if (data.hint_question_id && data.hint_question_id !== '0') {
        if (data.hint_question && data.hint_question !== '' && data.hint_question_id === '99') {
            form.hint_question = data.hint_question;
        } else {
            form.hint_question_id = data.hint_question_id;
        }

        form.hint_answer = data.hint_answer || '';
    }

    return form;
}

API.prototype.accountRegister = function(data) {
    var self = this;
    var options = {
        url: {
            pathname: '/1/account/register/simple/'
        },
        form: {
            url: data.url,
            mode: 'simplereg',
            login: data.login,
            firstname: data.firstname,
            lastname: data.lastname,
            password: data.password
        }
    };

    _.extend(options.form, contrQuestionParams(data));

    if (data.phone_number && data.phone_number !== '') {
        options.form.phone_number = data.phone_number;
    }

    if (data.gender && data.gender !== '') {
        options.form.gender = data.gender;
    }

    if (data.eula_accepted) {
        options.form.eula_accepted = 'true';
    }

    return when.join(this.params('language'), this.params('country'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.display_language = values[0].body.language;
        res.form.language = values[0].body.language;
        res.form.country = values[1].body.country[0];
        return self.fetch(res, 'register');
    });
};

/**
 * Ручка проверки ограничений и счетчика SMS
 * @see https://wiki.yandex-team.ru/passport/python/api/bundle/phone#proveritogranichenijaischetchikisms
 *
 * @param {Object} data
 * @param {String} data.track_id
 *
 */
API.prototype.getRegistrationLimits = function(data) {
    return this.prettyFetch({
        url: {
            pathname: '/1/bundle/account/register/limits/'
        },
        form: {
            track_id: data.track_id
        }
    });
};

/**
 * Ручка регистрации пользователя при обязательном наличии валидированного телефона
 *
 * @param {Object} data
 * @param {String} data.track_id
 * @param {String} data.login
 * @param {String} data.password
 * @param {String} data.firstname
 * @param {String} data.lastname
 * @param {String} data.eula_accepted Флаг принял ли пользователь пользовательское соглашение. "on" если да.
 *
 */
// eslint-disable-next-line max-len
// http://wiki.yandex-team.ru/passport/python/api/#registracijaakkauntapriobjazatelnomnalichiiprovalidirovannogotelefonaissozdaniemaliasasjetimnomerom
API.prototype.accountRegisterWithPhoneAlias = function(data) {
    var self = this;

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        var lang = values[0].body.language;
        var country = values[1].body.country[0];

        var options = {
            url: {
                pathname: '/1/bundle/account/register/phone_and_aliasify/'
            },
            form: {
                track_id: data.track_id,
                login: data.login,
                password: data.password,
                firstname: data.firstname,
                lastname: data.lastname,
                language: data.language || lang,
                country,
                eula_accepted: data.eula_accepted === 'on' ? 'y' : 'n'
            }
        };

        return self.fetch(options, 'register_and_aliasify');
    });
};

API.prototype.accountRegisterAlternative = function(data) {
    var self = this;

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        var lang = values[0].body.language;
        var country = values[1].body.country[0];

        var options = {
            url: {
                pathname: '/1/account/register/alternative/'
            },
            form: {
                track_id: data.track_id,
                login: data.login,
                password: data.password,
                firstname: data.firstname,
                lastname: data.lastname || data.surname,
                gender: data.gender,
                language: data.language || lang,
                display_language: lang,
                country,
                eula_accepted: data.eula_accepted === 'on' ? 'y' : 'n',
                validation_method: 'phone'
            }
        };

        if (data.force_clean_web) {
            options.form.force_clean_web = data.force_clean_web;
        }

        if (data.plus_promo_code) {
            options.form.plus_promo_code = data.plus_promo_code;
        }

        if (data.keepUnsubscribedValue || data.unsubscribe_from_maillists) {
            options.form.unsubscribe_from_maillists = true;
        }

        if (data.origin) {
            options.form.origin = data.origin;
        }

        return self.fetch(options, 'register_alternative');
    });
};

// https://wiki.yandex-team.ru/passport/api/bundle/registration/#registracijaneofonisha
API.prototype.neoPhonishAccountRegister = function(data) {
    const self = this;
    const options = {
        url: {
            pathname: '/1/bundle/account/register/neophonish/'
        },
        form: {
            track_id: data.track_id,
            firstname: data.firstname,
            lastname: data.lastname,
            gender: data.gender,
            language: data.language,
            eula_accepted: data.eula_accepted
        }
    };

    if (data.unsubscribe_from_maillists) {
        options.form.unsubscribe_from_maillists = data.unsubscribe_from_maillists;
    }

    if (data.origin) {
        options.form.origin = data.origin;
    }

    return this.params('country').then(function(countryResponse = {}) {
        const country = countryResponse.body && countryResponse.body.country[0];

        return self.fetch({...options, country}, 'neoPhonishAccountRegister');
    });
};

API.prototype.accountRegisterWithEmail = function(data) {
    var self = this;

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        var lang = values[0].body.language;
        var country = values[1].body.country[0];

        var options = {
            url: {
                pathname: '/1/account/register/email/'
            },
            form: {
                track_id: data.track_id,
                login: data.login,
                password: data.password,
                firstname: data.firstname,
                lastname: data.lastname,
                surname: data.surname,
                gender: data.gender,
                language: data.language || lang,
                display_language: lang,
                country,
                eula_accepted: data.eula_accepted === 'on' ? 'y' : 'n'
            }
        };

        if (data['human-confirmation'] === 'phone') {
            options.form.validation_method = 'phone';
        } else {
            options.form.validation_method = 'captcha';
            _.extend(options.form, contrQuestionParams(data));
        }

        if (data.force_clean_web) {
            options.form.force_clean_web = data.force_clean_web;
        }

        if (data.keepUnsubscribedValue) {
            options.form.unsubscribe_from_maillists = data.keepUnsubscribedValue;
        }

        if (data.origin) {
            options.form.origin = data.origin;
        }

        return self.fetch(options, 'accountRegisterWithEmail');
    });
};

API.prototype.confirmEmailForRegistrationByCode = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/register/confirm_email/by_code/'
        },
        form: {
            track_id: data.track_id,
            key: data.key
        }
    };

    return this.fetch(options, 'confirmEmailForRegistrationByCode');
};

API.prototype.accountRegisterExperimentEmailSubmit = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/register/lite/submit/'
        },
        form: {
            track_id: data.track_id,
            language: data.language,
            login: data.login
        }
    };

    return this.fetch(options, 'sendConfirmationEmailForLiteExperimentRegistration');
};

API.prototype.accountRegisterExperimentEmailCommit = function(data) {
    var self = this;

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        var lang = values[0].body.language;
        var country = values[1].body.country[0];
        var fields = {
            language: data.language || lang,
            display_language: lang,
            country,
            require_firstname: false,
            require_lastname: false
        };
        var {lastname, firstname} = data;
        var form = Object.assign({}, data, fields);

        if (firstname) {
            form.require_firstname = true;
        }

        if (lastname) {
            form.require_lastname = true;
        }

        if (data.force_clean_web) {
            form.force_clean_web = data.force_clean_web;
        }

        if (data.keepUnsubscribedValue) {
            form.unsubscribe_from_maillists = data.keepUnsubscribedValue;
            delete data.keepUnsubscribedValue;
        }

        if (data.origin) {
            form.origin = data.origin;
        }

        var options = {
            url: {
                pathname: '/1/bundle/account/register/lite/commit/'
            },
            form
        };

        return self.fetch(options, 'accountRegisterExperimentEmailCommit');
    });
};

API.prototype.captchaGenerate = function(data) {
    var self = this;

    return when
        .join(this.params('language'), {url: {pathname: '/1/captcha/generate/'}, voice: 'yes'})
        .then(function(values) {
            var res = values[values.length - 1];

            res.ignoreMissedTrack = true;

            res.form = {
                voice: true,
                display_language: values[0].body.language
            };

            if (data && data.use_cached) {
                res.form.use_cached = data.use_cached;
            }

            if (data && data.ocr) {
                res.form.type = 'wave';
            }

            if (data && data.wave) {
                res.form.type = 'wave';
            }

            if (data && data.track_id) {
                res.form.track_id = data.track_id;
            }

            if (data && data.scale_factor) {
                res.form.scale_factor = data.scale_factor;
            }

            res.extend = {
                mode: 'text'
            };

            return self.fetch(res, 'captcha');
        });
};

API.prototype.audioCaptchaGenerate = function() {
    var self = this;

    return when
        .join(this.params('language'), {url: {pathname: '/1/captcha/generate/'}, voice: 'yes'})
        .then(function(values) {
            var res = values[values.length - 1];

            res.form = {
                voice: true,
                display_language: values[0].body.language
            };

            res.extend = {
                mode: 'audio'
            };

            return self.fetch(res, 'captcha');
        });
};

API.prototype.captchaCheck = function(data) {
    var self = this;
    var track_id = data.track_id || self.track();
    var options = {
        url: {
            pathname: '/1/captcha/check/'
        },
        form: {
            answer: data.answer
        },
        passErrors: data.passErrors
    };

    if (data.key) {
        options.form.key = data.key;
    }

    if (track_id) {
        options.form.track_id = track_id;
    } else {
        return when.reject({field: 'captcha', body: {status: 'error'}});
    }

    return when.join(this.params('language'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.display_language = values[0].body.language;
        return self.fetch(res, 'captcha');
    });
};

API.prototype.captchaCheckStatus = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/captcha/status/'
        },
        form: {},
        passErrors: true
    };

    data = data || {};

    if (data.key) {
        options.form.key = data.key;
    }

    return this.fetch(options, 'captcha');
};

API.prototype.suggestTimezone = function() {
    var options;

    options = {
        url: {
            pathname: '/1/suggest/timezone/'
        }
    };

    return this.fetch(options, 'timezone');
};

API.prototype.suggestCountry = function() {
    var options;

    options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/suggest/country/'
        }
    };

    return this.fetch(options, 'country');
};

API.prototype.sessionCreate = function() {
    var self = this;
    var options = {
        url: {
            pathname: '/1/session/'
        },
        form: {}
    };

    return when.join(this.params('track_id'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.track_id = values[0].body.id;
        return self.fetch(res, 'session');
    });
};

API.prototype.sessionCheck = function(data) {
    var options = {
        url: {
            pathname: '/1/session/check/'
        },
        form: {
            session: data.session,
            track_id: data.track_id
        }
    };

    return this.fetch(options, 'session');
};

API.prototype.validateHint = function(data) {
    var options = {
        url: {
            pathname: '/1/validation/hint/'
        },
        form: {}
    };

    if (!(data.hint_answer || data.hint_question || data.hint_question_id)) {
        return when.resolve({field: 'hint', body: {status: 'ok'}});
    }

    if (data.hint_answer) {
        options.form.hint_answer = data.hint_answer;
    }

    if (data.hint_question && data.hint_question_id === '99') {
        options.form.hint_question = data.hint_question;
    } else {
        if (data.hint_question_id) {
            options.form.hint_question_id = data.hint_question_id;
        }
    }

    return this.fetch(options, 'hint');
};

API.prototype.validatePublicId = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/validate/public_id/'
        },
        form: {
            public_id: data.id,
            force_clean_web: data.force_clean_web
        }
    };

    return this.fetch(options, 'validatePublicId');
};

API.prototype.popUserMessages = function() {
    var self = this;

    var options = {
        url: {
            pathname: '/1/account/usermessages/pop/'
        },
        form: {}
    };

    return when.join(this.params('track_id'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.track_id = values[0].body.id;
        return self.fetch(res, 'usermessages');
    });
};

API.prototype.authSubmit = function(handle, data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: handle
        },
        form: data
    };

    if (data.asString) {
        options.encoding = 'ascii';
        delete options.form.asString;
    }

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.form.passErrors;
    }

    return this.fetch(options, 'submit');
};

API.prototype.getHistoryQuestions = function(data) {
    var self = this;
    var options = {
        method: 'GET',
        url: {
            pathname: '/1/account/questions/history/get/'
        },
        qs: data
    };

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.form.passErrors;
    }

    // return self.fetch(options, 'historyQuestions');
    return when.join(this.params('language'), options).then(function(values) {
        var res = values[values.length - 1];

        res.qs.display_language = values[0].body.language;
        res.qs.language = values[0].body.language;
        return self.fetch(options, 'historyQuestions');
    });
};

API.prototype.checkHistoryQuestions = function(data) {
    var self = this;
    var options = {
        ignoreMissedTrack: true,
        passErrors: true, // Whatever the error, pass it to the frontend
        url: {
            pathname: '/1/account/questions/history/check/'
        },
        form: data
    };

    return when.join(this.params('language'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.display_language = values[0].body.language;
        return self.fetch(options, 'historyQuestions');
    });
};

API.prototype.getCurrentQuestion = function() {
    var options = {
        method: 'GET',
        url: {
            pathname: '/1/account/questions/secure/'
        }
    };

    return this.fetch(options, 'getCurrentQuestion');
};

API.prototype.saveQuestion = function(data) {
    var options = {
        passErrors: true,
        url: {
            pathname: '/1/account/questions/change/'
        },
        form: data
    };

    return this.fetch(options, 'saveQuestion');
};

API.prototype.checkCurrentQuestion = function(data) {
    var options = {
        ignoreMissedTrack: true,
        passErrors: true,
        url: {
            pathname: '/1/bundle/account/questions/secure/check/'
        },
        form: data
    };

    return this.fetch(options, 'checkCurrentQuestion');
};

API.prototype.changepassSubmit = function(data) {
    const options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/change_password/submit/'
        },
        form: {
            is_pdd: data.is_pdd,
            retpath: data.retpath,
            origin: data.origin,
            track_id: data.track_id
        }
    };

    return this.fetch(options, 'changepass');
};

API.prototype.completeSubmit = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/complete/submit/'
        },
        form: data
    };

    return this.fetch(options, 'complete');
};

API.prototype.completeCommit = function(handle, data) {
    var handles = {
        complete_social: '/1/bundle/complete/commit_social/',
        complete_social_with_login: '/1/bundle/complete/commit_social_with_login/',
        complete_lite: '/1/bundle/complete/commit_lite/',
        force_complete_lite: '/1/bundle/complete/force_commit_lite/',
        complete_autoregistered: '/2/bundle/auth/password/complete_autoregistered/',
        complete_neophonish: '/1/bundle/complete/commit_neophonish/'
    };

    var pathname = handles[handle];
    var deferred;

    if (!pathname) {
        deferred = when.defer();
        deferred.reject(new Error('Unknown pathname for api.completeCommit'));
        return deferred.promise;
    }

    var options = {
        url: {
            pathname
        },
        form: {}
    };

    var self = this;

    [
        'login',
        'password',
        'firstname',
        'lastname',
        'surname',
        'gender',
        'birthday',
        'force_clean_web',
        'eula_accepted'
    ].forEach(function(field) {
        if (data[field]) {
            options.form[field] = data[field];
        }
    });

    options.form.validation_method = data['human-confirmation'];

    _.extend(options.form, contrQuestionParams(data));

    ['hint_question', 'hint_question_id', 'hint_answer'].forEach(function(field) {
        var new_name = field.replace('hint_', '');

        if (options.form[field]) {
            options.form[new_name] = options.form[field];
            delete options.form[field];
        }
    });

    if (data.current_password) {
        options.form.password = data.current_password;
        delete options.form.current_password;
    }

    if (data.origin) {
        options.form.origin = data.origin;
    }

    return when.join(this.params('language'), options).then(function(values) {
        var res = values[values.length - 1];

        res.form.display_language = values[0].body.language;
        return self.fetch(res, 'complete');
    });
};

API.prototype.changepassCommit = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/change_password/commit/'
        },
        form: {
            current_password: data.current_password,
            password: data.password,
            track_id: data.track_id,
            is_pdd: data.is_pdd,
            revoke_web_sessions: data.revoke_web_sessions,
            revoke_tokens: data.revoke_tokens,
            revoke_app_passwords: data.revoke_app_passwords
        }
    };

    if (data.check_revokers) {
        if (!data.logout) {
            options.form.revoke_web_sessions = false;
        }

        if (!data.revoke_accesses) {
            options.form.revoke_tokens = false;
            options.form.revoke_app_passwords = false;
        }
        delete data.check_revokers;
    }

    return this.fetch(options, 'changepass');
};

API.prototype.logout = function(data) {
    const optional = ['is_global', 'target', 'retpath', 'origin', 'yu', 'ci'];
    const options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/account/logout/'
        },
        form: {}
    };

    optional.forEach(function(opt) {
        if (data[opt]) {
            options.form[opt] = data[opt];
        }
    });

    return this.fetch(options, 'logout');
};

API.prototype.revokeTokensAndSessions = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/change_password/optional_logout/'
        },
        form: {
            track_id: data.track_id,
            revoke_web_sessions: data.revoke_web_sessions,
            revoke_tokens: data.revoke_tokens,
            revoke_app_passwords: data.revoke_app_passwords
        }
    };

    return this.fetch(options, 'optional_logout');
};

API.prototype.restoreSemiAutoEmailValidate = function(data) {
    var options = {
        passErrors: true,
        url: {
            pathname: '/1/bundle/restore/semi_auto/validate/'
        },
        form: {
            contact_email: data['feedback-email']
        }
    };

    return this.fetch(options, 'restore-fio-email-valiadte');
};

API.prototype.postChangeAvatar = function(data) {
    var options = {
        url: {
            pathname: '/1/change_avatar/'
        },
        form: {
            default: 'true',
            track_id: data.track_id,
            url: data.url,
            is_temp: data.is_temp
        }
    };

    if (data.file && data.file.size > 0) {
        delete data.url;
        delete options.form;

        options.body = tools.getFormData({...data, default: 'true'});
        options.headers = tools.getFormDataHeaders(options.body);
    }

    return this.fetch(options, 'loadAvatar');
};

API.prototype.genericAvatar = function(handle, data = {}) {
    var options = {
        url: {
            pathname: handle
        },
        method: 'GET',
        qs: data
    };

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.qs.passErrors;
    }

    return this.fetch(options, 'changeAvatar');
};

API.prototype.setAvatarByUid = function(data) {
    var options = {
        url: {
            pathname: '/2/change_avatar/'
        },
        form: {
            default: 'true',
            uid: data.uid,
            url: data.url
        }
    };

    if (data.file && data.file.size > 0) {
        delete data.url;
        delete options.form;

        options.body = tools.getFormData({...data, default: 'true'});
        options.headers = tools.getFormDataHeaders(options.body);
    }

    return this.fetch(options, 'setAvatarByUid');
};

API.prototype.setDefaultAvatarById = function(data) {
    var options = {
        url: {
            pathname: '/1/change_avatar/default/'
        },
        form: data
    };

    return this.fetch(options, 'setDefaultAvatarById');
};

API.prototype.deleteAvatar = function(data) {
    var options = {
        url: {
            pathname: '/1/change_avatar/default/'
        },
        method: 'DELETE',
        qs: data
    };

    return this.fetch(options, 'deleteAvatar');
};

API.prototype.restoreSemiAutoFromData = function() {
    var options = {
        url: {
            pathname: '/1/bundle/restore/semi_auto/form_data/'
        },
        form: {}
    };

    return this.fetch(options, 'restore-fio-form-data');
};

API.prototype.restoreSemiAutoCommit = function(data = {}) {
    const {version, body = {}} = data;
    const {language, ...params} = body;

    let options = {};

    if (version && (version === 3 || version === 4)) {
        options = {
            timeout: 12000,
            url: {
                pathname: `/${version}/bundle/restore/semi_auto/commit/`
            },
            form: {}
        };

        options.form.track_id = this.track();
        options.form.language = language;
        options.form.json_data = JSON.stringify(params);

        if (params.attach && params.attach.size > 0) {
            options.body = tools.getFormData(options.form);
            options.body.append('photo_file', fs.createReadStream(params.attach.path));
            options.headers = tools.getFormDataHeaders(options.body);

            delete options.form;
        }

        return this.fetch(options, 'restore-semiauto-v3');
    }

    options = {
        url: {
            pathname: '/2/bundle/restore/semi_auto/commit/'
        },
        form: {
            // required restore fileds
            language: data.language,
            eula_accepted: data.eula_accepted_confidential,
            firstname: data.firstname,
            lastname: data.lastname,
            birthday: data.birthday,
            registration_date: data.registration_date,
            registration_country: data['registration-country'],
            registration_city: data['registration-city'],
            contact_email: data['feedback-email'],
            track_id: data.track_id
        }
    };

    // опциональные поля, для которых актуальна проверка явного знания "не указано"
    // если "не указано", то передаем пустую строку
    var optionalFieldsMapping = {
        address: 'delivery_addresses',
        phones: 'phone_numbers',
        social_profiles: 'social_accounts',
        'additional-email': 'emails',
        'folders-mail': 'email_folders',
        'blacklist-mail': 'email_blacklist',
        'whitelist-mail': 'email_whitelist',
        'collector-mail': 'email_collectors',
        'sender-mail': 'outbound_emails'
    };

    Object.keys(optionalFieldsMapping).forEach(function(key) {
        if (data[`remember_${key}`] && data[`remember_${key}`] === 'empty') {
            options.form[optionalFieldsMapping[key]] = '';
        }
    });

    // опциональные поля, котрые передаем только если есть значение
    var optionalFieldsMappingNeedValue = {
        contact_reason: 'contact_reason',
        user_enabled: 'user_enabled',
        last_success_auth_date: 'password_auth_date',
        questions: 'question_id',
        questions_text: 'question',
        questions_answer: 'answer',
        'registration-country_id': 'registration_country_id',
        'last-passwords': 'password',
        'registration-city_id': 'registration_city_id'
    };

    [].concat(Object.keys(optionalFieldsMappingNeedValue), Object.keys(optionalFieldsMapping)).forEach(function(key) {
        if (data[key]) {
            options.form[optionalFieldsMappingNeedValue[key] || optionalFieldsMapping[key]] = data[key];
        }
    });

    if (data.attach && data.attach.size > 0) {
        options.body = tools.getFormData(options.form);
        options.body.append('photo_file', fs.createReadStream(data.attach.path));
        options.headers = tools.getFormDataHeaders(options.body);

        delete options.form;
    }

    return this.fetch(options, 'restore-fio');
};

API.prototype.restoreSemiAutoShortCommit = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/restore/semi_auto/short_form/'
        },
        form: {
            // required restore fileds
            eula_accepted: data.eula_accepted_confidential,
            firstname: data.firstname,
            lastname: data.lastname,
            birthday: data.birthday,
            registration_date: data.registration_date,
            registration_country: data['registration-country'],
            registration_country_id: data['registration-country_id'],
            registration_city: data['registration-city'],
            registration_city_id: data['registration-city_id']
        }
    };

    return this.fetch(options, 'restore-fio');
};

API.prototype.restoreGetTrackWithCaptcha = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/restore/semi_auto/submit_with_captcha/'
        },
        form: {
            login: data.body['login-simple'],
            request_source: 'directurl' // directurl передаем в бэк это значение по дефолту
        }
    };

    if (data.unconditional) {
        options.form.is_unconditional_pass = true;
        options.form.request_source = 'unconditional';
    }

    if (data.requestSource) {
        options.form.request_source = data.requestSource;
    }

    return this.fetch(options, 'restore-get-track');
};

API.prototype.restore2FAGetTrackWithCaptcha = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/restore/otp/submit_with_captcha/'
        },
        form: {
            login: data.body['login-simple']
        }
    };

    return this.fetch(options, 'restore-2fa-get-track');
};

API.prototype.restoreGetTrack = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/semi_auto/submit/'
            },
            form: {
                login: data.login,
                request_source: data.request_source || ''
            }
        },
        'restore-get-track'
    );
};

API.prototype.restoreSemiAutoGetState = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/3/bundle/restore/semi_auto/get_state/'
            },
            form: {
                track_id: data.track_id
            }
        },
        'restore-semiauto-get-state'
    );
};

API.prototype.authChangePassword = function(data) {
    var options = {
        url: {
            pathname: '/2/bundle/auth/password/change_password/'
        },
        form: {
            password: data.password
        }
    };

    return this.fetch(options, 'auth-changepass');
};

API.prototype.toggleEnterWithoutPassword = function(data) {
    var options = {
        method: 'POST',
        url: {
            pathname: '2/account/options/'
        },
        form: {
            qr_code_login_forbidden: data.qrCodeLoginForbidden,
            sms_code_login_forbidden: data.smsCodeLoginForbidden
        }
    };

    return this.fetch(options, 'enter-widthoutpassword');
};

API.prototype.toggleSms2fa = function(data) {
    const options = {
        method: 'POST',
        url: {
            pathname: '2/account/options/'
        },
        form: {
            sms_2fa_on: data.isEnabled
        }
    };

    return this.fetch(options, 'toggleSms2fa');
};

API.prototype.togglePersonalDataPublicAccess = function(data = {}) {
    const options = {
        method: 'POST',
        url: {
            pathname: '2/account/options/'
        },
        form: data
    };

    return this.fetch(options, 'togglePersonalDataPublicAccess');
};

API.prototype.feedbackForBigBro = function(data) {
    var broPaths = config.paths.bigbro;

    return this.fetch(
        {
            ignoreMissedTrack: true,
            url: {
                hostname: broPaths.hostname,
                port: broPaths.port,
                pathname: '/api/v1.0/feedback'
            },
            body: data,
            form: null
        },
        'feedback-for-bigbro'
    );
};

API.prototype.setKarma = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/session/karma/'
            },
            form: data,
            passErrors: true
        },
        'set-karma'
    );
};

API.prototype.OTPDisableSubmit = function() {
    return this.prettyFetch({
        url: {
            pathname: '/1/bundle/otp/disable/submit/'
        },
        ignoreMissedTrack: true // Specific track for this process is created by the handle itself
    });
};

API.prototype.OTPDisableCheckOtp = function(data) {
    // require('assert')(otp && typeof otp === 'string', 'OTP should be a string');
    return this.prettyFetch({
        form: {
            otp: data.otp
        },
        url: {
            pathname: '/1/bundle/otp/disable/check_otp/'
        }
    });
};

API.prototype.OTPDisableCommit = function(data) {
    assert(!data.password || typeof data.password === 'string', 'Password should be a string if defined');
    return this.prettyFetch({
        form: data,
        url: {
            pathname: '/1/bundle/otp/disable/commit/'
        }
    });
};

API.prototype.restore2faGetState = function() {
    var options = {
        url: {
            pathname: '/1/bundle/restore/otp/get_state/'
        },
        form: {}
    };

    return this.fetch(options, 'restore2fa-get-state');
};

// eslint-disable-next-line max-len
// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#proverkapin-kodatolkodlja2fa-vosstanovlenija
API.prototype.restore2faCheckPin = function(data) {
    // old path - '/1/bundle/restore/otp/check_pin/'

    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/restore/pin/check/'
            },
            form: {
                pin: data.pin
            }
        },
        'restore2fa-check-pin'
    );
};

API.prototype.restore2faCommit = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/otp/commit/'
            },
            form: {
                password: data.password
            }
        },
        'restore2fa-commit'
    );
};

API.prototype.OTPEnableSubmit = function(track, session_reissue_interval) {
    var options = {
        url: {
            pathname: '/2/bundle/otp/enable/submit/'
        },
        ignoreMissedTrack: true // Specific track for this process is created by the handle itself
    };

    if (track) {
        options.form = {
            track_id: track
        };
    }

    if (typeof session_reissue_interval === 'number') {
        if (!options.form) {
            options.form = {};
        }
        options.form.session_reissue_interval = session_reissue_interval;
    }

    return this.prettyFetch(options);
};

API.prototype.OTPEnableGetSecret = function() {
    return this.prettyFetch({
        url: {
            pathname: '/1/bundle/otp/enable/get_secret/'
        }
    });
};

API.prototype.OTPEnableGetState = function() {
    return this.prettyFetch({
        url: {
            pathname: '/1/bundle/otp/enable/get_state/'
        }
    });
};

API.prototype.OTPEnableSetPin = function(pin) {
    assert(pin && typeof pin === 'string', 'Pin should be a string');

    return this.prettyFetch({
        form: {
            pin
        },

        url: {
            pathname: '/1/bundle/otp/enable/set_pin/'
        }
    });
};
API.prototype.OTPEnableCheckOtp = function(otp) {
    assert(otp && typeof otp === 'string', 'OTP should be a string');

    return this.prettyFetch({
        form: {
            otp
        },

        url: {
            pathname: '/1/bundle/otp/enable/check_otp/'
        }
    });
};
API.prototype.OTPEnableCommit = function(password) {
    assert(!password || typeof password === 'string', 'Password should be a string if defined');

    return this.prettyFetch({
        form: omitEmpty({
            current_password: password
        }),

        url: {
            pathname: '/1/bundle/otp/enable/commit/'
        }
    });
};

API.prototype.OTPValidatePin = function(data) {
    assert(typeof data.pin === 'string', 'Pin should be a string');

    return this.prettyFetch({
        form: {
            track_id: data.track_id,
            pin: data.pin
        },

        url: {
            pathname: '/2/bundle/validate/otp_pin/'
        }
    });
};

API.prototype.OTPMigrateSubmit = function(data) {
    return this.prettyFetch({
        form: {
            track_id: data.track_id,
            retpath: data.retpath
        },

        url: {
            pathname: '/1/bundle/otp/migrate/submit/'
        }
    });
};

API.prototype.OTPMigrateGetSecret = function(track) {
    return this.prettyFetch({
        form: {
            track_id: track
        },

        url: {
            pathname: '/1/bundle/otp/migrate/get_secret/'
        }
    });
};

API.prototype.OTPMigrateCheckOtp = function(data) {
    assert(data.otp && typeof data.otp === 'string', 'OTP should be a string');

    return this.prettyFetch({
        form: {
            track_id: data.track_id,
            otp: data.otp
        },

        url: {
            pathname: '/1/bundle/otp/migrate/check_otp/'
        }
    });
};

API.prototype.OTPMigrateCommit = function(track) {
    assert(track && typeof track === 'string', 'track should be a string');

    return this.prettyFetch({
        form: {
            track_id: track
        },

        url: {
            pathname: '/1/bundle/otp/migrate/commit/'
        }
    });
};

API.prototype.socialProfileInit = function(handle) {
    var options = {
        url: {
            pathname: handle
        },
        method: 'GET'
    };

    return this.fetch(options, 'socialProfileInit');
};

API.prototype.socialSubscribeService = function(data) {
    var options = {
        url: {
            pathname: '/1/change_social/subscription/'
        },
        form: {
            track_id: data.track_id,
            profile_id: data.profile_id,
            sid: data.sid
        }
    };

    return this.fetch(options, 'socialProfileSubscribe');
};

API.prototype.socialUnsubscribeService = function(data) {
    var options = {
        url: {
            pathname: '/1/change_social/subscription/'
        },
        method: 'DELETE',
        qs: {
            track_id: data.track_id,
            profile_id: data.profile_id,
            sid: data.sid
        }
    };

    return this.fetch(options, 'socialProfileUnsubscribe');
};

API.prototype.socialAllowAuth = function(data) {
    var options = {
        url: {
            pathname: '/1/change_social/profile/'
        },
        form: {
            track_id: data.track_id,
            profile_id: data.profile_id,
            set_auth: data.set_auth,
            current_password: data.current_password
        }
    };

    if (data.passErrors) {
        options.passErrors = true;
    }

    return this.fetch(options, 'socialAllowAuth');
};

API.prototype.socialAllowAuthV2 = function(data) {
    var options = {
        url: {
            pathname: '/2/bundle/change_social/profile/'
        },
        form: {
            track_id: data.track_id,
            profile_id: data.profile_id,
            set_auth: data.set_auth
        }
    };

    return this.fetch(options, 'socialAllowAuth');
};

API.prototype.socialProfileDeleteV2 = function(data) {
    var options = {
        qs: {
            track_id: data.track_id,
            profile_id: data.profile_id
        },
        url: {
            pathname: '/2/bundle/change_social/profile/'
        },
        method: 'DELETE'
    };

    return this.fetch(options, 'socialProfileUnsubscribe');
};

API.prototype.socialProfileDelete = function(data) {
    var options = {
        qs: {
            track_id: data.track_id,
            profile_id: data.profile_id,
            current_password: data.current_password
        },
        url: {
            // pathname: '/2/bundle/change_social/profile/'
            pathname: '/1/change_social/profile/'
        },
        method: 'DELETE'
    };

    if (data.passErrors) {
        options.passErrors = true;
    }

    return this.fetch(options, 'socialProfileUnsubscribe');
};

API.prototype.socialProfileData = function(data) {
    assert(_.isObjectLike(data), 'Data should be a plain object');

    return this.fetch({
        passErrors: true,
        url: {
            pathname: '/1/bundle/social/thumbnail/'
        },

        form: data
    });
};

API.prototype.semiautoFieldValidate = function(data) {
    delete data.track_id;

    Object.keys(data).forEach(function(key) {
        if (/\[\]/.test(key)) {
            data[key.replace('[]', '')] = typeof data[key] === 'string' ? [data[key]] : data[key];
            delete data[key];
        }
    });

    return this.fetch({
        passErrors: true,
        url: {
            pathname: '/3/bundle/restore/semi_auto/validate/'
        },

        form: {
            json_data: JSON.stringify(data)
        }
    });
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/account/questions/
API.prototype.getUserQA = function() {
    return this.fetch(
        {
            url: {
                pathname: '/1/account/questions/secure/'
            },
            method: 'GET'
        },
        'getUserQA'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/NewAutoRestore/#poluchenievariantovloginadljavosstanovlenija
API.prototype.restoreLoginsSuggest = function() {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/2/bundle/restore/login_suggest/'
            }
        },
        'restoreLoginsSuggest'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#formatzaprosa1
API.prototype.restoreCheckLogin = function(data) {
    const dataToSend = {
        login: data.login
    };

    if (typeof data.retpath === 'string') {
        dataToSend.retpath = data.retpath;
    }

    if (typeof data.gps_package_name === 'string') {
        dataToSend.gps_package_name = data.gps_package_name;
    }

    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/submit/'
            },
            form: dataToSend
        },
        'restoreCheckLogin'
    );
};

// eslint-disable-next-line max-len
// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#polucheniesostojanijaproceduryvosstanovlenija
API.prototype.restoreGetState = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/restore/get_state/'
        },
        form: {}
    };

    if (data && data.track_id) {
        options.form.track_id = data.track_id;
    }
    return this.fetch(options, 'restoreGetState');
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#vybormetodavosstanovlenija
API.prototype.restoreSetRestoreMethod = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/select_method/'
            },
            form: {
                method: data.method
            }
        },
        'restoreSetRestoreMethod'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#formatzaprosa8
API.prototype.restoreCheckHintAnswer = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/hint/check/'
            },
            form: {
                answer: data.answer
            }
        },
        'restoreCheckHintAnswer'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#proverkaemail-adresa
API.prototype.restoreCheckEmail = function(data) {
    var self = this;

    return this.params('language').then(function(result) {
        return self.fetch(
            {
                url: {
                    pathname: '/1/bundle/restore/email/check/'
                },
                form: {
                    display_language: result.body && result.body.language,
                    email: data.email,
                    is_simple_format: data.simple
                }
            },
            'restoreCheckEmail'
        );
    });
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#proverkatelefonaiotpravkasms
API.prototype.restoreCheckPhone = function(data) {
    const self = this;
    const params = {
        phone_number: data.number,
        confirm_method: data.confirmMethod || 'by_sms'
    };

    if (data.code_format && data.code_format !== 'by_3') {
        params.code_format = data.code_format;
    }

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        return self._fetchForm('/1/bundle/restore/phone/check/', {
            ...params,
            display_language: values[0].body.language,
            country: values[1].body.country[0]
        });
    });
};

API.prototype.workspaceValidateDomain = function(body) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/validate/directory_domain/'
            },

            form: {
                domain: body.domain,
                strict: true
            },

            passErrors: true
        },
        'workspaceValidateDomain'
    );
};

API.prototype.workspaceRegistration = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/account/register/directory/'
            },

            form: data
        },
        'workspaceRegistration'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#podrobnoeopisaniekazhdogometoda
API.prototype.socialStart = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/start/'
            },
            form: data
        },
        'social-start'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#metodnativestart
API.prototype.socialStartNative = function(data, {hostname}) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/native_start/'
            },
            form: data,
            headers: {
                'Ya-Client-Host': hostname
            }
        },
        'social-start-native'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#metodnativestart
API.prototype.socialStartNativeThirdParty = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/third_party_native_start/'
            },
            form: data
        },
        'social-start-native-third-party'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#metodcallback
API.prototype.socialCallback = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/callback/'
            },
            form: data
        },
        'social-callback'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#metodsecurecallback
API.prototype.socialCallbackSecure = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/secure_callback/'
            },
            form: data
        },
        'social-callback-secure'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#metodchoose
API.prototype.socialChoose = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/choose/'
            },
            form: data
        },
        'social-choose'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/social/#metodthirdpartychoose
API.prototype.socialChooseNativeThirdParty = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/third_party_choose/'
            },
            form: data
        },
        'social-choose-native-third-party'
    );
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/social/#metodregister
API.prototype.socialRegister = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/register/'
            },
            form: data.form,
            method: data.method
        },
        'social-register'
    );
};

API.prototype.socialRegisterThirdParty = function(data) {
    return this.fetch(
        {
            passErrors: true,
            url: {
                pathname: '/1/bundle/auth/social/third_party_register/'
            },
            form: data.form,
            method: data.method
        },
        'social-register-third-party'
    );
};

API.prototype.socialIssueCredentials = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/auth/social/issue_credentials/'
            },
            form: data.form,
            method: data.method
        },
        'social-issue-credentials'
    );
};

filters.yasmsGetState = function(handleResponse) {
    var handleBody = handleResponse.body;
    var data = filters.base(handleResponse);

    if (handleBody.status === 'ok') {
        data.body.account = {
            phones: (handleBody.account && handleBody.account.phones) || {}
        };

        Object.keys(data.body.account.phones).forEach(function(key) {
            var _phone = data.body.account.phones[key];

            _phone.number = _phone.number.masked_international || null;
            data.body.account.phones[key] = _phone;
        });
    }

    return data;
};

filters.yasmsPhoneOperationSubmit = function(handleResponse) {
    var handleBody = handleResponse.body;
    var data = filters.base(handleResponse);

    if (handleBody.status === 'ok') {
        var phone = handleBody.phone;

        data.body.operation = {
            id: phone.operation && phone.operation.id,
            phone: phone.number && phone.number.masked_international,
            phoneId: phone.id
        };
    }

    return data;
};

filters.yasmsSecurePhoneReplaceSubmit = function(handleResponse) {
    var handleBody = handleResponse.body;
    var data = filters.base(handleResponse);

    if (handleBody.status === 'ok') {
        data.body.simple_phone = handleBody.simple_phone;
        data.body.secure_phone = handleBody.secure_phone;

        data.body.simple_phone.number =
            data.body.simple_phone.number && data.body.simple_phone.number.masked_international;
        data.body.secure_phone.number =
            data.body.secure_phone.number && data.body.secure_phone.number.masked_international;
    }

    return data;
};

API.prototype.yasmsGetState = function(data) {
    var config = _.assign(
        {
            filter: true
        },
        data
    );

    return this._fetchForm(
        '/1/bundle/phone/manage/get_state/',
        {},
        {
            filter: config.filter ? filters.yasmsGetState : null
        }
    );
};

API.prototype.yasmsEnableAliasAsEmail = function() {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/enable_alias_as_email/',
            {
                display_language: result.body && result.body.language
            },
            {
                filter: filters.base
            }
        );
    });
};

API.prototype.yasmsDisableAliasAsEmail = function() {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/disable_alias_as_email/',
            {
                display_language: result.body && result.body.language
            },
            {
                filter: filters.base
            }
        );
    });
};

API.prototype.yasmsSimplePhoneSubmit = function(data) {
    var self = this;

    return when.join(self.params('country'), self.params('language')).then(function(values) {
        return self._fetchForm(
            '/1/bundle/phone/manage/bind_simple/submit/',
            {
                country: values[0].body.country[0],
                display_language: values[1].body.language,
                number: data.number
            },
            {
                filter: filters.yasmsPhoneOperationSubmit
            }
        );
    });
};

API.prototype.yasmsSimplePhoneCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/bind_simple/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsSimplePhoneSecurifySubmit = function(data) {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/securify/submit/',
            {
                display_language: result.body && result.body.language,
                phone_id: data.id
            },
            {
                filter: filters.yasmsPhoneOperationSubmit
            }
        );
    });
};

API.prototype.yasmsSimplePhoneSecurifyCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/securify/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsSimplePhoneRemove = function(data) {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/remove_simple/',
            {
                display_language: result.body && result.body.language,
                phone_id: data.id
            },
            {
                filter: filters.base
            }
        );
    });
};

API.prototype.yasmsSecurePhoneSubmit = function(data) {
    var self = this;

    return when.join(self.params('country'), self.params('language')).then(function(values) {
        return self._fetchForm(
            '/1/bundle/phone/manage/bind_secure/submit/',
            {
                country: values[0].body.country[0],
                display_language: values[1].body.language,
                number: data.number,
                is_alias: data.aliasify
            },
            {
                filter: filters.yasmsPhoneOperationSubmit
            }
        );
    });
};

API.prototype.yasmsSecurePhoneCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/bind_secure/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsSecurePhoneRemoveSubmit = function(data) {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/remove_secure/submit/',
            {
                does_user_admit_secure_number: data.owner,
                display_language: result.body && result.body.language
            },
            {
                filter: filters.yasmsPhoneOperationSubmit
            }
        );
    });
};

// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#podtverzhdenietelefonaspomoshhjukodaizsms
API.prototype.restoreCheckCode = function(data) {
    return this._fetchForm('/1/bundle/restore/phone/confirm/', {
        code: data.code
    });
};

// eslint-disable-next-line max-len
// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#proverkakljuchavosstanovlenijadljaprodolzhenijaproceduryvosstanovlenija
API.prototype.restoreCheckEmailKey = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/restore/key/submit/'
        },
        form: {
            secret_key: data.key
        }
    };

    if (data.track_id) {
        options.form.track_id = data.track_id;
    }

    return this.fetch(options, 'restoreCheckEmailKey');
};

API.prototype.yasmsSecurePhoneRemoveCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/remove_secure/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsSecurePhoneReplaceSubmit = function(data) {
    var self = this;

    return when.join(self.params('language'), self.params('country')).then(function(values) {
        return self._fetchForm(
            '/1/bundle/phone/manage/replace/submit/',
            {
                display_language: values[0].body && values[0].body.language,
                country: values[1].body && values[1].body.country[0],
                does_user_admit_secure_number: data.owner,
                phone_id: data.phone_id,
                number: data.number
            },
            {
                filter: filters.yasmsSecurePhoneReplaceSubmit
            }
        );
    });
};

API.prototype.yasmsSecurePhoneAliasifySubmit = function() {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/aliasify/submit/',
            {
                display_language: result.body && result.body.language
            },
            {
                filter: filters.yasmsPhoneOperationSubmit
            }
        );
    });
};

API.prototype.yasmsSecurePhoneAliasifyCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/aliasify/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsSecurePhoneDealiasifySubmit = function() {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/dealiasify/submit/',
            {
                display_language: result.body && result.body.language
            },
            {
                filter: filters.yasmsPhoneOperationSubmit
            }
        );
    });
};

API.prototype.yasmsSecurePhoneDealiasifyCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/dealiasify/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsSecurePhoneReplaceCommit = function(data) {
    return this._fetchForm(
        '/1/bundle/phone/manage/replace/commit/',
        {
            operation_id: data.id
        },
        {
            filter: filters.base
        }
    );
};

API.prototype.yasmsProlong = function(data) {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/prolong_valid/',
            {
                display_language: result.body && result.body.language,
                phone_id: data.id
            },
            {
                filter: filters.base
            }
        );
    });
};

API.prototype.yasmsCheckPassword = function(data) {
    return this._fetchForm('/1/bundle/phone/manage/check_password/', {
        operation_id: data.operationId,
        current_password: data.password
    });
};

API.prototype.yasmsCheckCode = function(data) {
    return this._fetchForm('/1/bundle/phone/manage/check_code/', {
        operation_id: data.operationId,
        code: data.code
    });
};

// eslint-disable-next-line max-len
// https://beta.wiki.yandex-team.ru/passport/python/api/bundle/NewAutoRestore/#okonchatelnajaustanovkavsexdannyxdljaavtorizacii
API.prototype.restoreChangePassword = function(data) {
    var self = this;

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        return self.fetch(
            {
                passErrors: true,
                url: {
                    pathname: '/1/bundle/restore/commit/'
                },
                form: _.extend(omitEmpty(data), {
                    display_language: values[0].body.language,
                    language: values[0].body.language,
                    country: values[1].body.country[0]
                })
            },
            'restoreChangePassword'
        );
    });
};

API.prototype.yasmsResendCode = function(data) {
    var self = this;

    return self.params('language').then(function(result) {
        return self._fetchForm('/1/bundle/phone/manage/send_code/', {
            display_language: result.body && result.body.language,
            operation_id: data.id
        });
    });
};

API.prototype.yasmsCancelOperation = function(data) {
    return this._fetchForm('/1/bundle/phone/manage/cancel_operation/', {
        operation_id: data.id
    });
};

API.prototype.yasmsSetDefaultForNotify = function(data) {
    var self = this;

    return this.params('language').then(function(result) {
        return self._fetchForm(
            '/1/bundle/phone/manage/set_default/',
            {
                display_language: result.body && result.body.language,
                phone_id: data.id
            },
            {
                filter: filters.base
            }
        );
    });
};

API.prototype.userApproveSubmit = function({retpath}) {
    const options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/1/bundle/account/userapprove/submit/'
        },
        form: {
            retpath
        }
    };

    return this.fetch(options, 'userApproveSubmit');
};

API.prototype.userApproveCommit = function({track_id, text, retpath}) {
    const options = {
        url: {
            pathname: '/1/bundle/account/userapprove/commit/'
        },
        form: {
            track_id,
            text,
            retpath
        }
    };

    return this.fetch(options, 'userApproveCommit');
};

API.prototype.submitUnsubscribe = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/unsubscribe/submit/'
        },

        form: {
            service: data.service,
            retpath: data.retpath
        }
    };

    return this.fetch(options, 'submitUnsubscribe');
};

API.prototype.commitUnsubscribe = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/unsubscribe/commit/'
        },
        form: {
            track_id: data.track_id,
            password: data.password
        }
    };

    return this.fetch(options, 'commitUnsubscribe');
};

API.prototype.subscribe = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/account/subscribe/'
        },
        form: {
            consumer: data.consumer,
            service: data.service,
            retpath: data.retpath
        }
    };

    return this.fetch(options, 'subscribe');
};

API.prototype.appPasswordsActivate = function(track) {
    var options = {
        url: {
            pathname: '/1/account/app_passwords/activate/'
        },

        form: {
            track_id: track
        }
    };

    return this.fetch(options, 'appPasswordsActivate');
};

API.prototype.appPasswordsDeactivate = function(track) {
    var options = {
        url: {
            pathname: '/1/account/app_passwords/deactivate/'
        },

        form: {
            track_id: track
        }
    };

    return this.fetch(options, 'appPasswordsDeactivate');
};

// https://beta.wiki.yandex-team.ru/passport/api/bundle/newautorestore/#otpravkasmsnanovyjjnomerdljavosstanovlenija
API.prototype.restoreBindSumitPhone = function(data) {
    var self = this;

    return when.join(this.params('language'), this.params('country')).then(function(values) {
        return self._fetchForm('/1/bundle/restore/new_phone/check/', {
            display_language: values[0].body.language,
            country: values[1].body.country[0],
            phone_number: data.number
        });
    });
};

// eslint-disable-next-line max-len
// https://beta.wiki.yandex-team.ru/passport/api/bundle/newautorestore/#podtverzhdenienovogonomeratelefonaspomoshhjukodaizsms
API.prototype.restoreBindCheckCode = function(data) {
    return this._fetchForm('/1/bundle/restore/new_phone/confirm/', {
        code: data.code
    });
};

// https://wiki.yandex-team.ru/passport/api/bundle/NewAutoRestore/#proverkakorotkojjanketytolkodlja2fa-vosstanovlenija
API.prototype.restoreCheckTwoFaShortForm = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/restore/2fa_form/check/'
        },

        form: data
    };

    return this.fetch(options, 'restoreCheckTwoFaShortForm');
};

API.prototype.getAccountHistory = function(data) {
    var options = {
        url: {
            pathname: '/2/bundle/account/history/'
        },

        method: 'GET',
        qs: data
    };

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.qs.passErrors;
    }

    return this.fetch(options, 'getAccountHistory');
};

API.prototype.getChallengeInfo = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/account/history/challenge/'
        },

        method: 'GET',
        qs: {...data}
    };

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.qs.passErrors;
    }

    return this.fetch(options, 'getChallengeInfo');
};

API.prototype.getAccountHistoryEvents = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/events/'
        },

        method: 'GET',
        qs: data
    };

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.qs.passErrors;
    }

    return this.fetch(options, 'getAccountHistoryEvents');
};

API.prototype.profileGetState = function(params) {
    var options = {
        url: {
            pathname: '/1/bundle/account/'
        },
        method: 'GET'
    };

    if (params) {
        options.qs = params;
    }

    return this.fetch(options, 'profileGetState');
};

API.prototype.profileUpdate = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/person/'
        },
        form: data
    };

    if (data.passErrors) {
        options.passErrors = data.passErrors;
        delete options.form.passErrors;
    }

    return this.fetch(options, 'profileUpdate');
};

API.prototype.challengesSubmit = function(data) {
    const form = {
        track_id: data.track_id
    };

    if (typeof data.canSendSMS !== 'undefined') {
        form.can_send_sms = data.canSendSMS;
    }

    var options = {
        url: {
            pathname: '/1/bundle/auth/password/challenge/submit/'
        },
        form
    };

    return this.fetch(options, 'challengesSubmit');
};

API.prototype.challengesCommit = function(data) {
    const form = {
        track_id: data.track_id,
        challenge: data.challenge,
        answer: data.answer
    };

    if (typeof data.canSendSMS !== 'undefined') {
        form.can_send_sms = data.canSendSMS;
    }

    var options = {
        url: {
            pathname: '/1/bundle/auth/password/challenge/commit/'
        },
        form
    };

    return this.fetch(options, 'challengesCommit');
};

API.prototype.authPasswordCheckCookie = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/2/bundle/auth/password/check_cookie/'
        },

        form: _.pick(data, ['retpath', 'fretpath', 'clean'])
    };

    return this.fetch(options, 'authPaasswordCheckCookie');
};

API.prototype.authPasswordSubmit = function(data) {
    var options = {
        ignoreMissedTrack: true,
        url: {
            pathname: '/2/bundle/auth/password/submit/'
        },

        form: _.pick(data, ['track_id', 'retpath', 'fretpath', 'clean', 'service', 'origin', 'policy', 'with_code'])
    };

    return this.fetch(options, 'authPasswordSubmit');
};

API.prototype.authPasswordCommitPassword = function(data) {
    var options = {
        url: {
            pathname: '/2/bundle/auth/password/commit_password/'
        },

        form: _.pick(data, ['login', 'password', 'is_pdd'])
    };

    return this.fetch(options, 'authPasswordCommitPassword');
};

API.prototype.authPasswordCommitMagic = function(data) {
    var options = {
        url: {
            pathname: '/2/bundle/auth/password/commit_magic/'
        },

        form: _.pick(data, ['csrf_token'])
    };

    return this.fetch(options, 'authPasswordCommitMagic');
};

API.prototype.authSsoSubmit = function({track_id, code_challenge, code_challenge_method}) {
    var options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/auth/sso/submit/'
        },
        qs: {track_id, code_challenge, code_challenge_method}
    };

    return this.fetch(options, 'authSsoSubmit');
};

API.prototype.authSsoCommit = function({SAMLResponse, RelayState}) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/sso/commit/'
        },
        form: {
            saml_response: SAMLResponse,
            relay_state: RelayState
        }
    };

    return this.fetch(options, 'authSsoCommit');
};

API.prototype.authPasswordMultiStepStart = function(data) {
    const {
        isAm,
        appId,
        retpath,
        fretpath,
        clean,
        service,
        origin,
        policy,
        is_pdd,
        login,
        allow_scholar,
        process_uuid,
        social_track_id,
        old_track_id,
        with_2fa_pictures
    } = data;

    const options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/start/'
        },
        form: {
            retpath,
            fretpath,
            clean,
            service,
            origin,
            policy,
            is_pdd,
            login,
            allow_scholar,
            process_uuid,
            social_track_id,
            old_track_id,
            with_2fa_pictures
        },
        ignoreMissedTrack: true,
        passErrors: true
    };

    if (!isAm || !appId) {
        return this.fetch(options, 'authPasswordMultiStepStart');
    }

    const {
        am_version_name,
        app_id,
        app_platform,
        app_version_name,
        cloud_token,
        device_id,
        device_name,
        display_language,
        manufacturer,
        model,
        uuid,
        x_token_client_id
    } = data;

    options.form = Object.assign({}, options.form, {
        am_version_name,
        app_id,
        app_platform,
        app_version_name,
        cloud_token,
        device_id,
        device_name,
        display_language,
        manufacturer,
        model,
        uuid,
        x_token_client_id
    });

    return this.fetch(options, 'authPasswordMultiStepStart').then((result) => {
        const resolveMainResult = () => Promise.resolve(result);

        return this.writeTrack({app_id: appId})
            .then(resolveMainResult)
            .catch(resolveMainResult);
    });
};

API.prototype.authPasswordMultiStepCommitPassword = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/commit_password/'
        },
        form: _.pick(data, ['track_id', 'password'])
    };

    return this.fetch(options, 'authPasswordMultiStepCommitPassword');
};

API.prototype.authPasswordMultiStepCommitMagic = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/commit_magic/'
        },
        form: _.pick(data, ['track_id', 'csrf_token'])
    };

    return this.fetch(options, 'authPasswordMultiStepCommitMagic');
};

API.prototype.authPasswordMultiStepEmailCodeSubmit = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/email_code/submit/'
        },
        form: _.pick(data, ['track_id', 'csrf_token'])
    };

    return this.fetch(options, 'authPasswordMultiStepEmailCodeSubmit');
};

API.prototype.authPasswordMultiStepEmailCodeCommit = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/email_code/commit/'
        },
        form: _.pick(data, ['track_id', 'csrf_token', 'code'])
    };

    return this.fetch(options, 'authPasswordMultiStepEmailCodeCommit');
};

API.prototype.authLogout = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/logout/'
        },
        form: _.pick(data, ['track_id', 'uid'])
    };

    return this.fetch(options, 'authLogout');
};

API.prototype.authChangeDefault = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/change_default/'
        },
        form: _.pick(data, ['track_id', 'uid'])
    };

    return this.fetch(options, 'authChangeDefault');
};

API.prototype.authSuggest = function() {
    var options = {
        url: {
            pathname: '/1/bundle/auth/suggest/'
        },
        method: 'GET'
    };

    return this.fetch(options, 'authSuggest');
};

API.prototype.authSuggestForget = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/suggest/forget/'
        },
        form: _.pick(data, ['uid'])
    };

    return this.fetch(options, 'authSuggestForget');
};

// https://wiki.yandex-team.ru/passport/python/api/bundle/auth/suggest/Sadzhest-akkauntov/#getsuggestedinputlogin
API.prototype.authSuggestGetInputLogin = function(data) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/auth/suggest/get_input_login/'
        },
        qs: {
            uid: data.uid
        }
    };

    return this.fetch(options, 'authSuggestGetInputLogin');
};

API.prototype.authSendLinkLetter = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/magic_link/submit/'
        },

        form: {
            language: data.language,
            send_to: data.send_to
        }
    };

    return this.fetch(options, 'authSendLinkLetter');
};

API.prototype.getLetterPageInfo = function(track) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/magic_link/info/'
        },
        method: 'GET',
        qs: {
            track_id: track,
            avatar_size: 'islands-50'
        }
    };

    return this.fetch(options, 'getLetterPageInfo');
};

API.prototype.authWithLetterConfirm = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/magic_link/commit/'
        },

        form: {
            track_id: data.track,
            secret: data.secret,
            language: data.language
        }
    };

    if (data.redirect === 'true') {
        options.form.redirect = 'true';
    }

    return this.fetch(options, 'authWithLetterConfirm');
};

API.prototype.registerWithLetterConfirm = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/magic_link/commit_registration/'
        },

        form: {
            track_id: data.track,
            secret: data.secret,
            language: data.language
        }
    };

    if (data.redirect === 'true') {
        options.form.redirect = 'true';
    }

    return this.fetch(options, 'registerWithLetterConfirm');
};

API.prototype.authWithLetterInvalidate = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/magic_link/invalidate/'
        },

        form: {
            track_id: data.track_id
        }
    };

    return this.fetch(options, 'authWithLetterInvalidate');
};

API.prototype.checkStatusForAuthWithLetter = function(track) {
    var options = {
        url: {
            pathname: '/1/bundle/auth/password/multi_step/magic_link/status/'
        },
        method: 'GET',
        qs: {
            track_id: track
        }
    };

    return this.fetch(options, 'checkStatusForAuthWithLetter');
};

API.prototype.getYaMoneyAccountInfoShort = function(data) {
    var options = {
        retries: {
            timeout: 0
        },
        url: {
            port: config.paths.yamoney.port,
            hostname: config.paths.yamoney.hostname,
            pathname: '/yandex'
        },
        qs: {
            commandName: 'getAccountInfoShort'
        },
        form: data
    };

    return this.fetch(options, 'getYaMoneyAccountInfoShort');
};

API.prototype.getEmails = function() {
    var options = {
        url: {
            pathname: '/1/bundle/email/list_all/'
        },
        method: 'GET'
    };

    return this.fetch(options, 'getEmails');
};

API.prototype.deleteEmail = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/email/delete/'
        },
        form: {
            email: data.email
        }
    };

    return this.fetch(options, 'deleteEmail');
};

API.prototype.sendConfirmationEmail = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/email/send_confirmation_email/'
        },
        form: {
            email: data.email,
            retpath: data.retpath,
            language: data.language,
            track_id: data.track_id,
            is_safe: data.is_safe,
            code_only: data.code_only,
            validator_ui_url: data.validator_ui_url
        }
    };

    return this.fetch(options, 'sendConfirmationEmail');
};

API.prototype.confirmEmailByLink = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/email/confirm/by_link/'
        },
        form: {
            key: data.key,
            uid: data.uid
        }
    };

    return this.fetch(options, 'confirmEmailByLink');
};

API.prototype.confirmEmailByCode = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/email/confirm/by_code/'
        },
        form: {
            key: data.key
        }
    };

    return this.fetch(options, 'confirmEmailByCode');
};

API.prototype.setupConfirmedEmail = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/email/setup_confirmed/'
        },
        form: {
            email: data.email,
            is_safe: data.is_safe
        }
    };

    return this.fetch(options, 'setupConfirmedEmail');
};
// https://wiki.yandex-team.ru/passport/api/bundle/billing/#listpaymentmethods
API.prototype.billingListPaymentMethods = function(data) {
    return this.fetch(
        {
            ignoreMissedTrack: true,
            passErrors: true,
            url: {
                pathname: '/1/bundle/billing/list_payment_methods/'
            },
            form: {
                track_id: data.track_id
            }
        },
        'billingListPaymentMethods'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/billing/#createbinding
API.prototype.billingCreateBinding = function(data) {
    return this.fetch(
        {
            ignoreMissedTrack: true,
            passErrors: true,
            url: {
                pathname: '/1/bundle/billing/create_binding/'
            },
            form: {
                return_path: data.retpath,
                lang: data.language,
                domain_sfx: data.domain_sfx,
                track_id: data.track_id,
                template_tag: data.templateTag
            }
        },
        'billingCreateBinding'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/billing/#dobinding
API.prototype.billingDoBinding = function(data) {
    return this.fetch(
        {
            ignoreMissedTrack: true,
            passErrors: true,
            url: {
                pathname: '/1/bundle/billing/do_binding/'
            },
            form: {
                purchase_token: data.token,
                track_id: data.track_id
            }
        },
        'billingDoBinding'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/billing/#unbindcard
API.prototype.billingUnbindCard = function(data) {
    return this.fetch(
        {
            ignoreMissedTrack: true,
            passErrors: true,
            url: {
                pathname: '/1/bundle/billing/unbind_card/'
            },
            form: {
                card: data.id,
                track_id: data.track_id
            }
        },
        'billingUnbindCard'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/billing/#checkbinding
API.prototype.billingCheckBinding = function(data) {
    return this.fetch(
        {
            ignoreMissedTrack: true,
            passErrors: true,
            url: {
                pathname: '/1/bundle/billing/check_binding/'
            },
            form: {
                purchase_token: data.token,
                track_id: data.track_id
            }
        },
        'billingCheckBinding'
    );
};

// https://wiki.yandex-team.ru/passport/anti-hacking/ysa/
API.prototype.ysaGetEsecId = function(namespace) {
    var ysa = config.paths.ysa;

    return this.fetch(
        {
            timeout: 1000,
            url: {
                hostname: ysa.hostname,
                port: ysa.port,
                pathname: `/api/1/protector/${namespace}`
            },
            method: 'GET'
        },
        'ysaGetEsecId'
    );
};

API.prototype.ysaWriteUserEvent = function(data) {
    var ysa = config.paths.ysa;
    var events;

    if (typeof data.events === 'object') {
        events = JSON.stringify(data.events);
    } else {
        events = data.events;
    }

    return this.fetch(
        {
            timeout: 1000,
            url: {
                hostname: ysa.hostname,
                port: ysa.port,
                pathname: `/api/1/protector/${data.namespace}/${data.esecId}`
            },
            form: {
                events
            }
        },
        'ysaWriteUserEvent'
    );
};

API.prototype.ysaGetRobotness = function(data) {
    var ysa = config.paths.ysa;

    return this.fetch(
        {
            timeout: 1000,
            url: {
                hostname: ysa.hostname,
                port: ysa.port,
                pathname: `/api/1/protector/${data.namespace}/${data.esecId}`
            },
            method: 'GET'
        },
        'ysaGetRobotness'
    );
};

// https://wiki.yandex-team.ru/passport/api/bundle/LoginRestore/#inicializacijaprocessavosstanovlenijalogina

API.prototype.getRestoreLoginTrack = function() {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/login/submit/'
            }
        },
        'getRestoreLoginTrack'
    );
};

API.prototype.getRestoreLoginState = function(track) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/login/get_state/'
            },
            form: {
                track_id: track
            }
        },
        'getRestoreLoginState'
    );
};

API.prototype.getRestoreLoginCode = function(data) {
    var self = this;
    var options = {
        url: {
            pathname: '/1/bundle/restore/login/check_phone/'
        },
        form: {
            track_id: data.track_id,
            phone_number: data.phone_number,
            display_language: data.display_language,
            confirm_method: data.confirm_method || 'by_sms'
        }
    };

    return this.params('country').then(function(result) {
        options.form.country = (result.body && result.body.country[0]) || '';
        return self.fetch(options, 'getRestoreLoginCode');
    });
};

API.prototype.checkRestoreLoginCode = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/login/confirm_phone/'
            },
            form: {
                track_id: data.track_id,
                code: data.code
            }
        },
        'checkRestoreLoginCode'
    );
};

API.prototype.restoreLoginByName = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/restore/login/check_names/'
            },
            form: {
                track_id: data.track_id,
                firstname: data.firstname,
                lastname: data.lastname
            }
        },
        'restoreLoginByName'
    );
};

API.prototype.deleteAccountSubmit = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/submit/'
            },
            form: {
                track_id: data.track_id
            }
        },
        'deleteAccountSubmit'
    );
};

API.prototype.deleteAccountCommit = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/commit/'
            },
            form: {
                track_id: data.track_id,
                password: data.password
            }
        },
        'deleteAccountCommit'
    );
};

API.prototype.deleteAccountSendSms = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/send_sms/'
            },
            form: {
                track_id: data.track_id
            }
        },
        'deleteAccountSendSms'
    );
};

API.prototype.deleteAccountConfirmPhone = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/confirm_phone/'
            },
            form: {
                track_id: data.track_id,
                code: data.code
            }
        },
        'deleteAccountConfirmPhone'
    );
};

API.prototype.deleteAccountSendEmail = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/send_email/'
            },
            form: {
                track_id: data.track_id,
                email: data.email
            }
        },
        'deleteAccountSendEmail'
    );
};

API.prototype.deleteAccountConfirmEmail = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/confirm_email/'
            },
            form: {
                track_id: data.track_id,
                code: data.code
            }
        },
        'deleteAccountConfirmEmail'
    );
};

API.prototype.deleteCheckQuestion = function(data) {
    return this.fetch(
        {
            url: {
                pathname: '/2/bundle/delete_account/check_answer/'
            },
            form: {
                track_id: data.track,
                answer: data.answer
            }
        },
        'deleteCheckQuestion'
    );
};

API.prototype.suggestLanguage = function() {
    var options;

    options = {
        url: {
            pathname: '/1/suggest/language/'
        }
    };

    return this.fetch(options, 'language');
};

API.prototype.externalDataDisk = function() {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/disk/info/'
        }
    };

    return this.fetch(options, 'externalDataDisk');
};

API.prototype.externalDataMarket = function(pageSize) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/market/orders/'
        },
        qs: {
            page_size: pageSize
        }
    };

    return this.fetch(options, 'externalDataMarket');
};

API.prototype.externalDataMusic = function() {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/music/account_status/'
        }
    };

    return this.fetch(options, 'externalDataMusic');
};

API.prototype.externalDataVideo = function(pageSize, page) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/video/favourites/'
        },
        qs: {
            page_size: pageSize,
            page
        }
    };

    return this.fetch(options, 'externalDataVideo');
};

API.prototype.favExternalDataMarket = function(pageSize) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/market/favourites/'
        },
        qs: {
            page_size: pageSize
        }
    };

    return this.fetch(options, 'favExternalDataMarket');
};

API.prototype.externalDataCollections = function(pageSize) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/pictures/collections/'
        },
        qs: {
            page_size: pageSize
        }
    };

    return this.fetch(options, 'externalDataCollections');
};

API.prototype.externalDataMapBookmarks = function(pageSize) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/maps/bookmarks/'
        },
        qs: {
            page_size: pageSize
        }
    };

    return this.fetch(options, 'externalDataMapBookmarks');
};

API.prototype.externalDataMapBookmarkInfo = function(uri, language) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/bundle/account/external_data/maps/bookmark_info/'
        },
        qs: {
            uri,
            language
        }
    };

    return this.fetch(options, 'externalDataMapBookmarkInfo');
};

API.prototype.rfcTotpRecreateSecret = function() {
    const options = {
        url: {
            pathname: '/1/bundle/rfc_otp/recreate_secret/'
        }
    };

    return this.fetch(options, 'rfcTotpRecreateSecret');
};

API.prototype.getPhonishProfiles = function(trackId) {
    const options = {
        method: 'GET',
        url: {
            pathname: '/1/change_social/'
        },
        qs: {
            track_id: trackId
        }
    };

    return this.fetch(options, 'getPhonishProfiles');
};

API.prototype.confirmAndSubmitPhoneNumber = function(number, trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/phone/confirm_and_bind/submit/'
        },
        form: {
            number,
            track_id: trackId
        }
    };

    return when.join(this.params('language'), this.params('country')).then(
        ([
            {
                body: {language}
            },
            {
                body: {
                    country: [country]
                }
            }
        ]) => {
            options.form.display_language = language;
            options.form.language = language;
            options.form.country = country;
            return this.fetch(options, 'confirmAndSubmitPhoneNumber');
        }
    );
};

API.prototype.confirmAndBindPhoneNumber = function(code, trackId) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/phone/confirm_and_bind/commit/'
            },
            form: {
                code,
                track_id: trackId
            }
        },
        'confirmAndBindPhoneNumber'
    );
};

API.prototype.getTakeoutStatus = function(uid) {
    var options = {
        url: {
            pathname: '/1/bundle/takeout/extract/get_status/'
        },
        form: {uid}
    };

    return this.fetch(options, 'getTakeoutStatus');
};

API.prototype.getArchivePassword = function(uid) {
    var options = {
        url: {
            pathname: '/1/bundle/takeout/extract/get_archive_password/'
        },
        form: {uid}
    };

    return this.fetch(options, 'getArchivePassword');
};

API.prototype.getArchiveUrl = function(uid) {
    var options = {
        url: {
            pathname: '/1/bundle/takeout/extract/get_archive_url/'
        },
        form: {uid}
    };

    return this.fetch(options, 'getArchiveUrl');
};

API.prototype.startTakeoutRequest = function(uid) {
    var options = {
        url: {
            pathname: '/1/bundle/takeout/extract/request/'
        },
        form: {uid}
    };

    return this.fetch(options, 'startTakeoutRequest');
};

API.prototype.getUserSubs = function() {
    var options = {
        url: {
            pathname: '/1/bundle/account/mail_subscriptions/'
        },
        method: 'GET'
    };

    return this.fetch(options, 'getUserSubs');
};

API.prototype.changeUserSubs = function(data) {
    var options = {
        url: {
            pathname: '/1/bundle/account/mail_subscriptions/'
        },
        form: data
    };

    return this.fetch(options, 'changeUserSubs');
};

API.prototype.deviceSubmit = function(code) {
    const self = this;

    return when.join(this.params('language')).then(function(value) {
        const [languageField] = value;
        const {body = {}} = languageField;
        const {language} = body;
        const options = {
            url: {
                pathname: '/1/bundle/oauth/device_authorize/qr_code/submit/'
            },
            form: {
                code,
                language
            }
        };

        return self.fetch(options, 'deviceSubmit');
    });
};

API.prototype.deviceCommit = function(code) {
    const self = this;

    return when.join(this.params('language')).then(function(value) {
        const [languageField] = value;
        const {body = {}} = languageField;
        const {language} = body;
        const options = {
            url: {
                pathname: '/1/bundle/oauth/device_authorize/qr_code/commit/'
            },
            form: {
                code,
                language
            }
        };

        return self.fetch(options, 'deviceCommit');
    });
};

API.prototype.findAccountsByNameAndPhone = function(form) {
    const options = {
        url: {
            pathname: '/1/bundle/restore/login/check_names/simple/'
        },
        form
    };

    return this.fetch(options, 'findAccountsByNameAndPhone');
};

API.prototype.findAccountsByPhone = function(data) {
    const {track_id} = data;
    const options = {
        url: {
            pathname: '/1/bundle/auth/suggest/by_phone/list/'
        },
        qs: {
            track_id
        },
        method: 'GET'
    };

    return this.fetch(options, 'findAccountsByPhone');
};

API.prototype.validatePhoneBySquatter = function(data) {
    return when.join(this.params('country')).then((response = [{}]) => {
        const body = response[0].body || {};
        const {country = []} = body;

        return this._fetchForm('/1/bundle/validate/phone_number/by_squatter/', {
            country: country.length ? country[country.length - 1] : undefined,
            phone_number: data.phone,
            scenario: data.scenario
        });
    });
};

API.prototype.multiStepCommitSMSCode = function(form) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/auth/password/multi_step/commit_sms_code/'
            },
            form
        },
        'multiStepCommitSMSCode'
    );
};

API.prototype.supportCodeCreate = function(form) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/support_code/create/'
            },
            ignoreMissedTrack: true,
            form
        },
        'supportCodeCreate'
    );
};

API.prototype.webauthnList = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/webauthn/credentials/list/'
        },
        qs: {
            track_id: trackId,
            consumer: 'passport'
        },
        method: 'GET'
    };

    return this.fetch(options, 'webauthnList');
};

API.prototype.webauthnRemove = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/webauthn/credentials/remove/'
        },
        form: {
            ...data,
            consumer: 'passport'
        }
    };

    return this.fetch(options, 'webauthnRemove');
};

API.prototype.webauthnRegSubmit = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/webauthn/credentials/add/submit/'
        },
        qs: {
            track_id: trackId,
            consumer: 'passport'
        }
    };

    return this.fetch(options, 'webauthnRegSubmit');
};

API.prototype.webauthnRegCommit = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/webauthn/credentials/add/commit/'
        },
        form: {
            ...data,
            consumer: 'passport'
        }
    };

    return this.fetch(options, 'webauthnRegCommit');
};

API.prototype.webauthnAuthSubmit = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/webauthn/verify/submit/'
        },
        form: {
            ...data,
            consumer: 'passport'
        }
    };

    return this.fetch(options, 'webauthnAuthSubmit');
};

API.prototype.webauthnAuthCommit = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/webauthn/verify/commit/'
        },
        form: {
            ...data,
            consumer: 'passport'
        }
    };

    return this.fetch(options, 'webauthnAuthCommit');
};

API.prototype.getAccountWithFamilyInfo = function() {
    const options = {
        url: {
            pathname: '/1/bundle/account/'
        },
        qs: {
            need_display_name_variants: false,
            need_phones: true,
            need_emails: false,
            need_social_profiles: false,
            need_question: false,
            need_additional_account_data: false,
            need_family_info: true,
            need_family_members: true,
            need_family_kids: true,
            need_family_invites: true
        },
        method: 'GET'
    };

    return this.fetch(options, 'getAccountWithFamilyInfo');
};

API.prototype.createFamily = function() {
    const options = {
        url: {
            pathname: '/1/bundle/family/create/'
        }
    };

    return this.fetch(options, 'createFamily');
};

API.prototype.deleteFamily = function() {
    const options = {
        url: {
            pathname: '/1/bundle/family/delete/'
        }
    };

    return this.fetch(options, 'deleteFamily');
};

API.prototype.deleteFamilyMember = function(placeId) {
    const options = {
        url: {
            pathname: '/1/bundle/family/remove_member/'
        },
        form: {
            place_id: placeId
        }
    };

    return this.fetch(options, 'deleteFamilyMember');
};

API.prototype.leaveFamily = function() {
    const options = {
        url: {
            pathname: '/1/bundle/family/leave/'
        }
    };

    return this.fetch(options, 'leaveFamily');
};

API.prototype.neoPhonishAuth = function(form) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/auth/password/multi_step/after_login_restore/'
            },
            form
        },
        'neoPhonishAuth'
    );
};

API.prototype.neoPhonishRestoreAuth = function(form) {
    return this.fetch(
        {
            url: {
                pathname: '/1/bundle/auth/password/multi_step/after_suggest_by_phone/'
            },
            form
        },
        'neoPhonishRestoreAuth'
    );
};

API.prototype.createFamilyInvite = function(inviteRequest) {
    const options = {
        url: {
            pathname: '/1/bundle/family/issue_invite/'
        },
        form: inviteRequest
    };

    return this.fetch(options, 'createFamilyInvite');
};

API.prototype.confirmFamilyInvite = function(inviteId) {
    const options = {
        url: {
            pathname: '/1/bundle/family/accept_invite/'
        },
        form: {
            invite_id: inviteId
        }
    };

    return this.fetch(options, 'confirmFamilyInvite');
};

API.prototype.getFamilyInviteInfo = function(inviteId) {
    const options = {
        url: {
            pathname: '/1/bundle/family/invite_info/'
        },
        form: {
            invite_id: inviteId
        }
    };

    return this.fetch(options, 'createFamilyInvite');
};

API.prototype.cancelFamilyInvite = function(inviteId) {
    const options = {
        url: {
            pathname: '/1/bundle/family/revoke_invite/'
        },
        form: {
            invite_id: inviteId
        }
    };

    return this.fetch(options, 'cancelFamilyInvite');
};

API.prototype.createKiddish = function(kiddishInfo) {
    const options = {
        url: {
            pathname: '/1/bundle/family/create_kiddish/'
        },
        form: kiddishInfo
    };

    return this.fetch(options, 'createKiddish');
};

API.prototype.removeKiddish = function(kiddishInfo) {
    const options = {
        url: {
            pathname: '/1/bundle/family/delete_kiddish/'
        },
        form: kiddishInfo
    };

    return this.fetch(options, 'createKiddish');
};

API.prototype.editKiddish = function(kiddishInfo) {
    const options = {
        url: {
            pathname: '/1/bundle/family/change_kiddish/'
        },
        form: kiddishInfo
    };

    return this.fetch(options, 'createKiddish');
};

API.prototype.fetchNativeAmExperiments = function(deviceId) {
    const options = {
        url: config.paths.amExperiments,
        qs: {
            device_id: deviceId,
            consumer: 'passport'
        },
        method: 'GET',
        headers: {
            Host: config.paths.mobileproxy,
            'User-Agent': 'got',
            'content-type': 'application/x-www-form-urlencoded'
        }
    };

    return this.fetch(options, 'fetchNativeAmExperiments');
};

API.prototype.getUserValidateTrack = function(retpath, uid) {
    const options = {
        url: {
            pathname: '/1/bundle/challenge/standalone/create_track/'
        },
        form: {
            retpath,
            uid
        }
    };

    return this.fetch(options, 'getUserValidateTrack');
};

API.prototype.userValidateSubmit = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/challenge/standalone/submit/'
        },
        form: {
            track_id: trackId
        },
        passErrors: true
    };

    return this.fetch(options, 'userValidateSubmit');
};

API.prototype.userValidateCommit = function(data) {
    const {track_id, challenge, answer} = data;
    const options = {
        url: {
            pathname: '/1/bundle/challenge/standalone/commit/'
        },
        form: {
            track_id,
            challenge
        }
    };

    if (answer) {
        options.form.answer = answer;
    }

    return this.fetch(options, 'userValidateCommit');
};

API.prototype.userValidateSave = function(data) {
    const options = {
        url: {
            pathname: '/1/bundle/challenge/standalone/save/'
        },
        form: {
            track_id: data.track_id
        }
    };

    return this.fetch(options, 'userValidateSave');
};

API.prototype.challenge3dsVerifySubmit = function(data) {
    const {trackId, isNew3dsForm, useMobileLayout} = data;

    const options = {
        retries: {
            timeout: 5
        },
        url: {
            pathname: '/1/bundle/billing/3ds/verify/submit/'
        },
        form: {
            track_id: trackId,
            use_new_trust_form: isNew3dsForm,
            use_mobile_layout: useMobileLayout
        }
    };

    return this.fetch(options, 'challenge3dsVerifySubmit');
};

API.prototype.challenge3dsVerifyCommit = function(data) {
    const {track_id, errorsToRetry = []} = data;
    const options = {
        retries: {
            timeout: 10,
            errorsToRetry
        },
        url: {
            pathname: '/1/bundle/billing/3ds/verify/commit/'
        },
        form: {
            track_id
        },
        passErrors: true
    };

    return this.fetch(options, 'challenge3dsVerifyCommit');
};

API.prototype.getDeviceInfoQrAuth = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/oauth/device_authorize/qr_code/info/'
        },
        qs: {
            track_id: trackId
        },
        method: 'GET'
    };

    return this.fetch(options, 'getDeviceInfoQrAuth');
};

API.prototype.sendPush = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/push/2fa/send/'
        },
        form: {
            track_id: trackId
        },
        method: 'POST'
    };

    return this.fetch(options, 'sendPush');
};

API.prototype.getCode = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/push/2fa/get_code/'
        },
        qs: {
            track_id: trackId
        },
        method: 'GET'
    };

    return this.fetch(options, 'getCode');
};

API.prototype.authPrepareWithCred = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/auth/prepare_with_cred/'
        },
        form: {
            track_id: trackId
        },
        method: 'POST'
    };

    return this.fetch(options, 'authPrepareWithCred');
};

API.prototype.getAccountWithPhonesInfo = function() {
    const options = {
        url: {
            pathname: '/1/bundle/account/'
        },
        qs: {
            need_phones: true,
            need_display_name_variants: false,
            need_emails: false,
            need_social_profiles: false,
            need_question: false,
            need_additional_account_data: false
        },
        method: 'GET'
    };

    return this.fetch(options, 'getAccountWithPhonesInfo');
};

API.prototype.confirmSecureBoundAndAliasifySubmit = function(data) {
    const {trackId, enableAliasAsEmail} = data;
    const params = {
        track_id: trackId,
        enable_alias_as_email: enableAliasAsEmail
    };

    return this._genericPhoneConfirmSubmit('/2/bundle/phone/confirm_secure_bound_and_aliasify/submit/', params);
};

API.prototype.confirmSecureBoundAndAliasifyCommit = function(data) {
    const {trackId, code} = data;

    const options = {
        url: {
            pathname: '/2/bundle/phone/confirm_secure_bound_and_aliasify/commit/'
        },
        form: {
            track_id: trackId,
            code
        }
    };

    return this.fetch(options, 'confirmSecureBoundAndAliasifyCommit');
};

API.prototype.confirmSecureDeleteAliasSubmit = function(data) {
    const {trackId} = data;
    const params = {
        track_id: trackId
    };

    return this._genericPhoneConfirmSubmit('/2/bundle/phone/delete_alias/submit/', params);
};

API.prototype.confirmSecureDeleteAliasCommit = function(data) {
    const {trackId, password} = data;

    const options = {
        url: {
            pathname: '/2/bundle/phone/delete_alias/commit/'
        },
        form: {
            track_id: trackId,
            password
        }
    };

    return this.fetch(options, 'confirmSecureBoundAndAliasifyCommit');
};

API.prototype.sendCodeToEmail = function(trackId) {
    const options = {
        url: {
            pathname: '/1/bundle/email/check_ownership/send_code/'
        },
        form: {
            track_id: trackId
        },
        method: 'POST'
    };

    return this.fetch(options, 'sendCodeToEmail');
};

API.prototype.confirmEmailCode = function(data) {
    const {trackId, uid, code} = data;

    const options = {
        url: {
            pathname: '/1/bundle/email/check_ownership/confirm/'
        },
        form: {
            track_id: trackId,
            uid,
            code
        },
        method: 'POST'
    };

    return this.fetch(options, 'confirmEmailCode');
};

/**
 * Creates an API instance
 * @param {string} logID                    Log id to mark all the logs with
 * @param {Object<header, value>} headers   Headers hash
 * @param {string} [track_id]               Track id
 * @param {string} [language]               Two-letter language code. If none passed, a suggested language will be used
 * @param {string} [country]                Two-letter country code. If none passed, a suggested country will be used
 */
module.exports = inherit(API, {
    __constructor(logId, headers, trackId, lang, country) {
        assert(logId && typeof logId === 'string', 'Log id should be a string');
        assert(_.isObjectLike(headers), 'Headers should be passed as a dictionary');
        assert(!trackId || typeof trackId === 'string', 'Track id should be a string if defined');
        assert(
            !lang || (typeof lang === 'string' && lang.length === 2),
            'Lang should be a two-letter language code if defined'
        );
        assert(
            !country || (typeof country === 'string' && country.length === 2),
            'Country should be a two-letter code if defined'
        );

        // Hack to deal with cycle dependencies
        Controller = Controller || require('../controller');
        API.call(this, {
            logID: logId,
            headers,
            track_id: trackId,
            language: lang,
            country
        });
    }
});

module.exports.client = function(req) {
    Controller = Controller || require('../controller');
    var api = new API(req);

    return when(api);
};
