const config = require('../../configs/current');
const fs = require('fs');
const SERVICE_ALIASES = {
    ADDRS_SEARCH: 'addrs-search',
    ADFOX: 'adfox',
    AFISHA: 'afisha',
    AFISHADATA: 'afishadata',
    ALISA: 'alisa',
    ANTIFRAUD: 'antifraud',
    AUDIENCE: 'audience',
    APPMETRIKA: 'appmetrika',
    AVATARS: 'avatars',
    BLACKBOX: 'blackbox',
    BLACKBOX_MIMINO: 'mimino',
    BLOGS: 'blogs',
    BNPL: 'bnpl',
    BROWSER: 'browser',
    CARSHARING: 'carsharing',
    CLOUD: 'cloud',
    COLLECTIONS: 'collections',
    COMMENTATOR: 'messenger', // у них один хост
    CONTEST: 'contest',
    CRYPTA: 'crypta',
    DELI: 'taxi', // у них один хост
    DOCUMENTS: 'documents',
    EDADEAL: 'edadeal',
    EATS: 'taxi', // у них один хост
    FAMILYPAY: 'familypay',
    GAMES: 'games',
    FORMS: 'forms',
    GEOSMB: 'geosmb',
    IOT: 'iot',
    KINOPOISK: 'kinopoisk',
    LAVKA: 'taxi', // у них один хост
    MAPSBOOKING: 'mapsbooking',
    MAPSCONSTRUCTOR: 'mapsconstructor',
    MAPS: 'maps',
    MAPSHISTORY: 'mapshistory',
    MAPSMRC: 'mapsmrc',
    MAPSPANO: 'mapspano',
    MAPSPHOTO: 'mapsphoto',
    MAPSROADEVENTS: 'mapsroadevents',
    MAPSUGC: 'mapsugc',
    MARKET: 'market',
    MEDIABILLING: 'mediabilling',
    MESSENGER: 'messenger',
    METRIKA: 'metrika',
    MUSIC: 'music',
    NAVIGATOR: 'mapshistory', // у них один хост
    NMAPS: 'nmaps',
    OVERLOAD: 'overload',
    PLUS_LANDING: 'plus-landing',
    PORTAL: 'portal',
    PRAKTIKUM: 'praktikum',
    PROFI: 'profi',
    Q: 'q',
    REALTY: 'realty',
    SCHOOLBOOK: 'schoolbook',
    SITESEARCH: 'sitesearch',
    SURVEYS: 'surveys',
    TANKER: 'tanker',
    TAXI: 'taxi',
    TELEMED: 'telemed',
    TOLOKA: 'toloka',
    TRANSLATE: 'translate',
    TRAVELORCHESTRATOR: 'travelorchestrator',
    PLUSSTATISTIC: 'plusstatistic',
    TRUSTAPI: 'trustapi',
    UGC: 'ugc',
    USLUGI: 'uslugi',
    WEATHER: 'weather',
    WEBMASTER: 'webmaster',
    XMLSEARCH: 'xmlsearch',
    YASMS: 'yasms',
    ZEN: 'zen'
};

// promisify
const readFile = (path, options) =>
    new Promise((resolve, reject) => {
        fs.readFile(path, options, (err, data) => {
            if (err) {
                return reject(err);
            }
            return resolve(data);
        });
    });

/**
 * TicketsExtractor asynchronously extracts the tvm-tickets from external file.
 *
 * https://developer.mozilla.org/en-US/docs/Web/JavaScript/Reference/Global_Objects/Promise/resolve
 * Thenable
 */

class TicketsExtractor {
    constructor({ticketsFile = config.ticketsFile} = {}) {
        this.ticketsPath = ticketsFile;

        this.ticketsData = null;
        this.serviceTickets = {};

        this.bindAll();
    }

    bindAll() {
        this.readTicketsFile = this.readTicketsFile.bind(this);
        this.parseTickets = this.parseTickets.bind(this);
        this.extractTickets = this.extractTickets.bind(this);
        this.then = this.then.bind(this);
    }

    readTicketsFile() {
        return readFile(this.ticketsPath, 'utf8').then((ticketsData) => {
            this.ticketsData = ticketsData;
        });
    }

    parseTickets() {
        this.parsedTickets = JSON.parse(this.ticketsData);
    }

    extractTickets() {
        const parsedTickets = this.parsedTickets;

        if (!parsedTickets || !parsedTickets.tickets) {
            return;
        }

        const tickets = parsedTickets.tickets;

        Object.keys(tickets).forEach((key) => {
            const ticket = tickets[key];
            const alias = ticket.alias;

            this.serviceTickets[alias] = ticket.ticket;
        });
    }

    get tickets() {
        return Object.assign({}, this.serviceTickets);
    }

    then(resolve, reject) {
        this.readTicketsFile()
            .then(this.parseTickets)
            .then(this.extractTickets)
            .then(() => {
                resolve({tickets: this.tickets});
            })
            .catch(reject);
    }
}

function getTicketsAssoc() {
    return Promise.resolve(new TicketsExtractor()).then(({tickets}) => tickets);
}

module.exports = {
    SERVICE_ALIASES,
    TicketsExtractor,
    getTicketsAssoc
};
