var Api = require('inherit')(
    {
        __constructor: function(dao) {
            require('assert')(dao instanceof require('pdao'), 'Dao should be a data access object');
            this._dao = dao;
        },

        _mixDefaults: function(params) {
            return require('lodash').assign(params, {
                sender: 'passport-frontend'
            });
        },

        _responseHandler: function(responseBody = {}) {
            if (responseBody.doc && typeof responseBody.doc.errorcode === 'string') {
                throw new this.__self.ApiError(responseBody.doc.errorcode, responseBody.doc.error);
            }

            return responseBody;
        },

        sendToUid: function(uid, text) {
            require('assert')(uid && typeof uid === 'string', 'Uid should be a string');
            require('assert')(text && typeof text === 'string', 'Text should be a string');

            return this._dao
                .call(
                    'get',
                    '/sendsms',
                    this._mixDefaults({
                        text: text,
                        uid: uid,
                        utf8: 1
                    })
                )
                .then(this._responseHandler.bind(this));
        }
    },
    {
        ApiError: require('inherit')(Error, {
            __constructor: function(code, message) {
                this.name = 'YasmsApiError';
                this.message = message;
                this.code = code;
            },

            toString: function() {
                return require('util').format('%s %s: %s', this.name, this.code, this.message);
            }
        }),

        retryCondition: function(logger, response, body) {
            var logType = 'retryCondition';

            if (response.statusCode !== 200) {
                logger
                    .info()
                    .type(logType)
                    .write('Expected response status 200, got %s', response.statusCode);
                return new Api.retryCondition.Reason('Server response status was not 200 OK');
            }

            if (
                body.doc &&
                require('../../configs/current').api.yasms.dao.retryCodes.indexOf(body.doc.errorocode) > -1
            ) {
                logger
                    .info()
                    .type(logType)
                    .write('Error code %s was found in response %j', body.doc.errorocode, body);
                return new Api.retryCondition.Reason('Api returned an error code from retryCodes list');
            }
        }
    }
);

Api.retryCondition.Reason = require('inherit')(Error, {
    __constructor: function(message) {
        this.name = 'RetryReason';
        this.message = message;
    }
});

module.exports = require('inherit')(Api, {
    __constructor: function(logId, serviceTicket) {
        var config = require('../../configs/current');
        var dao = new (require('pdao/Http'))(
            logId,
            config.paths.yasms,
            config.api.yasms.dao.maxRetries,
            config.api.yasms.dao.retryAfter,
            config.api.yasms.dao.maxConnections,
            config.api.yasms.dao.timeout,
            Api.retryCondition
        )
            .setResponseType('text')
            .mixHeaders({
                'X-Ya-Service-Ticket': serviceTicket
            });

        this.__base(dao);
    }
});

module.exports.Api = Api;
