import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import {Modal} from '@components/Modal';
import {TextArea} from '@components/TextArea';
import {Input} from '@components/Input';
import {Spin} from '@components/Spin';
import {Captcha} from '@components/Captcha';
import reloadCaptcha from '@components/Captcha/actions/reloadCaptcha';
import {tldMapping} from '@blocks/utils';
import passport from '@plibs/pclientjs/js/passport';

const errorMessages = {
    captcha: {
        undefined: i18n('_AUTH_.errors.required'),
        empty: i18n('_AUTH_.errors.required'),
        notMatched: i18n('Profile.domik.captcha-not-matched')
    },
    message: {
        empty: i18n('_AUTH_.errors.required'),
        text_short: i18n('Errors.ErrorsTexts.shortbody')
    }
};

const sorry = i18n('UserApprove.form.no_phone')
    .replace('%link_start%', '<a class="b-link">')
    .replace('%link_finish%', '</a>');

class UserApprove extends Component {
    static mapStateToProps(state) {
        return {
            header: state.header,
            common: state.common,
            settings: state.settings,
            captcha: state.captcha
        };
    }

    constructor(props) {
        super(props);

        this.showModal = this.showModal.bind(this);
        this.maybeShowModal = this.maybeShowModal.bind(this);
        this.hideModal = this.hideModal.bind(this);
        this.toggleAttention = this.toggleAttention.bind(this);
        this.openPhones = this.openPhones.bind(this);
        this.submitMessage = this.submitMessage.bind(this);
        this.updateTextField = this.updateTextField.bind(this);
        this.hideError = this.hideError.bind(this);
        const mail = props.header.defaultAccount.mail;
        const tld = props.settings.tld;

        this.phonesLink = props.settings.phonesLink;
        this.answer = i18n('UserApprove.support.mail.answer').replace('%1', mail);
        this.restoreLink = i18n('UserApprove.denied.restore')
            .replace('%link_start%', `<a class='b-link' href="${this.phonesLink}">`)
            .replace('%link_finish%', '</a>');
        this.legalLink = i18n('UserApprove.denied.why.text').replace('legal.yandex.ru', function() {
            const newTld = tldMapping[tld] || 'ru';

            return `legal.yandex.${newTld}`;
        });

        this.state = {
            modalOpened: false,
            warningOpened: false,
            captchaError: null,
            messageError: null,
            loading: false,
            message: '',
            captcha: ''
        };
    }

    showModal() {
        this.setState({
            modalOpened: true
        });
    }

    maybeShowModal(e) {
        if (e.target.className.match('b-link')) {
            this.setState({
                modalOpened: true
            });
        }
    }

    hideModal() {
        this.setState({
            modalOpened: false,
            captchaError: null,
            messageError: null,
            loading: false,
            message: '',
            captcha: ''
        });
    }

    toggleAttention() {
        this.setState({
            warningOpened: !this.state.warningOpened
        });
    }

    openPhones() {
        location.replace(this.phonesLink);
    }

    updateTextField(event) {
        const {target = {}} = event;
        const {value, name} = target;

        this.setState({
            [name]: value
        });
    }

    hideError(event) {
        const fieldName = event.target.name;

        this.setState({
            [`${fieldName}Error`]: null
        });
    }

    redirectToRetpathOrFrontpage(retpath) {
        const host = this.props.settings.host;

        let resultUrl = location.href
            .split('.')
            .slice(1)
            .join('.');

        if (retpath) {
            resultUrl = `https://${host}/redirect?url=${encodeURIComponent(retpath)}`;
        }

        return location.replace(resultUrl);
    }

    submitMessage() {
        const self = this;
        const {message, captcha} = this.state;
        const match = location.search.match(/retpath=(.+)(&|$)/);
        const retpath = (match && match[1]) || this.props.settings.links.passport;
        const data = {
            captcha,
            track_id: this.props.common.track_id,
            text: message,
            retpath
        };

        if (message.length === 0 || captcha.length === 0) {
            return this.setState({
                messageError: message.length > 0 ? '' : 'empty',
                captchaError: captcha.length > 0 ? '' : 'empty'
            });
        }

        self.setState({
            loading: true
        });

        return passport.api
            .request('userapprove/commit', data)
            .then((res) => {
                if (res.status === 'ok') {
                    self.hideModal();
                    return location.replace(res.retpath);
                }

                return $.Deferred()
                    .reject(res.errors)
                    .promise();
            })
            .fail((errors) => {
                if (Array.isArray(errors)) {
                    if (errors.indexOf('captcha.not_matched') !== -1) {
                        self.setState({
                            captchaError: 'notMatched'
                        });
                    }

                    if (errors.indexOf('text.short') !== -1) {
                        self.setState({
                            messageError: 'text_short'
                        });
                    }

                    if (errors.indexOf('action.not_required') !== -1) {
                        self.redirectToRetpathOrFrontpage(retpath);
                    }
                }
            })
            .always(() => {
                self.setState({
                    loading: false
                });
                self.props.dispatch(reloadCaptcha());
            });
    }

    render() {
        const {dispatch, settings, captcha} = this.props;
        const {modalOpened, warningOpened, captchaError, messageError, message, loading} = this.state;
        const lang = settings.language;
        const ua = settings.ua;
        const isMobileCaptcha = Boolean(ua.isMobile || ua.isTablet);

        return (
            <div className='userapprove__wrapper'>
                <div className='userapprove'>
                    <h2 className='userapprove__title'>{i18n('UserApprove.title.denied')}</h2>
                    <p className='userapprove__subtitle'>
                        <span>{i18n('UserApprove.denied.some')} </span>
                        <span className='b-link' onClick={this.toggleAttention}>
                            {i18n('UserApprove.denied.why')}
                        </span>
                    </p>
                    <p
                        className='userapprove__attention'
                        style={{display: warningOpened ? '' : 'none'}}
                        dangerouslySetInnerHTML={{__html: this.legalLink}}
                    />
                    <div className='userapprove__bottom'>
                        <p dangerouslySetInnerHTML={{__html: this.restoreLink}} />
                        <p dangerouslySetInnerHTML={{__html: sorry}} onClick={this.maybeShowModal} />
                        <p>{i18n('UserApprove.form.sorry')}</p>
                        <div className='userapprove__buttons'>
                            <button className='userapprove__button' onClick={this.openPhones}>
                                {i18n('Enabling2fa.profile.access.2fa.entry-subtitle-no-phone')}
                            </button>
                            <button className='userapprove__button' onClick={this.showModal}>
                                {i18n('UserApprove.support.button')}
                            </button>
                        </div>
                    </div>
                </div>
                <Modal visible={modalOpened} onCloserClick={this.hideModal}>
                    <div className='userapprove__modal'>
                        <h3 className='userapprove__modal-title'>{i18n('UserApprove.support.title')}</h3>
                        <div>
                            <label className='userapprove__modal-label' htmlFor='message'>
                                {i18n('UserApprove.support.mail.title')}
                            </label>
                        </div>
                        <TextArea
                            size='l'
                            name='message'
                            id='message'
                            value={message}
                            onFocus={this.hideError}
                            onChange={this.updateTextField}
                        />

                        {Boolean(messageError) && (
                            <div className='control-error-block'>{errorMessages.message[messageError]}</div>
                        )}

                        <p className='userapprove__modal-text' dangerouslySetInnerHTML={{__html: this.answer}} />
                        <div className='userapprove__modal-captcha'>
                            <div className='userapprove__modal-input'>
                                <label className='userapprove__modal-label'>
                                    {i18n('_AUTH_.Errors.captcha.required')}:
                                </label>
                                <Input
                                    className='userapprove__modal-input'
                                    type='text'
                                    name='captcha'
                                    value={this.state.captcha}
                                    onFocus={this.hideError}
                                    onChange={this.updateTextField}
                                    size='l'
                                />
                                {Boolean(captchaError) && (
                                    <div className='control-error-block'>{errorMessages.captcha[captchaError]}</div>
                                )}
                            </div>
                            <Captcha
                                isMobile={isMobileCaptcha}
                                lang={lang}
                                introSound={captcha.introSound}
                                captchaSound={captcha.captchaSound}
                                imageUrl={captcha.imageUrl}
                                loading={captcha.loading}
                                loadingAudio={captcha.loadingAudio}
                                playing={captcha.playing}
                                type={captcha.type}
                                dispatch={dispatch}
                                visible={true}
                            />
                        </div>
                        {loading ? (
                            <Spin progress={true} size='m' />
                        ) : (
                            <button className='userapprove__modal-button' onClick={this.submitMessage}>
                                {i18n('UserApprove.support.send')}
                            </button>
                        )}
                    </div>
                </Modal>
            </div>
        );
    }
}

UserApprove.propTypes = {
    dispatch: PropTypes.func.isRequired,
    settings: PropTypes.object.isRequired,
    header: PropTypes.object.isRequired,
    common: PropTypes.object.isRequired,
    captcha: PropTypes.object.isRequired
};

const WrappedUserAppove = connect(UserApprove.mapStateToProps)(UserApprove);

export default WrappedUserAppove;
