/* eslint-env node */

var util = require('util');
var ApiError = require('papi/error');

module.exports = require('inherit')(
    require('../../../pages/AbstractPage'),
    {
        name: 'access-clientside',

        __constructor: function() {
            this.__base.apply(this, arguments);

            this._oauthApi = new (require('../../../lib/api/oauth'))(
                this._controller.getLogId(),
                this._controller.getHeaders(),
                this._lang,
                this._controller.getAuth().getUid()
            );

            this._router = new (require('prouter/Mapping'))()
                .map('/registration-validations/access/token/new', this._createAppPassword)
                .map('/registration-validations/access/pva', this._passwordVerificationAge)
                .map('/registration-validations/access/_testing/csrf', this._csrf);
        },

        open: function() {
            this._logger.info('opened');

            if (!this._controller.getMethod().isPost()) {
                this._logger.info(
                    'Expected POST, called with %s',
                    this._controller
                        .getMethod()
                        .toString()
                        .toUpperCase()
                );
                return this._controller.sendPage('POST request expected', 405);
            }

            if (!this._controller.getAuth().isLoggedIn()) {
                this._logger.info('User not logged in');
                return this._controller.sendPage('Pass valid session id', 401);
            }

            var method = this._router.match(this._controller.getUrl().pathname);

            if (method !== this._csrf && !this._controller.isCsrfTokenValidV2()) {
                return this._controller.sendPage('CSRF check failed', 403);
            }

            return method.call(this);
        },

        _createAppPassword: function() {
            var controller = this._controller;

            return this._oauthApi
                .issueAppPasswordVer2(controller.getFormData().scope, controller.getFormData().name, controller.getIp())
                .then(function(token) {
                    return controller.sendPage(
                        JSON.stringify({
                            alias: token.getAlias(),
                            clientSlug: token.getClientSlug()
                        })
                    );
                })
                .catch(function(err) {
                    var response;

                    if (typeof err === 'undefined') {
                        return; //Do nothing if there is no error, promise was rejected to stop the process
                    }

                    if (err instanceof ApiError) {
                        response = err.getResponse();

                        if (util.isArray(response.errors) && response.errors.indexOf('password.required') !== -1) {
                            return controller.sendPage(JSON.stringify(response));
                        }

                        if (util.isArray(response.errors) && response.errors.indexOf('token.limit_exceeded') !== -1) {
                            return controller.sendPage(JSON.stringify(response));
                        }
                    }

                    //Unknown error gets rethrown
                    throw err;
                });
        },

        _passwordVerificationAge: function() {
            var self = this;
            var age = self._controller.getAuth().getPasswordVerificationAge() || 86400;

            return self._controller.sendPage(
                JSON.stringify({
                    age: age
                })
            );
        },

        _csrf: function() {
            var env = this._controller.getEnv();
            var self = this;

            if (this.__self.CSRFHandleAllowedEnvs.indexOf(env) < 0) {
                this._logger.warn(
                    'Request to CSRF handle in %s, but only allowed in %s',
                    env,
                    this.__self.CSRFHandleAllowedEnvs
                );
                return this._controller.sendPage('Nah-uh!', 403);
            }

            return self._controller
                .getCsrfToken()
                .then(function(csrf_token) {
                    return self._controller.sendPage(csrf_token);
                })
                .catch(function() {
                    self._logger.error('Problem during CSRF-token generation');
                    return self._controller.sendPage('Nah-uh!', 500);
                });
        }
    },
    {
        CSRFHandleAllowedEnvs: ['development', 'testing']
    }
);
