/* eslint-env node */

var _ = require('lodash');
var util = require('util');
var ApiError = require('papi/error');
var config = require('../../../configs/current');
var tokensView = require('../../../blocks/access-tokens/tokensView');
var NonceView = require('../../../blocks/nonce/NonceView');
//TODO: delete-create-update notifications?

/**
 * @extends View
 * @typedef MessageView
 * @class MessageView
 */
var MessageView = require('inherit')(require('pview'), {
    name: 'message',

    /**
     * @param {string} message  Message localization key
     * @class MessageView
     * @constructor
     */
    __constructor: function(message) {
        require('assert')(message && typeof message === 'string', 'Message localization key should be defined');

        this.__base.apply(this, arguments);

        this._key = message;
        this._args = Array.prototype.slice.call(arguments);
    },

    /**
     * @param {string} lang Two-letter language code
     * @returns {{message: {localized: string, key: string}}}
     * @private
     */
    _compile: function(lang) {
        require('assert')(typeof lang === 'string' && lang.length === 2, 'Lang should be a two-letter language code');

        var i18n = require('putils').i18n;

        return {
            message: {
                localized: i18n.apply(i18n, [lang].concat(this._args)),
                key: this._key
            }
        };
    }
});

var NotificationView = require('inherit')(MessageView, {
    name: 'access.notification',
    _type: 'generic',
    _key: null,

    __constructor: function() {
        if (this._key) {
            this.__base(this._key);
        } else {
            this.__base.apply(this, arguments);
        }
    },
    _compile: function() {
        var result = this.__base.apply(this, arguments);

        result[Object.keys(result)[0]].type = this._type;
        return result;
    }
});
var TwofaTurnedOnNotification = require('inherit')(NotificationView, {
    _type: 'TwofaTurnedOn',
    _key: 'access.webpass.2fa.promo.success.title' //Key is irrelevant
});

//var TwofaChangepassNotification = require('inherit')(NotificationView, {
// TODO: PASSP-10745 — Текст для плашки о смене пароля
//    _type: 'TwofaChangepass',
//    _key: 'app.none_created' //TODO: needs proper text
//});

var TwofaMigrateNotification = require('inherit')(NotificationView, {
    _type: 'TwofaMigrationDone',
    _key: 'access.webpass.2fa.promo.success.title' //Key is irrelevant
});

var WebAccessView = require('inherit')(require('pview'), {
    name: 'access.web',
    __constructor: function(
        is2FAOn,
        canChangePassword,
        hasPassword,
        isIntranet,
        appPasswordsEnabled,
        urlParamOrigin,
        oauthUrl,
        activeTokenId
    ) {
        require('assert')(typeof is2FAOn === 'boolean', '2FA turned on flag should be a boolean');
        require('assert')(typeof canChangePassword === 'boolean', 'Password change forbidden flag should be a boolean');
        require('assert')(typeof hasPassword === 'boolean', 'Has password flag should be a boolean');
        require('assert')(typeof isIntranet === 'boolean', 'Intranet flag should be a boolean');
        require('assert')(typeof appPasswordsEnabled === 'boolean', 'App passwords enabled state should be a boolean');

        this.__base.apply(this, arguments);

        this._is2FAOn = is2FAOn;
        this._canChangePassword = canChangePassword;
        this._hasPassword = hasPassword;
        this._isIntranet = isIntranet;
        this._appPasswordsEnabled = appPasswordsEnabled;
        this._urlParamOrigin = urlParamOrigin;
        this._oauthUrl = oauthUrl;
        this._activeTokenId = activeTokenId;
    },

    _compile: function() {
        return {
            webAccess: {
                is2FAOn: this._is2FAOn,
                canChangePassword: this._canChangePassword,
                hasPassword: this._hasPassword,
                isIntranet: this._isIntranet,
                areAppPasswordsEnabled: this._appPasswordsEnabled,
                pageUrlOrigin: this._urlParamOrigin,
                oauthUrl: this._oauthUrl,
                activeTokenId: this._activeTokenId
            }
        };
    }
});

/* jshint unused:false */
var HistoryView = require('inherit')(
    require('pview'),
    {
        name: 'access.history',
        __constructor: function(passportApi, controller, is2FAOn, is2FAJustEnabled) {
            require('assert')(passportApi instanceof require('papi/Passport'), 'Api should be a Passport API');
            require('assert')(
                controller instanceof require('../../../lib/controller'),
                'Expected an instance of Controller'
            );

            this.__base.apply(this, arguments);

            this._api = passportApi;
            this._controller = controller;
            this._is2FAOn = is2FAOn;
            this._enabling2FAtrack = is2FAJustEnabled
                ? require('querystring').parse(this._controller.getUrl().query).enabling_2fa_track
                : undefined;
        },

        _compile: function() {
            if (this._controller.isIntranet()) {
                //PASSP-11438: 500ка при походе в ручку /lastauth
                return {
                    history: {
                        brokenAccessEntries: [],
                        tokens: []
                    }
                };
            }

            var that = this;
            var is2FAOn = this._is2FAOn;

            return this._api.getLastAuth(this._enabling2FAtrack).then(function(result) {
                var brokenEntries = (is2FAOn ? result.webApp : result.appPass).filter(function(entry) {
                    return entry.getTimestamp() > new Date(Date.now() - that.__self.BROKEN_AUTH_ENTRY_OLD_THRESHOLD);
                });

                var dateToChunks = that._controller.dateToChunks;

                return {
                    history: {
                        brokenAccessEntries: brokenEntries.map(function(entry) {
                            return {
                                ip: entry.getIp(),
                                timestamp: dateToChunks(entry.getTimestamp(), true),
                                authtype: entry.getAuthtype()
                            };
                        }),
                        tokens: result.tokens.concat(result.appPass).map(function(entry) {
                            return {
                                id: entry.getTokenId(),
                                lastUsage: dateToChunks(entry.getTimestamp(), false)
                            };
                        })
                    }
                };
            });
        }
    },
    {
        BROKEN_AUTH_ENTRY_OLD_THRESHOLD: 2592000000 //1 month in ms
    }
);

var AvailableSlugsView = require('inherit')(require('pview'), {
    name: 'access.clientslugs',
    _compile: function() {
        var slugs = _.map(require('../../../configs/current').appPasswordsClientIdMapping, function(name, clientId) {
            return {
                name: name,
                client_id: clientId
            };
        });

        return {clientSlugs: slugs.length ? slugs : null};
    }
});

var LoggerTokenView = require('inherit')(require('pview'), {
    name: 'access.logger.token',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);

        this._controller = controller;
    },
    _compile: function() {
        return {
            logger_token: this._controller.getLocalsField('logger_token')
        };
    }
});

module.exports = require('inherit')(
    require('../../AbstractPage'),
    {
        name: 'access',

        __constructor: function() {
            this.__base.apply(this, arguments);

            this._oauthApi = new (require('../../../lib/api/oauth'))(
                this._controller.getLogId(),
                this._controller.getHeaders(),
                this._lang,
                this._controller.getAuth().getUid()
            );

            var DiskApi = require('../../../lib/api/disk');

            this._diskApi = new DiskApi(this._controller.getLogId());
        },

        open: function() {
            var MAX_PASSWORD_AGE = 1800;

            this._logger.info('opened');

            if (!this._controller.getAuth().isLoggedIn()) {
                this._logger.info('User not logged in, redirecting to authorization');
                return this._controller.getAuth().authorize();
            }

            if (this._controller.getMethod().isPost()) {
                if (this._controller.getUrl().pathname === '/profile/access/delete') {
                    return this._deleteToken();
                }

                throw new URIError('No POST handler for the given url');
            } else {
                var passwordVerificationAge = this._controller.getAuth().getPasswordVerificationAge();

                if (passwordVerificationAge > MAX_PASSWORD_AGE) {
                    return this._controller.redirectToAuthVerify();
                }

                return this._startRenderingProcess();
            }
        },

        _deleteToken: function() {
            var that = this;
            var tokenID = Number(this._controller.getFormData()['token']);
            var controller = this._controller;

            controller
                .isCsrfTokenValidV2()
                .then(function(isValid) {
                    if (!isValid) {
                        return that._reopenViaRedirect();
                    } else {
                        that._logger.info('Deleting token %s', tokenID);

                        return that._oauthApi
                            .revokeSingleTokenVer3(tokenID)
                            .then(function() {
                                return that._reopenViaRedirect();
                            })
                            .catch(function(err) {
                                var response;

                                if (typeof err === 'undefined') {
                                    return; //Do nothing if there is no error, promise was rejected to stop the process
                                }

                                if (err instanceof ApiError) {
                                    response = err.getResponse();

                                    if (
                                        util.isArray(response.errors) &&
                                        response.errors.indexOf('password.required') !== -1
                                    ) {
                                        return controller.redirectToAuthVerify();
                                    }
                                }

                                //Unknown error gets rethrown
                                throw err;
                            });
                    }
                })
                .catch(function() {
                    return that._reopenViaRedirect();
                });
        },

        _reopenViaRedirect: function() {
            return this._controller.redirect('/profile/access');
        },

        _paramTimestampIsntOld: function(param) {
            param = parseInt(this._controller.getRequestParam(param), 10);
            if (param) {
                return Date.now() - param < this.__self.TIMESTAMPED_PARAM_OLD_THRESHOLD;
            }

            return null;
        },

        _handleRequestError: function(err, next) {
            if (err instanceof ApiError) {
                const response = err.getResponse();

                if (util.isArray(response.errors) && response.errors.indexOf('password.required') !== -1) {
                    return this._controller.redirectToAuthVerify();
                }
            } else if (!next) {
                throw err;
            }

            return next();
        },

        _startRenderingProcess: function() {
            const next = (canChangePassword) => this._getTokensListThenRender(canChangePassword || false);

            this._controller
                .getAuth()
                .canChangePassword()
                .then(next)
                .catch((err) => this._handleRequestError(err, next));
        },

        _getTokensListThenRender: function(canChangePassword) {
            const render = (allTokens) => this.render(canChangePassword, allTokens || []);

            this._oauthApi
                .listTokensVer3()
                .then(render)
                .catch((err) => this._handleRequestError(err, render));
        },

        // eslint-disable-next-line complexity
        render: function(canChangePassword = false, allTokens = []) {
            const controller = this._controller;
            const logger = this._logger;
            // eslint-disable-next-line global-require
            const Layout = require('./AccessLayout');
            const layout = new Layout(controller);
            const appPasswords = allTokens.filter((token) => token.isAppPassword());
            const is2FAOn = Boolean(controller.getAuth().getAttribute(1003));
            const areAppPasswordsEnabled =
                is2FAOn || Boolean(controller.getAuth().getAttribute(107)) || Boolean(appPasswords.length);

            this._api.writeStatbox(
                {
                    // Do not wait for the promise to resolve
                    action: 'opened',
                    mode: 'profile_access',
                    uid: controller.getAuth().getUid(),
                    origin: controller.getRequestParam('origin'),
                    is_2fa_on: is2FAOn ? 1 : 0,
                    app_passwords_enabled: areAppPasswordsEnabled ? 1 : 0,
                    app_passwords: appPasswords.length,
                    tokens: allTokens.length - appPasswords.length
                },
                false
            );

            this._logger.info(
                '2FA: %s, AppPasswords: %s, Can change password: %s',
                is2FAOn ? 'on' : 'off',
                areAppPasswordsEnabled ? 'on' : 'off',
                canChangePassword
            );

            layout.append(new LoggerTokenView(controller));
            layout.append(new AvailableSlugsView());
            layout.append(new NonceView(controller));
            layout.append(
                new WebAccessView(
                    is2FAOn,
                    canChangePassword,
                    controller.getAuth().hasPassword(),
                    controller.isIntranet(),
                    areAppPasswordsEnabled,
                    controller.getRequestParam('origin'),
                    config.paths.oauth.replace('%tld%', controller.getTld()),
                    controller.getRequestParam('token_id')
                )
            );

            layout.append(
                new HistoryView(
                    this._api,
                    controller,
                    is2FAOn,
                    is2FAOn && this._paramTimestampIsntOld('enabled') // 2fa just turned on
                )
            );

            // eslint-disable-next-line new-cap
            layout.append(new tokensView(areAppPasswordsEnabled, allTokens, controller));

            // check uid to prevent showing notification for another user: PASSP-13864
            if (is2FAOn && controller.getAuth().getUid() === controller.getRequestParam('user_id')) {
                if (this._paramTimestampIsntOld('enabled')) {
                    logger.info('Otp enabled recently, showing a success notification');
                    layout.append(new TwofaTurnedOnNotification());
                } else if (this._paramTimestampIsntOld('changepass')) {
                    logger.info('2FA user redirected from changepass, no notification yet, see PASSP-10745');
                    // logger.info('2FA user redirected from changepass, showing a notification');
                    // layout.append(new TwofaChangepassNotification());
                    // TODO: PASSP-10745 — Текст для плашки о смене пароля
                } else if (this._paramTimestampIsntOld('migrated')) {
                    logger.info('2FA migration done, showing a success notification');
                    layout.append(new TwofaMigrateNotification());
                }
            }

            logger.debug('Rendering');
            return layout.render(controller.getLanguage()).then(() => logger.info('Page sent'));
        }
    },
    {
        TIMESTAMPED_PARAM_OLD_THRESHOLD: 3600000 //1 hour in ms
    }
);
