import React, {Component} from 'react';
import PropTypes from 'prop-types';
import {connect} from 'react-redux';
import classnames from 'classnames';
import {Route, Switch, Redirect, withRouter} from 'react-router-dom';

import Header from '@components/Header/Header.jsx';
import {Footer} from '@components/Footer/New.js';
import {PageWrapper} from '@components/PageWrapper';
import {NoJS} from '@components/NoJS';

import {Domik} from '@blocks/morda/components/domik/domik.jsx';
import {RegPromo} from '@blocks/morda/components/reg_promo/reg_promo.jsx';
import {PageMessage} from '@blocks/morda/components/page_message/page_message.jsx';
import {changeCurrentPage} from '@blocks/common/actions';
import {hasExp} from '@blocks/selectors';
import api from '@blocks/api';
import metrics from '@blocks/metrics';
import {Greed} from '@plibs/greed';
import {Link} from '@components/Link';
import getRoutes from './routes';
import {IconsSvg} from './icons.jsx';

const ROUTES_WITHOUT_FOOTER = ['/profile/documents', '/profile/documents/pp'];
const ROUTES_WITHOUT_HEADER = ['/profile/documents/pp'];

class App extends Component {
    static sendDeleteMetrics() {
        metrics.send('Другие настройки: Удалить аккаунт');
    }

    constructor(props) {
        super(props);
        const isShowNoPasswordSettingsExp = hasExp(this.props, 'settings-nopassword-on');
        const {
            settings: {
                env: {type}
            },
            common: {isFamilyEnable}
        } = props;

        this.routes = getRoutes({
            isShowNoPasswordSettingsExp,
            envType: type,
            isFamilyEnable,
            isSupportEnabled: props && props.common && props.common.isSupportCenterEnable
        });
    }

    componentDidMount() {
        const dispatch = this.props.dispatch;
        const host = this.props.settings.host;

        this.props.history.listen((event) => {
            const {pathname, search} = event;
            const {action} = this.props.history;
            const url = `${host}${pathname}${search}`;

            dispatch(changeCurrentPage(url, action));
        });

        const {experiments = {}} = this.props.common;
        const {flags = []} = experiments;

        if (flags.includes('passport-greed-exp')) {
            this._sendFingerprint();
        }
    }

    async _sendFingerprint() {
        try {
            if (Greed && Greed.get) {
                const greedResult = await Greed.get();
                const {yandexuid, uid} = this.props.common;

                api.log(
                    {
                        action: 'fingerprint',
                        greedResult: JSON.stringify(greedResult),
                        yandexuid,
                        uid
                    },
                    {
                        encrypt: true
                    }
                );
            }
        } catch (error) {
            // DO NOTHING
        }
    }

    isPhone() {
        const {settings} = this.props;

        return (settings.ua.isMobile || settings.ua.isTouch) && !settings.ua.isTablet;
    }

    isYandexSearch() {
        const {settings: {ua: {BrowserName}} = {}} = this.props;

        return BrowserName && BrowserName === 'YandexSearch';
    }

    isFooterHidden() {
        const {history: {location: {pathname}} = {}} = this.props;

        return pathname && ROUTES_WITHOUT_FOOTER.includes(pathname) && (this.isPhone() || this.isYandexSearch());
    }

    isHeaderHidden() {
        const {history: {location: {pathname}} = {}} = this.props;

        return pathname && ROUTES_WITHOUT_HEADER.includes(pathname);
    }

    render() {
        const {settings, common = {}, location, pageMessage, dispatch, footer, router} = this.props;
        const {isSupportCenterEnable} = common;
        const {isLite, isNewLayout} = settings;
        const isPhone = this.isPhone();
        const isMainPage = router.location.pathname.split('/').filter(Boolean).length === 1;
        const isSupportPage = isSupportCenterEnable && router.location.pathname.match(/^\/support/);
        const isFooterHidden = this.isFooterHidden();
        const isHeaderHidden = this.isHeaderHidden();

        const showRemoveLink = !common.isPDD && !hasExp({common}, 'layout-footer-delete-account') && !isSupportPage;
        const deleteExp = hasExp({common}, 'delete_exp');
        const showDescRemoveLink = common.isPDD && !hasExp({common}, 'layout-footer-delete-account');

        return (
            <div
                className={classnames('layout-inner', {
                    'layout-inner_desktop': !isPhone
                })}
            >
                <NoJS />
                <PageWrapper isProfile={true} isNavigationHidden={isHeaderHidden} isHeaderHidden={isHeaderHidden}>
                    {!isNewLayout && <Header />}
                    <div
                        className={[
                            !isLite && 'profile',
                            isNewLayout && 'page-nl',
                            common.v2 && isMainPage && 'page-bg'
                        ]
                            .filter(Boolean)
                            .join(' ')}
                    >
                        <div>
                            {pageMessage && (
                                <PageMessage
                                    retpath={common.retpath}
                                    message={pageMessage.message}
                                    type={pageMessage.type}
                                />
                            )}

                            <Switch location={location}>
                                {!pageMessage &&
                                    this.routes.map((route) => (
                                        <Route
                                            key={route.path}
                                            path={route.path}
                                            exact={true}
                                            render={({match}) => {
                                                const ProfilePage = route.component;

                                                return <ProfilePage {...this.props} match={match} />;
                                            }}
                                        />
                                    ))}
                                <Redirect
                                    from={'/registration-validations/email/confirm-by-link'}
                                    to={'/profile'}
                                    exact={true}
                                    push={true}
                                />
                            </Switch>
                        </div>
                    </div>
                    <Domik />
                    <RegPromo dispatch={dispatch} common={common} />
                    <IconsSvg />
                </PageWrapper>

                {!isFooterHidden && !isLite && (
                    <div
                        className={classnames('profile__footer', {
                            'footer-bg': common.v2 && isMainPage
                        })}
                    >
                        <Footer
                            settings={settings}
                            env={settings.env}
                            location={location.pathname}
                            help={settings.help}
                            langlist={footer.langlist}
                            tld={settings.tld}
                            language={settings.language}
                        >
                            {showRemoveLink && (
                                <span className='n-footer__children-item' onClick={App.sendDeleteMetrics}>
                                    <Link
                                        href={
                                            deleteExp
                                                ? '/profile/delete?origin=passport_profile'
                                                : '/passport?mode=delete&origin=passport_profile'
                                        }
                                        className='footerDeleteLink'
                                    >
                                        {i18n('Profile.personal.hangover.remove')}
                                    </Link>
                                </span>
                            )}
                            {showDescRemoveLink && (
                                <span className='s-block__desc n-footer__children-item'>
                                    {i18n('Profile.personal.hangover.remove.pdd')}
                                </span>
                            )}
                        </Footer>
                    </div>
                )}
            </div>
        );
    }
}

App.propTypes = {
    pageMessage: PropTypes.object,
    dispatch: PropTypes.func.isRequired,
    dashboard: PropTypes.object.isRequired,
    common: PropTypes.object.isRequired,
    settings: PropTypes.object.isRequired,
    header: PropTypes.object.isRequired,
    footer: PropTypes.object.isRequired,
    router: PropTypes.object.isRequired,
    location: PropTypes.shape({
        pathname: PropTypes.string.isRequired
    }).isRequired,
    children: PropTypes.element,
    history: PropTypes.shape({
        listen: PropTypes.func.isRequired,
        action: PropTypes.string
    }).isRequired,
    isClosedPhoneAlert: PropTypes.bool,
    isClosed2faAlert: PropTypes.bool
};

export default withRouter(connect((state) => state)(App));
