/* global module */

(function(global, factory) {
    'use strict';

    if (typeof module === 'object' && module.exports) {
        module.exports = factory();
    } else {
        global.Metrika = factory();
    }
})(this, function() {
    var Metrika = {
        /**
         * @private
         */
        inited: false,

        /**
         * Кэш счётчиков, которые были добавлены, пока метрика ещё не догрузилась
         * @type Object
         * @private
         */
        counts: [],

        /**
         * Кэш целей, которые были добавлены, пока метрика ещё не догрузилась
         * @type Object
         * @private
         */
        goals: [],

        /**
         * Ссылка на счётчик
         * @type Ya.Metrika
         * @private
         */
        counter: null,

        /**
         * Инициализация метрики - загрузка скрипа и созание счётчика
         * @param {String} mcid id счётчика
         */
        init: function(mcid, experiments, params) {
            var self = this;
            var done = false;
            var script = document.createElement('script');
            var head = document.getElementsByTagName('head')[0];

            if (!mcid) {
                return this.clear();
            }

            self.infoParams = {};

            for (var key in params) {
                if (params[key]) {
                    self.infoParams[key] = params[key];
                }
            }

            // Грузим скрипт
            // noinspection FunctionWithInconsistentReturnsJS

            script.type = 'text/javascript';
            script.onload = script.onreadystatechange = function() {
                /* global Ya */
                if (
                    !done &&
                    (!this.readyState || this.readyState === 'loaded' || this.readyState === 'complete') &&
                    window.Ya &&
                    Ya.Metrika
                ) {
                    var settings = {
                        id: mcid,
                        clickmap: true,
                        trackLinks: true,
                        accurateTrackBounce: true,
                        webvisor: true,
                        trackHash: true
                    };

                    done = true;
                    script.onload = script.onreadystatechange = null;

                    if (head && script.parentNode) {
                        head.removeChild(script);
                    }

                    if (experiments) {
                        settings.experiments = experiments;
                    }

                    try {
                        self.counter = window['yaCounter' + mcid] = new Ya.Metrika(settings);
                    } catch (e) {
                        return self.clear();
                    }

                    // метрика проинициализирована
                    self.inited = true;

                    // выполняем ранее сохранённые цели
                    while (self.counts.length) {
                        self.count.apply(self, self.counts.shift());
                    }
                    while (self.goals.length) {
                        self.reachGoal.apply(self, self.goals.shift());
                    }
                }
            };
            script.src = 'https://mc.yandex.ru/metrika/watch.js';
            head.insertBefore(script, head.lastChild);
        },

        /**
         * Оборачивает переданные параметры в ещё один уровень
         * @param {Object} params
         * @param {String} name - имя обёртки
         * @private
         */
        wrap: function(params, name) {
            var wrapper = {};

            wrapper[name] = params;
            return wrapper;
        },

        /**
         * Сглаживает параметры и фильтрует пустые ячейки
         * @example [["Promo", "Some"], "click"] -> ["Promo", "Some", "click"]
         *
         * @param {Array} input     Данные для сглаживания
         * @returns {Array}
         */
        flatten: function(input) {
            // сглаживаем параметры
            // [["Promo", "Some"], "click"] -> ["Promo", "Some", "click"]
            var flatten = [];

            while (input.length) {
                flatten = flatten.concat(input.shift());
            }

            // удаляем пустые ячейки
            return flatten.filter(function(v) {
                return Boolean(v);
            });
        },

        /**
         * Считает в "Содержание" > "Параметры визитов"
         * @param {Object} params - https://yandex.ru/support/metrika/objects/params-method.xml
         *
         * вместо params можно мередать strig1, string2, string3,...
         * тогда оно склеится в {string1: {string2: string3}}
         * упрощает формирование структуры
         */
        count: function(params) {
            var args = Array.prototype.slice.call(arguments);

            // если метрика не инициализированна -
            // просто сохраним и обработаем позже
            if (!this.inited) {
                return this.counts.push(args);
            }

            args = this.flatten(args);

            // забираем последний параметр
            params = args.pop();

            // если остались ещё параметры - начинаем оборачивания
            while (args.length) {
                params = this.wrap(params, args.pop());
            }
            params.info = {};

            for (var key in this.infoParams) {
                if (this.infoParams[key]) {
                    params.info[key] = this.infoParams[key];
                }
            }

            this.counter.params(params);
        },

        /**
         * Считает в "Отчеты" > "Конверсии"
         * @param {String} target - https://yandex.ru/support/metrika/objects/reachgoal.xml
         *
         */
        reachGoal: function(target, params) {
            var args = Array.prototype.slice.call(arguments);

            // если метрика не инициализированна -
            // просто сохраним и обработаем позже
            if (!this.inited) {
                return this.goals.push(args);
            }

            this.counter.reachGoal(target, params);
        },

        /**
         * Нейтрализует метрику
         * @private
         */
        clear: function() {
            this.count = function() {};
            this.counts = [];
            this.goals = [];
        }
    };

    return Metrika;
});
