const url = require('url');
const crypto = require('crypto');
const fs = require('fs');
const httpCodes = require('http-codes');

// Routes
const blockRoutes = require('./routes/blocks');
const authRoutes = require('./routes/auth.js');
const retpathRoutes = require('./routes/retpath');
const logger = require('./routes/logger.js');
const authSilent = require('./routes/auth.silent').router;
// Profile
const langdetectDescription = require('./routes/info.langdetect');
const profilePasswordNew = require('./routes/profile.password.new').router;
const profilePassword = require('./routes/profile.password');
// const profileSocial = require('./routes/profile.social');
const profileEmails = require('./routes/profile.emails').router;
const ysa = require('./routes/ysa').router;
const restoreLogin = require('./routes/restore.login').router;
const a = require('./routes/a').router;
const deepSync = require('./routes/deepsync').router;
const modeLogout = require('./routes/logout');
const deleteUserAccount = require('./routes/profile.delete_account').router;
const subscribe = require('./routes/subscribe').router;
const {userapproveAjax, userapprovePage} = require('./routes/userapprove').router;
const multiAuthAccountsSetup = require('./routes/common/multiAuthAccountsSetup').getAccounts;
const modeErrorPage = require('./routes/mode.error').router;
const pushGetCode = require('./routes/push.get.code').router;
const pushQrSecureAuth = require('./routes/push.qr.secure.auth').router;
const pushTurboAppLogin = require('./routes/push.turbo.app.login').router;
const pushActivityAlert = require('./routes/push.activity.alert').router;
const pushTestPopupResize = require('./routes/push.test.popup.resize').router;
const takeOut = require('./routes/takeout').router;
const profileDeleteData = require('./routes/profile.delete-data').router;
const avatarDisplayname = require('./routes/avatarDisplayname').router;
const supportCode = require('./routes/supportCode').router;
const validateCSRF = require('./routes/common/validateCSRF.js');
const getCustomConfigByRequest = require('./routes/common/getCustomConfigByRequest.js');
const profilePublicId = require('./routes/profile.publicid').router;
const notMe = require('./routes/profile.notme').router;
const sessguardSet = require('./routes/sessguard/set').router;
const authSso = require('./routes/authSso').router;
const authUpdate = require('./routes/auth.update.js').router;
const subs = require('./routes/subs').router;
const userValidate = require('./routes/user-validate').router;
const checkData = require('./routes/check-data').router;
const loginMethod = require('./routes/login-method').router;

// New Auth
const authV2 = require('./routes/auth.v2').router;
const authSmartTv = require('./routes/auth.smarttv');
const authAskV2 = require('./routes/auth.ask.v2').router;
const authWithLetterLink = require('./routes/authv2/authWithLetterLink').router;
const authDevice = require('./routes/auth.device');
const smarttvGateway = require('./routes/smarttv-gateway/index');
const smarttvGatewayDebugIframe = require('./routes/smarttv-gateway/debug-iframe');

const FORCE_CLOSED = {
    PROFILE: false,
    REG: false,
    AUTH: false,
    RESTORATION: false,
    AJAX: false
};

try {
    FORCE_CLOSED.PROFILE = fs.statSync('./configs/common/PROFILE').isFile();
    console.error('Profile is closed');
} catch (e) {
    console.error('Profile is still open');
}

try {
    FORCE_CLOSED.REG = fs.statSync('./configs/common/REG').isFile();
    console.error('Registration is closed');
} catch (e) {
    console.error('Registration is still open');
}

try {
    FORCE_CLOSED.AUTH = fs.statSync('./configs/common/AUTH').isFile();
    console.error('Auth is closed');
} catch (e) {
    console.error('Auth is still open');
}

try {
    FORCE_CLOSED.RESTORATION = fs.statSync('./configs/common/RESTORATION').isFile();
    console.error('Restoration is closed');
} catch (e) {
    console.error('Restoration is still open');
}

try {
    FORCE_CLOSED.AJAX = fs.statSync('./configs/common/AJAX').isFile();
    console.error('AJAX is closed');
} catch (e) {
    console.error('AJAX is still open');
}

/* eslint-disable global-require */

module.exports = function(app) {
    // Вспомогательная плашка для тестингов в докере
    if (process.env.DOCKER) {
        app.all('*', function(req, res, next) {
            res.locals.dockerHost = req.hostname;
            res.locals.dockerHostColor = crypto
                .createHash('md5')
                .update(req.hostname)
                .digest('hex')
                .slice(0, 6);
            next();
        });
    }

    if (FORCE_CLOSED.AJAX) {
        app.all('/registration-validations/*', function(req, res) {
            res.sendStatus(400);
        });
    }

    app.get('/closewebview*', function(req, res) {
        return res.status(200).set('X-Frame-Options', 'DENY').send(`<!DOCTYPE html>
                    <html lang="en">
                        <head><meta charset="utf-8" /></head>
                        <body><p>Please, close this window</p></body>
                    </html>`);
    });

    // Refactored routes here
    app.use('/profile/password', profilePasswordNew.common);
    app.use('/registration-validations/email', profileEmails);
    app.use('/registration-validations/ysa', ysa);
    app.use('/registration-validations/userapprove', userapproveAjax);
    app.use('/restoration', restoreLogin);
    app.use('/a/?', a);
    app.use('/passport', subscribe);
    app.use('/passport', userapprovePage);
    app.use('/passport', modeErrorPage);

    app.use('/profile/delete', deleteUserAccount);
    app.use('/profile/getdata', takeOut);
    app.use('/profile/public', avatarDisplayname);
    app.use('/profile/publicid', profilePublicId);
    app.use('/profile/subs', subs);
    app.use('/profile/supportcode', supportCode);
    app.use('/profile/security/notme', notMe);
    app.use('/profile/data', profileDeleteData);
    app.use('/profile/loginmethod', loginMethod);

    modeLogout.route(app);

    // am pushes routes
    app.use('/am/push/getcode', pushGetCode);
    app.use('/am/push/qrsecure', pushQrSecureAuth);
    app.use('/am/push/turboapplogin', pushTurboAppLogin);
    app.use('/am/push/activityalert', pushActivityAlert);
    app.use('/am/push/changesalert', pushActivityAlert);

    /** ** Registration routes ****/

    // Frontend logger
    app.post('/registration-validations/logger', validateCSRF, function(req, res) {
        logger(req, res);
        res.send('');
    });
    app.options('/registration-validations/logger/?', function(req, res) {
        res.setHeader('Allow', 'POST');
        res.sendStatus(200);
    });
    // Done with logger

    // Routes for field validations
    for (const route in blockRoutes) {
        if (blockRoutes.hasOwnProperty(route)) {
            /* jshint loopfunc:true */
            (function(blockRoutes, route) {
                app.post(`/registration-validations/${route}/?`, blockRoutes[route]);
                app.options(`/registration-validations/${route}/?`, function(req, res) {
                    res.setHeader('Allow', 'POST');
                    res.sendStatus(200);
                });
            })(blockRoutes, route);
        }
    }

    // проверка загрузки css - https://st.yandex-team.ru/PASSP-10651
    require('./routes/checkcssload/checkcssload').route(app);

    // complete v2
    require('./routes/complete.v2').route(app);

    if (!FORCE_CLOSED.REG) {
        // registration v2
        require('./routes/registration.v2').route(app);

        // registration lite
        require('./routes/registration.lite').route(app);

        // registration.connect v2
        require('./routes/registration.connect').route(app);

        // На запросы к /passport?mode=simplereg тоже должны отвечать. В конфиге nginx'а:
        // location = /passport { ...
        // if ($arg_mode = 'simplereg') { proxy_pass http://127.0.0.1:3000; }
        // ... }
        app.all('/passport/?', function(req, res, next) {
            if (['simplereg', 'register', 'regone'].indexOf(req.query.mode) !== -1) {
                res.redirect(`/registration${url.parse(req.originalUrl).search}`);
            } else if (req.query.mode === 'delete') {
                res.redirect(`/profile/delete${url.parse(req.originalUrl).search}`);
            } else {
                next('route');
            }
        });
    }

    /** ** Done with registration routes ****/

    // embeddedauth
    require('./routes/embeddedauth.js').route(app);

    require('./routes/social').route(app);

    // retpath related routes
    retpathRoutes.route(app);

    // genauthtoken
    require('./routes/genauthtoken.js').route(app);

    // oauth
    require('./routes/oauth.js').route(app);

    // info
    require('./routes/info.js').route(app);

    // short restore fio
    require('./routes/restore.fio.js').route(app);

    // restore form
    require('./routes/restore.semiauto.js').route(app);

    // prerestore
    require('./routes/restore.prerestore.js').route(app);

    // get langdetect description
    app.get('/info/langdetect', langdetectDescription.getLangdetectDescription);

    // change password
    app.get('/for/:pdd_domain/profile/password', profilePassword.pass.enter);

    app.get(
        '/passport/?|/for/:pdd_domain*',
        function(req, res, next) {
            const mode = req.body.mode || (req.nquery && req.nquery.mode);

            if (mode === 'changepass') {
                next();
            } else {
                next('route');
            }
        },
        profilePassword.pass.enter
    );

    app.post(
        '/passport/?|/for/:pdd_domain*',
        function(req, res, next) {
            const mode = req.body.mode || (req.nquery && req.nquery.mode);

            if (mode === 'changepass') {
                next();
            } else {
                next('route');
            }
        },
        profilePassword.pass.submit
    );

    app.post('/profile/password', profilePassword.pass.submit);
    app.post('/for/:pdd_domain/profile/password', profilePassword.pass.submit);

    // suggest
    require('./routes/suggest.js').route(app);

    // multi accounts
    require('./routes/common/multiAuthAccountsSetup').route(app);

    // autologin
    require('./routes/autologin.js').route(app);
    //
    // complete
    require('./routes/complete.js').route(app);

    // confirm
    require('./routes/verify.js').route(app);

    // avatars
    require('./routes/avatars.js').route(app);

    // вас взломали
    // require('./routes/profile.hack').route(app);

    // after_restore
    require('./routes/after_restore.js').route(app);

    // auth/session
    require('./routes/session.js').route(app);

    app.use('/auth/guard/?', authUpdate);
    app.use('/set/?', sessguardSet);
    app.use('/auth/deepsync', deepSync);
    app.use('/auth/sso', authSso);

    // /auth/update
    require('./routes/auth.update.js').route(app);

    // require('./routes/simple-auth.js').route(app);

    // profile/social
    // app.use('/profile/social', profileSocial.router);
    // phones
    require('./routes/profile.phones.v2').route(app);
    require('./routes/profile.phones').route(app);

    // unsubscribe
    require('./routes/unsubscribe.js').route(app);

    // journal
    require('./routes/profile.journal.js').route(app);

    if (!FORCE_CLOSED.PROFILE) {
        // passport.v2
        require('./routes/profile.passport.v2').route(app);
    }

    // plus
    require('./routes/profile.plus').route(app);

    app.use('^/auth/?$', function(req, res, next) {
        const {track_id: trackId = ''} = req.query;
        const {fullscreen = false} = getCustomConfigByRequest(req) || {};

        if (trackId) {
            return next('route');
        }

        if (fullscreen) {
            return next('route');
        }

        res.set('X-Yandex-Passport-Authorize', '1');
        return next('route');
    });

    if (!FORCE_CLOSED.AUTH) {
        app.use('/auth/?', authV2);
        app.use('/auth/?', authAskV2);
        app.use('/auth/link/?', authWithLetterLink);
        app.use('/register/link/?', authWithLetterLink);
        app.use('/auth/silent', authSilent);
        app.use('/auth/smarttv', authSmartTv(app));
        app.use('/auth/smarttv-gateway', smarttvGateway(app));
        app.use('/auth/device', authDevice);
        app.use('/auth/register/lite/?', authV2);
        app.use('/auth/register/lite/form/?', authV2);
        app.use('/auth/register/lite/name/?', authV2);
        app.use('/auth/register/lite/eula/?', authV2);
        app.use('/auth/challenge/?', authV2);
        app.use('/auth/check-data', checkData);

        app.use(
            '/passport/?',
            function(req, res, next) {
                if (req.query.mode === 'auth') {
                    return next();
                }

                return next('route');
            },
            authV2
        );

        app.use(
            '/for/:pdd_domain',
            function(req, res, next) {
                res.locals.pdd_domain = req.params.pdd_domain;
                return next();
            },
            authV2
        );

        //
        // otp
        require('./routes/otp.js').route(app);
        require('./routes/authv2/authCheckMagicLetterStatus.js').route(app);
    }

    app.use('/auth/secure/?', (req) => req._controller.replaceToAuth());

    app.all('/auth/welcomepage', function(req, res) {
        return res
            .status(httpCodes.OK)
            .set('X-Frame-Options', 'SAMEORIGIN')
            .sendFile('WelcomeToYandex.html', {
                root: 'pages/public_html/'
            });
    });

    // Auth related routes
    authRoutes.route(app);

    // auth challenges
    require('./routes/auth.challenges.js').route(app);

    app.use('/auth/user-validate', userValidate);

    // security
    require('./routes/profile.security.js').route(app);

    if (app.get('env') === 'testing' || app.get('env') === 'development') {
        app.all('/test-auth', function(req, res) {
            const form =
                '<form method="post" action="/auth"><input name="login"/><br/><input type="password" ' +
                'name="passwd"/><br/><input name="retpath" value="https://yandex.ru"/><br/><button>ok</button></form>';

            res.send(form);
        });
    }

    // восстаноление
    require('./routes/restore.base.js').route(app);

    app.all(
        /\/profile\/access\/2fa\/migrate\/?.*/,
        multiAuthAccountsSetup,
        require('./routes/profile.access.2fa.migrate')
    );
    app.all(
        /\/profile\/access\/2fa\/disable\/?.*/,
        multiAuthAccountsSetup,
        require('./routes/profile.access.2fa.disable')
    );
    app.all(/\/profile\/access\/2fa\/?.*/, multiAuthAccountsSetup, require('./routes/profile.access.2fa'));

    require('./routes/profile.apppasswords').route(app);

    // profile/access routes should be _after_ profile/access/2fa routes,
    // since it matches everything under /profile/access
    require('./routes/profile.access').route(app);

    require('./routes/profile.verification').route(app);

    require('./routes/profile.email.confirm').route(app);

    require('./routes/phoneconfirm').route(app);

    require('./routes/am').route(['development', 'testing', 'rc'].indexOf(app.get('env')) !== -1)(app);

    require('./routes/family').route(app);

    // debug
    if (['development', 'testing', 'rc'].indexOf(app.get('env')) !== -1) {
        require('./routes/common/debug').route(app);
        require('./routes/webauthn').route(app);
        app.use('/auth/smarttv-gateway/frame', smarttvGatewayDebugIframe());
        app.use('/am/push/test-popup-resize', pushTestPopupResize);
    }
};
