const fs = require('fs');
const PLog = require('plog');
const apiSetup = require('../common/apiSetup');
const doCheck = require('../common/checkAuth/doCheck');
const avatarsApiSetup = require('../common/avatarsApiSetup');
const validateCSRF = require('../common/validateCSRF.js');
const multer = require('multer');
const config = require('../../configs/current');
const filePath = config.paths.logs.accountVerification;
const filePathPPL = config.paths.logs.accountVerificationPPL;

const {Validator} = require('express-json-validator-middleware');
const bodySchema = require('./body.schema.json');
const bodyPPLSchema = require('./body-ppl.schema.json');
const fileSchema = require('./file.schema.json');

const validator = new Validator({allErrors: true, removeAdditional: true}); // Pass in options to the Ajv instance
const {validate} = validator;

const storage = multer.diskStorage({
    destination(_req, _file, cb) {
        cb(null, '/tmp');
    },
    filename(_req, file, cb) {
        cb(null, `${Date.now()}-${file.originalname}`);
    }
});

const fileFilter = (_, file, cb) => {
    cb(null, ['image/gif', 'image/jpeg', 'image/png'].includes(file.mimetype));
};

const upload = multer({
    storage,
    fileFilter,
    limits: {
        fileSize: 10000000,
        files: 1
    }
});

const createPostRoute = ({validatorMiddleware, processorMiddleware}) => {
    return [
        (req, res, next) =>
            doCheck(req, res)
                .then(() => next())
                .catch(next),
        (req, _res, next) => {
            req._controller
                .getAuth()
                .sessionID({
                    multisession: 'yes',
                    full_info: 'yes',
                    get_family_info: 'yes',
                    get_public_id: 'yes',
                    get_public_name: 'yes',
                    emails: 'getdefault',
                    // @see https://wiki.yandex-team.ru/passport/dbmoving/#tipyatributov
                    attributes: '27,28,34,1003,1005,1011',
                    aliases: '1,5,6,13,21'
                })
                .then((sessionInfo) => {
                    const users = sessionInfo.users;
                    const defaultUid = sessionInfo.default_uid;

                    let defaultUser;

                    users.forEach(function(user) {
                        if (user.uid && user.uid.value === defaultUid) {
                            defaultUser = user;
                        }
                    });

                    req._defaultUser = defaultUser;

                    next();
                })
                .catch(next);
        },
        upload.single('attach_file'),
        validateCSRF,
        validatorMiddleware,
        (err, _req, res, next) => {
            if (err && err.validationErrors) {
                const {body = [], file = []} = err.validationErrors || {};

                const errors = [];

                if (file.length) {
                    errors.push('verification.photo');
                }
                if (body.length) {
                    body.forEach((errorInfo) => {
                        const {params = {}, dataPath = ''} = errorInfo || {};
                        const {missingProperty = ''} = params;

                        if (missingProperty.includes('social') || (dataPath.includes('social') && params.limit)) {
                            errors.push('verification.social');
                        }
                        if (
                            missingProperty.includes('default_email') ||
                            (dataPath.includes('default_email') && (params.limit || params.format))
                        ) {
                            errors.push('verification.default_email');
                        }
                        if (
                            missingProperty === 'additional_info' ||
                            (dataPath.includes('additional_info') && params.limit)
                        ) {
                            errors.push('verification.reason');
                        }
                    });
                }

                return res.json({status: 'error', errors: errors.length ? errors : ['verification.internal']});
            }
            next(err);
        },
        apiSetup,
        avatarsApiSetup,
        (req, res, next) => {
            const {file} = req;

            req._avatarsAPI
                .put({namespace: 'account-verification', file})
                .then((result) => {
                    res.locals.avatarsData = result;
                    return next();
                })
                .catch(next);
        },
        processorMiddleware
    ];
};

const postRoute = createPostRoute({
    validatorMiddleware: validate({
        body: bodySchema,
        file: fileSchema
    }),
    processorMiddleware: (req, res) => {
        const {'group-id': groupId, imagename: documentId} = res.locals.avatarsData || {};
        const date = `${Number.parseInt(new Date().getTime() / 1000, 10)}`;
        const {
            'address-list': addressList,
            login,
            display_name: {public_name} = {},
            public_id,
            attributes: {'27': firstname, '28': lastname} = {}
        } = req._defaultUser;

        const default_email =
            Array.isArray(addressList) &&
            typeof addressList[0] === 'object' &&
            typeof addressList[0].address === 'string' &&
            addressList[0].address;

        const obj = Object.assign({}, req.body, {
            date,
            login,
            public_name,
            public_id,
            firstname,
            lastname,
            default_email,
            groupId: String(groupId),
            documentId: String(documentId)
        });

        // eslint-disable-next-line compat/compat
        const data = new Uint8Array(Buffer.from(`${JSON.stringify(obj)}\n`));

        fs.writeFile(filePath, data, {flag: 'a'}, (err) =>
            res.json(err ? {status: 'error', errors: ['verification.write']} : {status: 'ok'})
        );
    }
});

const postRoutePPL = createPostRoute({
    validatorMiddleware: validate({
        body: bodyPPLSchema,
        file: fileSchema
    }),
    processorMiddleware: (req, res) => {
        const {'group-id': groupId, imagename: documentId} = res.locals.avatarsData || {};
        const date = `${Number.parseInt(new Date().getTime() / 1000, 10)}`;
        const {
            login,
            display_name: {public_name} = {},
            public_id,
            attributes: {'27': firstname, '28': lastname} = {}
        } = req._defaultUser;
        const serpUri = req.body.serp_uri;
        const handlers = [];

        if (serpUri) {
            handlers.push(req.api.validateRetpath({retpath: serpUri}));
        }

        const require_delete = Boolean(req.body.require_delete);

        Promise.all(handlers)
            .then(function(response) {
                const validatedSerpUri = response[0] || {};
                const processed = {};

                if (validatedSerpUri.body && validatedSerpUri.body.retpath) {
                    processed.serpUri = validatedSerpUri.body.retpath;
                }

                return processed;
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('verification_serp_uri_error')
                    .write(err);
                return {};
            })
            .then(function(processed) {
                const obj = Object.assign({}, req.body, {
                    date,
                    login,
                    public_name,
                    public_id,
                    firstname,
                    lastname,
                    require_delete,
                    serp_uri: processed.serpUri || '',
                    groupId: String(groupId),
                    documentId: String(documentId)
                });

                // eslint-disable-next-line compat/compat
                const data = new Uint8Array(Buffer.from(`${JSON.stringify(obj)}\n`));

                fs.writeFile(filePathPPL, data, {flag: 'a'}, (err) =>
                    res.json(err ? {status: 'error', errors: ['verification.write']} : {status: 'ok'})
                );
            });
    }
});

exports.postVerificationForm = postRoute;
exports.postVerificationFormPPL = postRoutePPL;
