const PLog = require('plog');
const cloudApiSetup = require('./common/cloudApiSetup');
const validateCSRF = require('./common/validateCSRF');
const setupUserTicket = require('./common/userTicketsSetup');
const serviceTicketsSetup = require('./common/serviceTicketsSetup');

const IotApi = require('../lib/api/iot');
const tvm = require('../lib/tvm');

const HELP_ENABLED_KEY = 'isHelpEnabled';
const DS_HELP_DISABLED_KEY = 'do_not_use_user_logs';

const handleError = (res) => (error) =>
    res.json({status: 'error', errors: error && error[0] ? [error[0]] : ['internal']});

const fetchAliceSettings = async (req) => {
    const body = await req._cloudAPI.getAliceSettings();

    if (body.error === 'PersonalityObjectNotFoundError') {
        return {[HELP_ENABLED_KEY]: true};
    }

    if (DS_HELP_DISABLED_KEY in body) {
        return {[HELP_ENABLED_KEY]: !body[DS_HELP_DISABLED_KEY]};
    }

    return {};
};

const getSettings = (req, res) =>
    fetchAliceSettings(req)
        .then((settings) => res.json(settings))
        .catch((err) => {
            PLog.warn()
                .logId(req.logID)
                .type('aliceSettings')
                .write(err);
            res.json(err);
        });

const setHelping = (req, res) => {
    const value = req.body.value === 'true';

    return req._cloudAPI
        .saveAliceSettings({[DS_HELP_DISABLED_KEY]: !value})
        .then(() => res.json({status: 'ok', [HELP_ENABLED_KEY]: value}));
};

const fetchDevices = async (req, deviceIds) => {
    const serviceTicket = req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.IOT];

    const [{payload: {devices = []} = {}} = {}, {items: flags = []} = {}] = await Promise.all([
        new IotApi(req.logID, req.userTicket, serviceTicket).getUserInfo(),
        req._cloudAPI.getAliceDeviceFlags()
    ]);
    const _devices = devices
        .filter((device) => device && device.quasar_info && device.quasar_info.device_id)
        .map(({quasar_info: {device_id: id}, name, type, icon_url}) => {
            const flagObj = flags.find((i) => i.id === id);

            return {
                icon_url,
                id,
                name,
                type,
                allowed: Boolean(flagObj && flagObj.allowed),
                current: deviceIds.includes(id)
            };
        });

    return {
        approved: _devices.filter((device) => device.allowed),
        disapproved: _devices.filter((device) => !device.allowed)
    };
};

const getDevices = (req, res) => {
    const {deviceId} = req.body;

    return fetchDevices(req, deviceId && typeof deviceId === 'string' ? deviceId.split(',') : [])
        .then((result) => res.json({status: 'ok', ...result}))
        .catch(handleError(res));
};

const setDeviceFlagFunc = (value) => (req, res) => {
    const {deviceId} = req.body;

    return req._cloudAPI
        .setAliceDeviceFlag(deviceId, value)
        .then((result) => res.json({status: 'ok', ...result}))
        .catch(handleError(res));
};

const middleWare = [validateCSRF, setupUserTicket, serviceTicketsSetup, cloudApiSetup];

module.exports = {
    getSettings: [middleWare, getSettings],
    setHelping: [middleWare, setHelping],
    getDevices: [middleWare, getDevices],
    setDeviceApproveStatus: [middleWare, setDeviceFlagFunc(true)],
    revokeDeviceAccess: [middleWare, setDeviceFlagFunc(false)]
};
