var apiSetup = require('./common/apiSetup');
var url = require('url');
var config = require('../configs/current');
var langs = config.langs;
var locs = require('../loc/auth.json');
var PLog = require('plog');
var _ = require('lodash');
var getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
var getUatraitsData = require('./common/getUatraitsData');
var isUnsupportedBro = require('./common/isUnsupportedBro');

var langSetup = [
    function(req, res, next) {
        if (!req.blackbox && req.body['language']) {
            req.blackbox = {
                dbfields: {
                    'userinfo.lang.uid': req.body['language']
                }
            };
        }
        next();
    },
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('challenges')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    function(req, res, next) {
        var language = langs[langs.indexOf(res.locals.language)] || 'ru';

        res.loc = locs[language];

        if (req.api) {
            req.api.language(language);
        }

        next();
    }
];

exports.routes = {};

exports.route = function(app) {
    var routes = this.routes;

    app.get('/auth/challenges/?', routes.main);
    app.post('/auth/challenges/?', routes.commit_challenge);
};

exports.routes.main = [
    apiSetup,
    langSetup,
    getYaExperimentsFlags,
    getUatraitsData,
    function redirectToNewChallenges(req, res, next) {
        var ua = res.locals.ua;

        if (!isUnsupportedBro(ua)) {
            return req._controller.redirectToLocalUrl({
                pathname: '/auth/challenge',
                query: req.query
            });
        }

        return next();
    },
    setHelpLink,
    initChallenges,
    function(_req, res) {
        res.render(`auth.challenges.${res.locals.language}.js`);
    }
];

exports.routes.commit_challenge = [
    apiSetup,
    langSetup,
    setHelpLink,
    initChallenges,
    function(req, res, next) {
        if (!res._challenge_number) {
            return next();
        }

        if (typeof req.body.sendCode !== 'undefined') {
            req.api
                .confirmPhoneSubmit({
                    display_language: res.locals.language,
                    number: res._challenge_number,
                    track_id: req.query.track_id
                })
                .then(() => {
                    res.locals.codeSent = true;
                    res.render(`auth.challenges.${res.locals.language}.js`);
                    return;
                })
                .catch(function(error) {
                    processError(error, req, res, next);
                    return;
                });
        } else {
            return next();
        }
    },
    function(req, res, next) {
        if (!res._challenge_number || typeof req.body.answer !== 'string') {
            return next();
        }

        req.api
            .confirmPhoneCommit(req.body.answer)
            .then((result) => {
                if (result.status === 'error') {
                    return processError(result.errors, req, res, next);
                }

                return next();
            })
            .catch(function(error) {
                processError(error, req, res, next);
            });
    },
    function(req, res, next) {
        var reqBody = req.body;

        if (reqBody) {
            var {track_id, challenge, answer} = reqBody;
            var data = {
                track_id,
                challenge,
                answer
                // canSendSMS: false
            };

            if (challenge !== 'phone_confirmation' && !answer) {
                return processError(['challenge.not_passed'], req, res);
            }

            req.api
                .challengesCommit(data)
                .then(function(result) {
                    if (result.body.status === 'ok') {
                        return next();
                    }
                })
                .catch(function(error) {
                    processError(error, req, res, next);
                });
        } else {
            return processError(['exception.unhandled'], req, res);
        }
    },
    function(req, res, next) {
        const controller = req._controller;

        req.api
            .bundleSession({
                track_id: req.body.track_id,
                retpath: req.body.retpath
            })
            .then(function(result) {
                var body = result.body;

                if (body.status === 'error') {
                    throw body.errors;
                }

                controller.augmentResponse(body);

                if (body.cookies) {
                    return res.redirect(
                        url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname: '/auth/finish',
                            query: {
                                track_id: body.track_id,
                                retpath: body.retpath
                            }
                        })
                    );
                }

                return res.redirect(body.retpath);
            })
            .catch(function(error) {
                processError(error, req, res, next);
            });
    }
];

function setHelpLink(req, res, next) {
    if (res.locals && ['ru', 'com', 'ua'].indexOf(res.locals.domain) > -1) {
        res.locals.helpLinkType = 'auth_challenges';
    }

    return next();
}

function initChallenges(req, res, next) {
    var track_id = req.nquery && req.nquery.track_id;

    if (track_id) {
        req.api
            .challengesSubmit({
                track_id
            })
            .then(function(result) {
                var body = result.body;
                var enabledChallenges = body.challenges_enabled;
                var challenge = {
                    default: body.default_challenge,
                    enabled_challenges: enabledChallenges,
                    phone_hint: enabledChallenges.phone ? enabledChallenges.phone.hint : '',
                    phone_confirmation_hint: enabledChallenges.phone_confirmation
                        ? enabledChallenges.phone_confirmation.hint
                        : '',
                    email_hint: enabledChallenges.email ? enabledChallenges.email.hint : '',
                    question_hint: enabledChallenges.question ? enabledChallenges.question.hint : ''
                };

                res._challenge_number =
                    (challenge.default === 'phone_confirmation' || challenge.default === 'push_2fa') &&
                    (challenge.enabled_challenges.phone_confirmation || {}).phone_number;

                res.locals.challenges = challenge;
                res.locals.track_id = body.track_id;

                next();
            })
            .catch(function(error) {
                processError(error, req, res, next);
            });
    } else {
        return res.redirect(
            url.format({
                protocol: req.headers['x-real-scheme'],
                host: req.hostname,
                pathname: '/auth/'
            })
        );
    }
}

function processError(error, req, res) {
    var challenge = req.body.challenge;
    var logger = new PLog(req.logID, 'passport', 'auth.challenges', 'processError');
    var basicUrlObj = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname
    };
    var errorCode;
    var redirectedErrors = [
        'track_id.empty',
        'track_id.invalid',
        'track.not_found',
        'unknowntrack',
        'account.auth_passed',
        'track.invalid_state',
        'action.impossible',
        'account.not_found',
        'account.disabled',
        'account.disabled_on_deletion',
        'account.global_logout'
    ];
    var lang = res.locals.language || 'ru';
    var loc = locs[lang];

    if (Array.isArray(error)) {
        errorCode = error[0];

        res.locals.error = errorCode;

        _.forEach(redirectedErrors, function(val) {
            if (errorCode === val) {
                logger.info('Challenges page error "%s", redirecting to auth', errorCode);
                res.redirect(
                    url.format(
                        _.extend(basicUrlObj, {
                            pathname: '/auth/'
                        })
                    )
                );
                return false;
            }
        });

        if (errorCode === 'challenge.not_passed' || errorCode === 'code.invalid') {
            logger.info('User could not pass the %s challenge', challenge);
            res.locals.isFailed = true;

            if (challenge === 'email') {
                res.locals.challenges.default = 'email';
            }
        }

        if (errorCode === 'challenge.limit_exceeded' || errorCode === 'confirmations_limit.exceeded') {
            logger.info('User exceed challenge limit');
            res.locals.error = 'limit_exceed';
        }

        if (errorCode === 'challenge.not_enabled') {
            logger.info('Challenge "%s" is not enabled for user, redirecting to self', challenge);
            return res.redirect(
                url.format(
                    _.extend(basicUrlObj, {
                        pathname: '/auth/challenges/',
                        query: {
                            track_id: res.locals.track_id
                        }
                    })
                )
            );
        }

        if (
            errorCode === 'exception.unhandled' ||
            errorCode === 'backend.yasms_failed' ||
            errorCode === 'backend.blackbox_failed'
        ) {
            res.locals.errorText = loc['Errors']['import.sms_internal_error'];
        }

        if (errorCode === 'sessionid.overflow') {
            res.locals.errorText = loc.Errors && loc.Errors['ErrorsTexts.sessionidoverflow'];
            res.locals.errorMend = loc.Mend && loc.Mend['MendTexts.sessionidoverflow'];
        }
    } else {
        res.locals.errorText = loc['Errors']['import.sms_internal_error'];
    }
    res.render(`auth.challenges.${lang}.js`);
}

exports.initChallenges = initChallenges;
exports.processError = processError;
