const express = require('express');
const PLog = require('plog');
const apiSetup = require('./common/apiSetup');
const checkAuth = require('./common/checkAuth');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const langSetup = require('./common/langSetup');
const writeStatbox = require('./common/writeStatbox');
const getMetrics = require('./common/getMetrics');
const rumCounterSetup = require('./common/rumCounterSetup');
const createState = require('./common/createState');
const validateCSRF = require('./common/validateCSRF');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');

const renderPage = (req, res) => {
    const {locals} = res;
    const {language} = locals;

    res.render(`react.auth-device.${language}.jsx`);
};

const NORMALIZE_REGEXP = / /g;

const deviceSubmit = (req, res, next) => {
    const {body = {}} = req;
    const {code} = body;
    const normalizedCode = code.replace(NORMALIZE_REGEXP, '');

    return req.api.deviceSubmit(normalizedCode).then(
        (submitResponse = {}) => {
            const {body: submitResponseBody = {}} = submitResponse;
            const {require_user_confirm: shouldShowConfirmation, client = {}} = submitResponseBody;
            const {title} = client;

            if (shouldShowConfirmation) {
                return res.json({
                    status: 'ok',
                    shouldShowConfirmation,
                    title
                });
            }

            return next();
        },
        (submitError = []) => res.json({status: 'error', errors: submitError})
    );
};

const deviceCommit = (req, res) => {
    const {body = {}} = req;
    const {code} = body;
    const normalizedCode = code.replace(NORMALIZE_REGEXP, '');

    return req.api.deviceCommit(normalizedCode).then(
        () => res.json({status: 'ok'}),
        () => res.json({status: 'error'})
    );
};

module.exports = express
    .Router()
    .get('/', [
        checkAuth,
        apiSetup,
        writeStatbox({
            action: 'opened',
            mode: 'auth_device'
        }),
        langSetup,
        getYaExperimentsFlags,
        multiAuthAccountsSetup,
        (req, res, next) => {
            req.api
                .profileGetState()
                .then((response = {}) => {
                    const {body = {}} = response;
                    const {account = {}} = body;

                    res.locals.account = account;

                    return next();
                })
                .catch((error) => {
                    PLog.warn()
                        .logId(req.logID)
                        .type('auth-device, getAccountInfo')
                        .write(error);
                    return next();
                });
        },
        createState,
        getMetrics({
            header: 'Страница ввода кода устройства'
        }),
        rumCounterSetup,
        renderPage,
        (error, req, res) => {
            PLog.warn()
                .logId(req.logID)
                .type('auth-device error')
                .write(error);

            if (!res.locals.store) {
                res.locals.store = {};
            }

            res.locals.store.error = error.message;

            return renderPage(req, res);
        }
    ])
    .post('/submit', [
        apiSetup,
        validateCSRF,
        writeStatbox({
            action: 'submit',
            mode: 'auth_device'
        }),
        deviceSubmit,
        writeStatbox({
            action: 'commit',
            mode: 'auth_device'
        }),
        deviceCommit
    ])
    .post('/commit', [
        apiSetup,
        validateCSRF,
        writeStatbox({
            action: 'commit',
            mode: 'auth_device'
        }),
        deviceCommit
    ]);
