var isIntranet = process.env.INTRANET === 'intranet';
var url = require('url');
var _ = require('lodash');
var apiSetup = require('./common/apiSetup');
var multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
var socialSetup = require('./common/socialSetup');
var getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
var errors = require('../lib/passport-errors');
var locs = require('../loc/auth.json');
var config = require('../configs/current');
var util = require('util');
var when = require('when');
var paths = config.paths;
var langs = config.langs;
var locMonths = require('../loc/months.json');
var PLog = require('plog');
var metricsRoute = require('./metrics.js');
var yakeyUrls = require('../lib/yakey-urls');
var DiskApi = require('../lib/api/disk');
var retpathRoutes = require('./retpath');
var getMDAOptions = require('./common/getMDAOptions.js');
var urlParse = require('./common/urlFormat').urlParse;
const processResponse = require('./authSso/processResponse');

const STAFF_LINK = 'https://doc.yandex-team.ru/help/diy/common/auth/change-password-staff.html#change-password-staff';
const YATEAM_LINK = 'https://passport.yandex-team.ru/profile';

const originsFromWebview = ['disk_app', 'telemost_app', 'telemost_chat_app', 'autofill'];

/* jshint -W079 */
var i18n = {};

langs.forEach(function(lang) {
    i18n[lang] = require('vm').runInNewContext(
        require('fs').readFileSync(`./lib/tanker/tanker.dynamic.${lang}.js`, 'utf-8')
    );
});

var getControls = (function() {
    var controls = {};

    return function(lang) {
        lang = langs[langs.indexOf(lang)] || 'ru';

        if (lang in controls) {
            return _.clone(controls[lang], true);
        }

        controls[lang] = require(`../lib/passport-form/template.${lang}.js`);
        return getControls(lang);
    };
})();

function modeFilter(req, res, next) {
    var mode = req.body.mode || req.nquery.mode;

    if (!mode) {
        return next();
    }

    if (mode && typeof mode !== 'string') {
        if (util.isArray(mode)) {
            mode = mode[0];
        } else {
            mode = '';
        }
    }

    if (['auth', 'mdauth', 'mauth', 'login', 'loginform'].indexOf(mode) > -1) {
        return next();
    } else {
        return next('route');
    }
}

function prepareStatData(req, res, next) {
    var controller = req._controller;
    var track_id = (req.nquery && req.nquery.track_id) || res.locals.track_id;
    var localsAccounts = res.locals.accounts;
    var experiments = res.locals.experiments;
    var log = {
        mode: 'auth',
        url: req.url
    };

    if (track_id) {
        log.track_id = track_id;
    }

    if (experiments && experiments.flagsString) {
        log.experiment_flags = experiments.flagsString;
    }

    if (localsAccounts && localsAccounts.accounts.length) {
        log.mode = 'ma_domik_opened';

        if (req.nquery && 'add-user' === req.nquery.mode) {
            log.mode += '_with_mode_add-user';
        }

        if (localsAccounts.defaultAccount && String(localsAccounts.defaultAccount.status) !== 'VALID') {
            log.referer = null;
            log.mode += `_error_${String(localsAccounts.defaultAccount.status).toLowerCase()}`;
        }
    }

    controller.prepareStatData(log);
    return next();
}

function elementyFilter(req, res, next) {
    if (req.nquery && req.nquery.domik === '1') {
        return next();
    } else {
        return next('route');
    }
}

exports.routes = {};

exports.route = function(app) {
    var routes = this.routes;
    var pddRoute = '/for/:pdd_domain*';

    // PASSP-9459 — Для Элементов нужно сделать показ домика на /auth при
    // наличии валидной куки Session_id и пришедшего логина
    app.get('/auth/?', elementyFilter, routes.retpath);
    app.get('/auth/?', elementyFilter, routes.enter);
    app.get(pddRoute, modeFilter, elementyFilter, routes.enter);

    // GET ?mode=auth дефолтный
    app.get('/passport/' + '?', modeFilter, routes.retpath);
    app.get('/passport/' + '?', modeFilter, routes.enter);

    // POST ?mode=auth дефолтный
    app.post('/passport/' + '?', modeFilter, routes.submit);

    app.all('/for/:pdd_domain/finish/?', routes.finish);

    app.get(
        pddRoute,
        function(req, res, next) {
            if (req.nquery && req.nquery.track_id) {
                next();
            } else {
                next('route');
            }
        },
        routes.tracksubmit
    );

    app.get(pddRoute, modeFilter, routes.retpath);
    app.get(pddRoute, modeFilter, routes.enter);
    app.post(pddRoute, modeFilter, routes.submit);

    app.all('/auth/finish/?', routes.finish);

    app.get(
        '/auth/?',
        function(req, res, next) {
            if (req.nquery && req.nquery.track_id) {
                next();
            } else {
                next('route');
            }
        },
        routes.tracksubmit
    );

    app.get(
        '/auth/confirm/?',
        function(req, res, next) {
            req.needConfirm = true;
            if (req.nquery && req.nquery.track_id) {
                res.isNumberUnavailable = true;
                next();
            } else {
                next('route');
            }
        },
        routes.tracksubmit
    );

    app.post('/auth/confirm/?', function(req, res, next) {
        req.needConfirm = true;
        return next('route');
    });

    app.get('/auth/confirm/?', routes.enter);
    app.post('/auth/confirm/?', routes.submit);

    app.get('/auth/?', routes.retpath);
    app.get('/auth/?', routes.enter);
    app.post('/auth/?', routes.submit);
};

/* {{{ AuthView */
function AuthView(name, language) {
    language = langs[langs.indexOf(language)] || 'ru';

    var loc = locs[language];
    var controls = getControls(language);
    var months = (locMonths[language] && locMonths[language]['months']) || locMonths.ru.months;
    var common = {
        pdd_title: loc.common['serviceTitles.DomainFor']
    };
    var roster = {
        submit: {
            controls: ['captcha', 'key'],
            process: enableCaptcha,
            view: 'auth.enter'
        },
        magic: {
            controls: ['captcha', 'key'],
            process: enableCaptcha,
            view: 'auth.magic'
        },
        password_required: {
            controls: ['captcha', 'key'],
            process: processSecure,
            view: 'auth.password_required'
        },
        confirm: {
            data: {
                loc: loc['Passwd'],
                cant_auto_restore: loc['Passwd']['changepassword_cant_auto_restore'],
                goto_support: loc['Passwd']['password_goto_support']
            },
            controls: [
                'confirm-method',
                'password',
                'password-confirm',
                'captcha',
                'key',
                'login',
                'phone-confirm',
                'history-question',
                'history-answer',
                'continue'
            ],
            process: processConfirm,
            view: 'auth.confirm'
        },
        message: {
            view: 'auth.message'
        },
        promo: {
            view: 'auth.promo'
        },
        change_password: {
            data: {
                loc: loc['Passwd']
            },
            controls: ['password', 'password-confirm', 'current-password', 'captcha', 'key', 'login', 'phone-confirm'],
            process: processPassword,
            view: 'auth.change_password'
        },
        complete_autoregistered: {
            data: {
                loc: loc['AdmRegSetPass'],
                btRegister: loc['MDFillInfo'].register
            },
            controls: [
                'password',
                'password-confirm',
                'question',
                'user-question',
                'answer',
                'phone-confirm',
                'eula',
                'restore-method',
                'login',
                'captcha',
                'key'
            ],
            process: processPDD,
            view: 'auth.complete_autoregistered'
        },
        complete_pdd: {
            data: {
                loc: loc['MDFillInfo'],
                months
            },
            controls: [
                'firstname',
                'lastname',
                'question',
                'user-question',
                'answer',
                'gender',
                'birthday',
                'captcha',
                'key',
                'eula'
            ],
            process: processPDD,
            view: 'auth.complete_pdd'
        },
        complete_federal: {
            data: {
                loc: loc['MDFillInfo'],
                hideTexts: true
            },
            controls: ['eula'],
            process: processFederal,
            view: 'auth.complete_pdd'
        },
        complete_pdd_with_password: {
            data: {
                loc: loc['MDFillInfo'],
                months
            },
            controls: [
                'firstname',
                'lastname',
                'password',
                'password-confirm',
                'question',
                'user-question',
                'answer',
                'birthday',
                'gender',
                'captcha',
                'key',
                'eula',
                'login'
            ],
            process: processPDD,
            view: 'auth.complete_pdd'
        },
        password_change_forbidden: {
            process(next) {
                var data = this.data;

                data.messages = [loc['Passwd']['denied']];
                data.retpath = '/passport?mode=passport';
                next();
            },
            view: 'auth.message'
        },
        get_state: {
            controls: ['captcha', 'key'],
            process: enableCaptcha,
            view: 'auth.enter'
        },
        complete_social: {
            controls: ['captcha', 'key'],
            process: processSecure,
            view: 'auth.password_required'
            // process: function(next) {
            // var data = this.data;

            // data.title = loc['Errors']['Common.choose_new_passwd'];
            // data.messages = [loc['Errors']['account_without_password']];
            // data.retpath = null;

            // next();
            // },
            // view: 'auth.message'
        },
        goto_support: {
            process(next) {
                var data = this.data;
                var req = this.req;
                var track_id = req.api.track();
                var login = req.authResult && req.authResult.account && req.authResult.account.login;
                var message = loc['Passwd']['changepassword_cant_auto_restore'];
                var cant_auto_restore_link = url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: '/restoration/fallback',
                    query: {
                        track_id
                    }
                });

                var reasons = {
                    account_hacked: 'fishing',
                    password_weak: 'IsPasswordSimple',
                    password_expired: 'expiredpw',
                    account_hacked_phone: 'account_hacked_phone',
                    account_hacked_no_phone: 'account_hacked_no_phone'
                };

                var reason = (req.authResult && req.authResult.change_password_reason) || req.body['reason'];

                if (reason) {
                    message = (loc['Passwd'][reasons[reason]] || '') + message;
                }

                message = message
                    .replace('%login_first_letter%', login.slice(0, 1))
                    .replace('%login_last_part%', login.slice(1))
                    .replace('%SUPPORT_LINK_BEGIN%', `<a href="${cant_auto_restore_link}">`)
                    .replace('%SUPPORT_LINK_END%', '</a>');
                data.title = loc['Passwd']['title'];
                data.messages = [message];
                data.retpath = null;

                next();
            },
            view: 'auth.message'
        }
    };

    if (!roster[name]) {
        throw new Error(`There is no place like ${name}`);
    }

    function processSecure(next) {
        var data = this.data;
        var req = this.req;
        var res = this.req.res;

        if (
            req.authResult &&
            req.authResult.account &&
            (req.authResult.account.display_login || req.authResult.account.display_name)
        ) {
            data.display_name = req.authResult.account.display_login || req.authResult.account.display_name.name;
        }

        if (req.authResult && req.authResult.state === 'complete_social') {
            data.hide_password = true;
        }

        if (req.authResult && req.authResult.social_providers && res.locals.social) {
            var old_social = res.locals.social;

            delete res.locals.social;
            res.locals.social = {
                providers: null
            };

            res.locals.social.providers = _.filter(old_social.providers, function(provider) {
                return req.authResult.social_providers.indexOf(provider.data.name) !== -1;
            });
        }

        data.secure = true;
        enableCaptcha.call(this, next);
    }

    function processFederal(next) {
        return next();
    }

    function processPDD(next) {
        var data = this.data;
        var req = this.req;
        var self = this;
        var account = req.authResult && req.authResult.account;
        var person = account && account.person;
        var hint = account && account.hint;
        var birthday;

        data.account = account;

        if (person && person.birthday) {
            birthday = person.birthday.split('-');
            data.bday = parseInt(birthday[2], 10) > 0 ? birthday[2] : '';
            data.bmonth = parseInt(birthday[1], 10) > 0 ? birthday[1] : '13';
            data.byear = parseInt(birthday[0], 10) > 0 ? birthday[0] : '';
        }

        if (data.form && data.form.control) {
            if (account && account.is_workspace_user) {
                _.remove(data.form.control, function(control) {
                    return ['password', 'password-confirm', 'eula'].indexOf(control.id) === -1;
                });
            }

            _.forEach(data.form.control, function(control) {
                if (person) {
                    if (control.name === 'lastname') {
                        control.value = person.lastname;
                        return;
                    }

                    if (control.name === 'firstname') {
                        control.value = person.firstname;
                        return;
                    }

                    if (control.name === 'gender') {
                        if (person.gender && person.gender !== 0) {
                            control.value = person.gender === 1 ? 'male' : 'female';
                        }
                        return;
                    }

                    if (control.name === 'birthday' && data.bmonth) {
                        control.value = {
                            day: data.bday,
                            month: data.bmonth,
                            year: data.byear
                        };
                        return;
                    }
                }

                if (hint) {
                    if (control.name === 'hint_answer') {
                        control.value = hint.answer;
                        return;
                    }

                    if (control.name === 'hint_question_id') {
                        control.value = hint.question && hint.question.id;
                        return;
                    }

                    if (control.name === 'hint_question') {
                        control.value = hint.question && hint.question.text;
                        return;
                    }
                }

                if (
                    control.id === 'eula' &&
                    (self.name === 'complete_pdd' || self.name === 'complete_pdd_with_password')
                ) {
                    if (account && account.is_workspace_user) {
                        control.isWorkspace = true;
                    } else {
                        control.isExtended = true;
                    }
                }

                if (control.id === 'phone-confirm') {
                    // пробрасываем в контест телефон пользователя
                    savePhoneConfirmState(control, req);
                }
            });
        }

        req.api.getQuestions().then(function(result) {
            data.questions = result.body.questions;
            if (
                self.name === 'complete_autoregistered' ||
                self.name === 'complete_pdd_with_password' ||
                self.name === 'complete_pdd'
            ) {
                self.req.captcha_required = true;
                return enableCaptcha.call(self, next);
            } else {
                return next();
            }
        });
    }

    function processPassword(next) {
        var self = this;
        var data = this.data;
        var req = this.req;
        var reasons = {
            account_hacked: 'fishing',
            password_flushed: 'password_flushed',
            password_weak: 'password_expired', // Специально такой же, как passwor_expired, это не опечатка
            password_expired: 'password_expired',
            account_hacked_phone: 'account_hacked_phone',
            account_hacked_no_phone: 'account_hacked_no_phone'
        };

        var reason = (req.authResult && req.authResult.change_password_reason) || req.body['reason'];
        var validation_method = req.authResult && req.authResult.validation_method;

        if (req.res && req.res.locals && ['ru', 'com', 'ua'].indexOf(req.res.locals.domain) > -1) {
            req.res.locals.helpLinkType = 'change_password_form';
        }

        if (req.res && req.res.locals && ['com.tr'].indexOf(req.res.locals.domain) > -1) {
            req.res.locals.helpLinkType = 'change_password';
        }

        if (reason) {
            data.reason = reason;
            data.message = loc.Passwd[reasons[reason]] || '';
        }

        data.title = data.loc.title;

        if (reason === 'password_expired') {
            data.title = loc.Passwd.title_expired;

            if (isIntranet) {
                data.isIntranet = true;
                data.message = loc.Passwd.password_expired_intranet;
                data.brickWarning = loc.Profile2['intranet.disclaimer'].replace('%href%', STAFF_LINK);
                req.res.locals.serviceUrl = YATEAM_LINK;
            }
        }

        if (reason === 'password_weak') {
            data.title = loc.Passwd.title_weak;
        }

        if (validation_method === 'captcha' || validation_method === 'captcha_and_phone') {
            req.captcha_required = true;
        }

        if (data.form && data.form.control) {
            if (!isIntranet) {
                _.remove(data.form.control, {id: 'current-password'});
            }

            if (reason === 'password_weak' || reason === 'password_expired') {
                data.form.control = _.forEach(data.form.control, function(item) {
                    if (item.id === 'password') {
                        if (reason === 'password_weak') {
                            item.label = loc.Frontend.field_password_createstrong;
                        } else {
                            item.label = loc.Frontend.field_password_createnew;
                        }
                    }
                });
            }

            if (validation_method === 'captcha_and_phone') {
                data.form.control = _.forEach(data.form.control, function(item) {
                    if (item.id === 'phone-confirm') {
                        // пробрасываем в контест телефон пользователя
                        savePhoneConfirmState(item, req);

                        if (item.value) {
                            data.reason = 'account_hacked_phone';
                        } else {
                            data.reason = 'account_hacked_no_phone';
                        }

                        data.message = loc['Passwd'][reasons[data.reason]] || '';

                        var tld = /.*.yandex\.(.*)$/.exec(req.hostname);

                        tld = (tld && tld[1]) || 'ru';
                        var feedbackLink = util.format(
                            '<a href="%s" target="_blank">',
                            loc['Passwd']['feedback_page_url'].replace('%TLD%', tld)
                        );

                        data.message = data.message.replace('%link_end%', '</a>').replace('%link_begin%', feedbackLink);
                    }
                });
            } else {
                _.remove(data.form.control, {id: 'phone-confirm'});
            }
        }

        return enableCaptcha.call(self, next);
    }

    function processConfirm(next) {
        var self = this;
        var data = this.data;
        var control = data.form && data.form.control;
        var req = this.req;
        var cant = data.cant_auto_restore;
        var goto_support = data.goto_support;
        var cant_auto_restore_link = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/restoration/fallback',
            query: {
                track_id: req.api.track()
            }
        });
        var login = req.authResult && req.authResult.account && req.authResult.account.login;
        var control_login;

        data.cant_auto_restore = cant
            .replace('%login_first_letter%', login.slice(0, 1))
            .replace('%login_last_part%', login.slice(1))
            .replace('%SUPPORT_LINK_BEGIN%', `<a href="${cant_auto_restore_link}">`)
            .replace('%SUPPORT_LINK_END%', '</a>');
        data.goto_support = goto_support
            .replace('%SUPPORT_LINK_BEGIN%', `<a href="${cant_auto_restore_link}">`)
            .replace('%SUPPORT_LINK_END%', '</a>');
        data.cant_auto_restore_link = cant_auto_restore_link;

        if (control) {
            control_login = _.find(control, {id: 'login'});
            control_login.value = login;
        }

        return processPassword.call(self, next);
    }

    function enableCaptcha(next) {
        var data = this.data;
        var captchaOpts = {
            asyncCheck: false
        };
        var req = this.req;
        var res = req.res;
        var userCountry =
            (req.authResult &&
                req.authResult.account &&
                req.authResult.account.person &&
                req.authResult.account.person.country) ||
            null;

        var expFlags = res && res.locals.experiments && res.locals.experiments.flags;
        var isOCRExp = expFlags && expFlags.includes('pwd_ocr_exp');

        if (userCountry) {
            data['country'] = userCountry;
        }

        if (this.name === 'change_password' || this.name === 'confirm') {
            if (isOCRExp) {
                captchaOpts.ocr = true;
            }

            captchaOpts.asyncCheck = true;
        }

        var captchaField = _.find(data.form.control, function(cntrl) {
            return cntrl.id === 'captcha';
        });

        if (captchaField) {
            var whitelist = config.audioCaptcha.whitelist;
            var enableAudioCaptcha = true;

            if (userCountry && whitelist.indexOf(userCountry.toUpperCase()) === -1) {
                enableAudioCaptcha = false;
            }

            data.form.control.forEach(function(cntrl) {
                if (cntrl.id === 'captcha') {
                    cntrl.countryFromAudioWhiteList = enableAudioCaptcha;
                }
            });

            captchaField.options = captchaOpts;
        }

        if (req.captcha_required) {
            data.captchaRequired = true;
            var getCaptcha = req.api.captchaGenerate;

            if (req.body.captcha_mode === 'audio') {
                getCaptcha = req.api.audioCaptchaGenerate;
            }
            when.reduce([getCaptcha.call(req.api, captchaOpts)], reduceAnswer, {}).then(
                function(results) {
                    if (results.captcha) {
                        results.captcha.mode = req.body.captcha_mode === 'audio' ? 'audio' : 'text';
                        data.captcha = results.captcha;

                        if (results.captcha.voice) {
                            data.captcha.voice_url = results.captcha.voice.url;
                            data.captcha.voice_intro_url = results.captcha.voice.intro_url;
                        }
                    }

                    next();
                },
                function(error) {
                    next(error);
                }
            );
        } else {
            data.form.control = _.filter(data.form.control, function(cntrl) {
                return !(cntrl.id === 'captcha' || cntrl.id === 'key');
            });
            next();
        }
    }

    function defaultProcess(next) {
        var data = this.data;
        var aliases = {
            login: {
                name: 'login'
            },
            hint_question_id: {
                name: 'question_id',
                missingvalue: 'invalid'
            },
            hint_question: {
                name: 'question',
                missingvalue: 'empty',
                process(data, control) {
                    var question_id;

                    if (!control.value) {
                        return;
                    }

                    if (data.form && data.form.control) {
                        question_id = _.filter(data.form.control, {id: 'question'})[0];

                        if (question_id) {
                            question_id.value = '99';
                        }
                    }
                }
            },
            hint_answer: {
                name: 'answer',
                missingvalue: 'empty'
            },
            password: {
                name: 'password',
                prohibitedsymbols: 'invalid',
                tooshort: 'short',
                toolong: 'long',
                missingvalue: 'empty',
                likeoldpassword: 'equals_previous',
                likelogin: 'likelogin',
                foundinhistory: 'found_in_history'
            },
            firstname: {
                name: 'firstname',
                missingvalue: 'empty'
            },
            lastname: {
                name: 'lastname',
                missingvalue: 'empty'
            },
            captcha: {
                name: 'captcha'
            },
            answer: {
                name: 'captcha'
            },
            phone_number: {
                name: 'phone',
                needsconfirmation: 'required'
            },
            birthday: {
                name: 'birthday',
                process(data) {
                    var value = data.fields && data.fields.birthday;
                    var splitted;

                    if (!value) {
                        return;
                    }

                    splitted = value.split('-');
                    data.byear = splitted[0];
                    data.bmonth = splitted[1];
                    data.bday = splitted[2];
                }
            },
            'restore-method': {
                name: 'restore-method',
                process(data, control) {
                    var validation_method = data.fields && data.fields.validation_method;

                    control.state = validation_method === 'captcha' ? 'question' : 'phone';
                }
            }
        };

        function load() {
            if (data.rawErrors || data.fields) {
                if (data.form && data.form.control) {
                    _.forEach(data.form.control, function(control) {
                        var alias = aliases[control.name];
                        var controlName = (alias && alias['name']) || control.name;

                        // Ошибки
                        if (data.rawErrors) {
                            _.forEach(control.error, function(error) {
                                var errorCode = (alias && alias[error.code]) || error.code;

                                error.active = false;

                                _.forEach(data.rawErrors, function(code) {
                                    if (code.match('phone_secure')) {
                                        code = code.replace('phone_secure', 'phone');
                                    }

                                    if (code === util.format('%s.%s', controlName, errorCode)) {
                                        error.active = true;
                                    }
                                });
                            });
                        }

                        // Предзаполненные
                        if (data.fields) {
                            if (controlName && data.fields[controlName]) {
                                control.value = data.fields[controlName];
                            }

                            if (alias && alias['process']) {
                                alias['process'](data, control);
                            }
                        }
                    });
                }
            }

            return next();
        }

        if (this.process) {
            this.process(load);
        } else {
            load();
        }
    }

    this.view = `${roster[name].view}.${language}.js`;
    this.data = _.merge({}, common);
    if (roster[name].data) {
        this.data = _.merge({}, this.data, roster[name].data);
    }

    if (roster[name].controls) {
        this.data.form = {
            control: _.filter(controls, function(cntrl) {
                return roster[name].controls.indexOf(cntrl.id) !== -1;
            })
        };
    }
    this.processData = defaultProcess;
    this.process = roster[name].process;
    this.lang = language;
    this.name = name;
}

AuthView.prototype.render = function(req, res, next, data) {
    var v = this;
    var pdd_domain = req.params.pdd_domain || req.body['pdd_domain'] || (req.nquery && req.nquery['pdd_domain']);

    // TODO
    if (pdd_domain) {
        v.data.pdd_title = v.data.pdd_title.replace('%domain%', pdd_domain);
    } else {
        delete v.data.pdd_title;
    }

    if (data.fields && data.fields.password) {
        delete data.fields.password;
    }

    v.data = _.merge({}, v.data, data);
    v.req = req;
    this.processData(function(err) {
        if (err) {
            next(err);
        } else {
            res.render(v.view, v.data);
        }
    });
};

/* }}} */

/* {{{ AuthState */
function AuthState(name, req, res) {
    var pdd_domain = req.params.pdd_domain || req.body['pdd_domain'] || (req.nquery && req.nquery['pdd_domain']);
    var roster = {
        submit: {
            // handle: '/1/bundle/auth/password/submit/',
            handle: '/2/bundle/auth/password/commit_password/',
            parse() {
                var form = processInput(req, {
                    login: {},
                    key: {},
                    answer: {
                        name: 'captcha_answer'
                    },
                    passwd: {
                        name: 'password'
                    },
                    origin: {
                        any: true
                    },
                    fretpath: {
                        any: true
                    },
                    clean: {
                        any: true
                    },
                    track_id: {
                        any: true
                    },
                    retpath: {
                        any: true
                    },
                    from: {
                        name: 'service',
                        any: true
                    }
                });

                if (!(form.login && form.password) && req.method === 'POST') {
                    return false;
                }

                var twoweeks = req.body['twoweeks'] || req.query['twoweeks'];

                if (twoweeks === 'no') {
                    form.policy = 'sessional';
                } else {
                    form.policy = 'long';
                }

                form.passErrors = true;

                return form;
            },
            preProcess: checkCaptcha,
            onError: submitOnError
        },
        password_required: {
            handle: '/1/bundle/auth/password/secure/',
            parse() {
                var form = processInput(req, {
                    key: {},
                    answer: {
                        name: 'captcha_answer'
                    },
                    passwd: {
                        name: 'password'
                    },
                    track_id: {
                        any: true
                    },
                    retpath: {
                        any: true
                    },
                    uid: {
                        any: true
                    },
                    service: {
                        any: true
                    },
                    from: {
                        name: 'service',
                        any: true
                    }
                });

                if (!form.password && req.method === 'POST') {
                    return false;
                }

                form.passErrors = true;

                return form;
            },
            preProcess: checkCaptcha,
            onError: secureOnError,
            recipient(result, s, next) {
                const controller = req._controller;
                var body = result.body;

                res.locals.track_id = body.track_id || s.form.track_id;
                req.authResult = body;

                if (!req.blackbox && body && body.account && body.account.person && body.account.person.language) {
                    req.blackbox = {
                        dbfields: {
                            'userinfo.lang.uid': body.account.person.language
                        }
                    };
                }

                res.account = body.account;

                if (!body.state) {
                    // авторизуем
                    controller.augmentResponse(body);

                    // больше никаких стадий
                    delete res.state;

                    if (body.status === 'ok') {
                        res.authFinished = true;
                    }
                } else {
                    // следующая стадия
                    res.state = body.state;
                    res.processedData = s.form;
                }
                res.locals.state = res.state;

                if (body.status === 'ok') {
                    return next();
                } else {
                    s.onError(s, body.errors, next);
                }
            }
        },
        confirm: {
            handle: '/2/bundle/auth/password/change_password/',
            parse(body) {
                return {
                    track_id: body['track_id'],
                    password: body['password'],
                    login: body['login'],
                    key: body['key'],
                    captcha_answer: body['answer'],
                    phone_number: body['phone_number'],
                    retpath: body['retpath'] || (req.nquery && req.nquery['retpath']),
                    answer: body['hint_answer'],
                    history_question: body['history_question']
                };
            },
            preProcess: confirmPreProcess
        },
        change_password: {
            handle: '/2/bundle/auth/password/change_password/',
            parse(body) {
                return {
                    track_id: body['track_id'],
                    password: body['password'],
                    current_password: body['current-password'],
                    login: body['login'],
                    key: body['key'],
                    captcha_answer: body['answer'],
                    phone_number: body['phone_number'],
                    retpath: body['retpath'] || (req.nquery && req.nquery['retpath']),
                    answer: body['hint_answer'],
                    history_question: body['history_question']
                };
            },
            preProcess: checkCaptcha
        },
        complete_autoregistered: {
            handle: '/2/bundle/auth/password/complete_autoregistered/',
            parse: parseComplete,
            preProcess: completeAutoregisteredPreProcess
        },
        complete_pdd: {
            handle: '/2/bundle/auth/password/complete_pdd/',
            parse: parseComplete,
            preProcess: completePDDPreProcess
        },
        complete_federal: {
            handle: '/1/bundle/auth/sso/complete_federal/',
            parse: parseComplete,
            recipient(result, s, next) {
                const apiResponse = result.body;
                const controller = req._controller;
                const api = req.api;

                if (apiResponse.status === 'ok') {
                    return processResponse({
                        api,
                        controller,
                        trackId: apiResponse.id
                    });
                } else {
                    s.onError(s, apiResponse.errors, next);
                }
            }
        },
        complete_pdd_with_password: {
            handle: '/2/bundle/auth/password/complete_pdd/',
            parse: parseComplete,
            preProcess: completePDDPreProcess
        },
        get_state: {
            handle: '/1/bundle/auth/password/get_state/',
            parse() {
                return {
                    track_id: req.nquery && req.nquery.track_id,
                    passErrors: true
                };
            },
            onError: submitOnError
        }
    };

    function defaultOnError(s, errors, next) {
        function load() {
            // эта же стадия с ошибками
            res.state = res.locals.state = s.state;
            res.processedData = s.form;
            res.rawErrors = (Array.isArray(res.rawErrors) && res.rawErrors.concat(errors)) || errors;
            next();
        }

        s.req.api.params('track_id', true).then(function(result) {
            res.locals.track_id = result.body.id;
            load();
        }, load);
    }

    function submitOnError(s, errors, next, body) {
        var retpath;
        var query;
        var controller;
        var passwordNotMatchedIndex;

        if (errors.indexOf('account.auth_passed') !== -1) {
            controller = req._controller;
            return controller.redirectToLocalUrl({
                pathname: 'auth'
            });
        }

        if (errors.indexOf('captcha.required') === -1 && body && body.account && body.account.is_2fa_enabled) {
            controller = req._controller;
            passwordNotMatchedIndex = errors.indexOf('password.not_matched');

            if (passwordNotMatchedIndex >= 0) {
                errors[passwordNotMatchedIndex] = 'password.not_matched.2fa';
            }

            return controller
                .getUatraits()
                .then(function(uatraits) {
                    return uatraits && uatraits.isMobile;
                })
                .catch(function(err) {
                    PLog.warn()
                        .logId(req.logID)
                        .type('auth')
                        .write(err);
                })
                .done(function(isMobile) {
                    if (isMobile) {
                        return defaultOnError(s, errors, next);
                    } else {
                        retpath = req.body['retpath'] || (req.nquery && req.nquery['retpath']);
                        query = {
                            mode: 'qr',
                            reason: Date.now()
                        };

                        if (retpath) {
                            query['retpath'] = retpath;
                        }

                        return s.res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: 'auth',
                                query
                            })
                        );
                    }
                });
        }

        if (errors.indexOf('captcha.required') !== -1) {
            req.captcha_required = true;
        }

        return defaultOnError(s, errors, next);
    }

    function secureOnError(s, errors, next) {
        var tempErrors;
        var req = s.req;
        var res = s.res;
        var query;
        var retpath;
        var track_id;
        var pdd_domain;

        tempErrors = errors.filter(function(val) {
            if (val === 'captcha.required' || val === 'password.not_matched') {
                return false;
            }
            return true;
        });

        if (tempErrors.length) {
            query = {};
            track_id = req.api.track();
            retpath = req.body['retpath'] || (req.nquery && req.nquery['retpath']);
            pdd_domain = req.params.pdd_domain;

            if (retpath) {
                query.retpath = retpath;
            }

            if (track_id) {
                query.track_id = track_id;
            }

            return res.redirect(
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: pdd_domain && !config.multiauth ? `for/${pdd_domain}` : 'auth',
                    query
                })
            );
        }

        return submitOnError(s, errors, next);
    }

    function defaultPreProcess(s, next) {
        next();
    }

    function checkCaptcha(s, next) {
        var passErrors = req.isSecure || false;

        if (s.form.key && s.form.captcha_answer) {
            s.req.captcha_required = true;
            s.req.api.captchaCheckStatus(s.form).then(
                function(result) {
                    var body = (result && result.body) || null;

                    if (body && body.is_recognized && body.is_checked) {
                        s.req.captcha_required = false;
                        delete s.form.captcha_answer;
                        return next();
                    }

                    s.req.api
                        .captchaCheck({
                            key: s.form.key,
                            answer: s.form.captcha_answer
                        })
                        .then(
                            function(result) {
                                delete s.form.captcha_answer;
                                if (result.body.correct) {
                                    s.req.captcha_required = false;
                                    next();
                                } else {
                                    next(['captcha.required', 'captcha.incorrect'], passErrors);
                                }
                            },
                            function() {
                                next(['captcha.required', 'captcha.captchalocate'], passErrors);
                            }
                        );
                },
                function() {
                    next();
                }
            );
        } else {
            next();
        }
    }

    function suggestOptionalFields(s, next) {
        var form = s.form;
        var api = s.req.api;

        if (!(form.country && form.timezone)) {
            when.join(api.suggestCountry(), api.suggestTimezone()).then(
                function(values) {
                    if (values.length === 2) {
                        form['country'] = values[0].body.country[0];
                        form['timezone'] = values[1].body.timezone[0];
                    }
                    next();
                },
                function() {
                    next();
                }
            );
        } else {
            next();
        }
    }

    function completePDDPreProcess(s, next) {
        suggestOptionalFields(s, function() {
            checkCaptcha(s, next);
        });
    }

    function completeAutoregisteredPreProcess(s, next) {
        suggestOptionalFields(s, function() {
            if (s.form.validation_method === 'captcha') {
                checkCaptcha(s, next);
            } else {
                next();
            }
        });
    }

    function confirmPreProcess(s, next) {
        var form = s.form;
        var api = s.req.api;
        var data = {
            track_id: form['track_id'],
            answer: form['answer']
        };

        if (form.history_question) {
            var qa = form.history_question.split(':');

            data.question_id = qa[0] || null;
            data.question = qa[1] || null;
        }

        api.checkHistoryQuestions(data).then(
            function() {
                checkCaptcha(s, next);
            },
            function() {
                checkCaptcha(s, next);
            }
        );
    }

    function parseComplete(body) {
        function zero(num) {
            if (num.length === 2) {
                return num;
            }
            if (!parseInt(num, 10)) {
                return '00';
            }
            return parseInt(num, 10) > 9 ? num : `0${parseInt(num, 10)}`;
        }

        var parsed = {
            track_id: body['track_id'],
            firstname: body['firstname'],
            lastname: body['lastname'],
            eula_accepted: body['eula_accepted'],
            force_clean_web: true,
            display_language: req.api.language()
        };

        if (body['byear'] && body['bmonth'] !== '13' && body['bday']) {
            parsed.birthday = util.format(
                '%s-%s-%s',
                body['byear'] || '0000',
                zero(body['bmonth']),
                zero(body['bday'])
            );
        }

        if (body['password']) {
            parsed.password = body['password'];
        }

        if (body['gender']) {
            parsed.gender = body['gender'];
        }

        if (body['hint_question_id']) {
            if (body['hint_question_id'] === '99') {
                parsed.question = body['hint_question'];
            } else {
                parsed.question_id = body['hint_question_id'];
            }
        }

        if (body['hint_answer']) {
            parsed.answer = body['hint_answer'];
        }

        if (body['key']) {
            parsed.key = body['key'];
        }

        if (body['answer']) {
            parsed.captcha_answer = body['answer'];
        }

        if (body['validation_method']) {
            parsed.validation_method = body['validation_method'];
        }

        return parsed;
    }

    function apiPost(handle, form, success, error) {
        req.api.authSubmit(handle, form).then(success, error);
    }

    this.post = req.apiPost || (req.apiPost = apiPost);
    this.state = name;
    this.form = roster[name].parse(req.body);
    this.handle = roster[name].handle;
    this.onError = roster[name].onError || defaultOnError;
    this.res = res;
    this.req = req;
    this.preProcess = roster[name].preProcess || defaultPreProcess;
    this.recipient = roster[name].recipient || null;

    if (pdd_domain && this.form.login && !/@/.test(this.form.login)) {
        this.form.login += `@${pdd_domain}`;
    }

    if (pdd_domain) {
        this.form.is_pdd = '1';
    }
}

AuthState.prototype.run = function(next) {
    var s = this;
    var res = s.res;
    var req = s.req;
    var recipient = null;
    const controller = req._controller;

    if (s.recipient) {
        recipient = function(results) {
            s.recipient(results, s, next);
        };
    }

    s.preProcess(s, function(errors, passErrors) {
        if (errors && !passErrors) {
            s.onError(s, errors, next);
        } else {
            if (passErrors && errors) {
                res.rawErrors = (Array.isArray(res.rawErrors) && res.rawErrors.concat(errors)) || errors;
            }

            if (s.form) {
                s.post(
                    s.handle,
                    s.form,
                    recipient ||
                        function(result) {
                            var body = result.body;

                            res.locals.track_id = body.track_id || s.form.track_id;
                            req.authResult = body;

                            // для определения pdd
                            res.account = body.account;

                            if (!body.state) {
                                // авторизуем
                                controller.augmentResponse(body);

                                // больше никаких стадий
                                delete res.state;
                                // все ОК в случае проверки только кук
                                if (body.status === 'ok') {
                                    res.authFinished = true;
                                }

                                if (!body.cookies && body.status === 'ok') {
                                    res.readyToGetSession = true;
                                }
                            } else {
                                // следующая стадия
                                res.state = body.state;
                                res.processedData = s.form;
                            }

                            res.locals.state = res.state;

                            if (body.is_replaced_phone_with_quarantine) {
                                req.flash('info', {
                                    id: 'id1',
                                    data: {
                                        timeout: body.secure_phone_pending_until
                                    }
                                });
                            }

                            if (body.is_conflicted_operation_exists || body.is_yasms_errors_when_replacing_phone) {
                                req.flash('info', {id: 'id2'});
                            }

                            if (body.enable_2fa_track_id) {
                                req.flash('enable_2fa_track_id', {id: body.enable_2fa_track_id});
                            }

                            if (body.status === 'ok') {
                                return next();
                            } else {
                                s.onError(s, body.errors, next, body);
                            }
                        },
                    function(errors) {
                        if (errors instanceof Error) {
                            next(errors);
                        } else {
                            s.onError(s, errors, next);
                        }
                    }
                );
            } else {
                // Показываем домик, если при разборе поста каких-то нужных
                // данных не оказалось
                res.state = 'submit';
                next();
            }
        }
    });
};

/* }}} */

/* {{{ Helpers */
var TLD_RE = /.*.yandex\.(.*)$/;
var setup = [
    function(req, res, next) {
        var tld = TLD_RE.exec(req.hostname);
        var retpath = req.body['retpath'] || (req.nquery && req.nquery['retpath']);
        var openstat = req.body['_openstat'] || (req.nquery && req.nquery['_openstat']);
        var from = req.body['from'] || (req.nquery && req.nquery['from']);
        var origin = req.body['origin'] || (req.nquery && req.nquery['origin']);

        res.locals.domain = (tld && tld[1]) || 'ru';
        res.locals.static = paths.static;

        res.locals.pdd_domain =
            req.params.pdd_domain || req.body['pdd_domain'] || (req.nquery && req.nquery['pdd_domain']);

        if (retpath) {
            res.locals.retpath = retpath;
        }

        _.forEach(['restore', 'register', 'restore_login'], function(mode) {
            var link = {
                protocol: req.headers['x-real-scheme'],
                host: req.hostname,
                pathname: '/passport',
                query: {
                    mode
                }
            };

            if (retpath) {
                link.query['retpath'] = retpath;
            }

            if (openstat) {
                link.query['_openstat'] = openstat;
            }

            if (from) {
                link.query['from'] = from;
            }

            if (mode === 'register') {
                link.query['origin'] = origin || 'passport_auth2reg';
                link.pathname = 'registration';
            }

            if (mode === 'restore_login') {
                link.query.origin = origin || 'passport_auth2reg';
                link.pathname = '/restoration/select';

                delete link.query.mode;
            }

            res.locals[`link_${mode}`] = url.format(link);
        });

        // Автозаполнение логина в домике из GET-параметра
        if (req.nquery && req.nquery['login'] && !(res.locals.fields && res.locals.fields.login)) {
            res.locals.fields = {
                login: req.nquery['login']
            };
        }

        if (!res.locals.metrics_id) {
            var Metrics = metricsRoute.Metrics;
            var m = new Metrics(config.metrics);

            res.locals.metrics_id = m.getCounterID('/auth');
        }

        next();
    }
];

var flashSetup = [
    require('cookie-session')({
        name: 'pf',
        keys: require('keygrip')(['bihiejkgeghofcpabilfdbagjjebpnkj'], 'sha256')
    }),
    require('connect-flash')()
];

function resign(req, res, next) {
    if (!res.locals.accounts) {
        return next();
    }

    var controller = req._controller;

    controller
        .getAuth()
        .loggedIn()
        .then(function() {
            next();
        })
        .catch(function(err) {
            if (err && err.code !== 'need_resign') {
                next();
            }
        });
}

function langSetup(req, res, next) {
    var controller = req._controller;

    controller
        .getLanguage()
        .then(function(lang) {
            res.locals.language = lang;
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('auth')
                .write(err);
        })
        .done(function() {
            if (req.api) {
                req.api.language(res.locals.language);
            }

            return next();
        });
}

function processErrors(errors, lang, data) {
    var loc = locs[lang || 'ru'];
    var resultErrors = [];
    var login = (data && data.login) || '';
    var errIds = {
        'password.change_forbidden': 'ErrorsTexts.password_change_forbidden',
        'password.not_matched': 'ErrorsTexts.badlog',
        'password.not_matched.2fa': 'ErrorsTexts.badlog_2fa',
        'account.disabled': 'ErrorsTexts.badlog_blocked',
        'account.not_found': 'ErrorsTexts.deleted',
        'account.disabled_on_deletion': 'ErrorsTexts.deleted',
        'account.disabled_with_money': 'ErrorsTexts.deleted',
        'account.registration_limited': 'import.sms_limit_exceeded',
        internal: 'ErrorsTexts.internal',
        retry: 'ErrorsTexts.retry',
        'backend.blackbox_failed': 'ErrorsTexts.auth_try_again',
        'backend.yasms_failed': 'ErrorsTexts.auth_try_again',
        'backend.database_failed': 'ErrorsTexts.auth_try_again',
        'backend.redis_failed': 'ErrorsTexts.auth_try_again',
        'login.invalid_format': 'ErrorsTexts.nodomain',
        'domain.not_hosted': 'ErrorsTexts.domainnotfound',
        'user.not_verified': 'field_human-confirmation_errors_unconfirmed',
        'sessionid.overflow': 'ErrorsTexts.sessionidoverflow',
        'account.compromised': 'ErrorsTexts.account_compromised',
        'session.invalid': 'session_invalid',
        'account.global_logout': 'restart_auth'
    };

    var mendIds = {
        'password.change_forbidden': 'MendTexts.password_change_forbidden',
        'password.not_matched': 'creg_misspasswd',
        'password.not_matched.2fa': 'creg_misspasswd_2fa',
        'account.disabled': 'disabled',
        'account.not_found': 'deleted',
        'account.disabled_on_deletion': 'deleted',
        'account.disabled_with_money': 'deleted',
        internal: 'crap',
        'session.invalid': 'session_invalid_mend',
        'sessionid.overflow': 'MendTexts.sessionidoverflow',
        'account.global_logout': 'restart_auth_process'
    };

    var skip = [
        'sessionid.invalid',
        'service.invalid',
        'password.long',
        'password.short',
        'password.weak',
        'password.prohibitedsymbols',
        'password.likelogin',
        'password.equals_previous',
        'password.found_in_history',
        'firstname.invalid',
        'lastname.invalid',
        'answer.long',
        'question.long',
        'question.inconsistent',
        'captcha.required',
        'captcha.incorrect',
        'captcha.captchalocate',
        'account.auth_passed',
        'phone.required',
        'phone_secure.bound_and_confirmed',
        'password.likephonenumber'
    ];

    var internal = [
        'auth.error',
        'ip.empty',
        'host.empty',
        'scheme.empty',
        'cookie.empty',
        'useragent.empty',
        'exception.unhandled',
        'track.not_found',
        'track_id.empty',
        'track_id.invalid',
        'track.invalid_state',
        'consumer.empty',
        'consumer.invalid',
        'action.not_required'
    ];

    _.forEach(errors, function(val) {
        var code = val;
        var errId = val;

        if (skip.indexOf(val) !== -1) {
            return true;
        }

        if (internal.indexOf(val) !== -1) {
            errId = 'internal';
        }

        if (!errIds[errId]) {
            errId = 'internal';
        }

        var error = {
            code,
            // TODO: проверки на наличие всех полей
            msg: loc['Errors'][errIds[errId]]
        };

        if (mendIds[errId]) {
            if (util.isArray(mendIds[errId])) {
                var res = [];

                _.forEach(mendIds[errId], function(id) {
                    res.push(loc['Mend'][id].replace('%1', login));
                });
                error.mend = res.join(' ');
            } else {
                // TODO: проверки на наличие всех полей
                error.mend = loc['Mend'][mendIds[errId]].replace('%1', login);
            }
        }

        resultErrors.push(error);
    });

    return resultErrors;
}

function renderError(errorId) {
    // eslint-disable-next-line no-unused-vars
    return function(err, req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function() {
                res.locals.language = 'ru';
            })
            .done(function() {
                res.render(`${'error' + '.'}${res.locals.language}.js`, {
                    retpath: config.paths.retpath,
                    logID: req.logID,
                    statusCode: 500,
                    date: new Date().toString(),
                    error: {
                        code: errorId,
                        msg: errors.get(errorId, res.locals.language)
                    }
                });
            });
    };
}

function reduceAnswer(currentValue, nextItem) {
    currentValue[nextItem.field] = nextItem.body;

    return currentValue;
}

function processInput(req, roster) {
    var form = {};

    _.forEach(roster, function(settings, field) {
        var trgt = settings.name;
        var input = '';
        var any = settings.any || false;

        if (!trgt) {
            trgt = field;
        }

        if (any) {
            // POST или GET
            input = req.body[field] || (req.nquery && req.nquery[field]);
        } else {
            // только POST
            input = req.body[field];
        }

        if (input) {
            // Вытаскиваем данные только из массивов, все вложенные структуры
            // дропаем
            if (typeof input !== 'string') {
                if (util.isArray(input)) {
                    input = String(input[0]);
                } else {
                    input = '';
                }
            }

            if (trgt !== 'password') {
                input = input.trim();
            }

            if (input !== '') {
                form[trgt] = input;
            }
        }
    });

    return form;
}

function showErrors(req, res, next) {
    var language = res.locals.language;
    var data = {};
    var v;

    if (res.state === 'force_complete_lite') {
        return res.redirect(
            url.format({
                protocol: req.headers['x-real-scheme'],
                host: req.hostname,
                pathname: '/profile/upgrade/lite/',
                query: {
                    origin: 'passport_profile',
                    track_id: req.api.track()
                }
            })
        );
    }

    data.errors = processErrors(res.rawErrors, language, res.processedData);
    data.rawErrors = res.rawErrors;
    data.fields = res.processedData;

    if (res.state === 'auth_challenge') {
        const retpath = req.query && req.query.retpath;
        const queryParams = {
            track_id: req.api.track()
        };

        if (retpath) {
            const parsedRetpath = urlParse(retpath);
            const retpathTrackIdFragment = parsedRetpath.search ? `&track_id=` : `?track_id=`;

            queryParams.retpath = `${retpath}${retpathTrackIdFragment}${req.api.track()}`; // PASSP-33850
        }

        return res.redirect(
            url.format({
                protocol: req.headers['x-real-scheme'],
                host: req.hostname,
                pathname: '/auth/challenges/',
                query: queryParams
            })
        );
    }

    if (res.state === 'magic' || (res.state === 'submit' && req.nquery && req.nquery.mode === 'qr')) {
        res.state = 'magic';
    }

    if (
        res.state === 'confirm' ||
        (res.state === 'change_password' &&
            req.authResult &&
            (req.authResult.change_password_reason === 'account_hacked' ||
                req.authResult.change_password_reason === 'password_pwned'))
    ) {
        if (!req.authResult.number || req.needConfirm) {
            res.state = res.locals.state = 'confirm';

            req.api
                .getHistoryQuestions({track_id: req.api.track()})
                .then(function(results) {
                    if (results.body.questions && results.body.questions.length === 0) {
                        if (req.needConfirm) {
                            return res.redirect(
                                url.format({
                                    protocol: req.headers['x-real-scheme'],
                                    host: req.hostname,
                                    pathname: '/restoration/fallback',
                                    query: {
                                        track_id: req.body['track_id'] || req.query['track_id']
                                    }
                                })
                            );
                        } else {
                            res.state = res.locals.state = 'goto_support';
                        }
                    } else {
                        delete req.authResult.number;
                        data.questions = results.body.questions;
                        data.questions.forEach(function(item) {
                            item.id += `:${item.text}`;
                        });
                    }
                    show();
                })
                .catch(function(errors) {
                    if (Array.isArray(errors) && errors[0] === 'account.global_logout') {
                        return res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                host: req.hostname,
                                pathname: '/auth'
                            })
                        );
                    }

                    if (Array.isArray(errors) && errors[0] === 'account.invalid_type') {
                        return res.redirect(
                            url.format({
                                protocol: req.headers['x-real-scheme'],
                                host: req.hostname,
                                pathname: '/restoration/fallback',
                                query: {
                                    track_id: req.api.track()
                                }
                            })
                        );
                    }

                    return next(errors);
                });
        } else {
            show();
        }
    } else {
        show();
    }

    function show() {
        v = new AuthView(res.state, language);
        v.render(req, res, next, data);
    }
}

function answerToEmbedded(req, res, next) {
    var response = {};

    res.setHeader('Access-Control-Expose-Headers', 'Location');

    response['status'] = 'other';
    response['url'] = url.format({
        protocol: req.headers['x-real-scheme'],
        host: req.hostname,
        pathname: '/auth/finish/',
        query: {
            track_id: req.api.track()
        }
    });
    req.api.readTrack().then(function(results) {
        var answerUrl = url.parse(results.body['retpath'], true);
        var retpathUrl = url.format({
            protocol: answerUrl.protocol,
            hostname: answerUrl.hostname
        });

        answerUrl.query = _.merge({}, answerUrl.query, response);
        // Переписываем retpath на хострут внешнего вызывающего сервиса, поскольку входная точка
        // embeddedauth бесполезна в качестве финального урла, а именно она
        // приходит в поле retpath на embeddedauth
        req.api
            .writeTrack({
                retpath: retpathUrl
            })
            .then(
                function() {
                    return res.redirect(302, url.format(answerUrl));
                },
                function() {
                    return res.redirect(302, url.format(answerUrl));
                }
            );
    }, next);
}

function savePhoneConfirmState(item, req) {
    var body = req.body;
    var number = req.authResult && req.authResult.number && req.authResult.number.masked_international;

    // пробрасываем в контест телефон пользователя

    if (!item.value) {
        item.value =
            number ||
            (body.validation_method !== 'captcha' && (body.phone_number || body.phone_number_confirmed)) ||
            null;
    }

    if (item.value) {
        // что?
        // item.value = item.value;
        item.options = {
            hasConfirmedPhone: true,
            mode: 'tracked'
        };
    }

    // Форму отправили — значит телефон подтверждали
    if (body.validation_method !== 'captcha' && body.phone_number) {
        if (item.options) {
            item.options.state = 'confirmed';
        } else {
            item.options = {
                state: 'confirmed'
            };
        }
    }

    // Начиная со второго сабмита верим первому сабмиту
    // с телефоном
    if (body['phone-confirm-state']) {
        if (item.options) {
            item.options.state = body['phone-confirm-state'];
        } else {
            item.options = {
                state: body['phone-confirm-state']
            };
        }
    }

    // На принудительной смене пароля требуем проверку капчи перед отправкой смс
    if (req.authResult && req.authResult.state === 'change_password') {
        if (item.options) {
            item.options.prevCheckCaptcha = true;
        } else {
            item.options = {
                prevCheckCaptcha: true
            };
        }
    }
}

function getQrCodeLink(req, res, next) {
    var retpath = req.body['retpath'] || (req.nquery && req.nquery['retpath']) || null;
    var query = _.extend({}, req.nquery);

    if (retpath) {
        query.retpath = retpath;
    }

    res.locals.qr_code_link = require('url').format({
        protocol: req.headers['x-real-scheme'],
        host: req.hostname,
        pathname: 'auth',
        query: _.extend({}, query, {mode: 'qr'})
    });

    next();
}

/* }}} */

/* {{{ Routes */

exports.routes.retpath = [
    function(req, res, next) {
        if (!isIntranet) {
            return next('route');
        }

        if (!req.query) {
            return next('route');
        }

        if (!req.query.retpath) {
            return next('route');
        }

        if (req.query.force_auth === '1') {
            return next('route');
        }

        if (['add-user', 'edit'].indexOf(req.query.mode) !== -1) {
            return next('route');
        }

        var controller = req._controller;

        controller
            .getAuth()
            .sessionID()
            .then(function(response) {
                if (response && response.status && response.status.id === 0) {
                    return next();
                } else {
                    return next('route');
                }
            })
            .catch(function(err) {
                if (err && err.code === 'need_resign') {
                    return controller.getAuth().resign();
                }

                if (err) {
                    return next('route');
                }
            });
    },
    retpathRoutes.routes.validate
];

exports.routes.enter = [
    setup,
    getQrCodeLink,
    langSetup,
    multiAuthAccountsSetup,
    apiSetup,
    getYaExperimentsFlags,
    socialSetup,
    prepareStatData,
    function checkIsWebview(req, res, next) {
        const origin = req.query && req.query.origin;

        if (originsFromWebview.includes(origin)) {
            res.locals.isWebView = true;
        }

        return next();
    },
    function showBigSocialButtons(req, res, next) {
        var controller = req._controller;
        var tld = controller.getTld();
        var socialButtons = req.query.social;

        if (socialButtons === 'extended' && tld === 'com') {
            res.locals.prettySocialButtons = true;
        }
        next();
    },
    function showDomik(req, res, next) {
        var language = res.locals.language;
        var data = {
            retpath: req.body['retpath'] || (req.nquery && req.nquery['retpath']),
            twoweeks: req.body['twoweeks'] || (req.nquery && req.nquery['twoweeks']),
            errors: processErrors(res.rawErrors, language),
            experiments: res.locals.experiments
        };

        var query = _.extend({}, req.nquery);
        var pdd_domain = req.params.pdd_domain;

        if (data.retpath) {
            query.retpath = data.retpath;
        }

        if (req.isSecure && res.rawErrors && res.rawErrors.length) {
            return res.redirect(
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: pdd_domain && !config.multiauth ? `for/${pdd_domain}` : 'auth',
                    query
                })
            );
        }

        var v = new AuthView(res.state || 'submit', language);

        v.render(req, res, next, data);
    },
    renderError('Complains.internal')
];

exports.routes.tracksubmit = [
    apiSetup,
    flashSetup,
    getYaExperimentsFlags,
    function(req, res, next) {
        if (!res.isNumberUnavailable) {
            return next();
        }

        const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
        const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

        req.api
            .statboxLogger({
                ignoreMissedTrack: true,
                mode: 'change_password_force',
                action: 'change_password_number_is_unavailable',
                track_id: req.nquery && req.nquery.track_id,
                yandexuid: req.cookies['yandexuid'],
                ip: req.headers['x-real-ip'],
                user_agent: req.headers['user-agent'],
                experiment_flags: experimentFlags,
                experiment_boxes: experimentBoxes
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('auth')
                    .write(err);
            });

        return next();
    },
    function(req, res, next) {
        var retpath = req.body.retpath || (req.nquery && req.nquery.retpath);

        // TODO: если есть retpath параметром, мы переходим на следующий шаг,
        // а там он все равно перепишется ретпасом из get_state, по сути
        // retpath тут как флажок не переписывать его в else
        if (retpath || req.query.one === 'yes') {
            return next();
        } else {
            req.api.readTrack().then(function(results) {
                if (!(results.body && results.body.retpath)) {
                    return next();
                }

                var answerUrl = url.parse(results.body.retpath, true);
                var retpathUrl = url.format({
                    protocol: answerUrl.protocol,
                    hostname: answerUrl.hostname
                });

                if (answerUrl.hostname === req.hostname) {
                    return next();
                }

                // Переписываем retpath на хострут внешнего вызывающего сервиса, поскольку входная точка
                // embeddedauth бесполезна в качестве финального урла, а именно она
                // приходит в поле retpath на embeddedauth
                req.api
                    .writeTrack({
                        retpath: retpathUrl
                    })
                    .then(
                        function() {
                            return next();
                        },
                        function() {
                            return next();
                        }
                    );
            }, next);
        }
    },
    function(req, res, next) {
        var s = new AuthState('get_state', req, res);

        s.run(next);
    },
    function(req, res, next) {
        if (!res.state || res.state === 'rfc_totp' || res.state === 'email_code') {
            next('route');
        } else {
            next();
        }
    },
    function checkPassportSubsEulaExp(req, res, next) {
        const experiments = res.locals.experiments && res.locals.experiments.flags;

        if (experiments.includes('passport-subs-eula-check-on')) {
            res.locals.keepUnsubscribedShown = true;
        }

        return next();
    },
    setup,
    getQrCodeLink,
    langSetup,
    multiAuthAccountsSetup,
    resign,
    socialSetup,
    showErrors,
    renderError('Complains.internal')
];

exports.routes.submit = [
    apiSetup,
    setup,
    getYaExperimentsFlags,
    getQrCodeLink,
    flashSetup,
    function submitForm(req, res, next) {
        var next_state = 'submit';
        var s;

        if (req.body['state'] && req.body['state'] !== 'get_state') {
            next_state = req.body['state'];
        } else if (res.state) {
            next_state = res.state;
        }

        s = new AuthState(next_state, req, res);
        s.run(next);
    },
    function getSession(req, res, next) {
        const controller = req._controller;

        if (!res.authFinished || !res.readyToGetSession) {
            return next();
        }

        var data = {
            track_id: req.api.track()
        };

        req.api
            .authSubmit('/1/bundle/session/', data)
            .then(function(result) {
                var body = result.body;
                var pathname = 'auth/finish';

                controller.augmentResponse(body);

                if (body.cookies) {
                    return res.redirect(
                        url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname,
                            query: {
                                track_id: data.track_id
                            }
                        })
                    );
                } else {
                    req.retpathUrl = body.retpath;
                    return next(body.errors);
                }
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('auth.submit.session')
                    .write(err);

                return next(err);
            });
    },
    function redirectToFinish(req, res, next) {
        var pdd = res.account && res.account.domain && res.account.domain.unicode;
        var pathname = pdd && !config.multiauth ? `/for/${encodeURIComponent(pdd)}/finish/` : '/auth/finish/';

        if (res.state) {
            next();
        } else {
            res.redirect(
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname,
                    query: {
                        track_id: req.body['track_id'] || res.locals.track_id
                    }
                })
            );
        }
    },
    langSetup,
    multiAuthAccountsSetup,
    resign,
    socialSetup,
    function writeStatBox(req, res, next) {
        if (
            !req.authResult ||
            req.authResult.state !== 'change_password' ||
            req.authResult.change_password_reason !== 'account_hacked' ||
            req.authResult.change_password_reason !== 'password_pwned'
        ) {
            return next();
        }

        const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
        const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;
        var statBoxData = {
            ignoreMissedTrack: true,
            mode: 'change_password_force',
            action: 'opened',
            track_id: req.authResult.track_id,
            uid: req.authResult.account && req.authResult.account.uid,
            yandexuid: req.cookies['yandexuid'],
            ip: req.headers['x-real-ip'],
            user_agent: req.headers['user-agent'],
            experiment_flags: experimentFlags,
            experiment_boxes: experimentBoxes
        };

        req.api.statboxLogger(statBoxData).catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('auth')
                .write(err);
        });

        return next();
    },
    function(req, res, next) {
        var track_id = req.api.track();

        if (req.authResult) {
            return next();
        }

        if (track_id) {
            req.api.authSubmit('/1/bundle/auth/password/get_state/', {track_id: req.api.track()}).then(
                function(results) {
                    req.authResult = results.body;
                    return next();
                },
                function() {
                    return next();
                }
            );
        } else {
            return next();
        }
    },
    prepareStatData,
    showErrors,
    renderError('Complains.internal')
];

exports.routes.finish = [
    apiSetup,
    setup,
    flashSetup,
    langSetup,
    // В сообщениях относительные ссылки
    // http://tanker.yandex-team.ru/keys/?keyset-id=Message&project-id=passport
    function legacyUrlSupport(req, res, next) {
        var pdd_domain = req.params.pdd_domain;

        if (req.nquery && req.nquery.mode === 'changepass') {
            return res.redirect(
                302,
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: pdd_domain ? `for/${pdd_domain}` : '/',
                    query: {
                        mode: 'changepass'
                    }
                })
            );
        } else {
            return next();
        }
    },
    function getMobileAndEnv(req, res, next) {
        var controller = req._controller;

        req.env = controller.getEnv();

        controller
            .getUatraits()
            .then(function(uatraits) {
                return uatraits && uatraits.isMobile;
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth', 'routes', 'finish', 'getMobileAndEnv')
                    .write(err);
            })
            .done(function(isMobile) {
                req.isMobile = isMobile;
                return next();
            });
    },
    function checkSession(req, res, next) {
        var session = req.cookies['Session_id'] || '';
        var track_id = req.api && req.api.track();

        if (!track_id) {
            var err = 'missing required field track_id';

            PLog.warn()
                .logId(req.logID)
                .type('auth', 'routes', 'finish', 'checkSession')
                .write(req.nquery);

            return next(err);
        }

        req.api.sessionCheck({session, track_id}).then(
            function(results) {
                if (results.body && results.body.session_is_correct) {
                    res.sessionCheck = results.body;
                    next();
                } else {
                    PLog.info()
                        .logId(req.logID)
                        .type('auth', 'routes', 'finish', 'checkSession', 'sessionCheckFailed')
                        .write('Session check failed with %j', results);

                    next(results);
                }
            },
            function(error) {
                next(error);
            }
        );
    },
    function choosePath(req, res, next) {
        var protocol = req.headers['x-real-scheme'];
        var sessionCheck = res.sessionCheck;
        var embedded = req.nquery && req.nquery.embedded;
        var explicitRetpath = req.nquery && req.nquery.retpath;
        var retpath;
        var mdaQuery;
        var fretpath;
        var clean;
        var parsedRetpath;
        var parsedQuery;

        res.hasUserMessages = Boolean(res.sessionCheck.usermessages);

        if (sessionCheck) {
            retpath = sessionCheck.retpath;
            fretpath = sessionCheck.fretpath;
            clean = sessionCheck.clean;
        }

        if (explicitRetpath) {
            retpath = url.format({
                protocol,
                hostname: req.hostname,
                pathname: 'redirect',
                query: {
                    retpath: explicitRetpath
                }
            });
        }

        if (!retpath) {
            retpath = url.format({
                protocol,
                hostname: req.hostname,
                pathname: 'profile'
            });
        }

        parsedRetpath = url.parse(retpath, true);
        mdaQuery = {};

        if (fretpath) {
            mdaQuery['fretpath'] = fretpath;
        }

        if (clean) {
            mdaQuery['clean'] = clean;
        }

        if (embedded) {
            mdaQuery['embedded'] = embedded;
        }

        const mdaOptions = getMDAOptions(retpath, mdaQuery);

        if (mdaOptions.needMDA) {
            req.exitURL = mdaOptions.redirectUrl;
        } else {
            if (embedded === 'yes') {
                parsedQuery = parsedRetpath.query || (parsedRetpath.query = {});
                parsedQuery['ncrnd'] = String(Math.round(Math.random() * 1000000));
                parsedQuery['status'] = 'ok';
                delete parsedRetpath.search;
                req.exitURL = url.format(parsedRetpath);
            } else {
                req.exitURL = retpath;
            }
        }

        return next();
    },
    function showUserMessages(req, res, next) {
        if (!res.hasUserMessages) {
            return next();
        }

        if (req.nquery && req.nquery.embedded === 'yes') {
            return answerToEmbedded(req, res, next);
        }

        var lang = res.locals.language;
        var loc = locs[lang || 'ru'];
        var msgs = loc['Message'];
        var data = {
            retpath: req.exitURL,
            messages: []
        };

        req.api.popUserMessages().then(function(result) {
            if (result.body.usermessages && result.body.usermessages.length) {
                _.forEach(result.body.usermessages, function(msgId) {
                    data.messages.push(msgs[`content.${msgId}`]);
                });

                var v = new AuthView('message', lang);

                v.render(req, res, next, data);
            } else {
                return next();
            }
        });
    },
    function showFlashMessages(req, res, next) {
        if (req.nquery && req.nquery.embedded === 'yes') {
            return next();
        }

        res.flashMessages = req.flash('info');

        if (!res.flashMessages.length) {
            return next();
        }

        var lang = res.locals.language || 'ru';
        var i18nLocal = i18n[lang];
        var loc = locs[lang];
        var msgs = loc['FlashMessages'];
        var data = {
            retpath: req.exitURL,
            title: loc['Passwd']['title'],
            messages: []
        };

        _.forEach(res.flashMessages, function(msg) {
            var messageVars = msgs[msg.id];
            var days =
                msg.data &&
                msg.data.timeout &&
                Math.ceil((new Date(msg.data.timeout * 1000).getTime() - Date.now()) / 1000 / 60 / 60 / 24);
            var message;

            if (_.isNumber(days) && util.isArray(messageVars)) {
                if (days === 0) {
                    message = messageVars[3];
                } else {
                    message = i18nLocal.plural_adv({
                        count: days,
                        one: messageVars[0],
                        some: messageVars[1],
                        many: messageVars[2]
                    });
                }
            } else {
                message = messageVars;
            }

            if (days) {
                message = message.replace('%%days%%', days);
            }

            data.messages.push(message);
        });

        var v = new AuthView('message', lang);

        v.render(req, res, next, data);
    },
    function show2FApromo(req, res, next) {
        if (req.nquery && req.nquery.embedded === 'yes') {
            return next();
        }

        res.show_2fa_promo = req.flash('enable_2fa_track_id');

        if (!res.show_2fa_promo.length) {
            return next();
        }

        var diskApi = new DiskApi(req.logID);
        var authInfo = req._controller.getAuth();

        authInfo
            .loggedIn()
            .then(function() {
                diskApi
                    .browserSyncEnabled(authInfo.getUid())
                    .then(function(syncOn) {
                        if (syncOn) {
                            return next();
                        }

                        var lang = res.locals.language || 'ru';
                        var enable_2fa_track_id = res.show_2fa_promo[0] && res.show_2fa_promo[0].id;
                        var loc = locs[lang];
                        var urls = yakeyUrls.getUrls({
                            lang,
                            domain: res.locals.domain,
                            host: req.hostname
                        });
                        var enable2faUrl = _.extend(url.parse(urls.enable2faUrl), {
                            query: {
                                track_id: enable_2fa_track_id,
                                from: 'changepass_2fa_promo'
                            }
                        });

                        delete enable2faUrl.search;

                        var data = {
                            retpath: req.exitURL,
                            applUrl: urls.applUrl,
                            googUrl: urls.googUrl,
                            promoUrl: urls.promoUrl,
                            enable2faUrl: url.format(enable2faUrl),
                            title: loc['Passwd']['title']
                        };

                        const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
                        const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

                        req.api
                            .statboxLogger({
                                ignoreMissedTrack: true,
                                mode: 'show_2fa_promo_after_changepassword',
                                action: 'opened',
                                track_id: req.api.track(),
                                enable_2fa_track_id,
                                yandexuid: req.cookies['yandexuid'],
                                ip: req.headers['x-real-ip'],
                                user_agent: req.headers['user-agent'],
                                experiment_flags: experimentFlags,
                                experiment_boxes: experimentBoxes
                            })
                            .catch(function(err) {
                                PLog.warn()
                                    .logId(req.logID)
                                    .type('show_2fa_promo_after_changepassword')
                                    .write(err);
                            });

                        var v = new AuthView('promo', lang);

                        v.render(req, res, next, data);
                    })
                    .catch(function(err) {
                        PLog.warn()
                            .logId(req.logID)
                            .type('show_2fa_promo_after_changepassword')
                            .write(err);

                        return next();
                    });
            })
            .catch(function(err) {
                if (err && err.code === 'need_resign') {
                    return;
                }

                PLog.warn()
                    .logId(req.logID)
                    .type('show_2fa_promo_after_changepassword')
                    .write(err);

                return next();
            });
    },
    function clearAndRedirect(req, res) {
        const {retpath, keep_track: keepTrack} = req.query;

        let isFromAmChallenge = false;

        if (retpath) {
            const parsedRetpathQuery = (urlParse(retpath) || {}).query;

            isFromAmChallenge = parsedRetpathQuery && parsedRetpathQuery.type === 'am_challenge';
        }

        if (!isFromAmChallenge && typeof keepTrack === 'undefined') {
            req.api.delTrack();
        }

        return res.redirect(req.exitURL);
    },
    function embeddedError(err, req, res, next) {
        if (req.nquery && req.nquery.embedded === 'yes') {
            answerToEmbedded(req, res, next);
        } else {
            next('no_cookie');
        }
    },
    renderError('ErrorsTexts.nocki')
];
/* }}} */
