const express = require('express');
const router = express.Router();
const apiSetup = require('./common/apiSetup');
const setCORSHeaders = require('./common/setCORSHeaders');
const {urlFormat} = require('./common/urlFormat');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const url = require('url');
const locs = require('../loc/auth.json');
const _ = require('lodash');
const PLog = require('plog');

const routes = (exports.routes = {});

exports.route = function(app) {
    app.options('/auth/update/?', routes.options);
    app.get('/auth/update/?', routes.authUpdate);
    app.post('/auth/update/?', routes.authUpdate);
};

exports.routes.options = [
    setCORSHeaders,
    function(_req, res) {
        return res.end();
    },
    function(err, req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(error) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('auth.update')
                    .write(error);
            })
            .done(function() {
                return next(err);
            });
    },
    errorHandler
];

exports.routes.authUpdate = [
    apiSetup,
    setCORSHeaders,
    filterNoCookieRequests,
    getYaExperimentsFlags,
    function(req, _res, next) {
        if (req.method !== 'GET' || !req._controller.hasExp('social-force-upgrade')) {
            return next();
        }

        req._controller
            .getAuth()
            .simpleSessionID({
                aliases: '1,6'
            })
            .then((response) => {
                const retpath = req.body.retpath || req.query.retpath;
                const isPortal = response.aliases && response.aliases.hasOwnProperty('1');
                const isSocial = response.aliases && response.aliases.hasOwnProperty('6') && !isPortal;

                if (!isSocial || !retpath) {
                    return next();
                }

                const {
                    display_name: {
                        social: {provider}
                    }
                } = response;

                if (['fb', 'tw'].includes(provider)) {
                    req.augmentedRetpath = urlFormat({
                        protocol: 'https',
                        hostname: req.hostname,
                        pathname: 'profile/upgrade',
                        query: {
                            retpath
                        }
                    });
                }

                return next();
            })
            .catch(() => {
                return next();
            });
    },
    doSessionUpdateRequest,
    function(err, req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(error) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('auth.update')
                    .write(error);
            })
            .done(function() {
                return next(err);
            });
    },
    errorHandler
];

function filterNoCookieRequests(req, res, next) {
    req.retpathUrl = req.body.retpath || req.query.retpath;
    const isAjax = !req.retpathUrl;

    if (!req.cookies.Session_id) {
        if (isAjax) {
            res.type('application/json');

            return res.send({status: 'error'});
        }

        return next(['sessionid.required']);
    }

    return next();
}

function doSessionUpdateRequest(req, res, next) {
    const controller = req._controller;
    const retpath = req.augmentedRetpath || req.body.retpath || req.query.retpath;

    const data = {
        passErrors: true,
        retpath
    };

    // PASSP-23949
    if (req.baseUrl === '/auth/guard') {
        data.force_prolong = 'yes';
    }

    res.removeHeader('X-Frame-Options');
    res.removeHeader('X-Download-Options');

    req.api
        .authSubmit('/1/bundle/session/update/', data)
        .then(function(result) {
            const body = result.body;
            const retpath =
                body.retpath ||
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: 'passport',
                    query: {
                        mode: 'passport'
                    }
                });

            const isAjax = !data.retpath;

            if (body.status === 'ok') {
                controller.augmentResponse(body);

                if (res.locals.sessguardContainer && res.locals.guardURL) {
                    const answer = `<html>
                            <body>
                                <form method="post" id="form" action="${res.locals.guardURL}">
                                   <input type="hidden" name="container" value="${res.locals.sessguardContainer}"/>
                                </form>
                                <script nonce="${res.locals.nonce}">
                                        document.getElementById("form").submit();
                                </script>
                            </body>
                        </html>`;

                    return res.send(answer);
                }

                if (isAjax) {
                    res.type('application/json');

                    return res.send({status: 'ok'});
                }

                return res.redirect(retpath);
            }

            if (isAjax) {
                res.type('application/json');

                return res.send({status: 'error'});
            }

            req.retpathUrl = data.retpath;
            return next(body.errors);
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('auth.update')
                .write(err);

            return next(err);
        });
}

// eslint-disable-next-line no-unused-vars
function errorHandler(err, req, res, next) {
    const retpath = req.retpathUrl;
    const lang = res.locals.language;
    const loc = locs[lang];
    const auth = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'auth',
        query: {}
    };

    if (retpath) {
        auth.query.retpath = _.escape(retpath);
    }

    const data = {
        title: loc.Errors.session_error_title,
        messages: [loc.Errors.session_error_message.replace('%url%', url.format(auth))]
    };

    if (err[0] === 'sessionid.required') {
        PLog.info()
            .logId(req.logID)
            .type('auth.update')
            .write(err);
    } else {
        PLog.warn()
            .logId(req.logID)
            .type('auth.update')
            .write(err);
    }

    return res.render(`auth.message.${lang}.js`, data);
}

router.options('/', routes.options);
router.get('/', routes.authUpdate);
router.post('/', routes.authUpdate);

exports.doSessionUpdateRequest = doSessionUpdateRequest;
exports.errorHandler = errorHandler;
exports.setCORSHeaders = setCORSHeaders;
exports.router = router;
