const express = require('express');
const router = express.Router();
const apiSetup = require('../../common/apiSetup');
const passportApiSetup = require('../../common/passportApiSetup');
const langSetup = require('../../common/langSetup');
const getUatraitsData = require('../../common/getUatraitsData');
const getYaExperimentsFlags = require('../../common/getYaExperimentsFlags');
const checkAuth = require('../../common/checkAuth');
const writeStatbox = require('../../common/writeStatbox');
const validateCSRF = require('../../common/validateCSRF.js');
const PLog = require('plog');
const rumCounterSetup = require('../../common/rumCounterSetup');
const getAdditionalDataRequest = require('../common/getStore/additionalDataRequest.js');
const getCommon = require('../common/getStore/getCommon.js');
const getSettings = require('../common/getStore/getSettings.js');
const getSocial = require('../common/getStore/getSocial.js');
const getMetrics = require('../../common/getMetrics.js');
const getMonitoring = require('../common/getStore/getMonitoring.js');
const getCaptcha = require('../../common/getCaptcha.js');
const getOneDomik = require('../common/getStore/getOneDomik.js');

function prepareStatData(req, res, next) {
    const locals = res.locals;
    const oneDomik = locals.store.one_domik;

    res.locals.preparedStat = Object.assign(
        {},
        {
            opened_state: oneDomik.mode,
            track_id: oneDomik.track_id || null,
            default_uid: oneDomik.authorizedAccountsDefaultUid || null,
            auth_accounts: String(oneDomik.authorizedAccounts.length),
            sugg_accounts: String(oneDomik.suggestedAccounts.length)
        }
    );

    return next();
}

function renderPage(req, res) {
    const locals = res.locals;
    const lang = locals.language;
    const additionalDataRequest = locals.store.additionalDataRequest;

    let pane = req.originalUrl;

    if (additionalDataRequest && additionalDataRequest.state) {
        pane = `/auth/${additionalDataRequest.state}`;
    } else {
        pane = '/auth/';
    }

    res.locals.store.common.pane = pane;
    res.locals.track_id = res.locals.store.additionalDataRequest.track_id;
    res.render(`auth.stepped.${lang}.jsx`);
}

const getRouteAsk = [
    checkAuth,
    (req, res, next) => next('route'),
    getUatraitsData,
    getYaExperimentsFlags,
    langSetup,
    getCommon(),
    getSettings(),
    getSocial(),
    getMetrics({
        header: 'Доввод данных'
    }),
    getMonitoring({
        page: 'auth.ask'
    }),
    getCaptcha(),
    getOneDomik(),
    getAdditionalDataRequest(),
    prepareStatData,
    writeStatbox({
        action: 'opened',
        mode: 'one_domik'
    }),
    rumCounterSetup,
    renderPage
];

router.get('/phone', getRouteAsk);
router.get('/email', getRouteAsk);
router.get('/social', getRouteAsk);

exports.phoneConfirmAndBindSecureSubmit = [
    langSetup,
    passportApiSetup,
    apiSetup,
    validateCSRF,
    (req, res, next) => {
        req.api
            .suggestCountry()
            .then(({body}) => {
                req.body.country = body && Array.isArray(body.country) && body.country[0];

                return next();
            })
            .catch(() => next());
    },
    (req, res) => {
        const {number, display_language, track_id, country, code_format} = req.body;
        const params = {
            number,
            country,
            displayLanguage: display_language,
            trackId: track_id
        };

        if (code_format) {
            params.code_format = code_format;
        }

        req._passportApi
            .phoneConfirmAndBindSecureSubmit(params)
            .then((result) =>
                res.json({
                    status: 'ok',
                    number: result.number,
                    deny_resend_until: result.deny_resend_until
                })
            )
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.phoneConfirmAndBindSecureSubmit')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.phoneConfirmAndBindSecureCommit = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    (req, res) => {
        const {code, password, track_id} = req.body;

        req._passportApi
            .phoneConfirmAndBindSecureCommit(code, password, track_id)
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.phoneConfirmAndBindSecureCommit')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.phoneProlongValid = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    (req, res) => {
        const {phone_id, display_language} = req.body;

        req._passportApi
            .phoneProlongValid(phone_id, display_language)
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.phoneProlongValid')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.sendConfirmationEmail = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    encodeEmailDomain,
    (req, res) => {
        const {email, code_only, validator_ui_url, retpath, track_id, uid} = req.body;

        req._passportApi
            .sendConfirmationEmail({
                email,
                retpath,
                uid,
                codeOnly: code_only,
                validatorUIUrl: validator_ui_url,
                trackId: track_id
            })
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.sendConfirmationEmail')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.confirmEmailByCode = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    (req, res) => {
        const {key, track_id, uid} = req.body;

        req._passportApi
            .confirmEmailByCode(key, track_id, uid)
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.confirmEmailByCode')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.setupConfirmedEmail = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    encodeEmailDomain,
    (req, res) => {
        const {email, is_safe, uid} = req.body;

        req._passportApi
            .setupConfirmedEmail(email, is_safe, uid)
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.setupConfirmedEmail')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.setSocialProfileAuth = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    (req, res) => {
        const {profile_id, set_auth, track_id} = req.body;

        req._passportApi
            .setSocialProfileAuth(profile_id, set_auth, track_id)
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.setSocialProfileAuth')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.additionalDataAsk = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    (req, res) => {
        req._passportApi
            .additionalDataAsk()
            .then((result) => {
                const answer = {
                    status: 'ok',
                    csrf_token: res.locals.csrf_token,
                    track_id: result.track_id,
                    state: result.state,
                    action: result.action
                };

                answer[result.state] =
                    (result.data && result.data[0]) ||
                    (result.addresses && result.addresses[0]) ||
                    (result.profiles && result.profiles[0]);

                if (result.state === 'phone' && result.action === 'add') {
                    answer.phone = {
                        number: '',
                        id: ''
                    };
                }

                if (result.state === 'email' && result.action !== 'add' && answer.email) {
                    try {
                        answer.email = req._controller.decodePunycodeEmail(answer.email);
                    } catch (e) {
                        return res.json({
                            status: 'error',
                            errors: ['email.invalid']
                        });
                    }
                }

                return res.json(answer);
            })
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.additionalDataAsk')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

exports.additionalDataFreeze = [
    langSetup,
    passportApiSetup,
    validateCSRF,
    (req, res) => {
        const {user_declined, track_id, uid} = req.body;

        req._passportApi
            .additionalDataFreeze(user_declined, track_id, uid)
            .then(() => res.json({status: 'ok'}))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('auth.new.additionalDataFreeze')
                    .write(errors);

                return res.json(errors._response);
            });
    }
];

function encodeEmailDomain(req, res, next) {
    if (req.body.email) {
        try {
            req.body.email = req._controller.encodeEmailToPunycode(req.body.email);
        } catch (e) {
            return res.json({
                status: 'error',
                errors: ['email.invalid']
            });
        }
    }

    return next();
}

exports.router = router;
