const express = require('express');
const router = express.Router();
const apiSetup = require('../common/apiSetup');
const writeStatbox = require('../common/writeStatbox');
const {urlFormat} = require('../common/urlFormat');

const {Validator} = require('express-json-validator-middleware');
const validator = new Validator({allErrors: true}); // Pass in options to the Ajv instance
const {validate} = validator;

const bodySchema = require('./body.schema.json');
const querySchema = require('./query.schema.json');
const doRequest = require('./doRequest');
const processResponse = require('./processResponse');

const locs = require('../../loc/auth.json');

const postRoute = [
    validate({
        body: bodySchema
    }),
    apiSetup,
    writeStatbox({
        action: 'opened',
        mode: 'auth_sso_commit'
    }),
    (req, res, next) => {
        doRequest(req, res)
            .then((apiResponse) => {
                const {track_id: trackId, state} = apiResponse;

                if (typeof state !== 'undefined') {
                    return req._controller.redirectToLocalUrl({
                        pathname: '/auth',
                        query: {
                            track_id: trackId,
                            one: 'yes'
                        }
                    });
                }

                return processResponse({
                    api: req.api,
                    controller: req._controller,
                    trackId
                });
            })
            .catch(next);
    },
    errorsHandler
];

const getRoute = [
    validate({
        query: querySchema
    }),
    apiSetup,
    writeStatbox({
        action: 'opened',
        mode: 'auth_sso_get'
    }),
    (req, _res, next) => {
        const {track_id, redirect_url, code_challenge, code_challenge_method} = req.query;
        const {api, _controller: controller} = req;

        let samlSsoUrl = '';

        api.authSsoSubmit({track_id, code_challenge, code_challenge_method})
            .then(({body = {}}) => {
                samlSsoUrl = body.redirect_to;

                if (redirect_url) {
                    return api
                        .writeTrack({
                            track_id,
                            frontend_state: 'retpath_rewritten',
                            retpath: redirect_url
                        })
                        .then(() => {
                            return samlSsoUrl;
                        });
                }

                return samlSsoUrl;
            })
            .then((samlSsoUrl) => {
                controller.redirect(samlSsoUrl);
            })
            .catch(next);
    },
    errorsHandler
];

// eslint-disable-next-line no-unused-vars
function errorsHandler(err, req, res, _next) {
    const lang = res.locals.language || 'ru';
    const loc = locs[lang];
    const retpath = urlFormat(req._controller.getAuthUrl());

    const data = {
        title: loc.Errors.session_error_title,
        messages: [getErrorMessage({err, lang, retpath})],
        retpath
    };

    return res.render(`auth.message.${lang}.js`, data);
}

function getErrorMessage({err, lang, retpath}) {
    const loc = locs[lang];
    const {error} = err;

    let errorCode = error[0] || '';

    if (err.name === 'JsonSchemaValidationError') {
        errorCode = 'required_params.missing';
    }

    const errorTexts = {
        'account.not_found': loc._AUTH_['SAML.Errors.account.not_found'],
        'domain.not_found': loc._AUTH_['SAML.Errors.domain.not_found'],
        'email.exists': loc._AUTH_['SAML.Errors.email.exists'],
        'email.no_in_response': loc._AUTH_['SAML.Errors.email.no_in_response'],
        'email.unsupportable_domain': loc._AUTH_['SAML.Errors.email.unsupportable_domain'],
        internal: loc.Errors.session_error_message.replace('%url%', retpath),
        'required_params.missing': loc._AUTH_['SAML.Errors.saml.required_params.missing'],
        'saml.registration_not_supported': loc._AUTH_['SAML.Errors.saml.registration_not_supported'],
        'samlrequest.invalid': loc._AUTH_['SAML.Errors.samlrequest.invalid'],
        'saml.request_your_admin': loc._AUTH_['SAML.Errors.saml.request_your_admin'],
        'samlresponse.invalid': loc._AUTH_['SAML.Errors.samlresponse.invalid'],
        'saml_settings.invalid': loc._AUTH_['SAML.Errors.saml_settings.invalid']
    };

    return errorTexts[errorCode] || errorTexts.internal;
}

router.post('/commit', postRoute);
router.get('/submit', getRoute);
module.exports.router = router;
