const express = require('express');
const router = express.Router();
const apiSetup = require('../common/apiSetup');
const setCORSHeaders = require('../common/setCORSHeaders.js');

const {Validator} = require('express-json-validator-middleware');
const querySchema = require('./authWithLetter.query.schema.json');
const bodySchema = require('./authWithLetter.body.schema.json');

const validator = new Validator({allErrors: true}); // Pass in options to the Ajv instance
const {validate} = validator;
const langSetup = require('../common/langSetup');
const getUatraitsData = require('../common/getUatraitsData');
const getSettings = require('../auth/common/getStore/getSettings');
const getMetrics = require('../common/getMetrics');
const getMonitoring = require('../auth/common/getStore/getMonitoring');
const rumCounterSetup = require('../common/rumCounterSetup');
const createState = require('./createState');
const createCommonState = require('../authv2/createCommonState');
const createCustomsState = require('../common/createCustomsState');
const renderPage = require('../authv2/renderPage');
const getYaExperimentsFlags = require('../common/getYaExperimentsFlags');
const config = require('../../configs/current');

const getRoute = [
    apiSetup,
    getUatraitsData,
    langSetup,
    getYaExperimentsFlags,
    createState,
    createCommonState(),
    getSettings(),
    getMetrics({
        header: 'Авторизация: страница входа через email'
    }),
    getMonitoring({
        page: 'auth.v2'
    }),
    createCustomsState,
    rumCounterSetup,
    getPageInfo,
    renderPage
];

const postRoute = [
    setCORSHeaders,
    validate({
        query: querySchema,
        body: bodySchema
    }),
    apiSetup,
    validateCSRF,
    (req, res) => {
        const {action, track_id, secret} = req.query;
        const data = {
            track_id
        };

        let handle = 'authWithLetterConfirm';

        if (action === 'accept') {
            data.secret = secret;
        }

        if (action === 'cancel') {
            handle = 'authWithLetterInvalidate';
        }

        req.api[handle](data)
            .then((result) => res.json(result.body || result))
            .catch((errors) =>
                res.json({
                    status: 'error',
                    errors: errors || []
                })
            );
    },
    errorsHandler
];

const optionsRoute = [
    setCORSHeaders,
    function(req, res) {
        return res.end();
    },
    errorsHandler
];

const csrfRoute = [
    setCORSHeaders,
    (req, res, next) => {
        req._controller
            .getScopedCsrfToken({key: config.cloudAPIToken})
            .then((csrfToken) =>
                res.status(200).json({
                    status: 'ok',
                    csrf_token: csrfToken
                })
            )
            .catch(next);
    },
    errorsHandler
];

router.get('/', getRoute);
router.post('/', postRoute);
router.options('/', optionsRoute);

router.post('/csrf', csrfRoute);
router.options('/csrf', optionsRoute);

exports.router = router;

// eslint-disable-next-line no-unused-vars
function errorsHandler(err, req, res, next) {
    if (err.name === 'JsonSchemaValidationError') {
        return res.status(400).json({
            status: 'error',
            code: 'invalid params'
        });
    }

    return res.status(500).json({
        status: 'error',
        code: 'internal'
    });
}

function getPageInfo(req, res, next) {
    const state = res.locals.store || (res.locals.store = {});
    const resultState = state.mailAuth || (state.mailAuth = {});
    const trackId = req.query.track_id;
    const secret = req.query.secret && req.query.secret.replace(/[^a-f0-9]/g, '');
    const isRedirect = req.query.redirect === 'true';
    const isRegister = /^\/register\/link/.test(req.originalUrl);
    const initialPageState = {
        isRedirect,
        isRegister,
        mailTrack: trackId,
        secret,
        isAuthConfirmed: false
    };

    res.locals.authPage = 'magiclink-auth';

    if (!(secret && trackId)) {
        Object.assign(resultState, initialPageState, {mailAuthError: 'invalid'});

        return next();
    }

    return req.api
        .getLetterPageInfo(trackId)
        .then((response) => {
            const {body = {}} = response;
            const {browser, login, code} = body;

            Object.assign(
                resultState,
                initialPageState,
                {
                    browser,
                    login
                },
                {
                    code: code && [...code]
                }
            );
            return next();
        })
        .catch((error) => {
            const errorCode = Array.isArray(error) ? error[0] : 'global';
            const errorsToRedirect = [
                'account.not_found',
                'account.disabled',
                'account.disabled_on_deletion',
                'request.credentials_all_missing', // - Требуется авторизация, но в запросе нет ни куки, ни oauth-токена
                'sessionid.invalid', // - Требуется авторизация, но переданная кука невалидна
                'oauth_token.invalid', // - Требуется авторизация, но переданный токен невалиден
                'sessionid.no_uid', // - Требуется авторизация, но в переданной куке нет нужного юзера
                'account.uid_mismatch' // - Требуется авторизация, но переданный токен принадлежит не тому юзеру
            ];

            if (errorsToRedirect.includes(errorCode)) {
                return req._controller.redirectToAuth();
            }
            Object.assign(resultState, initialPageState, {mailAuthError: errorCode});
            return next();
        });
}

function validateCSRF(req, res, next) {
    const controller = req._controller;

    controller
        .isScopedCsrfTokenValid({key: config.cloudAPIToken, token: req.body.csrf_token})
        .then(function(isValid) {
            if (isValid) {
                return next();
            }

            return res.status(403).json({});
        })
        .catch(function() {
            return res.status(403).json({});
        });
}
