const PLog = require('plog');
const {providers} = require('../common/providers.json');
const urlFormat = require('../common/urlFormat.js').urlFormat;
const getCustomsConfig = require('../common/getCustomConfig');
const putCustomsFieldsInStore = require('../common/putCustomsFieldsInStore');
const isNeoPhonishRegisterAvailable = require('../common/isNeoPhonishRegisterAvailable');
const config = require('../../configs/current');

const configs = {
    customs: getCustomsConfig('customs.js'),
    ios: getCustomsConfig('customs.ios.config.json'),
    android: getCustomsConfig('customs.android.config.json')
};

const MODE_TO_PANE = {
    upgrade: '/auth/complete',
    registration: '/auth/reg',
    relogin: '/auth',
    hintlogin: '/auth',
    welcome: '/auth',
    phoneconfirm: '/auth/phoneconfirm',
    turbo: '/auth/reg',
    phonish: '/auth/reg'
};

const RTL_LOCALES = ['he', 'ar', 'iw'];
const AM_LANGS = [...config.langs, 'iw'];

const DEFAULT_PUSH_SIZE_OPTIONS = {
    mode: 'bottom',
    cornerRadius: 20,
    horizontalMargins: 16,
    verticalMargins: 16,
    height: 300
};

const FULLSCREEN_PUSHES_REGEXP = [
    /^\/am\/push\/family\/?$/,
    /^\/am\/push\/family-limits\/?$/,
    /^\/am\/push\/family-limits-member\/?$/
];

const FULLSCREEN_PUSH_SIZE_OPTIONS = {
    mode: 'fullscreen',
    cornerRadius: 0,
    horizontalMargins: 0,
    verticalMargins: 0
};

function normalizePhone(phone) {
    if (!phone || typeof phone !== 'string') {
        return;
    }

    return phone.replace(/^p/, '+');
}

function normalizeExperiments(experimentsArray = []) {
    return experimentsArray.reduce((acc, curr) => {
        if (Array.isArray(((curr && curr.PASSPORT) || {}).flags)) {
            curr.PASSPORT.flags.forEach((flag) => {
                const [key, value] = flag.split('=');

                acc[key] = value === '1';
            });
        }

        return acc;
    }, {});
}

function setupExperiments(req, store = {}) {
    const {am = {}} = store;

    if (am.isAm && am.deviceId) {
        return req._controller
            .fetchNativeAmExperiments(am.deviceId)
            .then((response) => {
                return response && response.body && response.body.experiments ? response.body.experiments : [];
            })
            .catch((e) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('am.native-experiments')
                    .write(e);

                return [];
            })
            .then((experiments) => {
                if (Array.isArray(experiments)) {
                    am.experiments = normalizeExperiments(experiments);
                } else {
                    am.experiments = {};
                }
            });
    }

    return Promise.resolve({});
}

function createFinishRoute(req, status) {
    return urlFormat({
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: '/am/finish',
        query: {
            status
        }
    });
}

const getCustomConfig = (platform, appId) => {
    const {customs, [platform]: config} = configs;

    if (config[appId]) {
        return config[appId];
    }

    const appIdMatch = customs.appIdMatcher && customs.appIdMatcher.exec(appId);
    const matchedAppId = appIdMatch && appIdMatch[1];

    if (matchedAppId && config[matchedAppId]) {
        return config[matchedAppId];
    }

    return {};
};

const setLangFromAmParam = (res, lang) => {
    if (AM_LANGS.includes(lang)) {
        res.locals.language = lang === 'iw' ? 'he' : lang;
        res.locals.store.settings.language = lang === 'iw' ? 'he' : lang;
        res.locals.store.settings.dir = RTL_LOCALES.includes(lang) ? 'rtl' : 'ltr';
    }
};

function configureState(req, res) {
    // {
    //   "am_version_name":"7.22.1(722012081)",
    //   "app_id":"ru.yandex.taxi",
    //   "app_platform":"android",
    //   "app_version_name":"4.38.0",
    //   "auth_type":"yandex,lite,social",
    //   "device_id":"15005782685390a4783d53777c48a51d",
    //   "lang":"ru",
    //   "locale":"",
    //   "manufacturer":"Xiaomi",
    //   "model":"M2102J20SG",
    //   "mode":"phonish",
    //   "nosocial":"false",
    //   "reg_type":"neophonish"
    //   "theme":"light",
    // }

    const {
        am_version_name: amVersionName,
        app_id: appId,
        app_platform: platform,
        app_version_name: appVersionName,
        debug,
        enable_2fa: enableKey2fa,
        device_id: deviceId,
        device_name,
        editable,
        email,
        firstName,
        icloud_token,
        lang,
        lastName,
        locale,
        login,
        manufacturer,
        model,
        mode = 'welcome',
        nosocial,
        phone,
        reg_type: regType,
        scheme,
        siwa,
        source,
        theme,
        uid,
        uuid,
        x_token_client_id,
        track_id: trackId
    } = req.query;

    const initialData = {
        am_version_name: amVersionName,
        app_id: appId,
        app_platform: platform,
        app_version_name: appVersionName,
        cloud_token: icloud_token,
        device_id: deviceId,
        device_name,
        display_language: lang,
        manufacturer,
        model,
        uuid,
        x_token_client_id
    };

    return {
        isAm: true,
        platform,
        appId,
        enableKey2fa: enableKey2fa === 'true',
        appVersionName,
        amVersionName,
        deviceId,
        scheme,
        theme: theme || req._controller.getCookie('theme'),
        lang,
        locale,
        source,
        mode,
        regType,
        debug: Boolean(res.locals.isAmDebug),
        clearUrlParams: !debug,
        siwa: Boolean(siwa),
        nosocial: nosocial === 'true',
        login,
        suggestedLoginFromSWC: '',
        uid,
        phone: normalizePhone(phone),
        editable: editable === 'true',
        email,
        firstName,
        lastName,
        icloudToken: icloud_token,
        trackId,
        finishOkUrl: createFinishRoute(req, 'ok'),
        finishErrorUrl: createFinishRoute(req, 'error'),
        finishCancelUrl: createFinishRoute(req, 'cancel'),
        initialData
    };
}

function configureCustoms(req, res) {
    const {app_platform: platform, app_id: appId} = req.query;
    const {experiments: {flags = []} = {}} = res.locals;
    const origin = req.query.origin || req.body.origin;
    const storeDraft = res.locals.store || (res.locals.store = {});

    putCustomsFieldsInStore(
        getCustomConfig(platform, appId),
        storeDraft,
        isNeoPhonishRegisterAvailable(flags, origin),
        flags,
        res.locals.language
    );
}

function addAplSocialProvider(state = {}) {
    const {social = {}} = state;
    const aplProvider = providers.filter((provider) => provider.code === 'apl')[0];

    if (social.providers) {
        social.providers = [
            {
                id: aplProvider.id,
                enabled: true,
                primary: true,
                data: aplProvider
            }
        ].concat(social.providers);
    }
}

function setupTurboMode(store = {}) {
    const {am = {}, form = {}} = store;
    const {values = {}} = form;

    if (am.firstName && !values.firstname) {
        values.firstname = am.firstName;
    }

    if (am.lastName && !values.lastname) {
        values.lastname = am.lastName;
    }

    if (am.phone && !values.phone) {
        values.phone = am.phone;
    }
}

module.exports = function createAMState(req, res, next) {
    const locals = res.locals;
    const {pathname} = req._controller.getUrl() || {};
    const state = res.locals.store;
    const mode = req.query.mode || 'welcome';
    const appPlatform = req.query.app_platform || '';
    const lang = req.query && req.query.lang;

    let pane;

    if (typeof lang === 'string') {
        setLangFromAmParam(res, lang);
    }

    if (/^\/am\/debug\/?/.test(pathname)) {
        pane = '/am/debug';

        state.am = configureState(req, res);
        configureCustoms(req, res);
        state.common.isWebView = true;
        state.common.isWebViewWithSocialAuth = true;
    } else if (/^\/am\/finish\/?/.test(pathname)) {
        pane = '/am/finish';

        state.am = configureState(req, res);
        configureCustoms(req, res);
        state.common.isWebView = true;
        state.common.isWebViewWithSocialAuth = true;
    } else if (/^\/am\/push\/qrsecure\/?/.test(pathname)) {
        if (['ios', 'android'].includes(appPlatform)) {
            state.am = configureState(req, res);
            state.common.isWebView = true;
            state.common.isWebViewWithSocialAuth = true;

            state.am.isAmPopup = true;
            state.am.popupOptions = {
                mode: 'bottom',
                cornerRadius: 20,
                horizontalMargins: 16,
                verticalMargins: 16,
                height: 385
            };
        } else {
            state.am = configureState(req, res);
            state.am.isAm = false;
        }
    } else if (/^\/am\/?/.test(pathname)) {
        pane = '/auth';

        if (MODE_TO_PANE[mode]) {
            pane = MODE_TO_PANE[mode];
        }

        state.am = configureState(req, res);
        configureCustoms(req, res);
        state.common.isWebView = true;
        state.common.isWebViewWithSocialAuth = !state.am.nosocial;
        state.common.backpath = state.am.finishCancelUrl;

        if (/^\/am\/push\/?/.test(pathname)) {
            state.am.isAmPopup = true;

            if (FULLSCREEN_PUSHES_REGEXP.some((regex) => regex.test(pathname))) {
                state.am.popupOptions = FULLSCREEN_PUSH_SIZE_OPTIONS;
            } else {
                state.am.popupOptions = DEFAULT_PUSH_SIZE_OPTIONS;
            }
        }

        if (!state.common.retpath) {
            state.common.retpath = state.am.finishOkUrl;
        }

        if (state.am.siwa) {
            addAplSocialProvider(state);
        }

        if (state.am.login && !state.auth.form.login) {
            state.auth.form.login = state.am.login;
        }

        if (state.am.mode === 'turbo') {
            setupTurboMode(state);
        }

        return setupExperiments(req, state).then(() => {
            locals.aMpane = pane;

            next();
        });
    }

    locals.aMpane = pane;

    next();
};
