const PLog = require('plog');
const apiSetup = require('../common/apiSetup');
const validateCSRF = require('../common/validateCSRF');
const {urlFormat} = require('../common/urlFormat');
const getCustomConfigByRequest = require('../common/getCustomConfigByRequest');
const SAML_SSO_AUTH_METHOD = 'saml_sso';

const getParams = (req, params) => {
    const {allowScholar = false} = getCustomConfigByRequest(req) || {};

    return allowScholar ? {...params, allow_scholar: true} : params;
};

module.exports.route = [
    apiSetup,
    validateCSRF,
    (req, res) => {
        req.api
            .authPasswordMultiStepStart(getParams(req, req.body))
            .then((result) => {
                const {body: {magic_link_email = '', auth_methods = [], track_id} = {}} = result;

                if (magic_link_email) {
                    result.body.magic_link_email = req._controller.decodePunycodeEmail(magic_link_email);
                }

                if (auth_methods.includes(SAML_SSO_AUTH_METHOD)) {
                    result.body.samlSsoUrl = urlFormat({
                        ...req._controller.getUrl(),
                        pathname: '/auth/sso/submit',
                        query: {track_id}
                    });
                }

                return result;
            })
            .then((result) => res.json(result.body || result))
            .catch((errors) => {
                return res.json({
                    status: 'error',
                    errors: errors || []
                });
            });
    }
];

module.exports.middleware = function multiStepAuthStart(req, res, next) {
    if (res.locals.mode && res.locals.mode !== 'welcome') {
        return next();
    }
    const storeDraft = res.locals.store || {};
    const commonDraft = storeDraft.common || {};
    const authDraft = storeDraft.auth || {};
    const unitedAccounts = authDraft.unitedAccounts || {};
    const processedAccount = authDraft.processedAccount;

    if (!processedAccount || !unitedAccounts[processedAccount.uid]) {
        return next();
    }

    const account = unitedAccounts[processedAccount.uid];

    if (account.preferred_auth_method && account.allowed_auth_methods) {
        const isSocialAuth = account.preferred_auth_method.indexOf('social_') > -1;
        const isMultiStepAuth = account.allowed_auth_methods.some((method) =>
            ['password', 'magic_link', 'magic', 'magic_x_token', 'rfc_totp', 'otp', 'sms_code'].includes(method)
        );

        if (isSocialAuth && !isMultiStepAuth) {
            return next();
        }
    }

    // Кастомные случаи, когда приходится изменять предпочитаемый способ авторизации, полученный от бэкенда
    const setValidAuthMethod = (currentAccount, authMethods) => {
        if (!currentAccount.preferred_auth_method) {
            currentAccount.preferred_auth_method = (authMethods && authMethods[0]) || 'password';
        }

        if (!currentAccount.allowed_auth_methods) {
            currentAccount.allowed_auth_methods = (Array.isArray(authMethods) && authMethods) || ['password'];
        }

        // Если один пользователь в саджесте, он не авторизован и предпочитаемый способ авторизации через QR-код, то
        // по умолчанию показываем вход по одноразовому паролю
        // TODO: убрать в рамках тикета PASSP-22749
        if (currentAccount.preferred_auth_method === 'magic') {
            currentAccount.preferred_auth_method = 'otp';
            return;
        }

        // Если предпочитаемый способ авторизации QR-код через приложение "Яндекс", то необходимо проверить включен ли
        // эксперимент для данного способа авторизации
        // TODO: убрать, когда эксперимент выкатим на 100%
        if (currentAccount.preferred_auth_method === 'magic_x_token' && !authDraft.isQRAuthEnabled) {
            currentAccount.preferred_auth_method =
                currentAccount.allowed_auth_methods.filter((method) => method !== 'magic_x_token')[0] || 'password';
        }
    };

    const params = {
        login: account.login
    };

    if (authDraft.process_uuid) {
        params.process_uuid = authDraft.process_uuid;
    }

    ['retpath', 'fretpath', 'clean', 'origin'].forEach((param) => {
        if (commonDraft[param]) {
            params[param] = commonDraft[param];
        }
    });

    return req.api
        .authPasswordMultiStepStart(getParams(req, params))
        .then(({body}) => {
            if (body.status === 'error') {
                throw body.errors;
            }

            const {
                can_register: canRegister,
                login,
                csrf_token: csrf,
                track_id: trackId,
                magic_link_email: email,
                auth_methods: authMethods
            } = body;

            if (canRegister) {
                return req._controller.redirectToLocalUrl({
                    pathname: '/auth',
                    query: Object.assign({}, req.query, {
                        login
                    })
                });
            }

            authDraft.magicCSRF = csrf;
            commonDraft.track_id = trackId;

            if (email) {
                const decodedEmail = req._controller.decodePunycodeEmail(email);

                account.email = decodedEmail;
                processedAccount.email = decodedEmail;
            }

            setValidAuthMethod(account, authMethods);
            setValidAuthMethod(processedAccount, authMethods);

            if (authMethods.includes(SAML_SSO_AUTH_METHOD)) {
                processedAccount.samlSsoUrl = urlFormat({
                    ...req._controller.getUrl(),
                    pathname: '/auth/sso/submit',
                    query: {track_id: trackId}
                });
            }

            return next();
        })
        .catch((err) => {
            setValidAuthMethod(account, []);
            setValidAuthMethod(processedAccount, []);

            res.locals.mode = 'edit';

            PLog.warn()
                .logId(req.logID)
                .type('auth.v2.multiStepAuthStart')
                .write(err);

            return next();
        });
};
