const url = require('url');
const apiSetup = require('./common/apiSetup');
const config = require('../configs/current');
const langs = config.langs;
const locs = require('../loc/frontend.json');
const PLog = require('plog');
const got = require('got');
const multer = require('multer');
const validateCSRF = require('./common/validateCSRF.js');
const storage = multer.diskStorage({
    destination(req, file, cb) {
        cb(null, '/tmp');
    },
    filename(req, file, cb) {
        cb(null, `${Date.now()}-${file.originalname}`);
    }
});
const upload = multer({storage});
const langSetup = [
    function(req, res, next) {
        if (!req.blackbox && req.body.language) {
            req.blackbox = {
                dbfields: {
                    'userinfo.lang.uid': req.body.language
                }
            };
        }
        next();
    },
    function(req, res, next) {
        req._controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('avatars')
                    .write(err);
            })
            .done(function() {
                return next();
            });
    },
    function(req, res, next) {
        const language = langs[langs.indexOf(res.locals.language)] || 'ru';

        res.loc = locs[language];

        if (req.api) {
            req.api.language(language);
        }

        next();
    }
];

exports.routes = {};

exports.route = function(app) {
    const routes = this.routes;

    app.post('/profile/avatars/legacy/', routes.load); // isCropNotSupported
    app.get('/profile/avatars/legacy/', function(req) {
        return req._controller.redirectToFrontpage();
    });
    app.get('/profile/avatars/image/', routes.avatar); // image proxy
    app.post('/profile/avatars/load/', upload.single('file'), routes.loadAvatar); // load avatar
};

exports.routes.load = [
    // isCropNotSupported
    apiSetup,
    langSetup,
    validateCSRF,
    function(req, res) {
        const files = req.files || {};
        const file = files[Object.keys(files)[0]];

        req.api
            .postChangeAvatar({
                track_id: req.body.track_id,
                url: req.body.url,
                file
            })
            .then(function() {
                return res.redirect(url.format(req._controller.getModePassportUrl()));
            })
            .catch(function(error) {
                PLog.warn()
                    .logId(req.logID)
                    .type('legacy avatar load')
                    .write(error);
                return res.redirect(url.format(req._controller.getModePassportUrl()));
            });
    }
];

exports.routes.avatar = [
    // this is for proxy images with url loading
    function(req, res) {
        const imgUrl = req.query && req.query.path;
        const sendError = (error) => {
            PLog.warn()
                .logId(req.logID)
                .type('tried to proxy avatar image from avatars.mdst.yandex.net')
                .write(error);

            return res.status(500).sendFile('avatar-default.png', {
                root: 'public/i/',
                headers: {
                    'Content-Type': 'image/png'
                }
            });
        };

        if (imgUrl) {
            const stream = got.stream(imgUrl);

            stream.on('error', (error) => {
                sendError(error);
            });

            stream.pipe(res);
        } else {
            sendError('no img path provided');
        }
    }
];

exports.routes.loadAvatar = [
    apiSetup,
    langSetup,
    validateCSRF,
    function(req, res) {
        req.api
            .postChangeAvatar({
                track_id: req.body.track_id,
                url: req.body.url,
                file: req.file
            })
            .then(function(result) {
                res.json(result);
            })
            .catch(function(error) {
                res.json(error);
            });
    }
];

exports.loadImage = [
    // upload avatar
    apiSetup,
    function(req, res) {
        const files = req.files || {};
        const file = files[Object.keys(files)[0]];

        req.api
            .postChangeAvatar({
                track_id: req.body.track_id,
                url: req.body.url,
                file,
                is_temp: req.body.isTemp
            })
            .then(function(result) {
                res.json(result);
            })
            .catch(function(error) {
                res.json(error);
            });
    }
];

exports.deleteAvatar = [
    apiSetup,
    validateCSRF,
    function(req, res) {
        req.api
            .deleteAvatar({track_id: req.body.track_id, id: req.body.id})
            .then(function(result) {
                res.json({status: 'ok', track_id: result.body.track_id});
            })
            .catch(function(error) {
                res.json(error);
            });
    }
];

exports.getAvatarsTrack = [
    // avatars/init new avatars
    apiSetup,
    function(req, res) {
        req.api
            .genericAvatar('/1/change_avatar/init/', {uid: req.body.uid})
            .then(function(response) {
                res.json(response);
            })
            .catch(function(error) {
                res.json(error);
            });
    }
];

exports.setDefaultAvatarById = [
    // avatars/default new avatars
    apiSetup,
    validateCSRF,
    function(req, res) {
        req.api
            .setDefaultAvatarById({
                track_id: req.body.track_id,
                avatar_id: req.body.id
            })
            .then(function(result) {
                if (result && result.body.status === 'ok') {
                    res.json(result);
                }
            })
            .catch(function(error) {
                res.json(error);
            });
    }
];

function checkUidFromQuery(req, res, next) {
    const controller = req._controller;
    const queryUid = req.query && req.query.uid;

    if (queryUid) {
        return controller
            .getAuth()
            .sessionID({multisession: 'yes', allow_child: 'yes'})
            .then(function(sessionData) {
                if (sessionData && sessionData.users) {
                    const sessionUids = sessionData.users.map((user) => user.uid && user.uid.value);

                    if (sessionUids.length !== 0 && sessionUids.includes(queryUid)) {
                        res.locals.queryUid = queryUid;
                        PLog.info()
                            .logId(req.logID)
                            .type('profile.passport, checkUidFromQuery')
                            .write('uid from query string is found in sessionID data');
                    } else {
                        PLog.info()
                            .logId(req.logID)
                            .type('profile.passport, checkUidFromQuery')
                            .write('uid from query string not found in sessionID data');
                    }
                }
                return next();
            })
            .catch(function(error) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport, checkUidFromQuery')
                    .write(error);
                return next();
            });
    }

    return next();
}

exports.checkUidFromQuery = checkUidFromQuery;
