var _ = require('lodash');
var assert = require('assert');
var inherit = require('inherit');
var Interface = require('../lib/Interface');
const {SERVICE_ALIASES} = require('../lib/tvm');

var XMLRPCClient = inherit({
    __constructor: function(logger, host, port, path) {
        assert(logger instanceof require('plog'), 'Logger should be a logger instance');
        assert(host && typeof host === 'string', 'Host string required');
        assert(port && typeof port === 'string', 'Port string required');
        assert(path && typeof path === 'string', 'Path string required');

        logger.verbose(`Connecting to xmlrpc server at ${host}:${port}${path}`);
        this.connection = require('xmlrpc').createClient({
            host: host,
            port: port,
            path: path
        });

        this._logger = logger;
    },

    getConnection: function() {
        return this.connection;
    },

    call: function(method, params) {
        var deferred = require('when').defer();

        this._logger.verbose('Calling xmlrpc method', method, 'with', params);
        this.getConnection().methodCall(method, params, function(error, response) {
            if (error) {
                deferred.reject(error);
            } else {
                deferred.resolve(response);
            }
        });

        return deferred.promise;
    }
});

var BillingService = inherit(
    {
        __constructor: function(DAO, token, namespace) {
            assert(token && typeof token === 'string', 'Token string required');
            assert(namespace && typeof namespace === 'string', 'Namespace string required');
            assert(this.__self.daoInterface.implementedBy(DAO), 'DAO should comply with billing service dao interface');

            this._dao = DAO;
            this._token = token;
            this._namespace = namespace;
        },

        /**
         * Calls DAO with with params and handles status: error in response
         *
         * @param {string} method   Name of the method to call
         * @param {object} params   Params hash
         * @returns {when.promise}
         * @private
         */
        _call: function(method, params) {
            return this._dao.call(`${this._namespace}.${method}`, [this._token, params]).then(function(response) {
                if (response.status && response.status === 'error') {
                    throw new Error(`DAO responded with "status: error", response: ${JSON.stringify(response)}`);
                }

                return response;
            });
        },

        /**
         * Get payment methods available for user
         * Returns a promise, that resolves with the requested info
         * @see https://wiki.yandex-team.ru/Balance/Simple/XMLRPC#balancesimple.listpaymentmethods
         *
         * @param {string} uid      Passport user id
         * @param {string} userIp   Ip the user comes from
         * @returns {when.promise}
         */
        listPaymentMethods: function(uid, userIp) {
            return this._call('ListPaymentMethods', {
                user_ip: userIp,
                uid: uid
            });
        }
    },
    {
        //Static expected interface
        daoInterface: new Interface('call:2')
    }
);

var starRegexp = /\*+/;
var groupingRegexp = /.{1,4}/g;

/**
 * Parses single payment method from BillingService.listPyamentMethods response into type, name, details and a link
 * @param {Object} singleMethod   Object with BillingService.listPyamentMethods single method
 * @returns {}
 */
var listPaymentMethodsAdapter = function(singleMethod) {
    //Type
    var type = {
        yandex_money: 'yamoney',
        card: 'card',
        phone: 'phone'
    }[singleMethod.type];

    if (!type) {
        return null; //Ignore unknown types
    }

    //Name
    var name;

    if (singleMethod.name) {
        name = singleMethod.name;
    } else if (type === 'card') {
        name = singleMethod.system || 'Банковская карта';
    } else {
        //Generic by type
        name = {
            yamoney: 'Яндекс.Деньги',
            phone: 'Телефон'
        }[type];
    }

    if (!name) {
        throw new Error('Could not determine method name');
    }

    //Details
    var details = singleMethod.number;

    if (type === 'card') {
        var length = details.replace(starRegexp, '').length;
        var xes = new Array(16 - length + 1).join('x'); //As much x's as needed to fill length to 16 characters

        details = details
            .replace(starRegexp, xes) //Replace star with xxx's
            .match(groupingRegexp) //Split into groups of four
            .join(' ');
    }

    //Link
    var link = {
        yamoney: 'https://money.yandex.ru/',
        phone: 'http://phone-passport.yandex.ru/phones'
    }[type];

    //Result
    var result = {
        type: type,
        name: name,
        details: details
    };

    if (link) {
        result.link = link;
    }

    return result;
};

//Express route
module.exports = function(req, res, next) {
    var logger = new (require('plog'))(req.logID, 'passport', 'profile', 'billing');

    var domain = (req.headers.host && req.headers.host.split('yandex.')[1]) || 'ru';

    if (domain !== 'ru') {
        logger.debug('Domain is %s, should be "ru"', domain);
        return res.redirect('/passport');
    }

    var ExpressAuthController = require('pcontroller/Authentication');
    var authController = new ExpressAuthController(req, res, req.logID, SERVICE_ALIASES).setBlackbox(
        require('../configs/current').paths.blackbox
    );

    require('../lib/passport-api')
        .client({
            ...req,
            country: 'ru'
        })
        .then(function(api) {
            return api.params('language');
        })
        .then(function(langResponse) {
            return (langResponse && langResponse.body && langResponse.body.language) || 'ru';
        })
        .then(function(lang) {
            if (lang !== 'ru') {
                logger.debug('Language is %s, should be "ru"', lang);
                res.redirect('/passport');

                return require('when').reject();
            }

            return true;
        }, next) // If there was an error fetching lang, handle it
        .then(function() {
            logger.debug('Logging in');
            return authController.loggedIn();
        })
        .then(
            function(loggedIn) {
                if (!loggedIn) {
                    logger.debug('Not logged in, redirect to authorization');
                    authController.authorize();
                    return require('when').reject();
                }

                logger.debug('Logged in');
                return loggedIn;
            },
            function(error) {
                if (error && error.code !== 'need_resign') {
                    logger.warn('Error during authorization, redirecting to /auth');
                    authController.authorize();
                }
                return require('when').reject();
            }
        )
        .then(function() {
            if (authController.isHosted()) {
                logger.debug('Billing info not allowed for hosted users');
                res.redirect('/passport');
                return require('when').reject();
            }
        })
        .then(function() {
            var billingConfig = require('../configs/current').billing;
            var billingUrl = require('url').parse(billingConfig.url);

            var billing = new BillingService(
                new XMLRPCClient(logger, billingUrl.hostname, billingUrl.port, billingUrl.pathname),
                billingConfig.token,
                'BalanceSimple'
            );

            logger.debug('Fetching payment methods');
            return billing.listPaymentMethods(authController.getUid(), req.ip);
        })
        .then(function(paymentMethods) {
            logger.debug('Got payment methods:', paymentMethods);

            var PaymentMethodsView = require('../pages/profile/billing/billing.view');
            var view = new PaymentMethodsView();
            var Method = PaymentMethodsView.PaymentMethodsGroup.PaymentMethod;

            paymentMethods = paymentMethods.payment_methods || [];

            logger.debug('Adapting payment methods from billing to view, total of %d', _.size(paymentMethods));
            _(paymentMethods)
                .map(function(singleMethod) {
                    logger.verbose('Payment method: ', singleMethod);
                    return listPaymentMethodsAdapter(singleMethod);
                })
                .omit(function(parsedMethod) {
                    logger.verbose('Adapted payment method: ', parsedMethod);
                    return parsedMethod === null;
                })
                .each(function(parsedMethod) {
                    logger.verbose('Adding method to view');
                    view.addMethod(
                        new Method(parsedMethod.name, parsedMethod.details, parsedMethod.link),
                        parsedMethod.type
                    );
                });

            _.assign(res.locals, {
                language: 'ru',
                domain: 'ru'
            });

            logger.debug('Rendering');
            res.render('profile.billing.ru.js', view.compile());
        })
        .catch(function(error) {
            if (error && error.code !== 'need_resign') {
                logger.error(error);
                return next();
            }
        })
        .done();
};

module.exports.XMLRPCClient = XMLRPCClient;
module.exports.BillingService = BillingService;
module.exports.listPaymentMethodsAdapter = listPaymentMethodsAdapter;
