/* eslint-disable global-require */
const PUtils = require('putils');
const plog = require('plog');
const apiSetup = require('./common/apiSetup');
const getUid = require('./common/getUid');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const config = require('../configs/current');
const when = require('when');
const addresses = require('./profile.addresses');
const billing = require('./profile.billing');
const appPasswords = require('./profile.apppasswords');
const alice = require('./alice.settings.js');
const validateCSRF = require('./common/validateCSRF.js');
const getAccounts = require('./auth_intranet/middlewares/getAccounts');
const forgetAccount = require('./auth_intranet/middlewares/forgetAccount');
const getInputLogin = require('./auth_intranet/middlewares/getInputLogin');
const reAuthPasswordSubmit = require('./auth_intranet/middlewares/reAuthPasswordSubmit');
const restoreLogin = require('./restore.login');
const deleteAccount = require('./profile.delete_account');
const registration = require('./registration.v2');
const complete = require('./complete.v2');
const registerNewLiteWithEmail = require('./registration.lite').registerLiteEmail;
const registerLiteExperiment = require('./registration.lite').registerExperimentLite;
const qr = require('qr-image');
const connect = require('./registration.connect');
const profilePassport = require('./profile.passport.v2');
const takeOutHandles = require('./takeout').takeOutHandles;
const BillingApi = require('../lib/api/billing');
const lookup = require('../lib/getLookup.js');
const serviceTicketsSetup = require('./common/serviceTicketsSetup');
const userTicketsSetup = require('./common/userTicketsSetup');
const afishaApiSetup = require('./common/afishaApiSetup');
const getDisplaynameData = require('./avatarDisplayname').getDisplaynameData;
const getTrack = require('./common/getTrack');
const url = require('url');
const family = require('./profile.family');
const unsubscribe = require('./profile.unsubscribe');
const getAccountInfo = require('./security/getAccountInfo');
const social = require('./social').social;
const PraktikumApi = require('../lib/api/praktikum');
const {SERVICE_ALIASES} = require('../lib/tvm');
const sendSubs = require('./subs').sendSubs;
const getSubsList = require('./subs').getSubsList;
const langSetup = require('./common/langSetup');
const manageSms2fa = require('./profile.phones').manageSms2fa;
const secureBindCommit = require('./profile.phones').secureBindCommit;
const subscriptions = require('./profile.subscriptions');
const userValidateCommit = require('./user-validate').commitUserValidate;
const userValidateCommit3ds = require('./user-validate/3ds/api').commit3ds;
const userValidateSubmit3ds = require('./user-validate/3ds/api').submit3ds;
const userValidateGetRetpath3ds = require('./user-validate/3ds/api').getRetpath3ds;
const validateUserInput = require('./common/validateUserInput.js');
const checkSMS2faEnabled = require('./authv2/changePassword/checkSms2FAEnabled');
const setupUserTicket = require('./common/userTicketsSetup');
const documents = require('./documents/index.js');
const getPlusBalance = require('./common/getPlusBalance');
const {authVerifyCommitRoute} = require('./authv2/authVerifyCommitRoute');

const setup = [
    function trackFilter(req, res, next) {
        const trackId = req.body && req.body.track_id;

        if (!trackId) {
            return res.sendStatus(403);
        }

        return next();
    },
    function ajaxFilter(req, res, next) {
        const isAjax = req.headers['x-requested-with'] === 'XMLHttpRequest';

        if (!isAjax) {
            return res.sendStatus(403);
        }

        return next();
    },
    apiSetup
];

const createHoneyPotFilter = (data) => (req, res, next) => {
    if (!req._honeypot) {
        return next();
    }

    return res.status(200).send(data);
};

const afishaSetup = [serviceTicketsSetup, userTicketsSetup, afishaApiSetup];

const checkCaptcha = [
    apiSetup,
    validateCSRF,
    (req, res, next) => {
        if (req.body.answer) {
            return req.api
                .captchaCheck(req.body)
                .then((response = {}) => {
                    if (response.body && response.body.correct) {
                        return next();
                    }

                    res.json({
                        status: 'error',
                        errors: ['captcha.not_matched']
                    });
                })
                .catch((error = {}) => {
                    res.json({
                        status: 'error',
                        error
                    });
                });
        }
        return next();
    }
];

function processAfishaRequest(req, res) {
    if (!req.afishaApi) {
        return res.json({status: 'error'});
    }

    const {service, afishaMethods, orderId, page, pageSize, prepareData} = req.afishaReqParams;

    const apiMethod = afishaMethods[service];

    let apiFn = req.afishaApi[apiMethod];

    if (['getOrder'].includes(apiMethod)) {
        apiFn = apiFn.bind(req.afishaApi, orderId);
    } else {
        const pageOffset = page ? +pageSize * (+page - 1) : 0;

        apiFn = apiFn.bind(req.afishaApi, +pageSize, pageOffset);
    }

    return apiFn()
        .then((result) => {
            if (result.error || result.status !== 200) {
                return res.json({status: 'error', page});
            }

            const data = prepareData[service](result.data);

            if (page) {
                data.page = page - 1;
            }

            return res.json(data);
        })
        .catch((errors) => res.json({status: 'error', page, errors}));
}

const setupForceCleanWeb = [
    function(req, res, next) {
        req.body.force_clean_web = true;

        return next();
    }
];

function updateCookies(req, res) {
    const controller = req._controller;

    req.api
        .bundleSession({track_id: req.body.track_id, retpath: req.body.retpath})
        .then(function(result) {
            const body = result.body;

            controller.augmentResponse(body);

            if (body.status === 'ok') {
                res.json({
                    status: 'ok',
                    retpath: body.retpath
                });
            } else {
                res.json({
                    status: 'error',
                    errors: body.errors
                });
            }
        })
        .catch(function() {
            res.json({
                status: 'error'
            });
        });
}

function genericValidation(apiMethod) {
    return [
        setup,
        validateCSRF,
        function(req, res) {
            req.api[apiMethod](req.body).then(
                function(result) {
                    res.json(result.body || result);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ];
}

module.exports = {
    'auth-verify-commit': authVerifyCommitRoute,
    password: genericValidation('validatePassword'),
    phone: genericValidation('validatePhone'),
    'validate-firstname': genericValidation('validateFirstName'),
    'validate-lastname': genericValidation('validateLastName'),
    'validate-displayname': genericValidation('validateDisplayName'),
    'validate-phone': genericValidation('validatePhoneV2'),
    'phone-confirm-by-call-entry': genericValidation('validatePhoneV2'),
    getRestoreLoginCode: genericValidation('getRestoreLoginCode'),
    'toggle-enter-without-password': [getUid, genericValidation('toggleEnterWithoutPassword')],
    textcaptcha: [
        setup,
        validateCSRF,
        getYaExperimentsFlags,
        (req, res, next) => {
            const {experiments: {flags = []} = {}} = res.locals;

            req.body.ocr = true;

            if (flags.includes('reg_wave_exp')) {
                req.body.wave = true;
            }

            return next();
        },
        (req, res) => {
            req.api.captchaGenerate(req.body).then(
                function(result) {
                    res.json(result.body || result);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],
    audiocaptcha: genericValidation('audioCaptchaGenerate'),
    captcha: genericValidation('captchaCheck'),
    answer: genericValidation('validateHint'),
    'user-question': genericValidation('validateHint'),
    'phone-confirm-entry': genericValidation('validatePhone'),
    'write-track': genericValidation('writeTrack'),
    'phone-confirm-code-submit': [
        setup,
        validateCSRF,
        getYaExperimentsFlags,
        function(req, res, next) {
            if (req.body.isCodeWithFormat) {
                const expFlags = (res.locals.experiments && res.locals.experiments.flags) || [];

                if (expFlags.includes('rounded-input-exp')) {
                    req.body.code_format = 'by_3';
                    return next();
                }

                req.body.code_format = 'by_3_dash';
            }

            return next();
        },
        function(req, res, next) {
            if (req.body.checkCaptcha === 'false') {
                return next();
            }

            return req.api.captchaCheckStatus(req.body).then(
                (result = {}) => {
                    const {body} = result;

                    if (body && (!body.is_required || body.is_recognized)) {
                        return next();
                    }

                    return res.json({
                        status: 'error',
                        errors: ['captcha.not_checked']
                    });
                },
                (errors) =>
                    res.json({
                        status: 'error',
                        errors: errors || []
                    })
            );
        },
        function(req, res) {
            /* jshint unused:false */

            let confirmationRequest;

            switch (req.body.mode) {
                case 'confirmAndBindSecureWithAlias':
                    confirmationRequest = req.api.confirmAndBindSecureWithAliasPhoneSubmit(req.body);
                    break;
                case 'confirmAndBindSecure':
                    confirmationRequest = req.api.confirmAndBindSecurePhoneSubmit(req.body.number);
                    break;
                case 'tracked':
                    confirmationRequest = req.api.confirmTrackedPhoneSubmit(req.body);
                    break;
                case 'trackedWithUpdate':
                    confirmationRequest = req.api.confirmTrackedPhoneWithUpdateSubmit();
                    break;
                case 'confirmForRestore':
                    confirmationRequest = req.api.checkSecurePhoneRestore(req.body.number);
                    break;
                case 'restore':
                    confirmationRequest = req.api.restoreCheckPhone(req.body);
                    break;
                case 'restoreBind':
                    confirmationRequest = req.api.restoreBindSumitPhone({number: req.body.number});
                    break;
                default:
                    confirmationRequest = req.api.confirmPhoneSubmit(req.body);
                    break;
            }

            confirmationRequest.then(
                function(result) {
                    if (req.body.mode === 'confirmForRestore' && result.status === 'ok') {
                        // вот это странное поведение нужно для восстановения 2fa
                        // ручка валидации телефона не отправляет смс, поэтому дергаем ручку сами
                        req.api.confirmTrackedPhoneSubmit(req.body).then(
                            function(data) {
                                res.json(data);
                            },
                            function(errors) {
                                res.json({
                                    status: 'error',
                                    errors: errors || []
                                });
                            }
                        );
                    } else {
                        res.json(result);
                    }
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],
    'confirm-phone-check-status': [
        setup,
        validateCSRF,
        (req, res) => {
            req.api
                .confirmPhoneCheckStatus(req.body)
                .then((result) => res.json(result))
                .catch((error) => res.json({status: 'error', errors: error}));
        }
    ],
    'get-dashboard-external-data': [
        setup,
        validateCSRF,
        function(req, res, next) {
            const {pageSize, page, service, orderId, publicId, lang} = req.body;

            const prepareData = profilePassport.prepareFactory({
                req,
                lang,
                publicId
            });

            const afishaMethods = {
                favAfishaData: 'getFavoriteEvents',
                afishaData: 'getOrders',
                afishaDataInfo: 'getOrder'
            };

            if (service in afishaMethods) {
                if (!orderId && !pageSize) {
                    return res.json({status: 'error'});
                }

                req.afishaReqParams = {
                    service,
                    afishaMethods,
                    orderId,
                    page,
                    pageSize,
                    prepareData
                };

                return next();
            }

            const methods = {
                marketData: 'externalDataMarket', // @DEPRECATED PASSP-25254
                musicData: 'externalDataMusic',
                collectionsData: 'externalDataCollections',
                favMarketData: 'favExternalDataMarket',
                diskData: 'externalDataDisk',
                videoData: 'externalDataVideo',
                mapsBookmarksData: 'externalDataMapBookmarks',
                mapsBookmarksDataInfo: 'externalDataMapBookmarkInfo'
            };

            if (!methods.hasOwnProperty(service) || (!orderId && !pageSize)) {
                return res.json({status: 'error'});
            }

            return req.api[methods[service]](pageSize || orderId, page).then(
                (result) => {
                    if (!result.hasOwnProperty('body') || result.body.status !== 'ok') {
                        return res.json({status: 'error', page});
                    }

                    if (service === 'mapsBookmarksData') {
                        return prepareData[service](result.body)
                            .then((data) => res.json(data))
                            .catch(() => res.json({status: 'error', page}));
                    }

                    const data = prepareData[service](result.body);

                    if (page) {
                        data.page = page - 1;
                    }

                    return res.json(data);
                },
                (errors) => res.json({status: 'error', page, errors})
            );
        },
        afishaSetup,
        processAfishaRequest
    ],
    // Phone confirmation commit
    'phone-confirm-code': [
        setup,
        validateCSRF,
        checkCaptcha,
        function(req, res) {
            let confirmationRequest;

            switch (req.body.mode) {
                case 'confirmAndBindSecureWithAlias':
                    confirmationRequest = req.api.confirmAndBindSecureWithAliasPhoneCommit(
                        req.body.code,
                        req.body.password
                    );
                    break;
                case 'confirmAndBindSecure':
                    confirmationRequest = req.api.confirmAndBindSecurePhoneCommit(req.body.code, req.body.password);
                    break;
                case 'tracked':
                    confirmationRequest = req.api.confirmTrackedPhoneCommit(req.body.code);
                    break;
                case 'trackedWithUpdate':
                    confirmationRequest = req.api.confirmTrackedPhoneWithUpdateCommit(req.body.code);
                    break;
                case 'confirmForRestore':
                    confirmationRequest = req.api.confirmTrackedPhoneCommit(req.body.code);
                    break;
                case 'restore':
                    confirmationRequest = req.api.restoreCheckCode({code: req.body.code});
                    break;
                case 'restoreBind':
                    confirmationRequest = req.api.restoreBindCheckCode({code: req.body.code});
                    break;
                default:
                    confirmationRequest = req.api.confirmPhoneCommit(req.body.code);
                    break;
            }

            confirmationRequest.then(
                function(result) {
                    res.json(result);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || (!Array.isArray(errors) && errors.errors) || []
                    });
                }
            );
        }
    ],
    login: [
        setup,
        validateCSRF,
        function(req, res) {
            req.api.validateLogin(req.body).then(
                function(result) {
                    res.json(result.body);
                },
                function(error) {
                    res.json(error);
                }
            );
        }
    ],
    suggest: [
        setup,
        validateCSRF,
        function(req, res) {
            req.api.suggestLogin(req.body).then(
                function(result) {
                    res.json(result.body);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],
    suggestV2: [
        setup,
        validateCSRF,
        validateUserInput({
            bodySchema: {
                type: 'object',
                properties: {
                    firstname: {type: 'string', maxLength: 256},
                    lastname: {type: 'string', maxLength: 256},
                    login: {type: 'string', maxLength: 30},
                    track_id: {type: 'string', maxLength: 34},
                    csrf_token: {type: 'string', maxLength: 256},
                    timeout: {type: 'string', maxLength: 256}
                },
                additionalProperties: false
            },
            querySchema: {
                type: 'object',
                additionalProperties: false
            }
        }),
        function(req, res) {
            req.api.suggestLoginV2(req.body).then(
                function(result) {
                    const answer = result.body;

                    answer.logins = answer.suggested_logins;
                    delete answer.suggested_logins;
                    res.json(answer);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],
    'history-answer': [
        setup,
        validateCSRF,
        function(req, res) {
            const data = {
                track_id: req.body.track_id,
                answer: req.body.history_answer
            };

            if (req.body.history_question) {
                const qa = req.body.history_question.split(':');

                data.question_id = qa[0] || null;
                data.question = qa[1] || null;
            }

            req.api.checkHistoryQuestions(data).then(
                function(result) {
                    const answr = result.body;

                    answr.validation_errors = answr.errors;
                    delete answr.errors;

                    res.json(answr);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],

    'current-answer': [
        setup,
        validateCSRF,
        function(req, res) {
            const data = {
                track_id: req.body.track_id,
                answer: req.body.current_answer
            };

            req.api.checkCurrentQuestion(data).then(
                function(result) {
                    const answr = result.body;

                    answr.validation_errors = answr.errors;
                    delete answr.errors;
                    res.json(answr);
                },
                function(error) {
                    res.json({error});
                }
            );
        }
    ],

    'twofa-app-link-via-sms': require('./profile.access.2fa').sendYakeyLinkViaSms,
    'twofa-pin-entry': require('./profile.access.2fa').validatePin,
    'feedback-email': genericValidation('restoreSemiAutoEmailValidate'),
    pin: genericValidation('restore2faCheckPin'),
    'social-profiles': genericValidation('socialProfileData'),
    'access/token/new': require('./profile.access').clientside,
    'access/pva': require('./profile.access').clientside,
    'access/_testing/csrf': require('./profile.access').clientside,
    'profile/journal': require('./profile.journal').journal,
    'profile/actions': require('./profile.journal').actions,
    'profile/apppassword/create/': appPasswords.profileCreateApppassword,
    checkjsload: [
        setup,
        validateCSRF,
        function(req, res) {
            const dataForTrack = {
                track_id: req.body.track_id,
                language: req.body.language,
                check_js_load: true
            };

            req.api
                .writeTrack(dataForTrack)
                .then(function(result) {
                    res.json(result.body);
                })
                .catch(function(e) {
                    plog.warn()
                        .logId(req.logID)
                        .type('checkjsload_write-track-error')
                        .write(e);
                    res.json({
                        status: 'error',
                        errors: e || []
                    });
                });
        }
    ],
    enableSocialSubscribe: [
        setup,
        validateCSRF,
        function(req, res) {
            const data = {
                track_id: req.body.track_id,
                profile_id: req.body.profile_id,
                sid: req.body.sid
            };

            const controller = req._controller;

            req.api.socialSubscribeService(data).then(
                function(result) {
                    let setCookie;

                    controller.augmentResponse(result.body);

                    if (result.body.cookies && result.body.cookies.length) {
                        setCookie = {cookies: true, track_id: result.body.track_id};
                    }
                    res.json({
                        status: result.body.status,
                        state: result.body.state || 'normal',
                        user: setCookie ? setCookie : 'default'
                    });
                },
                function(error) {
                    res.json({error});
                }
            );
        }
    ],
    disableSocialSubscribe: [
        setup,
        validateCSRF,
        function(req, res) {
            const data = {
                track_id: req.body.track_id,
                profile_id: req.body.profile_id,
                sid: req.body.sid
            };

            const controller = req._controller;

            req.api.socialUnsubscribeService(data).then(
                function(result) {
                    let setCookie;

                    controller.augmentResponse(result.body);
                    if (result.body.cookies && result.body.cookies.length) {
                        setCookie = {cookies: true, track_id: result.body.track_id};
                    }
                    res.json({
                        status: result.body.status,
                        state: result.body.state || 'normal',
                        user: setCookie ? setCookie : 'default'
                    });
                },
                function(error) {
                    res.json({error});
                }
            );
        }
    ],
    allowSocialAuth: [
        setup,
        validateCSRF,
        function(req, res) {
            const body = req.body;
            const params = {
                track_id: body.track_id,
                profile_id: body.profile_id,
                set_auth: body.set_auth,
                current_password: body.current_password,
                passErrors: body.passErrors
            };
            const controller = req._controller;

            req.api.socialAllowAuth(params).then(
                function(result) {
                    const data = result.body;

                    let setCookie;

                    controller.augmentResponse(data);

                    if (data.cookies && data.cookies.length) {
                        setCookie = {cookies: true, track_id: data.track_id};
                    }

                    const response = {
                        status: data.status,
                        state: data.state || 'normal',
                        user: setCookie || 'default'
                    };

                    if (result.body.status === 'error') {
                        response.errors = data.errors;
                    }

                    res.json(response);
                },
                function(error) {
                    res.json({error});
                }
            );
        }
    ],
    delSocialProfile: [
        setup,
        validateCSRF,
        function(req, res) {
            const data = {
                track_id: req.body.track_id,
                profile_id: req.body.profile_id,
                current_password: req.body.current_password,
                passErrors: req.body.passErrors
            };
            const controller = req._controller;

            req.api.socialProfileDelete(data).then(
                function(result) {
                    const body = result.body;

                    let setCookie;

                    controller.augmentResponse(result.body);

                    if (body.cookies && body.cookies.length) {
                        setCookie = {cookies: true, track_id: body.track_id};
                    }

                    const response = {
                        status: body.status,
                        state: body.state || 'normal',
                        user: setCookie ? setCookie : 'default'
                    };

                    if (body.status === 'error') {
                        response.errors = body.errors;
                    }

                    res.json(response);
                },
                function(error) {
                    res.json({error});
                }
            );
        }
    ],
    semiauto: genericValidation('semiautoFieldValidate'),
    getTrackWithUid: [
        apiSetup,
        validateCSRF,
        function(req, res) {
            req.api.getTrackWithUid(req.body.track_id || '').then(
                function(result) {
                    res.json(result.body);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],
    appPwdActivate: [
        setup,
        validateCSRF,
        function(req, res) {
            req.api.appPasswordsActivate(req.body.track_id).then(
                function(result) {
                    res.json(result);
                },
                function(error) {
                    res.json({status: 'error', errors: error});
                }
            );
        }
    ],
    appPwdDeactivate: [
        setup,
        validateCSRF,
        function(req, res) {
            req.api.appPasswordsDeactivate(req.body.track_id).then(
                function(result) {
                    res.json(result);
                },
                function(error) {
                    res.json({status: 'error', errors: error});
                }
            );
        }
    ],
    track: [
        setup,
        (req, res, next) => {
            if (req.body.sk && req.body.sk === PUtils.csrf(null, req.cookies.yandexuid)) {
                return next();
            }

            if (req.body.csrf_token) {
                return validateCSRF(req, res, next);
            }

            return res.status(403).json({});
        },
        function(req, res) {
            const body = req.body;
            const trackType = body.type || 'authorize';

            req.api
                .getTrack(
                    {
                        type: trackType
                    },
                    body.isInit
                )
                .then(
                    function(result) {
                        res.json(result.body);
                    },
                    function() {
                        res.status(403).json({});
                    }
                );
        }
    ],
    accounts: [
        getAccounts,
        function(req, res) {
            return res.json({
                accounts: res.locals.store.one_domik,
                csrf: res.locals.updatedCSRF
            });
        }
    ],
    'accounts/forget': [validateCSRF, forgetAccount],
    'accounts/input-login': [validateCSRF, getInputLogin],
    reAuthPasswordSubmit: [validateCSRF, reAuthPasswordSubmit],
    orgdomain: (function() {
        const route = genericValidation('workspaceValidateDomain');

        route.unshift(function(req, res, next) {
            const controller = req._controller;
            const tld = controller.getTld();

            if (req.body && typeof req.body === 'object') {
                req.body.domain = `${req.body.domain}.${config.workspace.zone.replace('%TLD%', tld)}`;
            }

            next();
        });

        return route;
    })(),
    orgadmin: (function() {
        const route = genericValidation('validateLogin2');

        route.unshift(function(req, res, next) {
            if (req.body && typeof req.body === 'object') {
                req.body.isPdd = true;
                req.body.strict = true;
                req.body.requireDomainExistence = false;
            }

            next();
        });
        return route;
    })(),

    'yasms.secure.bind.submit': [genericValidation('yasmsSecurePhoneSubmit')],
    'yasms.secure.bind.commit': secureBindCommit,
    'yasms.secure.remove.submit': genericValidation('yasmsSecurePhoneRemoveSubmit'),
    'yasms.secure.remove.commit': genericValidation('yasmsSecurePhoneRemoveCommit'),
    'yasms.secure.replace.submit': genericValidation('yasmsSecurePhoneReplaceSubmit'),
    'yasms.secure.replace.commit': genericValidation('yasmsSecurePhoneReplaceCommit'),
    'yasms.secure.aliasify.submit': genericValidation('yasmsSecurePhoneAliasifySubmit'),
    'yasms.secure.aliasify.commit': genericValidation('yasmsSecurePhoneAliasifyCommit'),
    'yasms.secure.dealiasify.submit': genericValidation('yasmsSecurePhoneDealiasifySubmit'),
    'yasms.secure.dealiasify.commit': genericValidation('yasmsSecurePhoneDealiasifyCommit'),
    'yasms.simple.bind.submit': genericValidation('yasmsSimplePhoneSubmit'),
    'yasms.simple.bind.commit': genericValidation('yasmsSimplePhoneCommit'),
    'yasms.simple.securify.submit': [genericValidation('yasmsSimplePhoneSecurifySubmit')],
    'yasms.simple.securify.commit': genericValidation('yasmsSimplePhoneSecurifyCommit'),
    'yasms.secure.securify.commit': genericValidation('yasmsSimplePhoneSecurifyCommit'),
    'yasms.simple.remove': genericValidation('yasmsSimplePhoneRemove'),
    'yasms.operation.cancel': genericValidation('yasmsCancelOperation'),
    'yasms.password.check': [checkCaptcha, genericValidation('yasmsCheckPassword')],
    'yasms.code.resend': genericValidation('yasmsResendCode'),
    'yasms.code.check': genericValidation('yasmsCheckCode'),
    'yasms.prolong': genericValidation('yasmsProlong'),
    'yasms.notify': genericValidation('yasmsSetDefaultForNotify'),
    'yasms.state': genericValidation('yasmsGetState'),
    'yasms.alias.as.email.enable': genericValidation('yasmsEnableAliasAsEmail'),
    'yasms.alias.as.email.disable': genericValidation('yasmsDisableAliasAsEmail'),
    'yasms.sms2faCommit': genericValidation('toggleSms2fa'),
    'yasms.sms2fa': manageSms2fa,
    initTrack: genericValidation('initTrack'),
    'phones.alias.as.login.enable.submit': genericValidation('confirmSecureBoundAndAliasifySubmit'),
    'phones.alias.as.login.enable.commit': genericValidation('confirmSecureBoundAndAliasifyCommit'),
    'phones.alias.as.login.disable.submit': genericValidation('confirmSecureDeleteAliasSubmit'),
    'phones.alias.as.login.disable.commit': [
        apiSetup,
        validateCSRF,
        checkCaptcha,
        genericValidation('confirmSecureDeleteAliasCommit')
    ],

    // profile
    'update.profile': [
        setup,
        validateCSRF,
        setupForceCleanWeb,
        function(req, res) {
            const {body: data} = req;

            req.api
                .profileUpdate(data)
                .then(function(result) {
                    return res.json(result.body || result);
                })
                .catch(function(errors) {
                    return res.json({
                        status: 'error',
                        errors: errors || []
                    });
                });
        }
    ],

    // genericValidation('profileUpdate'),

    // suggest
    'suggest.city': require('./suggest').city,
    passwordConfirm: [
        apiSetup,
        validateCSRF,
        function(req, res, next) {
            const api = req.api;
            const trackId = req.body.track_id;
            const password = req.body.password;

            api.passwordConfirmSubmit({track_id: trackId})
                .then(function(data) {
                    if (data.body.status === 'ok') {
                        return api.passwordConfirmCommit({
                            track_id: trackId,
                            password
                        });
                    }
                    return when.reject(data.body);
                })
                .then(function(data) {
                    if (data.body.status === 'ok') {
                        return next();
                    }

                    return when.reject(data.body);
                })
                .catch(function(err) {
                    return res.json(err);
                });
        },
        updateCookies
    ],
    'registration-code.confirm': [
        apiSetup,
        validateCSRF,
        function(req, res) {
            req.api
                .confirmEmailForRegistrationByCode(req.body)
                .then((result) => {
                    res.json(result.body || result);
                })
                .catch((error) => {
                    res.json({
                        status: 'error',
                        error
                    });
                });
            // return res.json({status: 'ok'});
        }
    ],
    'registration-lite-experiment.submit': [
        apiSetup,
        validateCSRF,
        function(req, res) {
            req.api
                .accountRegisterExperimentEmailSubmit(req.body)
                .then((result) => {
                    res.json(result.body || result);
                })
                .catch((error) => {
                    res.json({
                        status: 'error',
                        error
                    });
                });
        }
    ],
    'registration-lite.submit': require('./registration.lite').sendConfirmationCode,
    rfcTotpRecreateSecret: [
        setup,
        validateCSRF,
        function(req, res) {
            req.api
                .rfcTotpRecreateSecret()
                .then(function(result) {
                    const rfc = result.body || result;

                    res.json({
                        // eslint-disable-next-line no-sync
                        qrSVG: qr.imageSync(rfc.totp_url, {
                            type: 'svg',
                            size: 5
                        }),
                        secret: rfc.secret
                    });
                })
                .catch(function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                });
        }
    ],
    'registration-alternative': [setup, validateCSRF, setupForceCleanWeb, registration.register],
    'registration-kiddish': [setup, validateCSRF, setupForceCleanWeb],
    'registration-neophonish': [setup, validateCSRF, setupForceCleanWeb, registration.registerNeoPhonish],
    'registration-complete': [setup, validateCSRF, setupForceCleanWeb, complete.save],
    'registration-connect': [setup, validateCSRF, setupForceCleanWeb, connect.register],
    'registration-lite': [setup, validateCSRF, setupForceCleanWeb, registerNewLiteWithEmail],
    'registration-lite-experiment.commit': [setup, validateCSRF, setupForceCleanWeb, registerLiteExperiment],
    'get.profile': profilePassport.getBundle,
    'family.get': [validateCSRF, apiSetup, setupUserTicket, family.getFamily],
    'family.leave': [validateCSRF, apiSetup, setupUserTicket, family.leaveFamily],
    'family.member.exclude': [validateCSRF, apiSetup, setupUserTicket, family.excludeMember],
    'family.invite.get': [validateCSRF, apiSetup, setupUserTicket, family.getInviteInfo],
    'family.invite.confirm': [validateCSRF, apiSetup, setupUserTicket, family.confirmInvite],
    'family.invite.cancel': [validateCSRF, apiSetup, setupUserTicket, family.cancelInvite],
    'family.invite.create': [validateCSRF, apiSetup, setupUserTicket, family.createInvite],
    'family.delete': [validateCSRF, apiSetup, setupUserTicket, family.deleteFamily],
    'family.offer.get': [validateCSRF, apiSetup, setupUserTicket, family.getSubscriptionOffer],
    'family.kiddish.create': [validateCSRF, apiSetup, setupUserTicket, family.createKiddish],
    'family.kiddish.edit': [validateCSRF, apiSetup, setupUserTicket, family.editKiddish],
    'family.kiddish.remove': [validateCSRF, apiSetup, setupUserTicket, family.removeKiddish],
    'family.pay.card.unbind': [validateCSRF, apiSetup, setupUserTicket, family.payUnbind],
    'family.pay.card.bind': [validateCSRF, apiSetup, getYaExperimentsFlags, setupUserTicket, family.payBind],
    'family.pay.limits.save': [validateCSRF, apiSetup, setupUserTicket, family.paySaveLimits],
    'family.pay.balance.fill': [validateCSRF, apiSetup, setupUserTicket, family.payResetUserExpenses],

    'get-plus-balance': [validateCSRF, apiSetup, setupUserTicket, getPlusBalance],

    'billing.cardsinfo': [
        createHoneyPotFilter({cards: [], status: 'success'}),
        setup,
        validateCSRF,
        billing.billinCardsInfo
    ],
    'billing.createbinding': [
        createHoneyPotFilter({cards: [], status: 'success', token: '2bda764f533a929f86216104cdd90736'}),
        setup,
        validateCSRF,
        billing.billingCreateBinding
    ],
    'billing.dobinding': [
        createHoneyPotFilter({
            status: 'success',
            origin: 'https://trust.yandex.ru',
            url: 'https://trust.yandex.ru/web/binding?purchase_token=2bda764f533a929f86216104cdd90736'
        }),
        setup,
        validateCSRF,
        billing.billingDoBinding
    ],
    'billing.unbindcard': [
        createHoneyPotFilter({
            status: 'success'
        }),
        setup,
        validateCSRF,
        billing.billingUnbindCard
    ],

    'billing.getnativeproducts': [validateCSRF, billing.bilingGetNativeProducts],
    'billing.submitnativeorder': [validateCSRF, billing.billingSubmitNativeOrder],
    'billing.orderInfo': [validateCSRF, billing.billingOrderInfo],
    'billing.getsubscriptionsinfo': [validateCSRF, billing.getSubscriptionsInfo],
    'billing.gettransitions': [validateCSRF, billing.getTransitions],
    'billing.downgradesubscription': [validateCSRF, billing.downgradeSubscription],
    'billing.yandexbalance': [validateCSRF, userTicketsSetup, billing.yandexBalance],

    'subscriptions.cards': [validateCSRF, langSetup, serviceTicketsSetup, subscriptions.getSubscriptionCards],

    saveQuestion: require('./morda').saveQuestion,
    getAccountData: require('./profile.security').updateData,
    getQuestions: genericValidation('getQuestions'),
    'get.addresses': [validateCSRF, addresses.getAddresses],
    'save.addresses': [validateCSRF, addresses.saveAddresses],
    'geocode.addresses': [validateCSRF, addresses.getLocationByGeoCode],
    'delete.addresses': [validateCSRF, addresses.deleteAddresses],
    'avatars/init': require('./avatars').getAvatarsTrack,
    'avatars/default': require('./avatars').setDefaultAvatarById,
    'avatars/upload': require('./avatars').loadImage,
    'avatars/delete': require('./avatars').deleteAvatar,
    'profile/request-data-archive': takeOutHandles.startTakeoutProcess,
    'profile/get-archive-link': takeOutHandles.getArchiveData,
    'profile/get-archive-password': takeOutHandles.getPassword,
    'profile/get-displayname': getDisplaynameData,
    'restore/login/submit.phone': setup.concat(validateCSRF, restoreLogin.submitPhone),
    'restore/login/submit.code': setup.concat(validateCSRF, restoreLogin.submitCode),
    'restore/login/submit.name': setup.concat(validateCSRF, restoreLogin.submitName),
    'apppasswords.count': appPasswords.getAppPasswordsNumber,
    'apppasswords.list': appPasswords.getTokensList,
    'apppasswords.token-revoke': appPasswords.revokeToken,
    'apppasswords.tokens-revoke': appPasswords.revokeTokens,
    'devices.revoke-tokens': appPasswords.revokeDeviceTokens,
    'get.grouped.tokens': appPasswords.getTokenGroups,
    'alice.settings': alice.getSettings,
    'alice.settings.setHelping': alice.setHelping,
    'alice.devices': alice.getDevices,
    'alice.devices.revoke': alice.revokeDeviceAccess,
    'alice.devices.approve': alice.setDeviceApproveStatus,
    'delete.account.commit': deleteAccount.deleteCommit,
    'delete.account.send.phone': deleteAccount.sendDeleteSms,
    'delete.account.confirm.phone': deleteAccount.checkPhoneCode,
    'delete.account.check.question': deleteAccount.checkQuestion,
    'delete.account.send.email': deleteAccount.sendEmailCode,
    'delete.account.confirm.email': deleteAccount.checkEmailCode,
    'phone/confirm_and_bind_secure/submit': require('./auth/ask').phoneConfirmAndBindSecureSubmit,
    'phone/confirm_and_bind_secure/commit': require('./auth/ask').phoneConfirmAndBindSecureCommit,
    'phone/manage/prolong_valid': require('./auth/ask').phoneProlongValid,
    'email/send_confirmation_email': require('./auth/ask').sendConfirmationEmail,
    'email/confirm/by_code': require('./auth/ask').confirmEmailByCode,
    'email/setup_confirmed': require('./auth/ask').setupConfirmedEmail,
    'change_social/profile': require('./auth/ask').setSocialProfileAuth,
    'auth/additional_data/ask': require('./auth/ask').additionalDataAsk,
    'auth/additional_data/freeze': require('./auth/ask').additionalDataFreeze,
    'phone/confirm_and_bind_secure/submit_v2': require('./authv2/phoneConfirmAndBindSecureSubmit'),
    'phone/confirm_and_bind_secure/commit_v2': require('./authv2/phoneConfirmAndBindSecureCommit'),
    'phone/bind_simple_or_confirm_bound/submit': require('./authv2/bindSimpleOrConfirmBoundPhoneSubmit'),
    'phone/bind_simple_or_confirm_bound/commit': require('./authv2/bindSimpleOrConfirmBoundPhoneCommit'),
    'phone/manage/prolong_valid_v2': require('./authv2/phoneProlongValid'),
    'email/send_confirmation_email_v2': require('./authv2/sendConfirmationEmail'),
    'email/confirm/by_code_v2': require('./authv2/confirmEmailByCode'),
    'email/setup_confirmed_v2': require('./authv2/setupConfirmedEmail'),
    'auth/additional_data/ask_v2': require('./authv2/additionalDataAsk'),
    'auth/additional_data/freeze_v2': require('./authv2/additionalDataFreeze'),
    'auth/multi_step/start': require('./authv2/multiStepAuthStart').route,
    'auth/multi_step/commit_password': require('./authv2/multiStepAuthCommitPassword'),
    'auth/multi_step/commit_magic': require('./authv2/multiStepAuthCommitMagic'),
    'auth/multi_step/submit_email_code': require('./auth_intranet/api/emailCodeSubmit'),
    'auth/multi_step/commit_email_code': require('./auth_intranet/api/emailCodeCommit'),
    'auth/logout': require('./authv2/authLogout').route,
    'auth/change_default': require('./authv2/authChangeDefault').route,
    'auth/password/submit': require('./authv2/authPasswordSubmit').route,
    'auth/accounts': require('./authv2/getAccounts').route,
    'auth/logout_and_forget': require('./authv2/authLogoutAndForget').route,
    'auth/send_magic_letter': require('./authv2/authSendMagicLetter'),
    'auth/confirm_magic_letter': require('./authv2/confirmMagicLetter'),
    'auth/invalidate_magic_letter': require('./authv2/invalidateMagicLetter'),
    'auth/restore_login/submit': require('./authv2/restoreLogin/getRestoreLoginTrack').route,
    'auth/restore_login/confirm_phone': require('./authv2/restoreLogin/checkRestoreLoginCode').route,
    'auth/restore_login/check_names': require('./authv2/restoreLogin/restoreLoginByName').route,
    'auth/restore_login/check_phone': require('./authv2/restoreLogin/getRestoreLoginCode').route,
    'auth/validate_phone': require('./authv2/validatePhone').route,
    'auth/validate_phone_by_id': require('./authv2/validatePhoneById').route,
    'auth/restore_pwd/check_login': require('./authv2/restorePassword/checkLogin'),
    'auth/restore_pwd/get_track': require('./authv2/restorePassword/client/getStartData'),
    'auth/restore_pwd/get_state': require('./authv2/restorePassword/client/getProcessState'),
    'auth/restore_pwd/select_method': require('./authv2/restorePassword/client/selectRestorationMethod'),
    'auth/restore_pwd/check_answer': require('./authv2/restorePassword/client/checkAnswer'),
    'auth/restore_pwd/send_email': require('./authv2/restorePassword/client/sendConfirmationEmail'),
    'auth/restore_pwd/confirm_email': require('./authv2/restorePassword/client/checkEmailCode'),
    'auth/restore_pwd/finish': require('./authv2/restorePassword/client/finishRestoration'),
    'auth/restore_pwd/finish-validate': require('./authv2/restorePassword/client/checkSession'),
    'auth/register_lite/get_track': require('./authv2/registerLite/setup'),
    'auth/challenge/submit': require('./authv2/challenge/getChallengeInfo'),
    'auth/challenge/commit': require('./authv2/challenge/commitChallenge'),
    'auth/challenge/send_push': require('./authv2/challenge/sendPush'),
    'auth/challenge/send_code_to_email': require('./authv2/challenge/sendCodeToEmail'),
    'auth/challenge/confirm_email_code': require('./authv2/challenge/confirmEmailCode'),
    'auth/register_lite/validate_name': require('./authv2/registerLite/validateName'),
    'auth/password/state': require('./authv2/changePassword/getPasswordState'),
    'geo/get_countries': require('./geo/getCountries'),
    'geo/get_timezones': require('./geo/getTimezones'),
    'update-csrf': require('./registration/getUpdatedCsrf'),

    'stop-subscription': [
        apiSetup,
        validateCSRF,
        (req, res) => {
            const {productId, id, points, type, region} = req.body;
            const controller = req._controller;
            const uid = controller.getAuth().getUid();

            if (!uid) {
                return res.json({
                    status: 'error',
                    errors: 'no_auth'
                });
            }

            return new BillingApi(req.logID)
                .stopSubscription({uid, productId})
                .then((data) => {
                    if (points) {
                        req.api.statboxLogger({
                            action: 'stop-subscription',
                            points,
                            type,
                            date_subscription_end: Date(),
                            uid,
                            region,
                            product_id: productId,
                            id
                        });
                    }

                    return res.send(data);
                })
                .catch((errors) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('profile.subscriptions.stop')
                        .write(errors);

                    res.status(500);
                    return res.send(errors);
                });
        }
    ],
    'restore-subscription': [
        validateCSRF,
        (req, res) => {
            const {productId} = req.body;
            const controller = req._controller;
            const uid = controller.getAuth().getUid();
            const ip = req.headers['x-real-ip'];

            if (!uid) {
                return res.json({
                    status: 'error',
                    errors: 'no_auth'
                });
            }

            return new BillingApi(req.logID)
                .restoreSubscription({uid, productId, ip})
                .then((data) => res.send(data))
                .catch((errors) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('profile.subscriptions.restore')
                        .write(errors);

                    if (errors.id != null) {
                        return res.send({status: 'error', id: errors.id});
                    }

                    res.status(500);
                    return res.send(errors);
                });
        }
    ],
    'update-subscriptions': [
        getYaExperimentsFlags,
        validateCSRF,
        serviceTicketsSetup,
        langSetup,
        (req, res) => {
            const {language, experiments: {flags = []} = {}} = res.locals;
            const serviceTicket = req.serviceTickets[SERVICE_ALIASES.PRAKTIKUM];
            const controller = req._controller;
            const uid = controller.getAuth().getUid();
            const billingApi = new BillingApi(req.logID);
            const praktikumApi = new PraktikumApi(req.logID, serviceTicket);

            if (!uid) {
                return res.json({
                    status: 'error',
                    errors: 'no_auth'
                });
            }

            const handlers = Object.entries({
                praktikum: flags.includes('passport-subs-praktikum-on')
                    ? praktikumApi.getSubscriptions({uid, language})
                    : Promise.resolve({}),
                billing: billingApi.getIntervals(uid)
            }).map(([type, promise]) =>
                promise.catch((error) => {
                    plog.warn()
                        .logId(req.logID)
                        .type(`profile.subscriptions.update.${type}`)
                        .write(error);
                    return null;
                })
            );

            return Promise.all(handlers)
                .then((response = []) => {
                    if (response.every((r) => !r)) {
                        return res.sendStatus(500);
                    }

                    const [praktikumSubscriptions, billingSubscriptions] = response;

                    res.send({
                        status: 'success',
                        intervals: [
                            ...((billingSubscriptions || {}).intervals || []),
                            ...((praktikumSubscriptions || {}).intervals || [])
                        ]
                    });
                })
                .catch((errors) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('profile.subscriptions.update')
                        .write(errors);

                    res.status(500);
                    return res.send(errors);
                });
        }
    ],
    'get-subscriptions': [
        validateCSRF,
        getYaExperimentsFlags,
        serviceTicketsSetup,
        langSetup,
        (req, res) => {
            const {language} = res.locals;
            const serviceTicket = req.serviceTickets[SERVICE_ALIASES.PRAKTIKUM];
            const controller = req._controller;
            const uid = controller.getAuth().getUid();
            const tld = controller.getTld();
            const {regionId} = res.locals || {};
            const billingApi = new BillingApi(req.logID);
            const praktikumApi = new PraktikumApi(req.logID, serviceTicket);
            const countryIdsWithExps = {
                181: 'plus-coil-enable'
            };

            let countryId;

            if (!uid) {
                return res.json({
                    status: 'error',
                    errors: 'no_auth'
                });
            }

            if (process.env.NODE_ENV !== 'production' && req.body.countryId) {
                countryId = req.body.countryId;
            } else {
                try {
                    countryId = lookup.getCountryId(regionId, tld) || null;
                } catch (error) {
                    plog.warn()
                        .logId(req.logID)
                        .type('profile.subscriptions.get.countryId')
                        .write(error);
                }
            }

            const expFlags = res && res.locals.experiments && res.locals.experiments.flags;
            const handlers = Object.entries({
                praktikum: expFlags.includes('passport-subs-praktikum-on')
                    ? praktikumApi.getSubscriptions({uid, language})
                    : Promise.resolve({}),
                billing: billingApi.getIntervals(uid)
            }).map(([type, promise]) =>
                promise.catch((error) => {
                    plog.warn()
                        .logId(req.logID)
                        .type(`profile.subscriptions.get.${type}`)
                        .write(error);
                    return null;
                })
            );

            return Promise.all(handlers)
                .then((response = []) => {
                    if (response.every((r) => !r)) {
                        return res.sendStatus(500);
                    }

                    const [praktikumSubscriptions, billingSubscriptions] = response;
                    const exp =
                        countryIdsWithExps[countryId] && controller.hasExp(countryIdsWithExps[countryId])
                            ? countryIdsWithExps[countryId]
                            : null;

                    res.send({
                        status: 'success',
                        intervals: [
                            ...((billingSubscriptions || {}).intervals || []),
                            ...((praktikumSubscriptions || {}).intervals || [])
                        ],
                        showcase: billingApi.getShowcase(countryId, exp)
                    });
                })
                .catch((errors) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('profile.subscriptions.get')
                        .write(errors);

                    res.status(500);
                    return res.send(errors);
                });
        }
    ],
    'do-complete-submit': [
        validateCSRF,
        apiSetup,
        function(req, res) {
            req.api.completeSubmit(req.body).then(
                function(result) {
                    res.json(result.body || result);
                },
                function(errors) {
                    res.json({
                        status: 'error',
                        errors: errors || []
                    });
                }
            );
        }
    ],
    'find-accounts-by-name-and-phone': genericValidation('findAccountsByNameAndPhone'),
    'find-accounts-by-phone': genericValidation('findAccountsByPhone'),
    'phone-validate-by_squatter': genericValidation('validatePhoneBySquatter'),
    'validate-public-id': [setupForceCleanWeb, genericValidation('validatePublicId')],
    'toggle-personal-data-access': genericValidation('togglePersonalDataPublicAccess'),
    'registration-submit': [
        validateCSRF,
        getTrack({type: 'register', scenario: 'register'}),
        (req, res) => {
            const {track_id: id} = res.locals;

            return res.json({
                status: id ? 'ok' : 'error',
                id
            });
        }
    ],
    'user-entry-flow-submit': [
        validateCSRF,
        getTrack({type: 'register'}),
        (req, res) => {
            const {track_id: id} = res.locals;

            return res.json({
                status: id ? 'ok' : 'error',
                id
            });
        }
    ],
    'neo-phonish-auth': [
        apiSetup,
        validateCSRF,
        (req, res, next) => {
            const apiHandle = req.body.useNewSuggestByPhone ? 'neoPhonishRestoreAuth' : 'neoPhonishAuth';

            req.api[apiHandle](req.body)
                .then((response = {}) => {
                    const {body = {}} = response;
                    const {state, track_id} = body;

                    if (state) {
                        const currentUrlObj = url.parse(url.format(req._controller.getUrl()), true);
                        const redirectUrl = url.parse(
                            url.format(
                                Object.assign({}, currentUrlObj, {
                                    search: null,
                                    pathname: '/auth',
                                    query: {
                                        track_id,
                                        one: 'yes'
                                    }
                                })
                            )
                        ).path;

                        return res.json({
                            status: 'ok',
                            state,
                            redirectUrl
                        });
                    }

                    return next();
                })
                .catch((errors) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('neo-phonish-auth')
                        .write(errors);

                    res.send({
                        status: 'error',
                        errors
                    });
                });
        },
        updateCookies
    ],
    'multi-step-commit-sms-code': [
        apiSetup,
        validateCSRF,
        (req, res, next) => {
            req.api
                .multiStepCommitSMSCode(req.body)
                .then((response = {}) => {
                    const {body = {}} = response;
                    const {state, track_id} = body;

                    if (state) {
                        const currentUrlObj = url.parse(url.format(req._controller.getUrl()), true);
                        const redirectUrl = url.parse(
                            url.format(
                                Object.assign({}, currentUrlObj, {
                                    search: null,
                                    pathname: '/auth',
                                    query: {
                                        track_id,
                                        one: 'yes'
                                    }
                                })
                            )
                        ).path;

                        return res.json({
                            status: 'ok',
                            state,
                            redirectUrl
                        });
                    }

                    return next();
                })
                .catch((errors) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('multi-step-commit-sms-code')
                        .write(errors);

                    res.status(500);
                    res.send(errors);
                });
        },
        updateCookies
    ],
    'change-pwd-commit': [
        apiSetup,
        validateCSRF,
        (req, res, next) => {
            req.api
                .authChangePassword(req.body)
                .then((result = {}) => {
                    const {body = {}} = result;
                    const {cookies, is_conflicted_operation_exists} = body;
                    const controller = req._controller;

                    if (cookies) {
                        controller.augmentResponse(body);
                        res.locals.changePwdResult = {
                            status: 'ok',
                            isPhoneConflictsExist: is_conflicted_operation_exists
                        };
                        return next();
                    }

                    return res.json({status: 'error', error: ['internal']});
                })
                .catch((error = {}) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('change-pwd-commit')
                        .write(error);
                    return res.json({status: 'error', error});
                });
        },
        checkSMS2faEnabled,
        (req, res) => {
            return res.json({...res.locals.changePwdResult, hasSms2faUpdated: res.locals.isSms2faEnabled});
        }
    ],
    statbox: [
        apiSetup,
        validateCSRF,
        (req, res) => {
            const params = Object.assign({}, req.body, {
                ip: req.headers['x-real-ip']
            });

            delete params.csrf_token;
            req.api.statboxLogger(params);
            res.send({status: 'ok'});
        }
    ],
    'change-password-submit': [
        apiSetup,
        validateCSRF,
        (req, res) => {
            req.api
                .changepassSubmit(req.body)
                .then((result = {}) => {
                    const {body = {}} = result;
                    const {status, revokers} = body;

                    if (status === 'ok') {
                        return res.json({status: 'ok', revokers});
                    }
                })
                .catch((error = {}) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('change-password-submit')
                        .write(error);
                    return res.json({status: 'error', error});
                });
        }
    ],
    'change-password-commit': [
        apiSetup,
        validateCSRF,
        (req, res, next) => {
            return req.api
                .captchaCheck(req.body)
                .then((response = {}) => {
                    if (response.body && response.body.correct) {
                        return next();
                    }

                    res.json({
                        status: 'error',
                        error: ['captcha.cannot_locate']
                    });
                })
                .catch((error = {}) => {
                    res.json({
                        status: 'error',
                        error
                    });
                });
        },
        (req, res) => {
            req.api
                .changepassCommit(req.body)
                .then((result = {}) => {
                    const {body = {}} = result;
                    const controller = req._controller;

                    if (body.status === 'ok') {
                        controller.augmentResponse(body);
                        return res.json({
                            status: 'ok'
                        });
                    }

                    return res.json({status: 'error', error: ['global']});
                })
                .catch((error = {}) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('change-password-commit')
                        .write(error);
                    return res.json({status: 'error', error});
                });
        }
    ],
    'unsubscribe.kinopoiskgetdefaultselection': [apiSetup, validateCSRF, unsubscribe.kinopoiskGetDefaultSelection],
    'update-account-info': [apiSetup, validateCSRF, getAccountInfo],
    'registration-social': [validateCSRF, social.asyncRegister],
    'callback-social': [validateCSRF, social.asyncCallback],
    'subs-submit': [validateCSRF, sendSubs],
    'subs-list': [validateCSRF, getSubsList],
    'user-validate/commit': userValidateCommit,
    'user-validate/commit3ds': userValidateCommit3ds,
    'user-validate/submit3ds': userValidateSubmit3ds,
    'user-validate/3ds-retpath': userValidateGetRetpath3ds,
    'auth-prepare-with-cred': [
        apiSetup,
        validateCSRF,
        (req, res) => {
            const trackId = req.body.trackId;

            req.api
                .authPrepareWithCred(trackId)
                .then(() => res.json({status: 'ok'}))
                .catch((error = {}) => {
                    plog.warn()
                        .logId(req.logID)
                        .type('auth-prepare-with-cred')
                        .write(error);
                    return res.json({status: 'error', error});
                });
        }
    ],
    'get-account-phones': genericValidation('getAccountWithPhonesInfo'),

    'documents.disk.getResourceMeta': documents.getResourceInfo,
    'documents.disk.createFolder': documents.createFolder,
    'documents.disk.getMeta': documents.getDiskInfo,
    'documents.disk.searchResource': documents.searchDiskResource,
    'documents.disk.copyResource': documents.copyDiskResource,
    'documents.disk.deleteResource': documents.deleteDiskResource,
    'documents.disk.createResource': documents.createDiskResource,
    // documents API
    'documents.api.getDocumentList': documents.getDocumentList,
    'documents.api.uploadDocument': documents.uploadDocument,
    'documents.api.deleteDocument': documents.deleteDocument,
    'documents.api.copyDocument': documents.copyDocument
};

module.exports.checkHuman = [
    setup,
    validateCSRF,
    function(req, res, next) {
        if ('answer' in req.body) {
            return req.api
                .captchaCheck(req.body)
                .then(function(response) {
                    const body = response.body;

                    if (Array.isArray(body)) {
                        body.forEach(function(item) {
                            if (item.code === 'captchalocate') {
                                res.json({
                                    status: 'error',
                                    errors: ['captcha.cannot_locate']
                                });
                            }
                        });
                    }

                    if (body.status === 'ok' && body.correct === true && 'password' in req.body) {
                        return next();
                    }

                    if (body.correct === false) {
                        return res.json({
                            status: 'error',
                            errors: ['captcha.not_matched']
                        });
                    }

                    return res.json({
                        status: 'ok'
                    });
                })
                .catch(function(err) {
                    return res.json({
                        status: 'error',
                        errors: err.errors
                    });
                });
        }

        return next();
    },
    module.exports.passwordConfirm
];
