var url = require('url');
var apiSetup = require('./common/apiSetup');
var config = require('../configs/current');
var PForm = require('pform');
var langSetup = require('./common/langSetup');
var util = require('util');
var inherit = require('inherit');
var langs = config.langs;
var _ = require('lodash');

var TrackField = require('../blocks/form/track.field');
var FirstnameField = require('../blocks/control/firstname/firstname.field');
var LastnameField = require('../blocks/control/lastname/lastname.field');
var PasswordField = require('../blocks/control/password/password.field');
var PasswordConfirmField = require('../blocks/control/password-confirm/password-confirm.field');
var PasswordCurrentField = require('../blocks/control/password-current/password-current.field');
var SubmitField = require('../blocks/control/submit/submit.field');
var HumanConfirmationField = require('../blocks/control/human-confirmation/human-confirmation.field');
var LoginField = require('../blocks/control/login/login.field');
var EulaField = require('../blocks/control/checkbox/eula/eula.field');
var CaptchaField = require('../blocks/control/captcha/captcha.field');
var PLog = require('plog');

var CompleteForm = inherit(
    {
        __constructor(body, api) {
            this.state = body.state;
            this.hasRecoveryMethod = body.has_recovery_method;
            this.body = body;
            this.api = api;
            this.humanConfirmationField = null;
            this.validationMethod = body['human-confirmation'];
            this.captchaField = null;
            this.form = null;

            // PASSP-12040: Лайт с телефоном не может дорегистрироваться
            // Если у нас нет явной информации про recovery_method, и если в данных
            // нет ничего про нет, считаем, что у пользователя он был, и мы с ним
            // не работаем
            if (this.hasRecoveryMethod === undefined && this.validationMethod === undefined) {
                this.hasRecoveryMethod = true;
            }
        },

        getForm: (function() {
            function F(args) {
                return PForm.apply(this, args);
            }

            F.prototype = PForm.prototype;

            return function() {
                return new F(arguments);
            };
        })(),

        getCaptcha() {
            this.captchaField = new CaptchaField().setHidden();
            return this.captchaField;
        },

        getHumanConfirmation() {
            var phoneNumber = this.body.phone_number || this.body.phone_number_confirmed;

            if (this.hasRecoveryMethod) {
                return null;
            }

            this.humanConfirmationField = new HumanConfirmationField();

            if (phoneNumber && this.validationMethod === 'phone') {
                this.humanConfirmationField.setOption('confirmed_phone_number', phoneNumber);
            }

            return this.humanConfirmationField;
        },

        extractCaptcha() {
            if (!this.hasRecoveryMethod) {
                this.humanConfirmationField.setOption('captchaExtracted', true);
            } else {
                this.captchaField.setVisible().setRequired();
            }
        },
        hideCaptcha() {
            if (!this.hasRecoveryMethod) {
                this.humanConfirmationField.setOption('captchaExtracted', false);
            } else {
                this.captchaField.setHidden().setOptional();
            }
        }
    },
    {}
);

var CompleteLiteForm = inherit(
    CompleteForm,
    {
        getForm() {
            var fields;
            var additionalField;

            if (this.form) {
                return this.form;
            }

            fields = [
                new TrackField(),
                new FirstnameField(),
                new LastnameField(),
                new LoginField().setRequired(),
                new PasswordCurrentField().setRequired()
            ];

            if (this.hasRecoveryMethod) {
                additionalField = this.getCaptcha();
            } else {
                additionalField = this.getHumanConfirmation();
            }

            if (additionalField) {
                fields.push(additionalField);
            }

            fields.push(new EulaField().setRequired());
            fields.push(new SubmitField());

            this.form = this.__base.apply(this, fields).setApi(this.api);

            return this.form;
        }
    },
    {}
);

var ForceCompleteLiteForm = inherit(
    CompleteForm,
    {
        getForm() {
            var fields;
            var additionalField;

            if (this.form) {
                return this.form;
            }

            fields = [new TrackField(), new FirstnameField(), new LastnameField(), new LoginField().setRequired()];

            if (this.hasRecoveryMethod) {
                additionalField = this.getCaptcha();
            } else {
                additionalField = this.getHumanConfirmation();
            }

            if (additionalField) {
                fields.push(additionalField);
            }

            fields.push(new EulaField().setRequired());
            fields.push(new SubmitField());

            this.form = this.__base.apply(this, fields).setApi(this.api);

            return this.form;
        }
    },
    {}
);

var CompleteSocialForm = inherit(
    CompleteForm,
    {
        getForm() {
            var fields;
            var humanConfirmation;

            if (this.form) {
                return this.form;
            }

            fields = [
                new TrackField(),
                new FirstnameField(),
                new LastnameField(),
                new PasswordField().setRequired(),
                new PasswordConfirmField().setRequired()
            ];

            humanConfirmation = this.getHumanConfirmation();

            if (humanConfirmation) {
                fields.push(humanConfirmation);
            }

            fields.push(new SubmitField());

            this.form = this.__base.apply(this, fields).setApi(this.api);

            return this.form;
        }
    },
    {}
);

var CompleteSocialWithLoginForm = inherit(
    CompleteForm,
    {
        getForm() {
            var fields;
            var humanConfirmation;

            if (this.form) {
                return this.form;
            }

            fields = [
                new TrackField(),
                new FirstnameField(),
                new LastnameField(),
                new LoginField().setRequired(),
                new PasswordField().setRequired(),
                new PasswordConfirmField().setRequired()
            ];

            humanConfirmation = this.getHumanConfirmation();

            if (humanConfirmation) {
                fields.push(humanConfirmation);
            }

            fields.push(new SubmitField());

            this.form = this.__base.apply(this, fields).setApi(this.api);

            return this.form;
        }
    },
    {}
);

var CompleteAutoregisteredForm = inherit(
    CompleteForm,
    {
        getForm() {
            var fields;
            var humanConfirmation;

            if (this.form) {
                return this.form;
            }

            fields = [new TrackField(), new PasswordField().setRequired(), new PasswordConfirmField().setRequired()];

            humanConfirmation = this.getHumanConfirmation();

            if (humanConfirmation) {
                fields.push(humanConfirmation);
            }

            fields.push(new EulaField().setRequired());
            fields.push(new SubmitField());

            this.form = this.__base.apply(this, fields).setApi(this.api);

            return this.form;
        }
    },
    {}
);

var forms = {
    complete_social: CompleteSocialForm,
    complete_autoregistered: CompleteAutoregisteredForm,
    complete_social_with_login: CompleteSocialWithLoginForm,
    complete_lite: CompleteLiteForm,
    force_complete_lite: ForceCompleteLiteForm
};

exports.routes = {};

exports.route = function(app) {
    var routes = this.routes;

    function modeFilter(req, res, next) {
        if (req.nquery && (req.nquery.mode === 'light2full' || req.nquery.mode === 'postregistration')) {
            return next();
        }

        return next('route');
    }

    app.get('/passport/?', modeFilter, routes.show);
    app.post('/passport/?', modeFilter, routes.save);

    app.get(
        '/auth/?',
        function(req, res, next) {
            if (req.query && req.query.key) {
                return next();
            }

            return next('route');
        },
        routes.authkey
    );

    app.post(
        '/auth/?',
        function(req, res, next) {
            if (req.query && req.query.key) {
                return next();
            }

            return next('route');
        },
        routes.save
    );

    app.get('/profile/upgrade/lite/?', routes.liteShow);
    app.post('/profile/upgrade/lite/?', routes.save);

    app.get('/profile/upgrade/?', routes.show);
    app.post('/profile/upgrade/?', routes.save);
};

exports.routes.liteShow = [
    apiSetup,
    langSetup,
    doAuthSubmit,
    showCompleteForm,
    redirectToRetpathIfNothingToDo,
    handleNonFormErrors
];

exports.routes.authkey = [
    apiSetup,
    langSetup,
    doKeyLinkSubmit,
    doCompleteSubmit,
    showCompleteForm,
    redirectToRetpathIfNothingToDo,
    handleNonFormErrors
];

exports.routes.show = [
    apiSetup,
    langSetup,
    doCompleteSubmit,
    showCompleteForm,
    redirectToRetpathIfNothingToDo,
    handleNonFormErrors
];

exports.routes.save = [apiSetup, submitCompleteForm, langSetup, showCompleteFormWithErrors, handleNonFormErrors];

function doAuthSubmit(req, res, next) {
    req.api.authSubmit('/1/bundle/auth/password/get_state/', {track_id: req.api.track()}).then(
        function(results) {
            var body = results.body;

            res.locals.knownData = body;
            res.locals.track_id = body.track_id;
            res.locals.state = body.state;

            return next();
        },
        function() {
            return next();
        }
    );
}

function doKeyLinkSubmit(req, res, next) {
    var secretKey = req.query && req.query.key;

    if (!secretKey) {
        return next(new Error());
    }

    return req.api
        .authKeyLinkSubmit({
            secret_key: secretKey
        })
        .then(function(results) {
            var body = results.body;
            var state = body.state;
            var trackId = body.track_id;

            res.locals.track_id = trackId;
            res.locals.state = state;

            res.locals.knownData = {
                track_id: trackId,
                state,
                account: {},
                has_recovery_method: false
            };

            if (state === 'complete_autoregistered') {
                res.locals.skipSubmit = true;
            }

            return next();
        }, next);
}

function doCompleteSubmit(req, res, next) {
    var retpath = req.body.retpath || (req.query && req.query.retpath);
    var trackId = res.locals.track_id;
    var data = {
        retpath
    };

    if (res.locals.skipSubmit) {
        return next();
    }

    if (trackId) {
        data.track_id = trackId;
    }

    return req.api.completeSubmit(data).then(function(result) {
        if (result.body && result.body.status === 'ok') {
            res.submitResult = result.body;
            return next();
        }

        return next(new Error());
    }, next);
}

function showCompleteForm(req, res, next) {
    var retpath = req.body.retpath || (req.query && req.query.retpath);
    var lang = langs[langs.indexOf(res.locals.language)] || 'ru';
    var data = {};
    var body = res.submitResult || res.locals.knownData || {};
    var account = body.account;
    const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
    const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;
    var statBoxData = {
        ignoreMissedTrack: true,
        mode: 'complete',
        action: 'opened',
        track_id: body.track_id,
        uid: account && account.uid,
        yandexuid: req.cookies.yandexuid,
        ip: req.headers['x-real-ip'],
        state: body.state,
        has_recovery_method: String(body.has_recovery_method),
        user_agent: req.headers['user-agent'],
        experiment_flags: experimentFlags,
        experiment_boxes: experimentBoxes
    };
    var pathToLightAuthToFull = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'passport',
        query: {
            mode: 'lightauth2full'
        }
    };
    var form;
    var formProxy;

    if (retpath) {
        pathToLightAuthToFull.query.retpath = retpath;
    }

    res.locals.state = body.state;

    if (!body.state) {
        return next();
    }

    if (body.state === 'upgrade_cookies') {
        return res.redirect(url.format(pathToLightAuthToFull));
    }

    formProxy = new forms[body.state](body, req.api);
    form = formProxy.getForm();

    if (account && account.person) {
        form.getField('firstname').setValue(account.person.firstname);
        form.getField('lastname').setValue(account.person.lastname);
    }

    return form.compile(lang).then(function(compiled) {
        res.locals.form = {
            control: compiled
        };

        req.api.statboxLogger(statBoxData).catch(function(err) {
            PLog.warn()
                .logId(req.logID)
                .type('complete')
                .write(err);
        });

        res.render(`complete.${lang}.js`, data);
    });
}

function submitCompleteForm(req, res, next) {
    var form;
    var reqBody = req.body;
    var api = req.api;
    const controller = req._controller;

    res.formProxy = new forms[reqBody.state](reqBody, api);
    form = res.formProxy.getForm();

    if (reqBody.answer) {
        res.formProxy.extractCaptcha();
    }

    form.validate(reqBody).then(
        function(isValid) {
            if (isValid) {
                return api.completeCommit(reqBody.state, reqBody).then(
                    function(results) {
                        var body = results.body;
                        var finishPathname = '/auth/finish/';
                        var target = url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname: 'passport',
                            query: {
                                mode: 'passport'
                            }
                        });

                        if (body.cookies && body.cookies.length) {
                            controller.augmentResponse(body);

                            target = url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: finishPathname,
                                query: {
                                    track_id: body.track_id || req.api.track()
                                }
                            });
                        } else if (body.retpath) {
                            target = body.retpath;
                        }

                        return res.redirect(target);
                    },
                    function(error) {
                        if (
                            util.isArray(error) &&
                            (error.indexOf('password.found_in_history') !== -1 ||
                                error.indexOf('password.equals_previous') !== -1 ||
                                error.indexOf('password.not_matched') !== -1 ||
                                error.indexOf('login.like_password') !== -1 ||
                                error.indexOf('captcha.required') !== -1 ||
                                error.indexOf('user.not_verified') !== -1)
                        ) {
                            res.rawErrors = error;
                            return next();
                        }

                        return next(error);
                    }
                );
            }

            return next();
        },
        function(error) {
            return next(error);
        }
    );
}

function showCompleteFormWithErrors(req, res, next) {
    var lang = res.locals.language;
    var form = res.formProxy.getForm();

    if (res.rawErrors) {
        res.formProxy.hideCaptcha();

        if (res.rawErrors.indexOf('password.not_matched') !== -1) {
            form.getField('current_password').setErrorsActive(['not_matched']);
        }

        if (res.rawErrors.indexOf('password.found_in_history') !== -1) {
            form.getField('password').setErrorsActive(['foundinhistory']);
        }

        if (res.rawErrors.indexOf('password.equals_previous') !== -1) {
            form.getField('password').setErrorsActive(['likeoldpassword']);
        }

        if (res.rawErrors.indexOf('login.like_password') !== -1) {
            form.getField('login').setErrorsActive(['notavailable']);
        }

        if (res.rawErrors.indexOf('captcha.required') !== -1 || res.rawErrors.indexOf('user.not_verified') !== -1) {
            res.formProxy.extractCaptcha();
        }
    }

    res.locals.state = req.body.state;

    form.compile(lang).then(
        function(compiled) {
            res.locals.form = {
                control: compiled
            };

            res.render(`complete.${lang}.js`, {});
        },
        function(error) {
            return next(error);
        }
    );
}

function handleNonFormErrors(err, req, res, next) {
    var currentUrl = url.parse(req.originalUrl, true);
    var retpathToComplete;
    var defaultRetpathObj = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'passport',
        query: {
            mode: 'passport'
        }
    };

    if (err instanceof Error) {
        return next(err);
    }

    if (
        util.isArray(err) &&
        (err.indexOf('action.not_required') !== -1 ||
            err.indexOf('account.disabled') !== -1 ||
            err.indexOf('account.global_logout') !== -1 ||
            err.indexOf('sessionid.invalid') !== -1 ||
            err.indexOf('sslsession.required') !== -1)
    ) {
        retpathToComplete = _.clone(defaultRetpathObj, true);

        if (err.indexOf('sessionid.invalid') !== -1 || err.indexOf('sslsession.required') !== -1) {
            retpathToComplete.pathname = 'auth';

            retpathToComplete.query.retpath = url.format(
                _.extend({}, defaultRetpathObj, {
                    pathname: currentUrl.pathname,
                    query: currentUrl.query,
                    search: null
                })
            );
            delete retpathToComplete.query.mode;
        }

        if (err.indexOf('sslsession.required') !== -1) {
            retpathToComplete.pathname = 'auth/secure';
        }

        return res.redirect(url.format(retpathToComplete));
    }

    return next(err);
}

function redirectToRetpathIfNothingToDo(req, res) {
    var validatedRetpath = res.submitResult && res.submitResult.retpath;
    var defaultRetpath = url.format({
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'passport',
        query: {
            mode: 'passport'
        }
    });

    if (validatedRetpath) {
        return res.redirect(validatedRetpath);
    }

    return res.redirect(defaultRetpath);
}
