const url = require('url');
const PLog = require('plog');
const config = require('../configs/current');
const apiSetup = require('./common/apiSetup');
const checkAuth = require('./common/checkAuth');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const getCountry = require('./common/getCountry');
const getMetrics = require('./common/getMetrics');
const rumCounterSetup = require('./common/rumCounterSetup');
const _ = require('lodash');
const _get = _.get;
const _clone = _.clone;
const _escape = _.escape;
const handlePhoneCallExperiment = require('./registration/handlePhoneCallExperiment');
const setRegistrationPhoneOnly = require('./registration/setRegistrationPhoneOnly');

const REGISTRATION_COMPLETE_GOAL_PREFIX = 'complete_v2';
const REGISTRATION_COMPLETE_MOBILE_GOAL_PREFIX = 'complete_v2_mobile';

function getLanguage(req, res, next) {
    req._controller
        .getLanguage()
        .then((lang) => {
            res.locals.language = lang;
        })
        .catch((err) => {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('registration.complete')
                .write(err);
        })
        .done(() => {
            next();
        });
}

function createForm(req, res, next) {
    let questionList = [];
    const formErrors = res.locals.formErrors || [];
    const captchaErrors = res.locals.captchaErrors || [];

    if (res.locals.questions && Array.isArray(res.locals.questions) && res.locals.questions.length > 1) {
        questionList = res.locals.questions.slice(1).map((question, index) => {
            return {
                id: `q${index}`,
                val: question.id,
                text: question.value
            };
        });
    }

    res.locals.form = {
        activeField: '',
        type: 'unknown',
        formErrors,
        captchaErrors,
        validation: {
            method: 'phone',
            phoneConfirmationType: 'sms',
            humanConfirmationDone: false,
            isValidPhoneForCall: false,
            magicConfirmationCode: '',
            callConfirmationTimer: null,
            isCallConfirmationInProcess: false,
            isForceCheckedConfirmationCode: false,
            confirmationCodeLength: 0
        },
        states: {
            firstname: '',
            lastname: '',
            login: '',
            password: '',
            password_confirm: '',
            hint_question_id: '',
            hint_question: '',
            hint_question_custom: 'valid',
            hint_answer: '',
            captcha: '',
            phone: '',
            phoneCode: '',
            phoneCodeStatus: ''
        },
        values: {
            firstname: '',
            lastname: '',
            surname: '',
            login: '',
            password: '',
            password_confirm: '',
            hint_question_id: '',
            hint_question: '',
            hint_question_custom: '',
            hint_answer: '',
            captcha: '',
            phone: '',
            phoneCode: ''
        },
        errors: {
            active: '',
            firstname: {
                code: '',
                text: ''
            },
            lastname: {
                code: '',
                text: ''
            },
            login: {
                code: '',
                text: ''
            },
            password: {
                code: '',
                text: ''
            },
            password_confirm: {
                code: '',
                text: ''
            },
            hint_question_id: {
                code: '',
                text: ''
            },
            hint_question: {
                code: '',
                text: ''
            },
            hint_question_custom: {
                code: '',
                text: ''
            },
            hint_answer: {
                code: '',
                text: ''
            },
            captcha: {
                code: '',
                text: ''
            },
            phone: {
                code: '',
                text: ''
            },
            phoneCode: {
                code: '',
                text: ''
            },
            phoneCodeStatus: ''
        },
        humanConfirmation: {
            questionList,
            isFetching: false
        },
        canSwitchConfirmationMethod: true
    };

    if (questionList.length) {
        res.locals.form.values.hint_question = questionList[0].text;
        res.locals.form.values.hint_question_id = questionList[0].val;
        res.locals.form.states.hint_question = 'valid';
        res.locals.form.states.hint_question_id = 'valid';
    }

    if (res.locals.experiments) {
        const experimentFlags = res.locals.experiments.flags;

        if (Array.isArray(experimentFlags)) {
            res.locals.form = Object.assign(
                {},
                res.locals.form,
                handlePhoneCallExperiment(experimentFlags, res.locals.language)
            );
        }
    }

    next();
}

function renderPage(req, res, next) {
    // eslint-disable-line
    const metricsHeaders = {
        complete_social: 'Дорегистрация социальщиков с логином',
        complete_social_with_login: 'Дорегистрация социальщиков',
        complete_lite: 'Дорегистрация лайтов',
        force_complete_lite: 'Дорегистрация лайтов (принудительная)',
        complete_autoregistered: 'Дорегистрация авторегистрированных',
        complete_neophonish: 'Дорегистрация неофониша'
    };
    const retpath = res.locals.store.common.retpath || req.body.retpath || (req.query && req.query.retpath);
    const lang = res.locals.language;
    const body = res.submitResult || res.locals.knownData || {};
    const account = body.account;
    const pathToLightAuthToFull = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'passport',
        query: {
            mode: 'lightauth2full'
        }
    };

    if (!body.state) {
        return next();
    }

    if (body.state === 'upgrade_cookies') {
        if (retpath) {
            pathToLightAuthToFull.query.retpath = retpath;
        }

        return res.redirect(url.format(pathToLightAuthToFull));
    }

    if (metricsHeaders.hasOwnProperty(body.state)) {
        res.locals.store.metrics.header = metricsHeaders[body.state];
    }

    const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
    const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

    req.api.statboxLogger({
        ignoreMissedTrack: true,
        track_id: body.track_id || res.locals.track_id,
        uid: account && account.uid,
        action: 'opened',
        mode: 'complete.v2',
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        yandexuid: res._yandexuid && res._yandexuid.replace(/[^0-9]/g, ''),
        state: body.state,
        has_recovery_method: String(body.has_recovery_method),
        experiment_flags: experimentFlags,
        experiment_boxes: experimentBoxes
    });

    return res.render(`react.complete.${lang}.jsx`);
}

const createStore = [
    (req, res, next) => {
        const langSwitcher = new LangSwitcherView(req._controller);
        const locals = res.locals;
        const tld = req._controller.getTld();
        const yandexuid = res._yandexuid && res._yandexuid.replace(/[^0-9]/g, '');
        const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
        const retpath = req.body.retpath || (req.query && (req.query.url || req.query.retpath));
        const account = locals.account || {};
        const captcha = res.locals.captcha || {};
        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {userType = {}} = locals;
        const {metricsUserType} = userType;
        const isAppMode = req.query && req.query.mode && req.query.mode === 'app';

        if (retpath) {
            handlers.push(req.api.validateRetpath({retpath}));
        }

        locals.store = {
            fetchingStatus: {
                isFetching: false
            },
            settings: {
                location: req.path,
                host: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname
                }),
                embeddedauth: config.paths.embeddedauth || '',
                links:
                    process.env.INTRANET === 'intranet'
                        ? config.links.intranet
                        : config.links[tld] || config.links.com || {},
                help: config.paths.help || {},
                tld,
                language: locals.language,
                country: locals.country,
                uatraits: {},
                isMobile: false,
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                staticPath,
                authURL: url.format(
                    Object.assign({}, req._controller.getAuthUrl(), {
                        query: {
                            retpath: url.format(req._controller.getUrl())
                        }
                    })
                ),
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType
            },
            common: {
                actionForRepeat: null,
                yandexuid,
                retpath: null,
                currentPage: req._controller.getUrl().href,
                dev: config.dev,
                origin: (req.query || {}).origin,
                experiments: res.locals.experiments || {
                    flags: [],
                    flagsString: '',
                    boxes: '',
                    encodedBoxes: ''
                },
                isAppMode
            },
            person: Object.assign({}, account.person, {
                isSocialchik: locals.isSocialchik,
                uid: account.uid,
                login: account.login,
                displayLogin: account.display_login,
                avatarId: (account.display_name && account.display_name.default_avatar) || '',
                displayNames: account.display_names || {},
                displayName: (account.display_name && account.display_name.name) || '',
                escapedDisplayName: (account.display_name && _escape(account.display_name.name)) || '',
                hasRecoveryMethod: false,
                errors: {}
            }),
            captcha: {
                loading: false,
                loadingAudio: false,
                playing: false,
                type: 'text',
                key: captcha.key,
                imageUrl: captcha.image_url,
                introSound: null,
                captchaSound: null
            },
            logins: {
                loginsList: [],
                showAll: false,
                isFetching: false
            },
            registrationErrors: {
                status: 'ok',
                code: '',
                text: ''
            },
            footer: {
                langlist: []
            },
            metrics: {
                header: 'Дорегистрация',
                experiments: res.locals.experiments ? res.locals.experiments.encodedBoxes : ''
            },
            monitoring: {
                page: 'registration.complete'
            },
            pageMessage: null,
            form: res.locals.form,
            registrationName: locals.registrationName
        };

        Promise.all(handlers)
            .then((response) => {
                const defaultRetpath = url.format(
                    Object.assign({}, req._controller.getUrl(), {
                        pathname: 'profile',
                        search: null,
                        query: null
                    })
                );
                const uatraits = response[0] || {};
                const langlist = (response[1] && response[1].langlist) || {};
                const validatedRetpath = response[2] || defaultRetpath;

                locals.store.settings.uatraits = uatraits;
                locals.store.footer.langlist = langlist;
                locals.store.settings.isMobile = uatraits.isMobile && !uatraits.isTablet;
                locals.store.common.retpath = defaultRetpath;

                if (validatedRetpath.body && validatedRetpath.body.retpath) {
                    locals.store.common.retpath = validatedRetpath.body.retpath;
                }

                if (req._controller.getRequestParam('from')) {
                    locals.store.common.from = req._controller.getRequestParam('from');
                }

                return next();
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.complete')
                    .write(err);

                return next();
            });
    },
    getMetrics({}),
    rumCounterSetup
];

function getState(req, res, next) {
    req.api
        .profileGetState()
        .then((response) => {
            const form = res.locals.form;
            const profile = response.body;
            const account = profile.account || {};
            const person = account.person || {};

            person.provider = _get(account, 'display_name.social.provider');
            res.locals.account = account;
            res.locals.isSocialchik = _get(account, 'password_info.strength', null) === -1 && !account.is_2fa_enabled;
            res.locals.isUserWithoutPassword = res.locals.isSocialchik;
            if (person.firstname) {
                form.values.firstname = person.firstname;
                form.states.firstname = 'valid';
            }

            if (person.lastname) {
                form.values.lastname = person.lastname;
                form.states.lastname = 'valid';
            }
            form.isEulaShowedInPopup = true;

            return next();
        })
        .catch((errors) => {
            res.locals.account = {};
            res.locals.isSocialchik = false;

            PLog.warn()
                .logId(req.logID)
                .type('registration.complete')
                .write(errors);

            return next();
        });
}

function correctForm({form, hasRecoveryMethod, isLiteWithoutPassword, isMobile, isNeoPhonishWithoutPersonal}) {
    const type = form.type;
    const keysToRemove = {
        complete_social: ['login'],
        complete_lite: isLiteWithoutPassword ? [] : ['password_confirm'],
        force_complete_lite: ['password', 'password_confirm'],
        complete_autoregistered: ['firstname', 'lastname', 'login'],
        complete_neophonish: [
            'phone',
            'phoneCode',
            'hint_question_id',
            'hint_question',
            'hint_question_custom',
            'hint_answer',
            'captcha',
            'phoneCodeStatus'
        ]
    };

    if (keysToRemove.hasOwnProperty(type)) {
        for (const key of keysToRemove[type]) {
            delete form.states[key];
            delete form.values[key];
            delete form.errors[key];
        }
    }

    if (isNeoPhonishWithoutPersonal) {
        ['firstname', 'lastname'].forEach((key) => {
            delete form.states[key];
            delete form.values[key];
            delete form.errors[key];
        });
    }

    if ((type === 'complete_lite' || type === 'force_complete_lite') && hasRecoveryMethod) {
        const removeForLites = [
            'hint_question_id',
            'hint_question',
            'hint_question_custom',
            'hint_answer',
            'phone',
            'phoneCode',
            'phoneCodeStatus'
        ];

        for (const key of removeForLites) {
            delete form.states[key];
            delete form.values[key];
            delete form.errors[key];
        }

        delete form.validation.method;
    }

    if (type === 'complete_neophonish') {
        delete form.validation.method;
    }

    if (!type) {
        form.prefix = isMobile ? REGISTRATION_COMPLETE_MOBILE_GOAL_PREFIX : REGISTRATION_COMPLETE_GOAL_PREFIX;
    } else {
        form.prefix = `${type}_v2${isMobile ? '_mobile' : ''}`;
    }
}

function doCompleteSubmit(req, res, next) {
    const {store = {}} = res.locals;
    const {settings = {}, common = {}, track_id: trackId, person} = store;
    const isMobile = settings.isMobile;
    const retpath = common.retpath || req.body.retpath || (req.query && req.query.retpath);
    const data = {retpath};

    if (res.locals.skipSubmit) {
        correctForm({form: res.locals.form, isMobile});
        return next();
    }

    if (trackId) {
        data.track_id = trackId;
    }

    return req.api.completeSubmit(data).then((result) => {
        if (result.body && result.body.status === 'ok') {
            const form = res.locals.form;
            const body = result.body;
            const phone = body.phone_number || body.phone_number_confirmed;
            const validationMethod = body['human-confirmation'];
            const isLiteWithoutPassword = body.state === 'complete_lite' && res.locals.isUserWithoutPassword;
            const isNeoPhonishWithoutPersonal =
                body.state === 'complete_neophonish' && !person.firstname && !person.lastname;

            let hasRecoveryMethod = body.has_recovery_method;

            if (hasRecoveryMethod === undefined && validationMethod === undefined) {
                hasRecoveryMethod = true;
            }

            res.submitResult = body;
            res.locals.store.person.hasRecoveryMethod = hasRecoveryMethod;
            res.locals.store.common.track_id = body.track_id;
            res.locals.track_id = body.track_id;
            form.type = body.state;

            if (isLiteWithoutPassword) {
                res.locals.store.person.isLiteWithoutPassword = true;
            }

            correctForm({form, hasRecoveryMethod, isLiteWithoutPassword, isMobile, isNeoPhonishWithoutPersonal});

            if (phone && validationMethod === 'phone') {
                form.values.phone = phone;
                form.states.phone = 'valid';
                form.validation.humanConfirmationDone = true;
            }

            return next();
        }

        return next(new Error());
    }, next);
}

function redirectToRetpathIfNothingToDo(req, res) {
    const validatedRetpath = res.submitResult && res.submitResult.retpath;
    const defaultRetpath = url.format({
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'passport',
        query: {
            mode: 'passport'
        }
    });

    if (validatedRetpath) {
        return res.redirect(validatedRetpath);
    }

    return res.redirect(defaultRetpath);
}

function handleNonFormErrors(err, req, res, next) {
    // eslint-disable-line complexity
    PLog.warn()
        .logId(req.logID)
        .type('registration.complete')
        .write(err);

    if (req.method === 'POST') {
        return res.json({status: 'error', error: err});
    }

    const currentUrl = url.parse(req.originalUrl, true);
    const defaultRetpathObj = {
        protocol: req.headers['x-real-scheme'],
        hostname: req.hostname,
        pathname: 'passport',
        query: {
            mode: 'passport'
        }
    };

    let retpathToComplete;

    if (err instanceof Error) {
        return next(err);
    }

    if (
        Array.isArray(err) &&
        (err.indexOf('action.not_required') !== -1 ||
            err.indexOf('account.disabled') !== -1 ||
            err.indexOf('account.global_logout') !== -1 ||
            err.indexOf('sessionid.invalid') !== -1 ||
            err.indexOf('sslsession.required') !== -1)
    ) {
        retpathToComplete = _clone(defaultRetpathObj, true);

        if (err.indexOf('sessionid.invalid') !== -1 || err.indexOf('sslsession.required') !== -1) {
            retpathToComplete.pathname = 'auth';
            retpathToComplete.query.retpath = url.format(
                Object.assign({}, defaultRetpathObj, {
                    pathname: currentUrl.pathname,
                    query: currentUrl.query,
                    search: null
                })
            );
            delete retpathToComplete.query.mode;
        }

        if (err.indexOf('sslsession.required') !== -1) {
            retpathToComplete.pathname = 'auth/secure';
        }

        return res.redirect(url.format(retpathToComplete));
    }

    return next(err);
}

function submitForm(req, res) {
    const controller = req._controller;

    req.body.force_clean_web = true;

    if (
        !res.locals.experiments.flags.includes('exp-no-phone-auth-complete') &&
        req.body['human-confirmation'] === 'captcha'
    ) {
        return res.json({status: 'error', error: ['user.not_verified']});
    }

    return req.api
        .completeCommit(req.body.state, req.body)
        .then((result) => {
            const body = result.body;
            const finishPathname = '/auth/finish/';
            const profileUrl = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: 'profile'
            });

            let target = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: 'passport',
                query: {
                    mode: 'passport'
                }
            });

            controller.augmentResponse(body);

            if (body.cookies && body.cookies.length) {
                target = url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: finishPathname,
                    query: {
                        track_id: body.track_id || req.body.track_id, // || res.locals.track_id,
                        retpath: req.body.retpath || profileUrl
                    }
                });
            } else if (body.retpath) {
                target = body.retpath;
            }

            return res.json({status: 'ok', target});
        })
        .catch((errors) => {
            PLog.warn()
                .logId(req.logID)
                .type('registration.complete')
                .write(errors);
            return res.json({status: 'error', error: errors});
        });
}

// function doAuthSubmit(req, res, next) {
//     req.api.authSubmit('/1/bundle/auth/password/get_state/', {
//         track_id: res.locals.track_id
//     }).then((results) => {
//         const body = results.body;
//
//         res.locals.knownData = body;
//         res.locals.track_id = body.track_id;
//         res.locals.state = body.state;
//
//         return next();
//     }).catch((err) => {
//         PLog.warn()
//             .logId(req.logID)
//             .type('registration.complete')
//             .write(err);
//
//         return next();
//     });
// }

function doKeyLinkSubmit(req, res, next) {
    const secretKey = req.query && req.query.key;

    if (!secretKey) {
        return next(new Error());
    }

    return req.api
        .authKeyLinkSubmit({
            secret_key: secretKey
        })
        .then((results) => {
            const {state, track_id: trackId} = results.body;

            res.locals.track_id = trackId;
            res.locals.state = state;
            res.locals.store.common.track_id = trackId;
            res.locals.form.type = state;

            res.locals.knownData = {
                track_id: trackId,
                state,
                account: {},
                has_recovery_method: false
            };

            if (state === 'complete_autoregistered') {
                res.locals.skipSubmit = true;
            }

            return next();
        }, next);
}

const setup = [
    getYaExperimentsFlags,
    getLanguage,
    apiSetup,
    getCountry,
    function(req, res, next) {
        res.locals.registrationName = 'complete';
        return next();
    },
    (req, res, next) => {
        return req.api
            .getQuestions()
            .then((response) => {
                res.locals.questions = response.body.questions;

                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.v2')
                    .write(errors);

                return next();
            });
    },
    (req, res, next) => {
        return req.api
            .captchaGenerate({})
            .then((response) => {
                res.locals.captcha = response.body;

                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('registration.v2')
                    .write(errors);

                return next();
            });
    },
    createForm
];

const enterSocial = [
    checkAuth,
    setup,
    getState,
    createStore,
    setRegistrationPhoneOnly,
    doCompleteSubmit,
    renderPage,
    redirectToRetpathIfNothingToDo,
    handleNonFormErrors
];

const enterLite = [
    checkAuth,
    setup,
    getState,
    createStore,
    // doAuthSubmit,
    doCompleteSubmit,
    renderPage,
    redirectToRetpathIfNothingToDo,
    handleNonFormErrors
];

const enterAuth = [
    setup,
    getState,
    createStore,
    doKeyLinkSubmit,
    doCompleteSubmit,
    renderPage,
    redirectToRetpathIfNothingToDo,
    handleNonFormErrors
];

exports.save = [apiSetup, getYaExperimentsFlags, submitForm, handleNonFormErrors];

exports.route = function(app) {
    const modeFilter = (req, res, next) => {
        if (req.query && (req.query.mode === 'light2full' || req.query.mode === 'postregistration')) {
            return next();
        }

        return next('route');
    };
    const newFilter = (req, res, next) => {
        // if (req.query && req.query.old === '1') {
        //     return next('route');
        // }

        if (req.query && req.query.new === '1') {
            res.locals.force = true;
        }

        return next();
    };
    const keyFilter = (req, res, next) => {
        if (req.query && req.query.key) {
            return next();
        }

        return next('route');
    };

    app.get('/passport/?', [modeFilter, newFilter], enterSocial);
    app.get('/auth/?', [keyFilter, newFilter], enterAuth);
    app.get('/profile/upgrade/lite/?', newFilter, enterLite);
    app.get('/profile/upgrade', newFilter, enterSocial);
    app.post('/profile/complete/submit', (req) => {
        const query = Object.assign({}, req.query, {
            // old: 1
        });

        delete query.new;
        delete query.mode;
        return req._controller.redirectToLocalUrl({
            pathname: '/profile/upgrade',
            query
        });
    });
};
