const multer = require('multer');

const setupUserTicket = require('../common/userTicketsSetup.js');
const serviceTicketsSetup = require('../common/serviceTicketsSetup');
const cloudApiSetup = require('../common/cloudApiSetup');
const documentsApiSetup = require('../common/documentsApiSetup');

const storage = multer.diskStorage({
    destination(_req, _file, cb) {
        cb(null, '/tmp');
    },
    filename(_req, file, cb) {
        cb(null, `${Date.now()}-${file.originalname}`);
    }
});

const fileFilter = (_, file, cb) => {
    cb(null, ['image/jpg', 'image/jpeg', 'image/png', 'image/heic'].includes(file.mimetype));
};

const upload = multer({
    storage,
    fileFilter,
    limits: {
        fileSize: 10000000,
        files: 1
    }
});

const handleError = (res) => (error) => {
    res.json({
        message: error.message,
        description: error.description,
        error: error.error
    });
};

const middleware = [setupUserTicket, serviceTicketsSetup, cloudApiSetup];

const documentsMiddleware = [setupUserTicket, serviceTicketsSetup, documentsApiSetup];

const getDocumentList = (req, res) => {
    const {
        body: {
            type,
            userId,
            options = {
                offset: 0,
                limit: 20
            }
        }
    } = req;

    return req._documentsAPI.getDocumentList(type, userId, options).then((result) => {
        return res.json(result);
    });
};

const uploadDocument = (req, res) => {
    const {
        file,
        body: {type, userId}
    } = req;

    return req._documentsAPI.uploadDocument(type, userId, file).then((result) => {
        return res.json(result);
    });
};

const deleteDocument = (req, res) => {
    const {
        body: {userId, docId}
    } = req;

    return req._documentsAPI.deleteDocument(userId, docId).then((result) => {
        return res.json(result);
    });
};

const copyDocument = (req, res) => {
    const {
        body: {userId, path, type}
    } = req;

    return req._documentsAPI.copyDocument(userId, path, type).then((result) => {
        return res.json(result);
    });
};

const createFolder = (req, res) => {
    const {
        body: {folderPath = '', redirect = false}
    } = req;

    return req._cloudAPI
        .createDiskFolder(folderPath)
        .then((result) => {
            if (redirect) {
                return res.redirect(result.href);
            }

            return res.json(result);
        })
        .catch(handleError(res));
};

const getResourceInfo = (req, res) => {
    const {
        body: {folderPath = '', ...options}
    } = req;

    return req._cloudAPI
        .readDiskResourceMeta(folderPath, options)
        .then((result) => res.json(result))
        .catch(handleError(res));
};

const getDiskInfo = (req, res) => {
    return req._cloudAPI
        .readDiskMeta()
        .then((result) => res.json(result))
        .catch(handleError(res));
};

const searchDiskResource = (req, res) => {
    const {
        body: {query = '', ...options}
    } = req;

    return req._cloudAPI
        .searchDiskResource(query, options)
        .then((result) => res.json(result))
        .catch(handleError(res));
};

const copyDiskResource = (req, res) => {
    const {
        body: {src, dst}
    } = req;

    return req._cloudAPI
        .copyDiskResource(src, dst)
        .then((result) => res.json(result))
        .catch(handleError(res));
};

const deleteDiskResource = (req, res) => {
    const {
        body: {path}
    } = req;

    return req._cloudAPI
        .deleteDiskResource(path)
        .then((result) => res.json(result))
        .catch(handleError(res));
};

const createDiskResource = (req, res) => {
    const {
        body: {path}
    } = req;

    return req._cloudAPI
        .createDiskResource(path)
        .then((result) => res.json(result))
        .catch(handleError(res));
};

module.exports = {
    createFolder: [...middleware, createFolder],
    getResourceInfo: [...middleware, getResourceInfo],
    getDiskInfo: [...middleware, getDiskInfo],
    searchDiskResource: [...middleware, searchDiskResource],
    copyDiskResource: [...middleware, copyDiskResource],
    deleteDiskResource: [...middleware, deleteDiskResource],
    createDiskResource: [...middleware, createDiskResource],
    // documents API
    getDocumentList: [...documentsMiddleware, getDocumentList],
    uploadDocument: [...documentsMiddleware, upload.single('document-uploader'), uploadDocument],
    deleteDocument: [...documentsMiddleware, deleteDocument],
    copyDocument: [...documentsMiddleware, copyDocument]
};
