const url = require('url');
const util = require('util');
const apiSetup = require('./common/apiSetup');
const got = require('got');
const _ = require('lodash');
const assert = require('assert');
const config = require('../configs/current');
const paths = config.paths;
const httpCodes = require('http-codes');
const punycode = require('punycode');
const PLog = require('plog');
const NOT_FOUND = -1;
const redirectStatusCode = httpCodes.MOVED_TEMPORARILY || httpCodes.FOUND;
const forcedMDADomains = require('../configs/common').forcedMDADomains;
const getMDAOptions = require('./common/getMDAOptions.js');

function embeddedModeFilter(req, res, next) {
    const mode = req.body.mode || req.query.mode;

    if (mode === 'embeddedauth') {
        return next();
    }

    return next('route');
}

exports.routes = {};

exports.getConfig = function() {
    return config;
};

exports.route = function(app) {
    const opts = {
        root: 'pages/auth/',
        headers: {
            'X-Frame-Options': 'SAMEORIGIN'
        }
    };
    const routes = this.routes;
    const embeddedSimpleRoute = [embeddedModeFilter, routes.submit];
    const embeddedGetRoute = [embeddedModeFilter, routes.getSubmit];
    const pddRoute = '/for/:pdd_domain*';

    app.post('/passport/?', embeddedSimpleRoute);
    app.post(pddRoute, embeddedSimpleRoute);

    app.get('/passport/?', embeddedGetRoute);
    app.get(pddRoute, embeddedGetRoute);

    app.get('/auth/captcha/?', routes.captcha);

    if (app.get('env') !== 'production') {
        app.get('/auth/accounts.json', routes.accounts);
        app.get('/auth/console/', function(req, res) {
            res.sendFile('iframe.html', opts);
        });
        app.get('/auth/console.js', function(req, res) {
            res.sendFile('iframe.js', {
                root: 'pages/auth/',
                headers: {
                    'Content-type': 'text/javascript'
                }
            });
        });
    }

    app.get('/auth/login-status.html', function(req, res) {
        return res
            .status(httpCodes.OK)
            .set('X-Frame-Options', 'SAMEORIGIN')
            .sendFile('login-status.html', {
                root: 'pages/public_html/'
            });
    });

    app.all('/auth/login-status_v2.html', function(req, res) {
        return res
            .status(httpCodes.OK)
            .set('X-Frame-Options', 'SAMEORIGIN')
            .sendFile('login-status_v2.html', {
                root: 'pages/public_html/'
            });
    });

    app.get('/auth/noop.html', function(req, res) {
        return res
            .status(200)
            .set('X-Frame-Options', 'SAMEORIGIN')
            .send('');
    });
};

exports.routes.submit = [
    apiSetup,
    setHeaders,
    validateRetpath,
    guard(isActionChangeDefault, multiAuth('change_default')),
    guard(isActionLogout, multiAuth('logout')),
    guard(hasCaptchaAnswer, checkCaptcha),
    guard(needSubmitData, submitData),
    guardNot(isAuthFinished, processResult),
    guard(needCaptchaUrl, setCaptchaUrl),
    guard(isReadyToGetSession, getSession),
    guard(isAuthFinished, redirectToFinish),
    setIdkey,
    patchRetpath,
    redirectWithNotOK,
    redirectWithError
];

exports.routes.getSubmit = [
    apiSetup,
    setHeaders,
    validateRetpath,
    guard(isActionChangeDefault, multiAuth('change_default')),
    guard(isActionLogout, multiAuth('logout')),
    guard(isReadyToGetSession, getSession),
    guard(isAuthFinished, redirectToCleanFinish),
    guardNot(isAuthFinished, function(req, res, next) {
        next(res.submitErrors);
    }),

    function redirectToAuthIfError(err, req, res, next) {
        /* eslint no-unused-vars: "off" */
        const pathname = 'auth';
        const retpath = req.query && req.query.retpath;
        const login = req.query && req.query.login;
        const query = {};

        if (err && err.indexOf('action.not_required') !== NOT_FOUND) {
            PLog.info()
                .logId(req.logID)
                .type('embeddedauth', 'getSubmit')
                .write(err);
            // retpath есть и провалидирован, иначе мы бы так далеко не зашли,
            // и уже 403 получили бы в самом начале
            return res.redirect(redirectStatusCode, retpath);
        }

        PLog.warn()
            .logId(req.logID)
            .type('embeddedauth', 'getSubmit')
            .write(err);

        if (retpath) {
            query.retpath = retpath;
        }

        if (login) {
            query.login = login;
        }

        return res.redirect(
            redirectStatusCode,
            url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname,
                query
            })
        );
    }
];

exports.routes.captcha = [
    apiSetup,
    sendCaptcha,
    function(err, req, res, next) {
        PLog.warn()
            .logId(req.logID)
            .type('embeddedauth', 'routes', 'captcha')
            .write(err);

        return res.status(httpCodes.INTERNAL_SERVER_ERROR).send('');
    }
];

exports.routes.accounts = [
    function(req, res, next) {
        const controller = req._controller;

        controller
            .getAuth()
            .sessionID({multisession: 'yes', regname: 'yes', allow_child: 'yes', full_info: 'yes'})
            .then(function(sessioninfo) {
                req.blackbox = sessioninfo || {};
                next();
            })
            .catch(function(error) {
                if (error && error.code === 'need_resign') {
                    return controller.getAuth().resign();
                }

                return next(error || new Error());
            });
    },
    function(req, res) {
        const result = req.blackbox;

        result.users = _.forEach(result.users, function(user) {
            const name = user.display_name && user.display_name.name;
            const domain = name && /@/.test(name) && name.split('@')[1];

            if (domain) {
                user.display_name.name = `${name.split('@')[0]}@${punycode.toUnicode(domain)}`;
            }
        });
        result.users = _.filter(result.users, 'uid');
        res.set('Content-type', 'application/json');
        res.status(httpCodes.OK).send(result);
    }
];

function cond(check, then, error) {
    return function() {
        const args = Array.prototype.slice.call(arguments);
        const one = then || noopMiddleware;
        const another = error || noopMiddleware;

        if (check.apply(null, args)) {
            one.apply(null, args);
        } else {
            another.apply(null, args);
        }
    };
}

function guard(check, then) {
    return cond(check, then);
}

function guardNot(check, then) {
    return cond(check, null, then);
}

function noopMiddleware(req, res, next) {
    return next();
}

function isAuthFinished(req, res) {
    return Boolean(res.authFinished);
}

function isReadyToGetSession(req, res) {
    return Boolean(res.readyToGetSession);
}

function needSubmitData(req, res) {
    return !(isAuthFinished(req, res) || (res.submitErrors && res.submitErrors.length));
}

function hasCaptchaAnswer(req) {
    return req.body && req.body.captcha_answer;
}

function needCaptchaUrl(req, res) {
    const status = res.result && res.result.status;

    return status && ['captcha-invalid', 'captcha-required'].indexOf(status) !== NOT_FOUND;
}

function isActionChangeDefault(req) {
    const action = (req.body && req.body.action) || (req.nquery && req.nquery.action);

    return Boolean(action && action === 'change_default');
}

function isActionLogout(req) {
    const action = (req.body && req.body.action) || (req.nquery && req.nquery.action);

    return Boolean(action && action === 'logout');
}

function sendCaptcha(req, res, next) {
    const cantread = req.nquery && req.nquery.cantread;
    const opts = {
        use_cached: 'yes'
    };

    assert(req.api, 'API should be inited');

    if (cantread === '1') {
        delete opts.use_cached;
    }

    req.api.captchaGenerate(opts).then(function(results) {
        const stream = got.stream(results.body.image_url);

        stream.on('error', next);
        stream.pipe(res);
    }, next);
}

function redirectWithError(err, req, res, next) {
    const answerUrl = url.parse(req.body.retpath || (req.nquery && req.nquery.retpath), true);

    PLog.warn()
        .logId(req.logID)
        .type('embeddedauth', 'redirectWithError')
        .write(err);

    delete answerUrl.search;
    answerUrl.query.status = 'internal-exception';

    if (req.body.one === '1' && Array.isArray(err) && err[0] === 'sessionid.overflow') {
        answerUrl.query.errors = 'sessionid.overflow';
    }

    res.redirect(redirectStatusCode, url.format(answerUrl));
}

function redirectWithNotOK(req, res, next) {
    const answerUrl = url.parse(req.body.retpath || (req.nquery && req.nquery.retpath), true);

    delete answerUrl.search;
    answerUrl.query = _.merge({}, answerUrl.query, res.result);
    res.redirect(redirectStatusCode, url.format(answerUrl));
}

function setHeaders(req, res, next) {
    res.setHeader('Access-Control-Expose-Headers', 'Location');
    return next();
}

function processField(field) {
    let res = '';

    if (typeof field !== 'string') {
        if (Array.isArray(field)) {
            res = field[0];
        }
    } else {
        res = field;
    }

    return res;
}

function submitData(req, res, next) {
    const reqBody = req.body;
    const reqQuery = req.query;
    const login = reqBody.login;
    const password = reqBody.password || reqBody.passwd;
    const retpath = reqBody.retpath || reqQuery.retpath;
    const from = reqBody.from || reqQuery.from;
    const fretpath = reqBody.fretpath || reqQuery.fretpath;
    const clean = reqBody.clean || reqQuery.clean;
    const origin = reqBody.origin || reqQuery.origin;
    const idkey = !res.clearTrackId && reqBody.idkey;
    const twoweeks = reqBody.twoweeks;
    const rfc = reqBody.rfc === '1';
    const emailCode = reqBody.email_code === '1';
    const pddDomain = req.params.pdd_domain || reqBody.pdd_domain || reqQuery.pdd_domain;
    const data = {};
    const controller = req._controller;

    // Явно задаем track_id, потому что он приходит в idkey
    // и в apiSetup не инициализируется
    if (idkey) {
        req.api.track(idkey);
        data.track_id = idkey;
    }

    function onError(error) {
        if (error instanceof Error) {
            return next(error);
        }
        res.submitErrors = error;
        return next();
    }

    function onSuccess(result) {
        const body = result.body;

        res.trackId = body.track_id || body.id;
        res.account = body.account;
        res.state = body.state || null;

        if (body.errors) {
            if (
                Array.isArray(body.errors) &&
                ['track.not_found', 'track_id.invalid', 'account.auth_passed'].indexOf(body.errors[0]) > -1
            ) {
                res.clearTrackId = true;
                return submitData(req, res, next);
            }

            return onError(body.errors);
        }

        if (!body.state && !body.errors) {
            // авторизуем
            controller.augmentResponse(body);
            res.authFinished = true;

            res.readyToGetSession = true;
        }

        return next();
    }

    res.authFinished = false;

    if (!rfc && !emailCode) {
        if (login) {
            data.login = processField(login).trim();
        } else {
            res.submitErrors = ['login.empty'];
            return next();
        }

        if (password) {
            data.password = processField(password);
        } else {
            res.submitErrors = ['password.empty'];
            return next();
        }
    }

    if (rfc) {
        const otp = password && processField(password).replace(/[^0-9]*/g, '');

        if (otp && otp.length === 6) {
            data.otp = otp;
        } else {
            if (password) {
                res.submitErrors = ['rfc_otp.invalid'];
            } else {
                res.submitErrors = ['password.empty'];
            }

            return next();
        }
    }

    if (emailCode) {
        data.code = password && processField(password).trim();
    }

    if (retpath) {
        data.retpath = processField(retpath).trim();
    }

    if (from) {
        data.service = processField(from).trim();
    }

    if (fretpath) {
        data.fretpath = processField(fretpath).trim();
    }

    if (clean) {
        data.clean = processField(clean).trim();
    }

    if (origin) {
        data.origin = processField(origin).trim();
    }

    if (twoweeks === 'yes' || twoweeks === '1') {
        data.policy = 'long';
    } else {
        data.policy = 'sessional';
    }

    if (reqBody.one === '1') {
        if (twoweeks === 'no') {
            data.policy = 'sessional';
        } else {
            data.policy = 'long';
        }
    }

    if (pddDomain && data.login && !/[^@]+@[^.]+\.[^.]+/.test(data.login)) {
        data.login = `${data.login}@${pddDomain}`;
    }

    if (pddDomain) {
        data.is_pdd = '1';
    }

    data.passErrors = true;

    if (emailCode) {
        return req.api.authPasswordMultiStepEmailCodeCommit(data).then(onSuccess, onError);
    }

    if (rfc) {
        return req.api.authSubmit('/1/bundle/auth/password/rfc_otp/check/', data).then(onSuccess, onError);
    }

    return req.api.authSubmit('/2/bundle/auth/password/commit_password/', data).then(onSuccess, onError);
}

function processResult(req, res, next) {
    const error = res.submitErrors;
    const response = res.result || (res.result = {});
    const rfc = req.body && req.body.rfc === '1';
    const emailCode = req.body && req.body.email_code === '1';
    const otherQuery = {};

    let pdd;

    let pathname;

    response.source = req.body.source === 'password' ? req.body.source : null;

    if (rfc) {
        response.state = 'rfc_totp';
    }

    if (emailCode) {
        response.state = 'email_code';
    }

    if (error) {
        if (req.body.extended) {
            response.errors = error.join(',');
        }

        if (error.indexOf('captcha.invalid') !== -1) {
            response.status = 'captcha-invalid';
            return next();
        }

        if (error.indexOf('captcha.captchalocate') !== -1) {
            response.status = 'captcha-captchalocate';
            return next();
        }

        if (error.indexOf('captcha.required') !== -1) {
            response.status = 'captcha-required';
            return next();
        }

        if (error.indexOf('account.not_found') !== -1) {
            response.status = 'account-not-found';
        }

        if (error.indexOf('account.auth_passed') !== -1) {
            response.status = 'account-auth-passed';
        }

        if (error.indexOf('password.not_matched') !== -1) {
            if (res.account && res.account.is_2fa_enabled) {
                response.magic = 'yes';

                if (req.body.source === 'otp' || req.body.source === 'magic') {
                    response.source = 'suggested';
                }
            } else {
                response.status = 'password-invalid';
            }
        }

        if (error.indexOf('rfc_otp.invalid') !== -1) {
            response.status = 'password-invalid';
        }

        if (error.indexOf('code.invalid') !== -1) {
            response.status = 'password-invalid';
        }

        if (error.indexOf('code.empty') !== -1) {
            response.status = 'password-empty';
        }

        if (error.indexOf('email_confirmations_limit.exceeded') !== -1) {
            response.status = 'email-confirmations-limit-exceeded';
        }

        if (error.indexOf('login.empty') !== -1) {
            response.status = 'login-empty';
        }

        if (error.indexOf('password.empty') !== -1) {
            response.status = 'password-empty';
        }

        if (error.indexOf('uid.empty') !== -1) {
            response.status = 'uid-empty';
        }

        if (error.indexOf('yu.empty') !== -1) {
            response.status = 'yu-empty';
        }

        if (error.indexOf('yu.invalid') !== -1) {
            response.status = 'yu-invalid';
        }

        if (error.indexOf('sessionid.invalid') !== -1) {
            response.status = 'sessionid-invalid';
        }

        if (error.indexOf('session.no_uid') !== -1 || error.indexOf('sessionid.no_uid') !== -1) {
            response.status = 'sessionid-no-uid';
        }

        if (error.indexOf('action.not_required') !== -1) {
            response.status = 'action-not-required';
        }

        if (error.indexOf('csrf_token.invalid') !== -1) {
            response.status = 'csrf-token-invalid';
        }
    }

    assert(req.api, 'API should be inited');

    if (!response.status) {
        response.status = 'other';
        pdd = res.account && res.account.domain && res.account.domain.unicode;
        pathname = pdd ? `/for/${encodeURIComponent(pdd)}/` : '/auth/';
        otherQuery.track_id = req.api.track();

        if (req.body.one === '1') {
            otherQuery.one = 'yes';

            if (res.state) {
                response.state = res.state;
            }
        }

        response.url = url.format({
            protocol: req.headers['x-real-scheme'],
            host: req.hostname,
            pathname,
            query: otherQuery
        });
    }

    return next();
}

function setCaptchaUrl(req, res, next) {
    const response = res.result;

    assert(req.api, 'API should be inited');
    assert(res.result, 'res.result should be defined');

    req.api.params('track_id').then(function(results) {
        response.captcha_url = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/auth/captcha/',
            query: {
                track_id: results.body.id,
                dnrcn: Math.floor(Math.random() * 1e5 + 1).toString()
            }
        });
        next();
    }, next);
}

function getSession(req, res, next) {
    const data = {
        track_id: res.trackId || req.api.track()
    };
    const controller = req._controller;

    req.api
        .authSubmit('/1/bundle/session/', data)
        .then(function(result) {
            const body = result.body;

            controller.augmentResponse(body);

            return next();
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('embeddedauth.submit.session')
                .write(err);

            return next(err);
        });
}

function redirectToFinish(req, res) {
    assert(res.trackId, 'track_id is required');

    if (req.body.one === '1') {
        const realRetpath = req.body && req.body.real_retpath;
        const parsedRetpath = url.parse(realRetpath, true);
        const matchedRetpath =
            parsedRetpath && parsedRetpath.hostname && /\.?([^.]*\.[^.]*$)/.exec(parsedRetpath.hostname);
        const mdaOptions = getMDAOptions(realRetpath, {});

        if (mdaOptions.needMDA) {
            return res.redirect(redirectStatusCode, mdaOptions.redirectUrl);
        }

        return res.redirect(
            redirectStatusCode,
            url.format({
                protocol: 'https',
                hostname: req.hostname,
                pathname: '/auth/finish/',
                query: {
                    track_id: res.trackId,
                    retpath: url.format(
                        Object.assign({}, url.parse(req.body.retpath, true), {
                            search: null,
                            query: {
                                status: 'ok',
                                source:
                                    ['otp', 'password', 'magic'].indexOf(req.body.source) > -1 ? req.body.source : null
                            }
                        })
                    )
                }
            })
        );
    }

    return res.redirect(
        redirectStatusCode,
        url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/auth/finish/',
            query: {
                track_id: res.trackId,
                embedded: 'yes'
            }
        })
    );
}

function checkCaptcha(req, res, next) {
    const captchaAnswer = req.body.captcha_answer;
    const trackId = req.body.idkey;

    assert(req.api, 'API should be inited');
    assert(trackId, 'track_id is required');

    // Явно задаем track_id, потому что он приходит в idkey
    // и в apiSetup не инициализируется
    req.api.track(trackId);

    req.api
        .captchaCheck({
            answer: captchaAnswer
        })
        .then(
            function(result) {
                let errors;

                if (!result.body.correct) {
                    errors = res.submitErrors || (res.submitErrors = []);
                    errors.push('captcha.invalid');
                }
                next();
            },
            function(error) {
                let errors;

                if (error && Array.isArray(error)) {
                    if (error[0].code && error[0].code === 'captchalocate') {
                        errors = res.submitErrors || (res.submitErrors = []);
                        errors.push('captcha.captchalocate');
                    }
                }
                next();
            }
        );
}

function setIdkey(req, res, next) {
    const response = res.result || (res.result = {});

    assert(req.api, 'API should be inited');

    if (!req.api.track()) {
        return req.api.getTrack({type: 'authorize'}).then(function(results) {
            response.idkey = results.body && results.body.id;
            return next();
        }, next);
    }

    response.idkey = req.api.track();
    return next();
}

function patchRetpath(req, res, next) {
    let realRetpath = req.body && req.body.real_retpath;

    let controller;

    assert(req.api, 'API should be inited');

    if (realRetpath === '') {
        controller = req._controller;
        realRetpath = url.format(controller.getModePassportUrl());
    }

    if (!realRetpath) {
        return next();
    }

    return req.api
        .writeTrack({
            retpath: realRetpath
        })
        .then(function() {
            return next();
        })
        .catch(function() {
            return next();
        });
}

function validateRetpath(req, res, next) {
    const retpath = req.body.retpath || (req.nquery && req.nquery.retpath);

    if (!retpath) {
        res.sendStatus(httpCodes.BAD_REQUEST);
        return;
    }

    assert(req.api, 'API should be inited');
    req.api
        .validateRetpath({
            retpath
        })
        .then(function(results) {
            const errors = results.body && results.body.validation_errors;

            if (!errors) {
                return next();
            }

            return res.sendStatus(httpCodes.FORBIDDEN);
        }, next);
}

function multiAuth(action) {
    const paths = {
        change_default: '/1/bundle/auth/change_default/',
        logout: '/1/bundle/auth/logout/'
    };

    const apiHandle = action && paths[action];

    if (!apiHandle) {
        return noopMiddleware;
    }

    return function(req, res, next) {
        const uid = req.body.uid || req.query.uid;
        const retpath = req.body.retpath || req.query.retpath;
        const yu = req.body.yu || req.query.yu;
        const controller = req._controller;

        function onError(error) {
            if (error instanceof Error) {
                return next(error);
            }
            res.submitErrors = error;
            return next();
        }

        function doSubmit() {
            req.api.writeTrack({retpath}).then(function(result) {
                const trackId = (res.trackId = result.body.id);

                req.api
                    .authSubmit(apiHandle, {
                        track_id: trackId,
                        uid
                    })
                    .then(function(submitResult) {
                        const body = submitResult.body;

                        res.account = body.account;
                        controller.augmentResponse(body);
                        res.authFinished = true;
                        return next();
                    }, onError);
            }, onError);
        }

        if (!yu) {
            res.submitErrors = ['yu.empty'];
            return next();
        }

        if (yu !== req.cookies.yandexuid) {
            res.submitErrors = ['yu.invalid'];
            return next();
        }

        if (!uid) {
            res.submitErrors = ['uid.empty'];
            return next();
        }

        if (!req.api.track()) {
            return req.api.getTrack({type: 'authorize'}).then(function(result) {
                req.api.track(result.body.id);

                return doSubmit();
            }, onError);
        }

        return doSubmit();
    };
}

function redirectToCleanFinish(req, res) {
    assert(res.trackId, 'track_id is required');

    res.redirect(
        redirectStatusCode,
        url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/auth/finish/',
            query: {
                track_id: res.trackId
            }
        })
    );
}

exports.checkCaptcha = checkCaptcha;
exports.redirectToFinish = redirectToFinish;
exports.multiAuth = multiAuth;
exports.submitData = submitData;
exports.getSession = getSession;
exports.redirectToFinish = redirectToFinish;
exports.redirectWithNotOK = redirectWithNotOK;
exports.redirectWithError = redirectWithError;
exports.processField = processField;
