const tvm = require('../../lib/tvm');
const FamilyPayApi = require('../../lib/api/familypay');
const AntifraudApi = require('../../lib/api/antifraud');
const {urlFormat} = require('../common/urlFormat.js');
const CONSTS = require('./const');

const mapFamilyPayBalance = ({amount = 0} = {}, {value = 0} = {}) => value - amount;

const sortMembers = (members = []) => {
    const firstMember = members.find((member) => member.isYou);

    return firstMember ? [firstMember, ...members.filter((member) => !member.isYou)] : members;
};

const createFamilyPayApi = (req, res) =>
    new FamilyPayApi(
        req.logID,
        res.locals.track_id,
        req.userTicket,
        req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.FAMILYPAY]
    );

const checkFamilyExist = (req, res) =>
    fetchFamilyInfo(req, res, {withBBUsers: false}).then(({familyId}) => {
        if (!familyId) {
            return req.api.createFamily();
        }
    });

const fetchFamilyInfo = async (req, res, {withUid = false, withAccount = false, withBBUsers = true} = {}) => {
    const controller = req._controller;
    const uid = Number(controller.getAuth().getUid());

    const {body = {}} = await req.api.getAccountWithFamilyInfo();

    const {
        account: {family_info: familyInfo},
        family_members: members = [],
        family_kids: kids = [],
        family_invites: invites = [],
        family_settings: {max_capacity: familyCapacity, max_kids_number: kiddishCapacity} = {}
    } = body;
    const adminUid = familyInfo && familyInfo.admin_uid;
    const familyId = familyInfo && familyInfo.family_id;

    const users = withBBUsers
        ? (
              await controller.getAuth().userInfo({
                  uid: members.map(({uid}) => uid).join(',') || uid,
                  attributes: [
                      CONSTS.HAVE_FAMILY_ROLE_ATTRIBUTE,
                      CONSTS.CHILDISH_ATTRIBUTE,
                      CONSTS.CONTENT_RATING_ATTRIBUTE
                  ].join(','),
                  returnAllUsers: true,
                  getphones: 'all',
                  phone_attributes: [CONSTS.HAVE_SECURE_PHONE_ATTRIBUTE]
              })
          ).map((i) => ({
              uid: Number(i.uid.value),
              attributes: i.attributes,
              phonesAttrs: i.phones.map((j) => j.attributes)
          }))
        : [];
    const hasFamilySubscription = users.some(({attributes = {}}) =>
        Boolean(attributes[CONSTS.HAVE_FAMILY_ROLE_ATTRIBUTE])
    );
    const result = {
        members: sortMembers(
            members.map((member) => {
                const user = users.find((i) => i.uid === member.uid);

                const newMember = {
                    placeId: member.place_id,
                    name: member.display_name,
                    avatar: member.default_avatar,
                    isYou: member.uid === uid,
                    isAdmin: Boolean(adminUid) && member.uid === adminUid,
                    hasPlus: Boolean(member.has_plus)
                };

                if (withUid) {
                    newMember.uid = member.uid;
                }

                if (withBBUsers) {
                    if (user.attributes[CONSTS.CHILDISH_ATTRIBUTE]) {
                        newMember.isChildish = true;
                        newMember.rating = Number(user.attributes[CONSTS.CONTENT_RATING_ATTRIBUTE] || 0);
                    }

                    newMember.isAbleToUsePay = user.phonesAttrs.some((attr) =>
                        Boolean(attr[CONSTS.HAVE_SECURE_PHONE_ATTRIBUTE])
                    );
                }

                return newMember;
            })
        ),
        kids: kids.map((kiddish) => ({
            uid: kiddish.uid,
            placeId: kiddish.place_id,
            name: kiddish.display_name && kiddish.display_name.name,
            avatar: kiddish.display_name && kiddish.display_name.default_avatar,
            gender: kiddish.person && CONSTS.KIDDISH_GENDER_MAP[kiddish.person.gender],
            birthday: kiddish.person && kiddish.person.birthday,
            rating: kiddish.content_rating_class,
            videoRating: kiddish.video_content_rating_class,
            musicRating: kiddish.music_content_rating_class,
            isYou: false,
            isAdmin: false,
            hasPlus: hasFamilySubscription
        })),
        invites: invites.map((invite) => ({
            inviteId: invite.invite_id,
            inviteMethod: invite.send_method,
            inviteContact: invite.contact
        })),
        familyId,
        hasFamilySubscription,
        kiddishCapacity,
        familyCapacity
    };

    if (familyId) {
        const api = createFamilyPayApi(req, res);
        const familyPayInfo = await api.fetchFamilyByIdSingleLimit(familyId).catch(() => ({error: true}));
        const {properties: {cardInfo = {}} = {}, userRestrictions = [], error, adminUid} = familyPayInfo;
        const adminInfo = userRestrictions.find((member) => member.uid === adminUid) || {};
        const {limit: {value: adminValue} = {}} = adminInfo;

        Object.assign(result, {
            pay: error
                ? {}
                : {
                      defaultValue: adminValue / CONSTS.PAY_PENNY_COUNT,
                      cardInfo,
                      usersInfo: userRestrictions
                          .map(
                              ({
                                  allowedServices,
                                  limit,
                                  limitCurrency,
                                  expenses,
                                  uid,
                                  allowAllServices,
                                  enabled
                              } = {}) => {
                                  const member = members.find((member) => member.uid === uid);

                                  if (!member) {
                                      // Участник удален из семьи но все еще не удален из familypay?
                                      return null;
                                  }

                                  const fullValue = limit.value / CONSTS.PAY_PENNY_COUNT;
                                  const limitMode =
                                      fullValue >= CONSTS.PAY_NO_LIMIT_VALUE_STARTS ? 'NOLIMIT' : limit.limitMode;
                                  const value = limitMode === 'NOLIMIT' ? 0 : fullValue;

                                  return {
                                      allowedServices,
                                      limit: {...limit, limitMode, value},
                                      currency: limitCurrency,
                                      allowAllServices: Boolean(allowAllServices),
                                      isEnabled: Boolean(enabled),
                                      balance: mapFamilyPayBalance(expenses, limit) / CONSTS.PAY_PENNY_COUNT,
                                      placeId: member.place_id
                                  };
                              }
                          )
                          .filter(Boolean)
                  }
        });
    }

    if (withAccount) {
        Object.assign(result, {account: body.account});
    }

    return result;
};

const postBindCard = async (req, res, trackId, cardId, retpath = '', origin = '') => {
    try {
        const controller = req._controller;
        const uid = Number(controller.getAuth().getUid());

        const {body: {billing_response: {payment_methods = {}} = {}} = {}} = await req.api.billingListPaymentMethods({
            track_id: trackId
        });
        const {number = '', system = '', currency = 'RUB'} =
            Object.values(payment_methods).find((i) => i.card_id === cardId) || {};

        if (!number || !system) {
            return {state: 'error', error: 'family.pay.no-card'};
        }

        if (
            req._controller.hasExp('profile-family-pay-3ds-exp') ||
            CONSTS.ORIGINS_WITHOUT_FAMILY_PAY_3DS_EXP.includes(origin)
        ) {
            const {account: {phones = {}} = {}} = await fetchFamilyInfo(req, res, {
                withAccount: true,
                withBBUsers: false
            });
            const {login_id: loginId} = await controller.getAuth().sessionID({get_login_id: 'yes'});

            const securePhoneId = Object.keys(phones).filter((phoneId) => phones[phoneId].bound)[0];
            const antifraudApi = new AntifraudApi(
                req.logID,
                req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.ANTIFRAUD]
            );

            const {action, tags: [{url: redirectUrl} = {}] = []} = await antifraudApi.score({
                extId: trackId,
                uid,
                cardId,
                phone: phones[securePhoneId].number.original,
                loginId,
                yandexUid: req.cookies['yandexuid'],
                retpath: urlFormat({
                    protocol: 'https',
                    hostname: req.hostname,
                    pathname: '/profile/family/after3ds',
                    query: {cardId, retpath, origin}
                }),
                paymentType: 'family-card-binding',
                serviceId: 'family-card-binding'
            });

            // Нельзя ни под каким предлогом
            if (action === 'DENY') {
                return {state: 'error', error: 'family.pay.deny', retpath};
            }

            // Можно после челленджа
            if (action === 'ALLOW' && redirectUrl) {
                return {state: 'challenge', redirectUrl, retpath};
            }
        }

        // Можно
        await checkFamilyExist(req, res);
        const {familyId, pay: {cardInfo: {cardId: oldCardId, bound} = {}} = {}} = await fetchFamilyInfo(req, res, {
            withBBUsers: false
        });

        const familyPayApi = createFamilyPayApi(req, res);

        if (oldCardId) {
            const isOnlyBoundUpdate = !bound && oldCardId === cardId;

            await familyPayApi.updateCardSingleLimit(
                familyId,
                isOnlyBoundUpdate
                    ? {bound: true}
                    : {
                          cardId,
                          maskedNumber: number,
                          paySystem: system,
                          currency,
                          bound: true
                      }
            );
            return {state: 'bind', retpath};
        }

        await familyPayApi.bindCardSingleLimit(familyId, uid, cardId, number, system, currency);

        return {state: 'bind', retpath};
    } catch (err) {
        return {state: 'error', error: 'family.error', retpath};
    }
};

const fetchInviteInfo = (req, res, inviteId) =>
    req.api.getFamilyInviteInfo(inviteId).then(({body: {issuer_uid: uid, family_id: familyId}}) =>
        req._controller
            .getAuth()
            .userInfo({
                uid,
                regname: 'yes',
                attributes: [CONSTS.HAVE_PLUS_ATTRIBUTE]
            })
            .then(({display_name: {name, avatar: {default: avatar}}, attributes}) => ({
                admin: {
                    avatar,
                    name,
                    familyId,
                    isUser: true,
                    hasPlus: Boolean(attributes[CONSTS.HAVE_PLUS_ATTRIBUTE])
                }
            }))
    );

module.exports = {
    createFamilyPayApi,
    checkFamilyExist,
    postBindCard,
    fetchFamilyInfo,
    fetchInviteInfo
};
