var putils = require('putils');
var api = require('../lib/passport-api/index.js');
var crypto = require('crypto');
var url = require('url');
var PLog = require('plog');
var trimSlashesRegexp = /(^\/|\/$)/g;
var spacesRegexp = /\s+/g;

var token = {
    cipherAlgorightm: 'aes128',
    token_lifetime: 3600000, //One hour in ms
    salt: 'reen2Orie4phu5oogah6eecheiKigo5eaS6ahMaechuz7Maebah7VooL6Xe8wai5',

    /**
     * Create a key valid for a day for a given yandexuid
     * to sign timestamp for logger requests token
     *
     * @param {String} yandexuid - Yandex cookie to calculate the token
     * @returns {String} key - key is used to create token and check token validity
     */
    getKey: function(yandexuid) {
        var uid = yandexuid || 0;

        return `${token.salt}::${uid}`;
    },

    /**
     * Create token to sign logging requests.
     *
     * @param {String} yandexuid Yandex cookie to calculate the token
     * @returns {String} ciphered timestamp in hex encoding
     */
    create: function(yandexuid) {
        var key = token.getKey(yandexuid);
        var timestamp = new Date().getTime().toString();
        var cipher = crypto.createCipher(token.cipherAlgorightm, key);

        cipher.update(timestamp, 'ascii', 'hex');

        return cipher.final('hex');
    },
    /**
     * Check whether a token was created on yandex page
     *
     * @param {string} token -Token to check whether yandex initiated logging
     * @param {string} yandexuid - Yandex cookie to calculate the token
     * @return {boolean}
     */
    check: function(token, yandexuid) {
        var key = this.getKey(yandexuid);
        var now = new Date().getTime();
        var decipher = crypto.createDecipher(this.cipherAlgorightm, key);
        var timestamp;

        decipher.update(token, 'hex', 'ascii');
        try {
            timestamp = parseInt(decipher.final('ascii'), 10);
        } catch (e) {
            return false;
        }
        return now - timestamp < this.token_lifetime;
    }
};

var utils = {
    /**
     * Try to get log object
     *
     * @param {JSON|Base64String}    json - JSON string, encrypted or not
     * @param {String}               json.action
     * @param {String}               json.url
     * @param {String}               json.token
     * @param {String}               yandexuid
     * @param {String}               secret
     *
     * @returns {Object|null}  returns log object or null if failed
     */
    tryToGetLog: function(json, secret) {
        var log;
        var isLoggable;
        var encrypted = utils.checkIfEncrypted(json);

        try {
            if (encrypted) {
                json = putils.simpleCipher.decode(secret, json);
            }
            log = JSON.parse(json);
        } catch (e) {
            return null;
        }

        isLoggable =
            log &&
            (!log.withTrackId || log.track_id) &&
            utils.notAnEmptyString(log.url) &&
            utils.notAnEmptyString(log.action);
        if (isLoggable) {
            return log;
        } else {
            return null;
        }
    },
    /**
     * Check if data was encrypted
     *
     * @param {String} data
     * @returns {boolean}
     */
    checkIfEncrypted: function(data) {
        return Boolean(data && data.indexOf('{') === -1);
    },
    /**
     * Trim url slashes and add one at start
     *
     * @param {String} fullUrl
     * @returns {String}
     */
    normalizeUrl: function(fullUrl) {
        var pathname = url.parse(fullUrl).pathname;

        if (typeof pathname === 'string') {
            return `/${pathname.replace(trimSlashesRegexp, '')}`;
        }
        return '';
    },
    /**
     * Convert camel case to snake case
     *
     * @param {String} action
     * @return {String}
     */
    normalizeAction: function(action) {
        return action.replace(spacesRegexp, '_').toLowerCase();
    },
    /**
     * Check whether string is empty or not
     *
     * @param {String} str
     * @returns {Boolean}
     */
    notAnEmptyString: function(str) {
        return typeof str === 'string' && str;
    },
    /**
     * Check whether token looks as expected
     *
     * @param {String} str
     * @returns {Boolean}
     */
    looksLikeToken: function(str) {
        return /^[0-9a-fA-F]{32}$/.test(str);
    }
};

/**
 * Logs frontend events
 *
 * @param {Object}      req - http req object, required to write to passport-api
 * @returns {boolean}   Whether anything had been logged
 */
module.exports = function(req) {
    var data = req.body.log;
    var trackId = req.body.track_id;
    var yandexuid = req.cookies.yandexuid;
    var log = utils.tryToGetLog(data, trackId);

    if (log) {
        var normalizedUrl = utils.normalizeUrl(log.url) || 'unknown';
        var normalizedAction = utils.normalizeAction(log.action);
        var commonInfo = {
            ignoreMissedTrack: true,
            ip: req.headers['x-real-ip'],
            user_agent: req.headers['user-agent'],
            host: req.hostname,
            yandexuid: yandexuid,
            url: normalizedUrl,
            action: normalizedAction
        };
        var info = Object.assign({}, log, commonInfo);

        PLog.debug()
            .logId(req.logID)
            .type('loggerroute')
            .write('CLIENTSIDE %s from url %s with track %s', log.action, normalizedUrl, log.track_id || 'unknown');
        delete info.withTrackId;
        delete info.token;
        api.client(req).then(function(api) {
            api.statboxLogger(info);

            if (log.action === 'fingerprint' && typeof trackId === 'string') {
                api.writeTrack({
                    track_id: trackId,
                    js_fingerprint: info.greedResult
                });
            }
        });

        return true;
    }

    return false;
};

module.exports.token = token;
module.exports.utils = utils;
module.exports.getToken = token.create;
module.exports.checkToken = token.check;
module.exports.normalizeUrl = utils.normalizeUrl;
module.exports.normalizeAction = utils.normalizeAction;
