const PLog = require('plog');
const express = require('express');
const PController = require('../lib/controller');
const config = require('../configs/current');
const langs = config.langs;
const PApi = require('../lib/passport-api');

exports.route = function(app) {
    const router = express.Router();

    app.use('/passport', router);

    router.get('/', [chooseWay, doRequest, handleError, renderError]);

    router.post('/', [chooseWay, doRequest, handleError, renderError]);
};

function doRequest(req, res, next) {
    const query = req.query;
    const yu = /^[0-9]*$/.test(query.yu) && query.yu;
    const ci = typeof query.ci === 'string' && query.ci;
    const isGlobal = query.global === '1';
    const origin = query.origin;
    const retpath = query.retpath;
    const successHandler = exports.getSuccessHandler(req, res, next);
    const errorHandler = exports.getErrorHandler(req.logID, next);
    const doAPI = exports.doAPILogout;

    let token = yu;

    let sample = req.cookies && req.cookies.yandexuid;

    if (ci) {
        token = ci;
        sample = req.blackbox && req.blackbox.connection_id;
    }

    if (!token || token !== sample) {
        return next(['csrf_token.invalid']);
    }

    const logoutResult = doAPI(req, {
        yu,
        ci,
        is_global: isGlobal,
        origin,
        retpath
    });

    return logoutResult.then(successHandler).catch(errorHandler);
}

function doAPILogout(req, opts) {
    return PApi.client(req).then((api) => api.logout(opts));
}

function getController(req, res) {
    if (!req._controller) {
        req._controller = new PController(req, res, req.logID);
    }

    return req._controller;
}

function getSuccessHandler(req, res, next) {
    const controller = exports.getController(req, res);

    return function(response) {
        const body = response && response.body;

        if (!body) {
            return next(new Error('Unexpected response from API'));
        }

        controller.augmentResponse(body);

        return controller.redirectToFinish(body.track_id);
    };
}

function getErrorHandler(logID, next) {
    return function(err) {
        PLog.warn()
            .logId(logID)
            .type('logout')
            .write(err);

        return next(err);
    };
}

function chooseWay(req, res, next) {
    const mode = req.query.mode;

    if (mode === 'logout') {
        return next();
    }

    return next('route');
}

function handleError(err, req, res, next) {
    // eslint-disable-line
    const controller = exports.getController(req, res);

    if (err instanceof Array) {
        if (err.includes('csrf_token.invalid')) {
            controller.writeStatbox({
                action: 'csrf',
                mode: 'logout',
                url: req.originalUrl,
                yandexuid: res._yandexuid
            });
            return controller.redirectToFrontpage();
        }

        if (err.includes('sessionid.invalid')) {
            return controller.redirectToLocalUrl({
                pathname: 'auth' // not redirectToAuth(), because we don't need retpath to logout after auth
            });
        }

        if (err.includes('sessionid.no_uid')) {
            return controller.redirectToLocalUrl({
                pathname: 'auth' // not redirectToAuth(), because we don't need retpath to logout after auth
            });
        }
    }

    return next(err);
}

// eslint-disable-next-line no-unused-vars
function renderError(err, req, res, next) {
    const controller = exports.getController(req, res);
    const context = {
        retpath: '/profile',
        statusCode: 400
    };

    if (err instanceof Array) {
        context.msg = err[0];
    }

    const render = exports.getRender(res, context);

    return controller
        .getLanguage()
        .then((lang) => render(lang))
        .catch(() => render());
}

function getRender(res, context) {
    return function(lang) {
        const language = (lang && langs[langs.indexOf(lang)]) || 'ru';

        return res.status(context.statusCode).render(`error.${language}.js`, context);
    };
}

exports.doAPILogout = doAPILogout;
exports.getController = getController;
exports.doRequest = doRequest;
exports.getSuccessHandler = getSuccessHandler;
exports.getErrorHandler = getErrorHandler;
exports.chooseWay = chooseWay;
exports.handleError = handleError;
exports.renderError = renderError;
exports.getRender = getRender;
