/**
 * Creates a route, that writes metrics_id to locals according to the given config
 *
 * @param {Object} metricsConfig @see module.exports.Metrics
 */
module.exports = function(metricsConfig) {
    var metrics = new module.exports.Metrics(metricsConfig);

    return metrics.getRoute();
};

/**
 * Config containing urls and their matching metricsIDs
 *
 * @param {Object} config Keys stand for pathnames, values stand for metrics counter IDs
 * @example {
 *  '/registration': '123',
 *  '/registration/mail': '345'
 * }
 *
 * @constructor
 */
module.exports.Metrics = function(config) {
    this.config = config || {};
};

module.exports.Metrics.prototype = {
    /**
     * Gets the counter id for the url according to config
     *
     * If there is no id in the config for this exact url
     * it looks for the id for parent url, so that /a/b/c
     * would receive an id from /a/b or /a, whichever comes first
     * or null if no suitable parent found in config
     *
     * @param {String} url  Page url to get counterID for
     * @returns {String|Null}
     */
    getCounterID: function(url) {
        if (!url) {
            if (this.config['/']) {
                return this.config['/'];
            }

            return null;
        }

        url = require('./logger').normalizeUrl(url);

        var counterID = this.config[url];

        if (counterID) {
            return counterID;
        }

        url = url.split('/');
        url.pop();

        return this.getCounterID(url.join('/'));
    },

    /**
     * Gets the Express route that sets metrics_id to locals
     * @returns {Function}
     */
    getRoute: function() {
        var metrics = this;

        return function(req, res, next) {
            res.locals.metrics_id = metrics.getCounterID(req.url);
            next();
        };
    }
};
