const url = require('url');
const when = require('when');
const express = require('express');
const _ = require('lodash');
const PLog = require('plog');
const apiSetup = require('./common/apiSetup');
const langSetup = require('./common/langSetup');
const getMetrics = require('./common/getMetrics');
const rumCounterSetup = require('./common/rumCounterSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const config = require('../configs/current');
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const modeErrorPage = express.Router();

const routerHandlers = [
    function(req, res, next) {
        const mode = req.query.mode;

        if (mode === 'error') {
            return next();
        }

        return next('route');
    },
    langSetup,
    apiSetup,
    getYaExperimentsFlags,
    function getStore(req, res, next) {
        const langSwitcher = new LangSwitcherView(req._controller);
        const locals = res.locals;
        const tld = req._controller.getTld();
        const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {userType = {}} = locals;
        const {metricsUserType} = userType;

        locals.store = {
            settings: {
                host: req.hostname,
                embeddedauth: config.paths.embeddedauth || '',
                links:
                    process.env.INTRANET === 'intranet'
                        ? config.links.intranet
                        : config.links[tld] || config.links.ru || {},
                help: config.paths.help || {},
                tld,
                language: locals.language,
                ua: {},
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType,
                staticPath
            },
            common: {
                yandexuid: res._yandexuid && res._yandexuid.replace(/[^0-9]/g, ''),
                track_id: locals.track_id,
                currentPage: req._controller.getUrl().href,
                experiments: res.locals.experiments || {
                    flags: [],
                    flagsString: '',
                    boxes: '',
                    encodedBoxes: ''
                },
                defaultPage: url.format({
                    pathname: '/profile',
                    query: req.query
                })
            },
            error: {
                code: setErrorCode(req.query.error),
                errorEncoded: encodeErrorInfo(req.query, req.logID),
                badretpath: handleBadretpathParam(req.query.badretpath)
            },
            footer: {
                langlist: []
            },
            monitoring: {
                page: 'mode=error'
            }
        };

        when.all(handlers)
            .then(function(response) {
                const uatraits = response[0] || {};
                const langlist = (response[1] && response[1].langlist) || {};

                locals.store.settings.ua = uatraits;
                locals.store.footer.langlist = langlist;
                return next();
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('mode_error')
                    .write(err);
            });
    },
    getMetrics({header: 'Страница mode=error'}),
    rumCounterSetup,
    function renderPage(req, res) {
        const lang = res.locals.language;

        res.render(`react.mode-error.${lang}.jsx`);
    }
];

modeErrorPage.get('/', routerHandlers);

exports.router = modeErrorPage;
exports.routerHandlers = routerHandlers;

function encodeErrorInfo(query = {}, logID) {
    const {trace, mode, error} = query;
    const strToEncode = trace ? [trace, Date.now()].join('.') : [logID, mode, error, Date.now()].join('.');
    const bt = Buffer.from(strToEncode);
    const btEncoded = bt.toString('base64');

    return [...btEncoded].filter((symbol) => symbol !== '=').join('');
}

function setErrorCode(queryError) {
    const errorCodes = ['internal', 'nojavascript', 'nocki', 'badretpath'];

    return errorCodes.find((item) => item === queryError) || 'internal';
}

function handleBadretpathParam(badretpath) {
    if (badretpath && (badretpath.startsWith('https://') || badretpath.startsWith('http://'))) {
        return _.escape(badretpath);
    }

    return undefined;
}
