var _ = require('lodash');
var apiSetup = require('./common/apiSetup');
const validateCSRF = require('./common/validateCSRF.js');
const checkAuth = require('./common/checkAuth');
var qr = require('qr-image');
var url = require('url');
var config = require('../configs/current');
var langs = config.langs;
var locs = require('../loc/auth.json');
const {otpEnableCheckOtp, otpEnableCommit, otpEnableGetSecret, otpEnableSubmit} = require('./authOtp');
var when = require('when');

var getControls = (function() {
    var controls = {};

    return function(lang) {
        lang = langs[langs.indexOf(lang)] || 'ru';

        if (lang in controls) {
            return _.clone(controls[lang], true);
        }

        controls[lang] = require(`../lib/passport-form/template.${lang}.js`);
        return getControls(lang);
    };
})();

exports.routes = {};
exports.route = function(app) {
    var routes = this.routes;

    app.post('/auth/otp/enable/check-otp', [apiSetup, checkAuth, validateCSRF, otpEnableCheckOtp]);
    app.post('/auth/otp/enable/commit', [apiSetup, checkAuth, validateCSRF, otpEnableCommit]);
    app.post('/auth/otp/enable/submit', [apiSetup, checkAuth, validateCSRF, otpEnableSubmit]);
    app.post('/auth/otp/enable/get-secret', [apiSetup, checkAuth, validateCSRF, otpEnableGetSecret]);

    app.get('/account/short/?', otpRoute('account'));
    app.post('/auth/otp/magic/?', otpRoute('prepare'));

    app.get('/auth/magic/code/?', routes.magicCode);
    app.get('/auth/magic/code.svg', routes.magicCode);

    app.get(
        '/auth/magic/code.png',
        function(req, res, next) {
            req.pngFallback = true;
            next();
        },
        routes.magicCode
    );

    app.post('/auth/magic/status/?', routes.magicStatus);
    app.post(
        '/auth/verify/magic/status/?',
        function(req, res, next) {
            req.handle = '/1/bundle/auth/password/confirm/commit_magic/';
            next();
        },
        routes.verifyMagicStatus
    );

    app.post(
        '/auth/new/magic/status/?',
        function(req, res, next) {
            req.handle = '/2/bundle/auth/password/commit_magic/';
            next();
        },
        routes.oneDomikMagicStatus
    );

    app.get(
        '/auth/?',
        function(req, res, next) {
            if (req.nquery && req.nquery.mode === 'qr') {
                return next();
            } else {
                return next('route');
            }
        },
        routes.magic
    );
};

function otpRoute(action) {
    return [apiSetup, otpProxy(action), errorHandler];
}

function otpProxy(action) {
    var paths = {
        account: '/1/bundle/account/otp/short_info/',
        prepare: '/1/bundle/auth/otp/prepare/'
    };

    var url = action && paths[action];

    if (!url) {
        return function(req, res, next) {
            next();
        };
    }

    return function(req, res, next) {
        var data = _.extend({}, req.nquery, req.body);

        if (action === 'account') {
            data.asString = true;
        }

        req.api.authSubmit(url, data).then(
            function(result) {
                if (action === 'account') {
                    res.set('Content-type', 'application/json; charset=utf-8');
                }

                return res.send(result.body);
            },
            function(errors) {
                return next(errors);
            }
        );
    };
}

// eslint-disable-next-line no-unused-vars
function errorHandler(err, req, res, next) {
    var result = {
        status: 'error'
    };

    if (err instanceof Error) {
        result.errors = ['internal.permanent'];
    } else {
        result.errors = err;
    }

    return res.send(result);
}

exports.routes.magic = [
    apiSetup,
    function(req, res, next) {
        new (require('../lib/controller'))(req, res, req.logID)
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                require('plog')
                    .warn()
                    .logId(req.logID)
                    .type('otp magic')
                    .write(err);
            })
            .done(function() {
                if (req.api) {
                    req.api.language(res.locals.language);
                }

                return next();
            });
    },
    function(req, res, next) {
        var query = req.query;
        var data = {};
        var twoweeks;

        if (query) {
            data = _.pick(query, ['retpath', 'service', 'origin', 'fretpath', 'clean', 'from']);
            twoweeks = query.twoweeks;

            if (data.from) {
                data['service'] = data.from;
                delete data.from;
            }

            if (twoweeks === 'no') {
                data.policy = 'sessional';
            } else {
                data.policy = 'long';
            }

            if (query.reason && Date.now() - query.reason <= 10000) {
                res.locals.from_default_auth = true;
            }
        }

        req.api.authSubmit('/2/bundle/auth/otp/submit/', data).then(
            function(results) {
                res.locals.track_id = results.body.track_id;
                res.locals.csrf_token = results.body.csrf_token;
                next();
            },
            function(error) {
                next(error);
            }
        );
    },
    function(req, res, next) {
        const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
        const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;

        var log = {
            ignoreMissedTrack: true,
            action: 'opened',
            mode: 'qr',
            track_id: res.locals.track_id,
            url: req.url,
            user_agent: req.headers['user-agent'],
            yandexuid: res.yandexuid || req.cookies.yandexuid,
            ip: req.headers['x-real-ip'],
            from: (req.nquery && req.nquery.from) || null,
            origin: (req.nquery && req.nquery.origin) || null,
            referer: req.headers.referer || null,
            retpath: (req.nquery && req.nquery.retpath) || null,
            host: req.hostname,
            experiment_flags: experimentFlags,
            experiment_boxes: experimentBoxes
        };

        req.api.statboxLogger(log);
        next();
    },
    function(req, res) {
        var lang = res.locals.language;
        var loc = locs[lang || 'ru'];
        var query = url.parse(req.originalUrl, true).query;
        // Автозаполнение логина в домике из GET-параметра

        if (req.nquery && req.nquery['login'] && !(res.locals.fields && res.locals.fields.login)) {
            res.locals.fields = {
                login: req.nquery['login']
            };
        }

        if (res.locals.from_default_auth) {
            _.assign(res.locals, {
                errors: [
                    {
                        code: 'account.2fa_enabled',
                        msg: loc['Errors']['ErrorsTexts.badlog_2fa'],
                        mend: loc['Mend']['creg_misspasswd_2fa']
                    }
                ]
            });
        }

        query['mode'] = 'add-user';

        res.locals.add_user_link = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: 'auth',
            query
        });

        res.render(`auth.magic.${lang}.js`);
    },
    errorHandler
];

exports.routes.magicCode = [
    apiSetup,
    function(req, res) {
        const nodeEnv = process.env.NODE_ENV;
        const MAGIC_CONST = '888AA27A5584C396DD6933116DF2EAC2';
        var tld = /.*.yandex\.(.*)$/.exec(req.hostname);
        var domain = (tld && tld[1]) || 'ru';
        const domainMap = {
            production: `passport.yandex.${domain}`,
            rc: `passport-rc.yandex.${domain}`,
            testing: `passport-test.yandex.${domain}`
        };
        var string = require('url').format({
            protocol: 'https',
            hostname: domainMap[nodeEnv] || `passport.yandex.${domain}`,
            pathname: 'am/push/qrsecure',
            query: {
                track_id: req.api.track(),
                magic: MAGIC_CONST
            }
        });

        var qrCodeStream = null;

        if (req.pngFallback) {
            qrCodeStream = qr.image(string, {type: 'png', margin: 2, size: 5});
            res.set('Content-type', 'image/png');
        } else {
            qrCodeStream = qr.image(string, {type: 'svg', margin: 2, size: 5});
            // Opera 12.16 не дает ставить фоном, если DENY
            res.setHeader('Content-Type', 'image/svg+xml');
            res.setHeader('X-Frame-Options', 'SAMEORIGIN');
        }

        qrCodeStream.pipe(res);
    },
    errorHandler
];

exports.routes.magicStatus = [
    apiSetup,
    function(req, res, next) {
        var csrf_token = req.body['csrf_token'];

        if (!csrf_token) {
            return res.sendStatus(403);
        }

        return next();
    },
    function(req, res, next) {
        var data = _.extend({passErrors: true}, req.nquery, req.body);
        const controller = req._controller;

        req.api.authSubmit(req.handle || '/2/bundle/auth/otp/commit/', data).then(
            function(results) {
                var body = (res.body = results.body);
                var errors = body.errors;

                controller.augmentResponse(body);
                if (body.cookies) {
                    delete body.cookies;
                }

                if (!errors && body.state === 'otp_auth_not_ready') {
                    body = {};
                }

                if (errors && errors.indexOf('captcha.required') !== -1) {
                    return next();
                }

                if (errors && errors.indexOf('csrf_token.invalid') !== -1) {
                    return res.sendStatus(403);
                }

                if (errors) {
                    body = {
                        errors
                    };
                }

                res.send(body);
            },
            function(error) {
                next(error);
            }
        );
    },
    function(req, res, next) {
        new (require('../lib/controller'))(req, res, req.logID)
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                require('plog')
                    .warn()
                    .logId(req.logID)
                    .type('otp magic status')
                    .write(err);
            })
            .done(function() {
                if (req.api) {
                    req.api.language(res.locals.language);
                }

                return next();
            });
    },
    function setCaptcha(req, res, next) {
        var body = {
            errors: res.body.errors
        };
        var captchaOpts = {
            asyncCheck: false
        };

        var captchaField = _.find(getControls(res.locals.language), function(cntrl) {
            return cntrl.id === 'captcha';
        });

        if (captchaField) {
            var enableAudioCaptcha = true;

            captchaField.countryFromAudioWhiteList = enableAudioCaptcha;
            captchaField.options = captchaOpts;
        }

        var getCaptcha = req.api.captchaGenerate;

        if (req.body.captcha_mode === 'audio') {
            getCaptcha = req.api.audioCaptchaGenerate;
        }
        when.reduce([getCaptcha.call(req.api, captchaOpts)], reduceAnswer, {}).then(
            function(results) {
                var captcha = _.extend({}, captchaField, results.captcha);

                delete captcha.status;
                captcha.id = 'captcha';
                captcha.static = config.paths.static;
                body.captcha = captcha;
                res.send(body);
            },
            function(error) {
                next(error);
            }
        );
    },
    errorHandler
];

exports.routes.verifyMagicStatus = [
    apiSetup,
    function(req, res, next) {
        var csrf_token = req.body['csrf_token'];

        if (!csrf_token) {
            return res.sendStatus(403);
        }

        return next();
    },
    function(req, res, next) {
        const data = _.extend({passErrors: true}, req.nquery, req.body);
        const controller = req._controller;

        req.api.authSubmit('/1/bundle/auth/password/confirm/commit_magic/', data).then(
            function(results) {
                var body = (res.body = results.body);
                var errors = body.errors;

                if (body.status === 'ok' && body.state !== 'otp_auth_not_ready') {
                    req.api
                        .bundleSession({
                            track_id: results.track_id,
                            retpath: data.retpath
                        })
                        .done(function(result) {
                            var body = result.body;

                            if (body.state) {
                                return res.redirect(
                                    url.format({
                                        protocol: req.headers['x-real-scheme'],
                                        hostname: req.hostname,
                                        pathname: 'auth',
                                        query: {
                                            track_id: body.track_id,
                                            retpath: body.retpath
                                        }
                                    })
                                );
                            }

                            controller.augmentResponse(body);

                            body.state = 'otp_auth_finished';
                            res.send(body);
                        });
                } else {
                    if (!errors && body.state === 'otp_auth_not_ready') {
                        body = {};
                    }

                    if (errors && errors.indexOf('captcha.required') !== -1) {
                        return next();
                    }

                    if (errors && errors.indexOf('csrf_token.invalid') !== -1) {
                        return res.sendStatus(403);
                    }

                    if (errors && errors.indexOf('password.not_matched') !== -1) {
                        errors = ['password.not_matched_2fa'];
                    }

                    if (errors) {
                        body = {
                            errors
                        };
                    }

                    res.send(body);
                }
            },
            function(error) {
                next(error);
            }
        );
    },
    function(req, res, next) {
        new (require('../lib/controller'))(req, res, req.logID)
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                require('plog')
                    .warn()
                    .logId(req.logID)
                    .type('otp magic status')
                    .write(err);
            })
            .done(function() {
                if (req.api) {
                    req.api.language(res.locals.language);
                }

                return next();
            });
    },
    function setCaptcha(req, res, next) {
        var body = {
            errors: res.body.errors
        };
        var captchaOpts = {
            asyncCheck: false
        };

        var captchaField = _.find(getControls(res.locals.language), function(cntrl) {
            return cntrl.id === 'captcha';
        });

        if (captchaField) {
            var enableAudioCaptcha = true;

            captchaField.countryFromAudioWhiteList = enableAudioCaptcha;
            captchaField.options = captchaOpts;
        }

        var getCaptcha = req.api.captchaGenerate;

        if (req.body.captcha_mode === 'audio') {
            getCaptcha = req.api.audioCaptchaGenerate;
        }
        when.reduce([getCaptcha.call(req.api, captchaOpts)], reduceAnswer, {}).then(
            function(results) {
                var captcha = _.extend({}, captchaField, results.captcha);

                delete captcha.status;
                captcha.id = 'captcha';
                captcha.static = config.paths.static;
                body.captcha = captcha;
                res.send(body);
            },
            function(error) {
                next(error);
            }
        );
    },
    errorHandler
];

exports.routes.oneDomikMagicStatus = [
    apiSetup,
    function(req, res, next) {
        var csrf_token = req.body['csrf_token'];

        if (!csrf_token) {
            return res.sendStatus(403);
        }

        return next();
    },
    function(req, res, next) {
        var data = _.extend({passErrors: true}, req.nquery, req.body);
        const controller = req._controller;

        req.api
            .authSubmit(req.handle, data)
            .then(function(results) {
                var body = (res.body = results.body);
                var errors = body.errors;

                if (body.status === 'ok' && body.state !== 'otp_auth_not_ready' && body.state !== 'auth_challenge') {
                    req.api
                        .bundleSession({
                            track_id: results.track_id,
                            retpath: data.retpath
                        })
                        .then(function(result) {
                            var body = result.body;

                            if (body.state) {
                                return res.redirect(
                                    url.format({
                                        protocol: req.headers['x-real-scheme'],
                                        hostname: req.hostname,
                                        pathname: 'auth',
                                        query: {
                                            track_id: body.track_id,
                                            retpath: body.retpath
                                        }
                                    })
                                );
                            }

                            controller.augmentResponse(body);

                            body.state = 'otp_auth_finished';
                            delete body.cookies;

                            res.send(body);
                        })
                        .catch((error) => next(error));
                } else {
                    if (!errors && body.state === 'otp_auth_not_ready') {
                        body = {};
                    }

                    if (!errors && body.state === 'auth_challenge') {
                        body = {
                            state: 'auth_challenge',
                            redirectUrl: url.format({
                                protocol: req.headers['x-real-scheme'],
                                hostname: req.hostname,
                                pathname: '/auth/challenge',
                                query: {
                                    track_id: req.body.track_id
                                }
                            })
                        };
                    }

                    if (errors && errors.indexOf('captcha.required') !== -1) {
                        return next();
                    }

                    if (errors && errors.indexOf('csrf_token.invalid') !== -1) {
                        return res.sendStatus(403);
                    }

                    if (errors) {
                        body = {
                            errors
                        };
                    }

                    res.send(body);
                }
            })
            .catch((error) => next(error));
    },
    function(req, res, next) {
        new (require('../lib/controller'))(req, res, req.logID)
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(err) {
                res.locals.language = 'ru';

                require('plog')
                    .warn()
                    .logId(req.logID)
                    .type('otp magic status')
                    .write(err);
            })
            .done(function() {
                if (req.api) {
                    req.api.language(res.locals.language);
                }

                return next();
            });
    },
    function setCaptcha(req, res, next) {
        var body = {
            errors: res.body.errors
        };
        var captchaOpts = {
            asyncCheck: false
        };

        var captchaField = _.find(getControls(res.locals.language), function(cntrl) {
            return cntrl.id === 'captcha';
        });

        if (captchaField) {
            var enableAudioCaptcha = true;

            captchaField.countryFromAudioWhiteList = enableAudioCaptcha;
            captchaField.options = captchaOpts;
        }

        var getCaptcha = req.api.captchaGenerate;

        if (req.body.captcha_mode === 'audio') {
            getCaptcha = req.api.audioCaptchaGenerate;
        }
        when.reduce([getCaptcha.call(req.api, captchaOpts)], reduceAnswer, {}).then(
            function(results) {
                var captcha = _.extend({}, captchaField, results.captcha);

                delete captcha.status;
                captcha.id = 'captcha';
                captcha.static = config.paths.static;
                body.captcha = captcha;
                res.send(body);
            },
            function(error) {
                next(error);
            }
        );
    },
    errorHandler
];

function reduceAnswer(currentValue, nextItem) {
    currentValue[nextItem.field] = nextItem.body;

    return currentValue;
}
