var fs = require('fs');
var express = require('express');
var path = require('path');
var when = require('when');
var apiSetup = require('./common/apiSetup');
var setCORSHeaders = require('./common/setCORSHeaders');
var yasmsStateParser = require('./common/yasmsStateParser');
var router = express.Router();
var translations = {};
var locPath = './loc/';
var loaderPath = path.join(__dirname, '../node_modules/phone-actualization-client/build/loader.js');

fs.readdir(locPath, function(err, files) {
    files.forEach(function(file) {
        var match = file.match(/Plashka\.(\w\w)\.js/);

        if (match) {
            translations[match[1]] = require(`../loc/${file}`)[match[1]];
        }
    });
});

function getLanguage(req, res, next) {
    req._controller(req, res, req.logID)
        .getLanguage()
        .then(function(lang) {
            res.locals.lang = lang;
        })
        .catch(function() {
            res.locals.lang = 'ru';
        })
        .done(function() {
            next();
        });
}

function initTrack(req, res, next) {
    req.api
        .initTrack()
        .then(function(response) {
            res.locals.trackId = response.body.track_id || '';
            next();
        })
        .catch(function(errors) {
            res.json({
                status: 'error',
                errors: errors
            });
        });
}

function checkCodePassword(req, res, next) {
    var body = req.body;
    var passwordPromise = req.api.yasmsCheckPassword({
        operationId: body.operation_id,
        password: body.password
    });
    var codePromise = req.api.yasmsCheckCode({
        operationId: body.operation_id,
        code: body.code
    });

    when.all([passwordPromise, codePromise])
        .then(function(results) {
            var errorJSON = {
                status: 'error',
                errors: []
            };

            results.forEach(function(result) {
                if (result.status === 'error') {
                    errorJSON.errors = errorJSON.errors.concat(result.errors);
                }
            });
            if (errorJSON.errors.length !== 0) {
                res.json(errorJSON);
            } else {
                next();
            }
        })
        .catch(function(err) {
            res.json(err);
        });
}

function getPlashkaState(req, res) {
    req.api
        .yasmsGetState({filter: false})
        .then(function(response) {
            var json = yasmsStateParser.parse(response.account);

            json.simple.forEach(function(phone) {
                phone.number = phone.number.e164;
            });

            if (res.locals && res.locals.plashkaInit) {
                json.trackId = res.locals.trackId;
                json.lang = res.locals.lang;
                json.translations = translations[json.lang];
            }

            if (!json.secure && json.simple.length === 0) {
                json.type = 'noPhone';
            }

            if (json.simple.length !== 0 && !json.secure) {
                json.type = 'confirmPhone';
            }

            if (json.secure) {
                json.secure.number = json.secure.number.masked_e164;

                if (json.secure.needProlong) {
                    json.type = 'needProlong';
                }

                if (json.secure.operation) {
                    json.secure.operation.phone = json.secure.operation.phone.masked_e164;
                    json.type = 'smsSent';
                }
                if (!json.secure.needProlong && !json.secure.operation) {
                    json.type = 'complete';
                }
            }

            res.json(json);
        })
        .catch(function() {
            res.json({
                status: 'error'
            });
        });
}

router.use(setCORSHeaders);

router.get('/', [
    apiSetup,
    initTrack,
    getLanguage,
    function(req, res, next) {
        res.locals.plashkaInit = true;
        next();
    },
    getPlashkaState
]);

router.get('/loader', function(req, res) {
    res.sendFile(loaderPath, function(err) {
        if (err) {
            res.status(err.status).end();
        }
    });
});

router.post('/secure.bind.submit', [
    apiSetup,
    function(req, res, next) {
        req.api
            .yasmsSecurePhoneSubmit(req.body)
            .then(function(response) {
                if (response.status === 'error') {
                    res.json({
                        status: response.status,
                        errors: response.errors
                    });
                } else {
                    next();
                }
            })
            .catch(function(err) {
                res.json(err);
            });
    },
    getPlashkaState
]);

router.post('/secure.bind.commit', [
    apiSetup,
    checkCodePassword,
    function(req, res) {
        req.api
            .yasmsSecurePhoneCommit({id: req.body.operation_id})
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                res.json(err);
            });
    }
]);

router.post('/simple.securify.submit', [
    apiSetup,
    function(req, res, next) {
        req.api
            .yasmsSimplePhoneSecurifySubmit(req.body)
            .then(function(response) {
                if (response.status === 'error') {
                    res.json({
                        status: response.status,
                        errors: response.errors
                    });
                } else {
                    next();
                }
            })
            .catch(function(err) {
                res.json(err);
            });
    },
    getPlashkaState
]);

router.post('/simple.securify.commit', [
    apiSetup,
    checkCodePassword,
    function(req, res) {
        req.api
            .yasmsSimplePhoneSecurifyCommit({id: req.body.operation_id})
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                res.json(err);
            });
    }
]);

router.post('/code.resend', [
    apiSetup,
    function(req, res) {
        req.api
            .yasmsResendCode({id: req.body.operation_id})
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                res.json(err);
            });
    }
]);

router.post('/operation.cancel', [
    apiSetup,
    function(req, res, next) {
        req.api
            .yasmsCancelOperation({id: req.body.operation_id})
            .then(function(response) {
                if (response.status === 'ok') {
                    next();
                } else {
                    res.json(response);
                }
            })
            .catch(function(err) {
                res.json(err);
            });
    },
    getPlashkaState
]);

router.post('/prolong', [
    apiSetup,
    function(req, res) {
        req.api
            .yasmsProlong(req.body)
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                res.json(err);
            });
    }
]);

module.exports = router;
