var assert = require('assert');
var when = require('when');
var _ = require('lodash');
var url = require('url');
var inherit = require('inherit');
var PView = require('pview');
var PLog = require('plog');
var tokensView = require('../blocks/access-tokens/tokensView');
var API = require('../lib/passport-api');
var PassportApi = require('../lib/api/passport');
var OauthApi = require('../lib/api/oauth');
var NonceView = require('../blocks/nonce/NonceView');
var Layout = require('inherit')(require('../blocks/layout/RenderingLayoutView'), {
    template: 'profile.access.2fa.disable.%lang%.js'
});

var StepSelector = inherit(PView, {
    name: 'otpEnableStep',
    __constructor: function(panel) {
        this.__base.apply(this, arguments);
        this._step = panel;
    },

    _compile: function() {
        return {step: this._step};
    }
});

var LoginView = inherit(PView, {
    name: 'login',
    __constructor: function(login) {
        this.__base.apply(this, arguments);
        this._login = login;
    },

    _compile: function() {
        return {login_val: this._login};
    }
});

var TrackView = inherit(PView, {
    name: 'track',
    __constructor: function(api) {
        assert(api instanceof API);
        this._api = api;

        this.__base.apply(this, arguments);
    },

    _compile: function() {
        return this._api.params('track_id').then(function(track) {
            return {track_id: {id: track.body.id}};
        });
    }
});

var UidView = inherit(PView, {
    name: 'user-uid',
    __constructor: function(uid) {
        this.__base.apply(this, arguments);
        this._uid = uid;
    },

    _compile: function() {
        return {
            uid: this._uid
        };
    }
});

/* jshint unused:false */
var HistoryView = inherit(PView, {
    name: 'access.history',
    __constructor: function(papi, controller) {
        assert(papi instanceof PassportApi, 'Api should be a Passport API');

        this.__base.apply(this, arguments);

        this._api = papi;
        this._controller = controller;
    },

    _compile: function() {
        var dateToChunks = this._controller.dateToChunks;

        if (this._controller.isIntranet()) {
            // PASSP-11438: 500ка при походе в ручку /lastauth
            return {
                history: {
                    tokens: []
                }
            };
        }

        return this._api.getLastAuth().then(function(result) {
            return {
                historyTokens: result.tokens.concat(result.appPass).map(function(entry) {
                    return {
                        id: entry.getTokenId(),
                        lastUsage: dateToChunks(entry.getTimestamp())
                    };
                })
            };
        });
    }
});

var RevokersView = inherit(PView, {
    name: 'revokers',
    __constructor: function(revokersData) {
        this.__base.apply(this, arguments);
        this._revokers = revokersData;
    },

    _compile: function() {
        return {
            revokers: this._revokers
        };
    }
});

var InvalidInputView = inherit(PView, {
    name: 'invalidInput',
    __constructor: function(code) {
        this.__base.apply(this, arguments);
        this._code = code || undefined;
    },
    _compile: function() {
        return {
            invalidInput: true,
            invalidInputCode: this._code
        };
    }
});

var CaptchaField = inherit(PView, {
    name: 'captcha',
    __constructor: function(compiled) {
        this.__base.apply(this, arguments);
        assert(compiled && typeof compiled === 'object', 'compiled should be an object');
        this._compiled = compiled;
    },

    _compile: function() {
        return {
            captcha_is_required: 'required',
            captcha: this._compiled
        };
    }
});

var TwoFADisablePage = inherit({
    name: null, // Should be overwritten
    needsTrack: true,
    __constructor: function(controller, api, papi) {
        assert(api instanceof API);
        assert(this.name && typeof this.name === 'string', 'Page name should be defined for logging');

        var that = this;

        this._controller = controller;
        this._api = api;
        this._papi = papi;
        this._logger = new PLog(controller.getLogId(), 'profile', 'access', '2fa', 'disable', 'page', this.name);

        this._controller.getLanguage().then(function(lang) {
            that._lang = lang;
        });
    },

    getLayout: function() {
        return new Layout(this._controller).append(new NonceView(this._controller)).append(new TrackView(this._api));
    },

    _open: function() {
        throw new Error('Method should be overwritten');
    },

    open: function() {
        var that = this;
        var auth = this._controller.getAuth();

        return auth
            .loggedIn()
            .then(function(isLoggedIn) {
                if (!isLoggedIn) {
                    that._logger.info('User not logged in');
                    auth.authorize();
                    return when.reject();
                }
                that._logger.info('User is logged in');
            })
            .then(function() {
                if (that.needsTrack) {
                    return auth
                        .sessionID({
                            attributes: '1003'
                        })
                        .then(function() {
                            var is2FAOn = Boolean(auth.getAttribute(1003));
                            var userId = that._controller.getRequestParam('user_id');

                            if (userId && userId !== auth.getUid()) {
                                that._logger.info('Default user has changed, process interrupted');
                                if (is2FAOn) {
                                    that._logger.info('User has 2FA enabled, redirecting to start disabling process');
                                    that._controller.redirect('/profile/access/2fa/disable');
                                } else {
                                    that._logger.info('2FA not enabled, redirecting to /profile');
                                    that._controller.redirect('/profile');
                                }
                            }
                        });
                }
            })
            .then(function() {
                that._logger.info('Opening %s', that.name);
                return that._open();
            })
            .then(function() {
                that._controller.writeStatbox(
                    {
                        // DO NOT RETURN THE PROMISE, page should not depend on the promise being resolved or rejected
                        action: 'opened',
                        mode: 'disable_otp',
                        step: that.name,
                        uid: auth.getUid()
                    },
                    false
                ); // Api should already know track by that point
            })
            .catch(function(err) {
                if (typeof err === 'undefined' || (err && err.code === 'need_resign')) {
                    return; // Do nothing if there is no error, promise was rejected to stop the process
                }

                if (Array.isArray(err) && err.length === 1) {
                    if (err[0] === 'action.not_required') {
                        that._logger.info('Action not required');

                        return that._controller.redirect(
                            url.format(
                                _.assign(that._controller.getUrl(), {
                                    pathname: '/profile'
                                })
                            )
                        );
                    }

                    if (err[0] === 'sslsession.required') {
                        that._logger.info('Needs SSL session');
                        return that._controller.getAuth().obtainSecureCookie();
                    }
                }

                throw err;
            });
    }
});

var EntryPage = inherit(TwoFADisablePage, {
    name: 'entry',
    needsTrack: false,
    _open: function() {
        if (this._controller.getMethod().isPost()) {
            this._logger.info('Post request');
            return this._post();
        } else {
            this._logger.info('Get request');
            return this._get();
        }
    },

    _post: function() {
        var that = this;

        return this._api
            .OTPDisableCheckOtp(this._controller.getFormData())
            .then(function(state) {
                that._logger.info('Otp has not been checked yet, but is provided in a current request', state);
                return that._nextStep();
            })
            .catch(function(err) {
                var layout = that.getLayout();
                var uid = that._controller.getFormData().uid;

                that._logger.info('OTP error');

                if (Array.isArray(err) && err.length === 1 && err[0] === 'password.not_matched') {
                    that._logger.info('OTP is wrong and error occurs');
                    layout.append(new InvalidInputView()).append(new UidView(uid));
                    return layout.append(new StepSelector('entry')).render();
                }
                if (Array.isArray(err) && err.length === 1 && err[0] === 'captcha.required') {
                    that._logger.info('Captcha required');
                    return that.createCaptcha();
                }
                if (
                    Array.isArray(err) &&
                    err.length === 2 &&
                    err[0] === 'captcha.required' &&
                    err[1] === 'password.not_matched'
                ) {
                    that._logger.info('Otp is wrong and captcha required');
                    layout.append(new InvalidInputView()).append(new UidView(uid));
                    return layout.append(new StepSelector('entry')).render();
                }
                throw err;
            });
    },

    _renderCaptcha: function(compiled) {
        var layout = this.getLayout();

        this._logger.info('Rendering page with captcha');

        if (compiled) {
            layout.append(new CaptchaField(compiled));
        }

        return layout.append(new StepSelector('entry')).render();
    },

    createCaptcha: function(invalidCaptcha) {
        var that = this;
        var api = this._api;
        var captcha = new (require('../blocks/control/captcha/captcha.field'))().setRequired();

        this._logger.info('Creating captcha');

        if (invalidCaptcha) {
            captcha.setErrorsActive(invalidCaptcha);
        }

        return this._controller.getLanguage().then(function(lang) {
            captcha
                .setMode()
                .compile(lang, api)
                .then(function(compiled) {
                    return that._renderCaptcha(compiled);
                });
        });
    },

    _get: function() {
        var that = this;

        return this._api.OTPDisableSubmit().then(function(result) {
            var uid = result.account.uid;
            var layout = that.getLayout();

            return layout
                .append(new UidView(uid))
                .append(new StepSelector('entry'))
                .render();
        });
    },

    _nextStep: function() {
        this._logger.info('Proceeding to a next step - password page');
        return this._controller.redirect(
            url.format(
                _.assign(this._controller.getUrl(), {
                    pathname: '/profile/access/2fa/disable/newpassword',
                    search: null,
                    query: {
                        track_id: this._controller.getFormData().track_id,
                        user_id: this._controller.getFormData().uid,
                        retpath: this._controller.getAuth().getRequestParam('retpath')
                    }
                })
            )
        );
    }
});

var setNewPassPage = inherit(TwoFADisablePage, {
    name: 'change-password',
    needsTrack: true,
    _open: function() {
        if (this._controller.getMethod().isPost()) {
            this._logger.info('Post request');
            this._post();
        } else {
            this._logger.info('Get request');
            this._get();
        }
    },
    _get: function() {
        var that = this;
        var controller = this._controller;
        var papi = this._papi;
        var revokersData = {};
        var login;

        return this._api
            .OTPDisableSubmit()
            .then(function(result) {
                var revokers = result.revokers;

                login = result.account.login;
                revokersData = {
                    allow_select: revokers.allow_select,
                    tokens: revokers.default.tokens,
                    web_sessions: revokers.default.web_sessions,
                    app_passwords: revokers.default.app_passwords
                };
            })
            .then(
                controller.getAuth().sessionID({
                    // glogouttime, revoker.tokens, revoker.app_passwords and revoker.web_sessions,
                    // see https://beta.wiki.yandex-team.ru/passport/dbmoving
                    attributes: '4,134,135,136',
                    allow_child: 'yes'
                })
            )
            .then(function() {
                var oauthApi = new OauthApi(
                    controller.getLogId(),
                    controller.getHeaders(),
                    that._lang,
                    controller.getAuth().getUid()
                );

                return oauthApi.listTokensVer3().then(function(response) {
                    var layout = that.getLayout();

                    that._logger.info('Rendering 2fa disable/newpassword page');

                    if (response.length) {
                        layout.append(new tokensView(false, response, that._controller));
                        layout.append(new HistoryView(papi, controller));
                    }

                    return layout
                        .append(new LoginView(login))
                        .append(new RevokersView(revokersData))
                        .append(new StepSelector('change-password'))
                        .render();
                });
            })
            .catch(function(err) {
                that._logger.info('Error occured while opening - %s', err);
                return that._controller.redirect('/profile/access/2fa/disable');
            });
    },
    _post: function() {
        var that = this;
        var controller = this._controller;
        var formData = this._controller.getFormData();
        var data = {
            password: formData.password
        };
        var revokersData = {
            allow_select: Boolean(formData.allow_select_revokers),
            tokens: Boolean(formData.tokens_revokers),
            web_sessions: Boolean(formData.web_sessions_revokers),
            app_passwords: Boolean(formData.app_passwords_revokers)
        };

        if (formData.allow_select_revokers && !formData.logout) {
            data.revoke_web_sessions = false;
        }
        if (formData.allow_select_revokers && !formData.revoke_accesses) {
            data.revoke_tokens = false;
            data.revoke_app_passwords = false;
        }

        return this._api
            .OTPDisableCommit(data)
            .then(function(response) {
                var queryRetpath = that._controller.getAuth().getRequestParam('retpath');
                var finishRetpath = url.format(
                    _.assign(that._controller.getUrl(), {
                        pathname: '/profile',
                        query: _.omitBy(
                            {
                                retpath: response.retpath
                            },
                            function(arg) {
                                return !arg;
                            }
                        ),
                        search: null
                    })
                );

                that._logger.info('Commit successful, creating session');
                that._controller
                    .getAuth()
                    .setSession(
                        response.cookies,
                        that._controller.getFormData().track_id,
                        queryRetpath || finishRetpath
                    );
            })
            .catch(function(err) {
                that._logger.info('New password set ERROR');

                if (Array.isArray(err) && err.length === 1) {
                    if (err[0] === 'track.invalid_state' || err[0] === 'track.not_found') {
                        return that._controller.redirect('/profile/access/2fa/disable');
                    }

                    if (err[0] === 'sslsession.required') {
                        that._logger.info('Needs SSL session');
                        return that._controller.getAuth().obtainSecureCookie();
                    }

                    controller
                        .getAuth()
                        .sessionID({
                            // glogouttime, revoker.tokens, revoker.app_passwords and revoker.web_sessions,
                            // see https://beta.wiki.yandex-team.ru/passport/dbmoving
                            attributes: '4,134,135,136',
                            allow_child: 'yes'
                        })
                        .then(function() {
                            var oauthApi = new OauthApi(
                                controller.getLogId(),
                                controller.getHeaders(),
                                that._lang,
                                controller.getAuth().getUid()
                            );

                            return oauthApi.listTokensVer3().then(function(response) {
                                var layout = that.getLayout();
                                var papi = that._papi;

                                that._logger.info('Rendering 2fa disable/newpassword page');

                                if (response.length) {
                                    layout.append(new tokensView(false, response, controller));
                                    layout.append(new HistoryView(papi, controller));
                                }

                                return layout
                                    .append(new InvalidInputView(err[0]))
                                    .append(new RevokersView(revokersData))
                                    .append(new StepSelector('change-password'))
                                    .render();
                            });
                        });
                }

                throw err;
            });
    }
});

var initPage = function(Page, req) {
    var controller = req._controller;
    var api = new API(
        controller.getLogId(),
        controller.getHeaders({withPassportServiceTicket: true}),
        controller.getRequestParam('track_id')
    );

    return new Page(controller, api);
};

var TypedMappingRouter = require('prouter/TypedMapping');
var router = new TypedMappingRouter(TwoFADisablePage);

router.map(
    '/profile/access/2fa/disable',
    new TypedMappingRouter(TwoFADisablePage).map('/', EntryPage).map('/newpassword', setNewPassPage)
);

module.exports = function(req, res, next) {
    var logger = new PLog(req.logID, 'profile', 'access', '2fa', 'disable');
    var controller = req._controller;
    var api = new API(
        controller.getLogId(),
        controller.getHeaders({withPassportServiceTicket: true}),
        controller.getRequestParam('track_id')
    );
    var Page = router.match(controller.getUrl().href);

    logger.info('Request received');

    if (req.body.key) {
        return module.exports.checkCaptcha(req, res);
    }

    controller.getLanguage().then(function(lang) {
        var papi = new PassportApi(
            req.logID,
            controller.getHeaders({withPassportServiceTicket: true}),
            lang,
            controller.getAuth().getUid()
        );

        return new Page(controller, api, papi)
            .open()
            .then(function() {
                logger.info('Request handled');
            })
            .catch(function(error) {
                if (error) {
                    logger.error(error);
                    next(error);
                }
            })
            .finally();
    });
};

module.exports.checkCaptcha = function(req, res) {
    var controller = req._controller;
    var logger = new PLog(req.logID, 'profile', 'access', '2fa', 'disable');
    var page = initPage(EntryPage, req, res);
    var captcha = new (require('../blocks/control/captcha/captcha.field'))().setRequired();
    var api = new API(
        controller.getLogId(),
        controller.getHeaders({withPassportServiceTicket: true}),
        controller.getRequestParam('track_id')
    );
    var data = controller.getFormData();

    logger.info('Validate captcha');

    return captcha.validate(data, api).then(function(response) {
        var invalidCaptcha;

        if (response && response[0] !== 'incorrect') {
            return page._post();
        } else {
            invalidCaptcha = response;
            delete req.body.answer;
            return page.createCaptcha(invalidCaptcha);
        }
    });
};
