const assert = require('assert');
const when = require('when');
const util = require('util');
const PassportApi = require('../lib/passport-api');
const {getTicketsAssoc, SERVICE_ALIASES} = require('../lib/tvm');
const NonceView = require('../blocks/nonce/NonceView');

const FakePassportApi = require('inherit')(
    PassportApi,
    {
        params: function(arg) {
            if (arg === 'track_id') {
                return when.resolve({
                    body: {
                        id: this.__self.fakeTrack
                    }
                });
            }

            return this.__base.apply(this, arguments);
        },

        OTPEnableGetSecret: function() {
            return when.resolve({
                status: 'ok',
                app_secret: this.__self.fakeAppSecret,
                pin: this.__self.fakePin,
                track_id: this.__self.fakeTrack,
                id: this.__self.fakeTrack
            });
        },

        OTPEnableSubmit: function() {
            return when.resolve({
                status: 'ok',
                skip_phone_check: false,
                account: {
                    person: {
                        lastname: 'test',
                        language: 'ru',
                        firstname: 'arseny',
                        country: 'ru'
                    },
                    login: 'arseny-test',
                    display_name: {
                        default_avatar: '4000397283',
                        name: 'arseny.test'
                    },
                    uid: 4000397283,
                    display_login: 'arseny.test'
                },
                secure_number: {
                    masked_e164: '+7495359****',
                    masked_original: '+7495359****',
                    masked_international: '+7 495 359-**-**',
                    e164: '+74953590323',
                    is_deleting: false,
                    international: '+7 495 359-03-23',
                    original: '+74953590323'
                },
                profiles: [],
                track_id: this.__self.fakeTrack
            });
        },

        OTPEnableGetState: function() {
            return when.resolve({
                status: 'ok',
                skip_phone_check: false,
                is_password_required: false,
                pin: '***',
                is_otp_checked: false,
                retpath: null,
                app_secret: this.__self.fakeAppSecret,
                track_id: this.__self.fakeTrack,
                secure_number: {
                    masked_e164: '+7495359****',
                    masked_original: '+7495359****',
                    masked_international: '+7 495 359-**-**',
                    e164: '+74953590323',
                    international: '+7 495 359-03-23',
                    original: '+74953590323'
                },
                id: this.__self.fakeTrack
            });
        },

        OTPEnableSetPin: function() {
            return when.resolve();
        },

        OTPEnableCheckOtp: function() {
            return when.resolve({});
        },

        OTPEnableCommit: function() {
            return when.reject(new Error('Nuh-uh. Fake api for screenshooter.'));
        }
    },
    {
        fakeTrack: 'screenshooter',
        fakeAppSecret: 'fakeAppSecret00031114hcu',
        fakePin: '1122'
    }
);

const Layout = require('inherit')(require('../blocks/layout/RenderingLayoutView'), {
    template: 'profile.access.2fa.%lang%.js'
});

const StepSelector = require('inherit')(require('pview'), {
    name: 'otpEnableStep',
    __constructor: function(panel) {
        this.__base.apply(this, arguments);
        this._step = panel;
    },

    _compile: function() {
        return {step: this._step};
    }
});

const CaptchaView = require('inherit')(require('pview'), {
    name: 'captcha',
    __constructor: function(controller, api, isRequired) {
        this.__base.apply(this, arguments);
        this._isRequired = isRequired;
        this._controller = controller;
        this._api = api;
    },

    _compile: function() {
        const captcha = new (require('../blocks/control/captcha/captcha.field'))().setOption('asyncCheck', true);

        if (this._isRequired) {
            captcha.setRequired();
        }

        return this._controller
            .getLanguage()
            .then((lang) => {
                return captcha
                    .setMode()
                    .compile(lang, this._api)
                    .then((compiled) => {
                        return {
                            captcha: compiled
                        };
                    });
            })
            .catch((err) => {
                throw err;
            });
    }
});

const SetLanguage = require('inherit')(require('pview'), {
    name: 'language',
    __constructor: function(controller) {
        this.__base.apply(this, arguments);
        this._controller = controller;
    },

    _compile: function() {
        return this._controller.getLanguage().then((lang) => {
            return {
                language: lang
            };
        });
    }
});

const PhoneControlView = require('inherit')(require('pview'), {
    name: 'phoneConfirmationControl',

    __constructor: function(currentPhone, isDeleting, needsCaptcha, controller) {
        this.__base.apply(this, arguments);
        this._currentPhone = currentPhone;
        this._controller = controller;
        this._isDeleting = isDeleting;
        this._needsCaptcha = needsCaptcha;
    },

    _compile: function() {
        const PhoneConfirmator = require('../blocks/control/phone-confirm/phone-confirm.field');
        const phoneConfirmator = new PhoneConfirmator();

        phoneConfirmator.setLabel('');
        phoneConfirmator.setCodeLabel('%phone-confirm_code_label_short');

        if (this._needsCaptcha) {
            phoneConfirmator.setOption('needsCaptcha', true);
        }

        if (this._currentPhone) {
            phoneConfirmator.setValue(this._currentPhone);

            phoneConfirmator.setOption('hasConfirmedPhone', true);
            phoneConfirmator.setOption('mode', 'trackedWithUpdate');

            if (this._isDeleting) {
                phoneConfirmator.setOption('isDeleting', true);
            }
        } else {
            phoneConfirmator.setOption('mode', 'confirmAndBindSecure');
        }

        return this._controller.getLanguage().then((lang) => {
            return {phoneconfirm: phoneConfirmator.compile(lang)};
        });
    }
});

const TrackView = require('inherit')(require('pview'), {
    name: 'track',
    __constructor: function(api) {
        assert(api instanceof require('../lib/passport-api'));
        this._api = api;

        this.__base.apply(this, arguments);
    },

    _compile: function() {
        return this._api.params('track_id').then((track) => {
            return {track_id: {id: track.body.id}};
        });
    }
});

const CommonEntryIsWrongFlag = require('inherit')(require('pview'), {
    name: 'EntryIsWrongFlag',
    _compile: function() {
        return {
            entryIsWrong: true
        };
    }
});

const PinView = require('inherit')(require('pview'), {
    name: '2fa.pin',
    __constructor: function(pin, showInstantly) {
        this.__base.apply(this, arguments);

        assert(pin && typeof pin === 'string', 'Pin should be a string');
        assert(typeof showInstantly === 'boolean', 'Show-instantly flag should be a boolean');

        this._pin = pin;
        this._instantly = showInstantly;
    },

    _compile: function() {
        return {
            pin: {
                value: this._pin,
                showInstantly: this._instantly
            }
        };
    }
});

const OTPPage = require('inherit')({
    name: null, //Should be overwritten
    needsTrack: true,
    __constructor: function(controller, api) {
        assert(api instanceof require('../lib/passport-api'));
        assert(this.name && typeof this.name === 'string', 'Page name should be defined for logging');

        this._controller = controller;
        this._api = api;

        this._logger = new (require('plog'))(controller.getLogId(), 'profile', 'access', '2fa', 'page', this.name);
    },

    getLayout: function() {
        return new Layout(this._controller).append(new NonceView(this._controller)).append(new TrackView(this._api));
    },

    _open: function() {
        throw new Error('Method should be overwritten');
    },

    open: function() {
        const auth = this._controller.getAuth();

        return auth
            .loggedIn()
            .catch((err) => {
                if (err && err.code === 'need_resign') {
                    return when.reject();
                }

                if (err && err instanceof Error && err.message === 'password change required') {
                    this._logger.info('Blackbox responded with "password change required", redirecting to auth');
                    auth.authorize();
                    return when.reject();
                }

                this._logger.warn('Unknown error, rethrowing');
                throw err;
            })
            .then((isLoggedIn) => {
                if (!isLoggedIn) {
                    this._logger.info('User not logged in');
                    auth.authorize();
                    return when.reject();
                }

                this._logger.info('User is logged in');
            })
            .then(() => {
                if (this.needsTrack) {
                    return this._api.params('track_id', true).catch((err) => {
                        if (!err) {
                            // No error means there was an empty reject — exactly the behaviour of 'rejectOnEmpty'
                            this._logger.info('User has no track, redirecting to entry');

                            this._controller.redirect('/profile/access/2fa');
                            return when.reject();
                        }
                    });
                }
            })
            .then(() => {
                if (this.needsTrack) {
                    const uid = auth.getUid();

                    return this._api.OTPEnableGetState().then((result) => {
                        const processUid = result.account.uid.toString();

                        if (processUid !== uid) {
                            this._logger.info('Default user is changed, redirecting to start');
                            this._controller.redirect('/profile/access/2fa');
                        }
                    });
                }
            })
            .then(() => {
                this._logger.info('Opening %s', this.name);
                return this._open();
            })
            .then(() => {
                this._controller.writeStatbox(
                    {
                        //DO NOT RETURN THE PROMISE, page should not depend on the promise being resolved or rejected
                        action: 'opened',
                        mode: 'enable_otp',
                        step: this.name,
                        uid: auth.getUid(),
                        from: this._controller.getRequestParam('from'),
                        track_id: this._api.track(),
                        origin: this._controller.getRequestParam('origin')
                    },
                    false
                );
            })
            .catch((err) => {
                if (err) {
                    if (Array.isArray(err) && err.length === 1) {
                        if (['action.not_required', 'account.sms_2fa_enabled'].includes(err[0])) {
                            this._logger.info(
                                err[0] === 'action.not_required' ? 'Action not required' : 'Sms 2fa enabled'
                            );

                            return this._controller.redirect(
                                require('url').format(
                                    require('lodash').assign(this._controller.getUrl(), {
                                        pathname: '/profile'
                                    })
                                )
                            );
                        }

                        if (err[0] === 'sslsession.required') {
                            this._logger.info('Needs SSL session');

                            return this._controller.getAuth().obtainSecureCookie();
                        }
                    }
                    throw err; //Unknown error gets rethrown
                }
            });
    }
});

const EntryPage = require('inherit')(
    OTPPage,
    {
        name: 'entry',
        needsTrack: false,
        _open: function() {
            return this._api
                .OTPEnableSubmit(this._controller.getRequestParam('track_id'))
                .then((otpResponse) => {
                    let from, pathToPin;

                    this._api.track(otpResponse.track_id);

                    if (otpResponse.skip_phone_check) {
                        from = this._controller.getRequestParam('from');
                        pathToPin = `/profile/access/2fa/pin?track_id=${otpResponse.track_id}`;

                        if (typeof from !== 'undefined') {
                            pathToPin = `${pathToPin}&from=${from}`;
                        }

                        this._logger.info('Skip phone check');
                        return this._controller.redirect(pathToPin);
                    }

                    const layout = this.getLayout();

                    switch (otpResponse.state) {
                        case 'complete_social':
                            this._logger.info('Social light user, gets redirected to /profile');
                            return this._controller.redirect('/profile');

                        case 'password_change_forbidden':
                            this._logger.info('Password change forbidden');

                            if (otpResponse.retpath) {
                                layout.append(new this.__self.RetpathView(otpResponse.retpath));
                            }

                            return layout.append(new StepSelector('password_change_forbidden')).render();
                    }

                    if (otpResponse.secure_number) {
                        if (otpResponse.secure_number.is_deleting) {
                            this._logger.info('Cancel deleting or delete phone');
                        } else {
                            this._logger.info('Confirming if user owns the number');
                        }

                        layout
                            .append(new StepSelector('entry-has-phone'))
                            .append(
                                new PhoneControlView(
                                    otpResponse.secure_number.masked_international,
                                    otpResponse.secure_number.is_deleting,
                                    false,
                                    this._controller
                                )
                            );
                    } else {
                        this._logger.info('Asking user to confirm new phone number');

                        layout
                            .append(new StepSelector('entry-no-phone'))
                            .append(new SetLanguage(this._controller))
                            .append(new CaptchaView(this._controller, this._api))
                            .append(new PhoneControlView(null, null, true, this._controller));
                    }

                    return layout.render();
                })
                .catch((err = []) => {
                    if (Array.isArray(err) && err.includes('account.without_password')) {
                        return this._controller.redirectToFrontpage();
                    }

                    throw err;
                });
        }
    },
    {
        RetpathView: require('inherit')(require('pview'), {
            name: 'retpath',
            __constructor: function(retpath) {
                assert(retpath && typeof retpath === 'string', 'Retpath should be a string');
                this._retpath = retpath;

                this.__base.apply(this, arguments);
            },

            _compile: function() {
                return {retpath: this._retpath};
            }
        })
    }
);

const GetAppPage = require('inherit')(OTPPage, {
    name: 'get-app',
    _open: function() {
        return this.getLayout()
            .append(new StepSelector('get-app'))
            .render();
    },

    linkViaSms: function() {
        const YasmsApi = require('../lib/yasms/api');
        const auth = this._controller.getAuth();

        return auth
            .loggedIn()
            .then((loggedIn) => {
                if (!loggedIn) {
                    throw new Error('Should be logged in');
                }

                return this._controller.getLanguage();
            })
            .then((lang) => {
                lang = ['ru', 'en', 'tr', 'uk'].indexOf(lang) > -1 ? lang : 'ru';
                this.lang = lang;

                return getTicketsAssoc();
            })
            .then((tickets) => {
                return new YasmsApi(this._controller.getLogId(), tickets[SERVICE_ALIASES.YASMS]).sendToUid(
                    auth.getUid(),
                    require('putils').i18n(this.lang, 'profile.access.2fa.smslink', 'https://m.ya.ru/key')
                );
            })
            .then(() => {
                //Nobody cares for the response as long as operation was successful,
                //if it wasn't, deferred would reject.
                return {status: 'ok'};
            })
            .catch((err) => {
                if (err && err.code === 'need_resign') {
                    return;
                }

                if (typeof err === 'undefined') {
                    return; //Do nothing if there is no error, promise was rejected to stop the process
                }

                if (err instanceof YasmsApi.ApiError) {
                    switch (err.code) {
                        case 'LIMITEXCEEDED':
                            return {
                                status: 'error',
                                error: 'limitexceeded'
                            };
                        case 'PHONEBLOCKED':
                        case 'PERMANENTBLOCK':
                            return {
                                status: 'error',
                                error: 'blocked'
                            };

                        default:
                            return {
                                status: 'error',
                                error: 'internal'
                            };
                    }
                }

                //Unknown error gets rethrown
                throw err;
            });
    }
});

const GetSecretPage = require('inherit')(
    OTPPage,
    {
        name: 'get-secret',

        getSecretQRImageStream: function() {
            return this._api.OTPEnableGetState().then((result) => {
                const secret = result.app_secret;
                const track_id = result.track_id;

                const login = result.account && result.account.login ? encodeURIComponent(result.account.login) : '';
                const uid = result.account && result.account.uid;
                const name =
                    result.account && result.account.display_name && result.account.display_name.name
                        ? encodeURIComponent(result.account.display_name.name)
                        : '';
                const pin_length = result.pin_length;

                /*
                 otpauth://yaotp/vasiliy.pupkin
                 ?secret=AVC5JUBYMGUMTZIIJNWWOJ3H
                 [&name=%D0%93%D1%80%D1%83%D1%81%D1%82%D0%BD%D1%8B%D0%B9%20%D0%9F%D0%B0%D0%BD%D0%B4%D0%B0]
                 [&track_id=e67302ce6a7beefa5ac55e041b5f676e7f]
                 [&uid=00000000000000]
                 [&image=https%3A%2F%2Favatars.yandex.net%2Fget-yapic%2F12032654%2Fislands-retina-50]
                 [&chars=latin]
                 [&length=8]
                 */

                let format = 'otpauth://yaotp/%s?secret=%s&name=%s&track_id=%s&uid=%s&pin_length=%s';

                let appsecret = util.format(format, login, secret, name, track_id, uid, pin_length);

                if (name.length > 100) {
                    this._logger.info('Name is over 100 characters long, removing it from the qr-code');
                    format = 'otpauth://yaotp/%s?secret=%s&track_id=%s&uid=%s&pin_length=%s';
                    appsecret = util.format(format, login, secret, track_id, uid, pin_length);
                }

                this._logger.verbose('Sending %s', appsecret);

                const masked = util.format(
                    format,
                    login,
                    secret.slice(0, 2) + secret.slice(2, -2).replace(/./g, '*') + secret.slice(-2),
                    name
                );

                this._logger.info('Sending %s', masked);
                return require('qr-image').image(appsecret, {type: 'png'});
            });
        },

        _open: function() {
            return this._api.OTPEnableGetState().then((result) => {
                return this.getLayout()
                    .append(new this.__self.SecretView(result.app_secret_container))
                    .append(new StepSelector('get-secret'))
                    .render();
            });
        }
    },
    {
        SecretView: require('inherit')(require('pview'), {
            name: '2fa.secret',
            __constructor: function(secret) {
                this.__base.apply(this, arguments);

                assert(secret && typeof secret === 'string', 'Secret should be a string');
                this._secret = secret;
            },

            _compile: function() {
                const split = this._secret.match(/.{1,4}/g);
                const midpoint = Math.ceil(split.length / 3);

                return {
                    secret: `${split.slice(0, midpoint).join('&nbsp;&nbsp')}<br />${split
                        .slice(midpoint, midpoint * 2)
                        .join('&nbsp;&nbsp')}<br />${split.slice(midpoint * 2, midpoint * 3).join('&nbsp;&nbsp')}`
                };
            }
        })
    }
);

const GetPinPage = require('inherit')(OTPPage, {
    name: 'get-pin',
    _open: function() {
        if (this._controller.getMethod().isPost()) {
            return this._post();
        } else {
            return this._api.OTPEnableGetSecret().then(() => {
                return this._get(this._controller.getRequestParam('showPin') === 'show');
            });
        }
    },

    _get: function(showInstantly) {
        return this._api.OTPEnableGetState().then((state) => {
            return this.getLayout()
                .append(new PinView(state.pin, showInstantly))
                .append(new StepSelector('get-pin'))
                .render();
        });
    },

    _nextStep: function() {
        return this._controller.redirect(
            require('url').format(
                require('lodash').assign(this._controller.getUrl(), {
                    pathname: '/profile/access/2fa/secret',
                    query: {
                        track_id: this._controller.getFormData().track_id
                    }
                })
            )
        );
    },

    _post: function() {
        return this._api.OTPEnableGetState().then((state) => {
            const pin = this._controller.getFormData().pin;

            if (pin === state.pin) {
                return this._nextStep();
            }

            return this._api.OTPEnableSetPin(pin).then(() => {
                return this._nextStep();
            });
        });
    },

    validatePin: function(data) {
        return this._api
            .OTPValidatePin(data)
            .then(() => {
                if (data.pin.trim().length < 4) {
                    return ['tooshort'];
                }

                return []; //Is valid
            })
            .catch((err) => {
                return err; //Not valid, converting to resolved promise
            });
    }
});

const CheckOTPPage = require('inherit')(
    OTPPage,
    {
        name: 'check-otp',
        _open: function() {
            return this._api.OTPEnableGetState().then((state) => {
                if (state.is_otp_checked) {
                    this._logger.info('OTP is checked in that track');
                    if (!state.is_password_required) {
                        this._logger.info('Password not required');
                        return this._commit();
                    } else if (this._controller.getFormData().password) {
                        this._logger.info('Password is required and was provided');
                        return this._commit(this._controller.getFormData().password);
                    } else {
                        this._logger.info('Password was not provided');
                        return this._renderPassword();
                    }
                } else if (this._controller.getFormData().otp) {
                    this._logger.info('Otp has not been checked yet, but is provided in a current request');
                    return this._submit(this._controller.getFormData().otp);
                } else {
                    this._logger.info('Otp has not been checked yet and is not provided');
                    return this._renderOtp();
                }
            });
        },

        /**
         * Checks the otp
         *
         * @param {string} otp
         * @private
         */
        _submit: function(otp) {
            this._logger.info('Submitting otp');

            return this._api
                .OTPEnableCheckOtp(otp)
                .then((response) => {
                    this._logger.info('Otp was valid');
                    if (response.is_password_required) {
                        this._logger.info('Password is required');
                        return this._renderPassword();
                    }

                    this._logger.info('Password not required');
                    return this._commit();
                })
                .catch((err) => {
                    if (Array.isArray(err) && err.length === 1 && err[0] === 'otp.not_matched') {
                        this._logger.info('Otp not matched');
                        return this._renderOtp(true); //TODO: link to restart if there were several of those
                    }

                    throw err;
                });
        },

        /**
         * Commit the otp setup process
         * @param {string} [password]   Optional password, if it was required when checking otp
         * @private
         */
        _commit: function(password) {
            this._logger.info('Committing', password ? 'with password' : 'without password');

            return this._api
                .OTPEnableCommit(password)
                .then((response) => {
                    const _ = require('lodash');
                    const finishRetpath = require('url').format(
                        _.assign(this._controller.getUrl(), {
                            pathname: '/profile',
                            query: _.omitBy(
                                {
                                    retpath: response.retpath
                                },
                                (arg) => {
                                    return !arg;
                                }
                            ),
                            search: null
                        })
                    );

                    this._logger.info('Commit successful, creating session');
                    this._controller
                        .getAuth()
                        .setSession(response.cookies, this._controller.getFormData().track_id, finishRetpath);
                })
                .catch((err) => {
                    const containsError = (code) => (Array.isArray(err) ? err.indexOf(code) > -1 : err === code);

                    if (containsError('captcha.required')) {
                        this._logger.info('Captcha required');
                        return this._renderPassword(false, true);
                    }

                    if (containsError('password.not_matched')) {
                        this._logger.info('Password not matched');
                        return this._renderPassword(true, false);
                    }

                    throw err;
                });
        },

        _renderOtp: function(isWrong) {
            this._logger.info('Rendering otp input');

            const layout = this.getLayout();

            if (isWrong) {
                layout.append(new CommonEntryIsWrongFlag());
            }

            return layout.append(new StepSelector('check-otp')).render();
        },

        _renderPassword: function(isWrong, captchaRequired) {
            this._logger.info('Rendering password input');

            const layout = this.getLayout();

            layout.append(new StepSelector('check-password'));

            if (isWrong) {
                layout.append(new CommonEntryIsWrongFlag());
            }

            if (captchaRequired) {
                layout
                    .append(new this.__self.CaptchaRequiredFlag())
                    .append(new CaptchaView(this._controller, this._api, true));
            }

            return layout.render();
        }
    },
    {
        CaptchaRequiredFlag: require('inherit')(require('pview'), {
            name: 'CaptchaRequiredFlag',
            _compile: () => ({captchaRequired: true})
        })
    }
);

const TypedMappingRouter = require('prouter/TypedMapping');
const router = new TypedMappingRouter(OTPPage);

router.map(
    '/profile/access/2fa',
    new TypedMappingRouter(OTPPage)
        .map('/', EntryPage)
        .map('/app', GetAppPage)
        .map('/secret', GetSecretPage)
        .map('/pin', GetPinPage)
        .map('/otp', CheckOTPPage)
);

const errhandler = (logger, next) => (error) => {
    if (error) {
        logger.error(error);
        next(error);
    }
};

module.exports = (req, res, next) => {
    const logger = new (require('plog'))(req.logID, 'profile', 'access', '2fa');

    logger.info('Request received');

    const controller = req._controller;

    if (controller.getUrl().pathname === '/profile/access/2fa/qrsecret') {
        return module.exports.sendSecretQRImage(req, res, next);
    }

    let API = PassportApi;

    if (controller.getRequestParam('track_id') === FakePassportApi.fakeTrack) {
        //PASSP-11662: Сделать возможность пройти включение 2fa скриншутилкой
        API = FakePassportApi;
    }

    const api = new API(
        controller.getLogId(),
        controller.getHeaders({withPassportServiceTicket: true}),
        controller.getRequestParam('track_id')
    );

    const Page = router.match(controller.getUrl().href);

    return new Page(controller, api)
        .open()
        .then(() => {
            logger.info('Request handled');
        })
        .catch(errhandler(logger, next))
        .finally();
};

const initPage = (Page, req) => {
    const API = require('../lib/passport-api');
    const controller = req._controller;

    return new Page(
        controller,
        new API(
            controller.getLogId(),
            controller.getHeaders({withPassportServiceTicket: true}),
            controller.getRequestParam('track_id')
        )
    );
};
const getLogger = (page) => new (require('plog'))(page._controller.getLogId(), 'profile', 'access', '2fa', page.name);

module.exports.sendYakeyLinkViaSms = (req, res, next) => {
    const page = initPage(GetAppPage, req, res);
    const logger = getLogger(page);

    logger.info('Opened to send YaKey link via sms');

    return page
        .linkViaSms()
        .then((response) => {
            logger.info('Sms sent');
            return page._controller.sendPage(JSON.stringify(response));
        })
        .catch((err) => {
            if (err instanceof Error || typeof error !== 'object') {
                throw err;
            }

            return page._controller.sendPage(JSON.stringify(err));
        })
        .catch(errhandler(logger, next))
        .finally();
};

module.exports.validatePin = (req, res, next) => {
    const page = initPage(GetPinPage, req, res);
    const logger = getLogger(page);
    const data = {
        track_id: req.body.track_id,
        pin: req.body.pin
    };

    logger.info('Opened to validate pin');

    return page
        .validatePin(data)
        .then((errors) => {
            const resp = {
                status: errors.length ? 'invalid' : 'valid'
            };

            if (errors.length) {
                resp.errors = errors;
            }

            res.end(JSON.stringify(resp));
        })
        .catch(errhandler(logger, next))
        .finally();
};

module.exports.sendSecretQRImage = (req, res, next) => {
    const page = initPage(GetSecretPage, req, res);
    const logger = getLogger(page);

    logger.info('Opened to send secret as QR code');

    return page
        .getSecretQRImageStream()
        .then((stream) => {
            logger.info('Piping QR code into response');

            res.set('Content-type', 'image/png');
            stream.pipe(res);
        })
        .catch(errhandler(logger, next))
        .finally();
};
