var when = require('when');

var assert = require('assert');
var url = require('url');
var util = require('util');
var _ = require('lodash');
var inherit = require('inherit');
var pview = require('pview');
var qrImage = require('qr-image');
var PLog = require('plog');
var PassportApi = require('../lib/passport-api');
var mappingRouter = require('prouter/TypedMapping');
var Layout = require('inherit')(require('../blocks/layout/RenderingLayoutView'), {
    template: 'profile.access.2fa.migrate.%lang%.js'
});
var NonceView = require('../blocks/nonce/NonceView');

var StepSelector = inherit(pview, {
    name: 'otpMigrateStep',
    __constructor: function(panel) {
        this.__base.apply(this, arguments);
        this._step = panel;
    },

    _compile: function() {
        return {step: this._step};
    }
});

var TrackView = inherit(pview, {
    name: 'track',
    __constructor: function(api) {
        assert(api instanceof PassportApi);
        this._api = api;

        this.__base.apply(this, arguments);
    },

    _compile: function() {
        return this._api.params('track_id').then(function(track) {
            return {track_id: {id: track.body.id}};
        });
    }
});

var UidView = require('inherit')(require('pview'), {
    name: 'user-uid',
    __constructor: function(uid) {
        this.__base.apply(this, arguments);
        this._uid = uid;
    },

    _compile: function() {
        return {
            uid: this._uid
        };
    }
});

var CommonEntryIsWrongFlag = inherit(pview, {
    name: 'EntryIsWrongFlag',
    _compile: function() {
        return {
            entryIsWrong: true
        };
    }
});

var TwoFAMigratePage = inherit({
    name: null, //Should be overwritten
    needsTrack: true,
    __constructor: function(controller, api) {
        assert(api instanceof PassportApi);
        assert(this.name && typeof this.name === 'string', 'Page name should be defined for logging');

        this._controller = controller;
        this._api = api;
        this._logger = new PLog(controller.getLogId(), 'profile', 'access', '2fa-migrate', 'page', this.name);
    },

    getLayout: function() {
        return new Layout(this._controller).append(new NonceView(this._controller)).append(new TrackView(this._api));
    },

    errorProcessing: function(error) {
        var logger = this._logger;
        var controller = this._controller;
        var retpath = url.format(
            _.assign(this._controller.getUrl(), {
                pathname: '/profile/access/2fa/migrate'
            })
        );

        if (error) {
            var errorCode = error[0];

            if (typeof error === 'undefined') {
                return; //Do nothing if there is no error, promise was rejected to stop the process
            }
            if (Array.isArray(error) && error.length === 1) {
                if (
                    errorCode === 'track.not_found' ||
                    errorCode === 'track_id.invalid' ||
                    errorCode === 'track_id.empty'
                ) {
                    logger.info('Track is invalid or empty, redirecting to start');
                    controller.redirect(
                        url.format(
                            _.assign(controller.getUrl(), {
                                pathname: '/profile/access/2fa/migrate'
                            })
                        )
                    );
                }

                if (errorCode === 'action.not_required' || errorCode === 'action.impossible') {
                    logger.info('Action not required or impossible - redirecting to /profile');
                    return controller.redirect(
                        url.format(
                            _.assign(controller.getUrl(), {
                                pathname: '/profile'
                            })
                        )
                    );
                }

                if (errorCode === 'sslsession.required' || errorCode === 'sessionid.invalid') {
                    logger.info('Needs SSL session');
                    return controller.getAuth().obtainSecureCookie();
                }

                if (errorCode === 'sessionid.no_uid') {
                    logger.info('uid from track not equal the one from the cookies');
                    return controller.redirect(
                        url.format(
                            _.assign(controller.getUrl(), {
                                pathname: '/auth',
                                query: {
                                    retpath: retpath
                                }
                            })
                        )
                    );
                }

                if (errorCode === 'password.required') {
                    this._logger.info('Password required to continue');
                    return this._controller.redirectToAuthVerify();
                }
            }

            throw error; //Unknown error gets rethrown
        }
    },

    _open: function() {
        throw new Error('Method should be overwritten');
    },

    open: function() {
        var that = this;
        var controller = this._controller;
        var logger = this._logger;
        var auth = controller.getAuth();

        return auth
            .loggedIn()
            .catch(function(err) {
                if (err && err.code === 'need_resign') {
                    return when.reject();
                }

                if (err && err instanceof Error && err.message === 'password change required') {
                    logger.info('Blackbox responded with "password change required", redirecting to auth');
                    auth.authorize();
                    return when.reject();
                }

                logger.warn('Unknown error, rethrowing');
                throw err;
            })
            .then(function(isLoggedIn) {
                if (!isLoggedIn) {
                    logger.info('User not logged in');
                    auth.authorize();
                    return when.reject();
                }

                logger.info('User is logged in');
            })
            .then(function() {
                if (that.needsTrack) {
                    return auth
                        .sessionID({
                            attributes: '1003'
                        })
                        .then(function() {
                            var is2FAOn = Boolean(auth.getAttribute(1003));

                            if (that._controller.getRequestParam('user_id')) {
                                if (that._controller.getRequestParam('user_id') !== auth.getUid()) {
                                    that._logger.info('Default user has changed, process interrupted');
                                    if (is2FAOn) {
                                        that._logger.info(
                                            'User has 2FA enabled, redirecting to start migrating process'
                                        );
                                        that._controller.redirect('/profile/access/2fa/migrate');
                                    } else {
                                        that._logger.info('2FA not enabled, redirecting to /profile');
                                        that._controller.redirect('/profile');
                                    }
                                }
                            }
                        });
                }
            })
            .then(function() {
                logger.info('Opening %s', that.name);
                return that._open();
            })
            .then(function() {
                controller.writeStatbox(
                    {
                        //DO NOT RETURN THE PROMISE, page should not depend on the promise being resolved or rejected
                        action: 'opened',
                        mode: 'migrate_otp',
                        step: that.name,
                        uid: auth.getUid(),
                        from: controller.getRequestParam('from'),
                        origin: controller.getRequestParam('origin'),
                        track_id: that._api.track()
                    },
                    false
                );
            })
            .catch(function(err) {
                that.errorProcessing(err);
            });
    }
});

var GetSecretPage = inherit(
    TwoFAMigratePage,
    {
        name: 'get-secret',
        needsTrack: false,
        getSecretQRImageStream: function(track) {
            var that = this;
            var data = {
                track_id: track
            };
            var retpath = that._controller.getRequestParam('retpath');

            if (retpath) {
                data.retpath = retpath;
            }

            return that._api
                .OTPMigrateSubmit(data)
                .then(function(response) {
                    var login = response.account.login;
                    var track = response.track;

                    return that._api
                        .OTPMigrateGetSecret(track)
                        .then(function(result) {
                            var secret = result.app_secret;
                            var track_id = result.track_id;
                            var pin_length = result.pin_length;

                            /*
                         otpauth://yaotp/vasiliy.pupkin
                         ?secret=AVC5JUBYMGUMTZIIJNWWOJ3H
                         [&track_id=e67302ce6a7beefa5ac55e041b5f676e7f]
                         [&uid=00000000000000]
                         [&image=https%3A%2F%2Favatars.yandex.net%2Fget-yapic%2F12032654%2Fislands-retina-50]
                         [&chars=latin]
                         [&length=8]
                         */

                            var format = 'otpauth://yaotp/%s?secret=%s&track_id=%s&pin_length=%s';
                            var appsecret = util.format(format, login, secret, track_id, pin_length);

                            that._logger.verbose('Sending %s', appsecret);
                            var masked = util.format(
                                format,
                                secret.slice(0, 2) + secret.slice(2, -2).replace(/./g, '*') + secret.slice(-2)
                            );

                            that._logger.info('Sending %s', masked);
                            return qrImage.image(appsecret, {type: 'png'});
                        })
                        .catch(function(err) {
                            that.errorProcessing(err);
                        });
                })
                .catch(function(err) {
                    that.errorProcessing(err);
                });
        },

        _open: function() {
            var that = this;
            var controller = this._controller;
            var uid = controller.getAuth().getUid();

            return this._api
                .getTrackWithUid()
                .then(function(response) {
                    return that._api
                        .OTPMigrateGetSecret(response.body.track_id)
                        .then(function(result) {
                            return that
                                .getLayout()
                                .append(new that.__self.SecretView(result.app_secret_container))
                                .append(new UidView(uid))
                                .append(new StepSelector('get-secret'))
                                .render();
                        })
                        .catch(function(err) {
                            that.errorProcessing(err);
                        });
                })
                .catch(function(err) {
                    that.errorProcessing(err);
                });
        }
    },
    {
        SecretView: inherit(pview, {
            name: '2fa.migrate.secret',
            __constructor: function(secret) {
                this.__base.apply(this, arguments);

                assert(secret && typeof secret === 'string', 'Secret should be a string');
                this._secret = secret;
            },

            _compile: function() {
                var split = this._secret.match(/.{1,4}/g);
                var midpoint = Math.ceil(split.length / 3);

                return {
                    secret: `${split.slice(0, midpoint).join('&nbsp;&nbsp')}<br />${split
                        .slice(midpoint, midpoint * 2)
                        .join('&nbsp;&nbsp')}<br />${split.slice(midpoint * 2, midpoint * 3).join('&nbsp;&nbsp')}`
                };
            }
        })
    }
);

var CheckOTPPage = inherit(TwoFAMigratePage, {
    name: 'check-otp',
    needsTrack: true,
    _open: function() {
        var that = this;
        var logger = this._logger;

        if (that._controller.getFormData().otp) {
            logger.info('Otp has not been checked yet, but is provided in a current request');
            return that._submit(that._controller.getFormData());
        } else {
            logger.info('Otp has not been checked yet and is not provided');
            return that._renderOtp();
        }
    },

    /**
     * Checks the otp
     * @param {object} data
     * @param {string} data.otp - OTP to check
     * @param {string} data.track - track_id
     * @private
     */
    _submit: function(data) {
        this._logger.info('Submitting otp');

        var that = this;

        return this._api
            .OTPMigrateCheckOtp(data)
            .then(function(response) {
                that._logger.info('Otp was valid');
                return that._commit(response.track_id);
            })
            .catch(function(err) {
                if (Array.isArray(err) && err[0] === 'otp.not_matched') {
                    that._logger.info('Otp not matched');
                    return that._renderOtp(true);
                } else {
                    that.errorProcessing(err);
                }
            });
    },

    /**
     * Commit the otp migrate process
     * @param {string} [track]  required when checking otp
     * @private
     */
    _commit: function(track) {
        this._logger.info('Committing 2fa migrate with track', track);
        var that = this;
        var controller = this._controller;

        return this._api
            .OTPMigrateCommit(track)
            .then(function(response) {
                var finishRetpath = url.format(
                    _.assign(controller.getUrl(), {
                        pathname: '/profile',
                        query: _.omitBy(
                            {
                                retpath: response.retpath
                            },
                            function(arg) {
                                return !arg;
                            }
                        ),
                        search: null
                    })
                );

                that._logger.info('Commit otp migrate successful');
                controller.redirect(finishRetpath);
            })
            .catch(function(err) {
                that.errorProcessing(err);
            });
    },

    _renderOtp: function(isWrong) {
        this._logger.info('Rendering otp input');

        var layout = this.getLayout();

        if (isWrong) {
            layout.append(new CommonEntryIsWrongFlag());
        }

        return layout.append(new StepSelector('check-otp')).render();
    }
});

var TypedMappingRouter = mappingRouter;
var router = new TypedMappingRouter(TwoFAMigratePage);

router.map(
    '/profile/access/2fa/migrate',
    new TypedMappingRouter(TwoFAMigratePage).map('/', GetSecretPage).map('/otp', CheckOTPPage)
);

var errhandler = function(logger, next) {
    return function(error) {
        if (error) {
            logger.error(error);
            next(error);
        }
    };
};

module.exports = function(req, res, next) {
    var logger = new PLog(req.logID, 'profile', 'access', '2fa', 'migrate');

    logger.info('Request received');

    var controller = req._controller;

    if (controller.getUrl().pathname === '/profile/access/2fa/migrate/qrsecret') {
        return module.exports.sendSecretQRImage(req, res, next);
    }

    var api = new PassportApi(
        controller.getLogId(),
        controller.getHeaders({withPassportServiceTicket: true}),
        controller.getRequestParam('track_id')
    );

    var Page = router.match(controller.getUrl().href);

    return new Page(controller, api)
        .open()
        .then(function() {
            logger.info('Request handled');
        })
        .catch(errhandler(logger, next))
        .finally();
};

var initPage = function(Page, req) {
    var controller = req._controller;

    return new Page(
        controller,
        new PassportApi(
            controller.getLogId(),
            controller.getHeaders({withPassportServiceTicket: true}),
            controller.getRequestParam('track_id')
        )
    );
};
var getLogger = function(page) {
    return new PLog(page._controller.getLogId(), 'profile', 'access', '2fa/migrate', page.name);
};

module.exports.sendSecretQRImage = function(req, res, next) {
    var page = initPage(GetSecretPage, req, res);
    var controller = req._controller;
    var track = controller.getRequestParam('track_id');
    var logger = getLogger(page);

    logger.info('Opened to send secret as QR code');

    return page
        .getSecretQRImageStream(track)
        .then(function(stream) {
            logger.info('Piping QR code into response');

            res.set('Content-type', 'image/png');
            stream.pipe(res);
        })
        .catch(errhandler(logger, next))
        .finally();
};
