const PLog = require('plog');
const when = require('when');
const config = require('../configs/current');
const CloudAPI = require('../lib/api/cloud');
const GeoAPI = require('../lib/api/geo');
const {getTicketsAssoc, SERVICE_ALIASES} = require('../lib/tvm');

const NOT_FOUND = -1;
const DEFAULT_DELIVERY_ADDRESS_ID = 'defaultDelivery';

const doTVM = [
    (req, res, next) => {
        const localsSaveTickets = (tickets) => {
            if (tickets.hasOwnProperty(SERVICE_ALIASES.CLOUD)) {
                res.locals.cloudServiceTicket = tickets[SERVICE_ALIASES.CLOUD];
            }
            if (tickets.hasOwnProperty(SERVICE_ALIASES.ADDRS_SEARCH)) {
                res.locals.addrsSearchServiceTicket = tickets[SERVICE_ALIASES.ADDRS_SEARCH];
            }
        };

        getTicketsAssoc()
            .then(localsSaveTickets)
            .then(() => {
                next();
            })
            .catch((err) => {
                return next(err);
            });
    }
];

exports.getAddresses = [
    doTVM,
    (req, res) => {
        const addresses = {
            delivery: [],
            home: {
                id: 'home',
                editedAddressLine: ''
            },
            work: {
                id: 'work',
                editedAddressLine: ''
            },
            defaultDelivery: {
                id: DEFAULT_DELIVERY_ADDRESS_ID,
                editedAddressLine: '',
                editedAddressFlat: ''
            },
            successConnect: true
        };

        return req._controller
            .getAuth()
            .sessionID()
            .then((session) => {
                const uid = session && session.uid && session.uid.value;

                if (!uid) {
                    return when.reject('sessionid.invalid');
                }

                req._cloudAPI = new CloudAPI(req.logID, uid, res.locals.cloudServiceTicket);

                return req._cloudAPI.getAddresses();
            })
            .then((response) => {
                if (response && response.items && response.items.length) {
                    response.items.forEach((address) => {
                        addresses[address.address_id] = {
                            id: address.address_id,
                            longitude: address.longitude,
                            latitude: address.latitude,
                            addressLine: address.address_line,
                            editedAddressLine: address.address_line,
                            addressLineShort: address.address_line_short,
                            isValid: true
                        };
                    });
                }

                return req._cloudAPI.getDeliveryAddresses();
            })
            .then((response) => {
                const items =
                    response &&
                    response.items &&
                    response.items.length &&
                    response.items.filter((item) => item.addressLine || item.title);

                if (items && items.length) {
                    addresses.delivery = items.map((address) => ({
                        id: address.id,
                        country: address.country,
                        city: address.city,
                        street: address.street,
                        building: address.building,
                        flat: address.flat,
                        editedAddressFlat: address.flat,
                        addressLine: address.addressLine,
                        editedAddressLine: address.addressLine,
                        addressLineShort: address.title,
                        isValid: true
                    }));
                }
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.getAddresses')
                    .write(err);

                addresses.successConnect = false;
            })
            .finally(() => res.json(addresses));
    }
];

exports.saveAddresses = [
    doTVM,
    (req, res) => {
        const handlers = [];

        let response = {
            status: 'ok'
        };

        let addresses = [];

        try {
            addresses = JSON.parse(req.body.addresses);
        } catch (error) {
            PLog.warn()
                .logId(req.logID)
                .type('profile.passport.saveAddress')
                .write(`can't parse "req.body.addresses" ${error}`);
        }

        if (!addresses.length) {
            return res.json({
                status: 'error',
                errors: ['addresses.empty']
            });
        }

        return req._controller
            .getAuth()
            .sessionID()
            .then((session) => {
                const uid = session && session.uid && session.uid.value;

                if (!uid) {
                    return when.reject('sessionid.invalid');
                }

                const cloudAPI = new CloudAPI(req.logID, uid, res.locals.cloudServiceTicket);

                addresses.forEach((address) => {
                    // @see https://wiki.yandex-team.ru/disk/personality/addresses/
                    if (['home', 'work'].indexOf(address.id) !== NOT_FOUND) {
                        if (address.latitude && address.longitude && address.addressLineShort) {
                            handlers.push(
                                cloudAPI.saveAddress({
                                    address_id: address.id,
                                    tags: [address.id],
                                    title: address.addressLineShort,
                                    address_line: address.addressLine,
                                    address_line_short: address.addressLineShort,
                                    latitude: Number(address.latitude),
                                    longitude: Number(address.longitude)
                                })
                            );
                        }
                    } else {
                        if (address.id && address.addressLineShort) {
                            handlers.push(
                                cloudAPI.saveDeliveryAddress({
                                    id: address.id,
                                    title: address.addressLineShort,
                                    addressLine: address.addressLine,
                                    flat: address.flat || '',
                                    country: address.country || '',
                                    street: address.street || '',
                                    city: address.city || '',
                                    building: address.building || ''
                                })
                            );
                        }
                    }
                });

                if (!handlers.length) {
                    response = {
                        status: 'error',
                        errors: ['incorrect-address']
                    };

                    return;
                }

                return when.all(handlers);
            })
            .catch((err) => {
                err = (err && err._response) || err;
                delete err.stacktrace;

                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.saveAddress')
                    .write(err);

                response = {
                    status: 'error',
                    errors: [err]
                };
            })
            .finally(() => res.json(response));
    }
];

exports.deleteAddresses = [
    doTVM,
    (req, res) => {
        const addressesID = req.body.id;

        let response = {
            status: 'ok'
        };

        if (!addressesID) {
            return res.json({
                status: 'error',
                errors: ['addressesID.empty']
            });
        }

        return req._controller
            .getAuth()
            .sessionID()
            .then((session) => {
                const uid = session && session.uid && session.uid.value;

                if (!uid) {
                    return when.reject('sessionid.invalid');
                }

                const cloudAPI = new CloudAPI(req.logID, uid, res.locals.cloudServiceTicket);

                if (['home', 'work'].indexOf(addressesID) !== NOT_FOUND) {
                    return cloudAPI.deleteAddress(addressesID);
                }

                return cloudAPI.deleteDeliveryAddress(addressesID);
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.deleteAddress')
                    .write(err);

                response = {
                    status: 'error',
                    errors: [err]
                };
            })
            .finally(() => res.json(response));
    }
];

exports.getLocationByGeoCode = [
    doTVM,
    (req, res) => {
        const geoAPI = new GeoAPI(req.logID, res.locals.addrsSearchServiceTicket);
        const geoCode = req.body && req.body.geoCode;
        const detail = req.body && req.body.detail === 'true';
        const lang = (req.body && req.body.lang) || 'ru_RU';
        const locationParams = {
            origin: config.geoCoderOrigin,
            results: 1,
            text: geoCode,
            type: 'geo',
            ms: 'pb',
            lang
        };

        return geoAPI
            .getLocation(locationParams)
            .then((response = {}) => {
                const {error, reply: responseBody} = response;

                if (error) {
                    return res.json({
                        status: 'error',
                        errors: [error.message]
                    });
                }

                const location = parseLocationFromGeoData(responseBody, detail);

                if (!location) {
                    return res.json({
                        status: 'error',
                        errors: ['geocoder.empty']
                    });
                }

                return res.json({
                    status: 'success',
                    body: location
                });
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.getLocationByGeoCode')
                    .write(error);

                return res.json(error);
            });
    }
];

exports.parseLocationFromGeoData = parseLocationFromGeoData;

function parseLocationFromGeoData(data = {}, detail) {
    const geoObject = (data.geoObject && data.geoObject[0]) || {};
    const {name, description, geometry = [{}]} = geoObject;
    const coords = (geometry[0] && geometry[0].point) || {};
    const {lon, lat} = coords;

    if (!name) {
        return null;
    }

    const fullAddress = description
        ? `${description
              .split(', ')
              .reverse()
              .join(', ')}, ${name}`
        : name;
    const location = {
        addressLine: fullAddress,
        addressLineShort: name,
        longitude: Number(lon),
        latitude: Number(lat)
    };

    if (!detail) {
        return location;
    }

    const {metadata = []} = geoObject;
    const geoObjectMeta = (metadata[0] && metadata[0]['.yandex.maps.proto.search.geocoder.GEO_OBJECT_METADATA']) || {};
    const {address = {}} = geoObjectMeta;
    const {component: components} = address;

    const availableDetailsMapping = {
        COUNTRY: 'country',
        LOCALITY: 'city',
        STREET: 'street',
        HOUSE: 'building'
    };

    const fallbacksMapping = {
        AREA: 'city'
    };

    const addressDetails = {
        country: '',
        city: '',
        street: '',
        building: ''
    };

    components.forEach((component) => {
        const detailKind = component.kind && availableDetailsMapping[component.kind[0]];
        const fallbackKind = component.kind && fallbacksMapping[component.kind[0]];

        if (detailKind) {
            addressDetails[detailKind] = component.name;
        }

        if (fallbackKind && !addressDetails[fallbackKind]) {
            addressDetails[fallbackKind] = component.name;
        }
    });

    return Object.assign({}, location, addressDetails);
}
