const url = require('url');
const PLog = require('plog');
const _ = require('lodash');

const Api = require('../lib/api/passport');
const apiSetup = require('./common/apiSetup');
const langSetup = require('./common/langSetup');
const config = require('../configs/current');
const OauthApiLib = require('../lib/api/oauth');
const TokenApiLib = require('papi/OAuth/models/Token');
const validateCSRF = require('./common/validateCSRF.js');
const dheaderSetup = require('./common/dheaderSetup');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');

exports.routes = {};
exports.route = function(app) {
    const routes = this.routes;

    app.get('/profile/access/apppasswords/create/', routes.main);
};

exports.routes.main = [
    langSetup,
    function checkIfAppPasswordsEnabled(req, res, next) {
        req._controller
            .getAuth()
            .sessionID({
                // account.enable_app_password, glogouttime, revoker.tokens, revoker.app_passwords and
                // revoker.web_sessions see https://beta.wiki.yandex-team.ru/passport/dbmoving
                attributes: '107,4,134,135,136'
            })
            .then(function(response) {
                if (!response) {
                    res.locals.appPasswordsEnabled = false;
                    return next();
                }

                res.locals.appPasswordsEnabled = Boolean(req._controller.getAuth().getAttribute(107));
                res.locals.uid = req._controller.getAuth().getUid();
                return next();
            })
            .catch(function(error) {
                if (error && error.code === 'need_resign') {
                    return req._controller.getAuth().resign();
                }

                return next(error);
            });
    },
    apiSetup,
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    function operateWithTrack(req, res, next) {
        const trackId = req.query.track_id;
        const logger = new PLog(req.logID, 'passport', 'create app password', 'operateWithTrack');

        if (trackId) {
            res.locals.track_id = trackId;

            req.api
                .readTrack(trackId)
                .then(function(result) {
                    const body = result && result.body;

                    if (body && body.frontend_state && body.uid) {
                        const data = JSON.parse(body.frontend_state);
                        const uidFromTrack = body.uid;

                        if (res.locals.uid !== uidFromTrack || data === 'done') {
                            logger.info(
                                'Uid from track is not equal the one from blackbox' +
                                    'or track %s has been modified before, redirecting to start',
                                res.locals.track_id
                            );
                            return redirectToStart(req, res);
                        }
                        res.locals.skipForm = true;
                        res.locals.isAppPasswordsEnabled = data.isAppPasswordsEnabled;
                        res.locals.appPasswordValue = data.enteredName || '';
                    }

                    return next();
                })
                .catch(function(error) {
                    logger.info(
                        'Failed to read track %s when creating password' + ' for scope "%s", error: %s',
                        res.locals.track_id,
                        req.nquery && req.nquery.scope,
                        error
                    );
                    return redirectToStart(req, res);
                });
        } else {
            req.api
                .getTrack(null, true)
                .then(function(result) {
                    res.locals.track_id = result && result.body && result.body.track_id;
                    res.locals.skipForm = false;
                    return next();
                })
                .catch(function(errors) {
                    res.locals.error = true;

                    if (Array.isArray(errors)) {
                        const retpath = url.format(req._controller.getUrl());

                        if (errors.indexOf('sessionid.invalid') !== -1) {
                            return res.redirect(
                                url.format(
                                    _.extend({}, req._controller.getAuthUrl(), {
                                        query: {
                                            retpath
                                        }
                                    })
                                )
                            );
                        }

                        return next(errors);
                    }

                    return next(errors);
                });
        }
    },
    getSessionAttributes,
    function(req, res, next) {
        if (!res.locals.skipForm) {
            return next();
        }

        if (res.locals.isAppPasswordsEnabled) {
            return next();
        }

        return req.api
            .appPasswordsActivate(res.locals.track_id)
            .then(function() {
                next();
            })
            .catch(function(error) {
                next(error);
            });
    },
    function(req, res, next) {
        if (!res.locals.skipForm) {
            return next();
        }

        const oauthApi = new OauthApiLib(
            req._controller.getLogId(),
            req._controller.getHeaders(),
            res.locals.language,
            req._controller.getAuth().getUid()
        );

        const clientId = returnClientIdByName(req.query.scope || 'mail');

        // create Apppassword
        return oauthApi
            .issueAppPasswordVer2(clientId, res.locals.appPasswordValue, req._controller.getIp())
            .then(function(response) {
                res.locals.passwordGenerated = {
                    alias: response.getAlias(),
                    clientSlug: response.getClientSlug()
                };
                next();
            })
            .catch(function(err) {
                const logger = new PLog(req.logID, 'passport', 'create app password', 'create app password');

                logger.info('Failed create app password, error: %s, redirecting to start', err);
                return redirectToStart(req, res);
            });
    },
    function(req, res, next) {
        if (!res.locals.skipForm) {
            return next();
        }

        const logger = new PLog(req.logID, 'passport', 'create app password', 'rewriteTrackField');
        const data = {
            track_id: res.locals.track_id,
            frontend_state: 'done'
        };

        return req.api
            .writeTrack(data)
            .then(function(response) {
                if (!response) {
                    return redirectToStart(req, res);
                }
                return next();
            })
            .catch(function() {
                logger.info('Failed to rewrite track data with %s, redirecting to start', data);
                return redirectToStart(req, res);
            });
    },
    function(req, res) {
        const scope = req.query.scope || 'mail';

        res.locals.retpath = req.query.retpath || url.format(req._controller.getModePassportUrl());
        res.locals.clientId = returnClientIdByName(scope);
        res.render(`profile.access.apppasswords.${res.locals.language}.js`);
    }
];

function returnClientIdByName(name) {
    const idList = config.appPasswordsClientIdMapping;

    let clientId;

    _.forIn(idList, function(key, val) {
        if (key === name) {
            clientId = val;
        }
    });

    return clientId;
}

function redirectToStart(req, res) {
    return res.redirect(
        url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: req.path,
            query: {
                retpath: req.query.retpath,
                scope: req.query.scope
            }
        })
    );
}

function getSessionAttributes(req, res, next) {
    req._controller
        .getAuth()
        .sessionID({
            // glogouttime, revoker.tokens, revoker.app_passwords and revoker.web_sessions
            // see https://beta.wiki.yandex-team.ru/passport/dbmoving
            attributes: '4,134,135,136'
        })
        .then(function(response) {
            if (!response) {
                const logger = new PLog(req.logID, 'passport', 'profile count app passwords', 'getAppPasswordsNumber');

                logger.warn('Could not get attributes from session');
                res.json({status: 'error', errors: ['sessionid.invalid']});
            }

            return next();
        });
}

// create app password from /profile
const profileCreateApppassword = [
    apiSetup,
    validateCSRF,
    getSessionAttributes,
    function(req, res, next) {
        const logger = new PLog(req.logID, 'passport', 'profile create app password', 'profileCreateApppassword');

        if (req.body.appPasswordsStatus === 'enabled') {
            logger.info('Application passwords enabled');
            return next();
        }

        return req.api
            .appPasswordsActivate(req.body.track_id)
            .then(function() {
                logger.info('Application passwords activated');
                return next();
            })
            .catch(function(errors) {
                logger.warn('Could not activate application passwords: %s', errors);

                if (Array.isArray(errors)) {
                    if (errors[0] === 'action.not_required') {
                        return next();
                    }

                    if (errors[0] === 'password.required' || errors[0] === 'token.limit_exceeded') {
                        return res.json({
                            status: 'error',
                            errors
                        });
                    }
                }

                return next(errors);
            });
    },
    function(req, res) {
        const controller = req._controller;
        const {lang, clientId, deviceName} = req.body;

        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            lang,
            controller.getAuth().getUid()
        );

        // create Apppassword
        return oauthApi
            .issueAppPasswordVer2(clientId, deviceName, controller.getIp())
            .then(function(response) {
                if (response._alias) {
                    res.json({status: 'ok', alias: response._alias});
                }
            })
            .catch(function(err) {
                const logger = new PLog(req.logID, 'passport', 'profile create app password', 'create app password');

                logger.info('Failed create app password, error: %s', err);
                res.json(err._response);
            });
    }
];

// count app passwords from /profile
const getAppPasswordsNumber = [
    apiSetup,
    validateCSRF,
    getSessionAttributes,
    function(req, res) {
        const controller = req._controller;
        const lang = req.body.lang;
        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            lang,
            controller.getAuth().getUid()
        );

        return oauthApi
            .countAppPasswordsVer2()
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                const logger = new PLog(req.logID, 'passport', 'profile count app passwords', 'getAppPasswordsNumber');

                logger.info('Failed app passwords number, error: %s', err);
                res.json(err._response);
            });
    }
];

// get appPasswords tokens
const getTokensList = [
    apiSetup,
    validateCSRF,
    getSessionAttributes,
    function(req, res) {
        const controller = req._controller;
        const lang = req.body.lang;
        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            lang,
            controller.getAuth().getUid()
        );

        oauthApi.listTokensVer3().then(function(response) {
            const appPasswordsList = [];

            if (response && Array.isArray(response)) {
                response.forEach((token) => {
                    const appPassword = new TokenApiLib(token._token);

                    if (appPassword.isAppPassword()) {
                        appPasswordsList.push({
                            tokenId: appPassword.getId(),
                            tabOpened: false,
                            deviceId: appPassword.getDeviceId(),
                            deviceName: appPassword.getDeviceName(),
                            createTime: token._token.create_time,
                            issueTime: token._token.issue_time,
                            client: appPassword.getClientSlug(),
                            scopes: appPassword.getScopes().map((scope, index) => {
                                return {
                                    scope: scope.getTitle(),
                                    key: index + 1
                                };
                            })
                        });
                    }
                });
            }
            res.json(appPasswordsList.sort((i, j) => j.createTime - i.createTime));
        });
    }
];

const revokeToken = [
    apiSetup,
    validateCSRF,
    getSessionAttributes,
    function(req, res) {
        const controller = req._controller;
        const lang = req.body.lang;
        const tokenId = parseInt(req.body.tokenId, 10);
        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            lang,
            controller.getAuth().getUid()
        );

        return oauthApi
            .revokeSingleTokenVer3(tokenId)
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                const logger = new PLog(req.logID, 'passport', 'revoke token', 'revokeToken');

                logger.info('Failed to revoke single token, error: %s', err);
                res.json(err._response);
            });
    }
];

const revokeTokens = [
    apiSetup,
    validateCSRF,
    getSessionAttributes,
    function(req, res) {
        const controller = req._controller;
        const lang = req.body.lang;
        const tokens = (req.body.tokens || '').split(',').map((t) => parseInt(t, 10));
        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            lang,
            controller.getAuth().getUid()
        );

        return oauthApi
            .revokeSingleTokenVer3(tokens)
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                const logger = new PLog(req.logID, 'passport', 'revoke token', 'revokeToken');

                logger.info('Failed to revoke single token, error: %s', err);
                res.json(err._response);
            });
    }
];

const getTokenGroups = [
    langSetup,
    getHistoryTokens,
    apiSetup,
    validateCSRF,
    function(req, res, next) {
        const controller = req._controller;
        const auth = controller.getAuth();
        const authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        authRequest.then(function() {
            return next();
        });
    },
    function(req, res, next) {
        const controller = req._controller;

        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            req.body.lang,
            controller.getAuth().getUid()
        );

        return oauthApi
            .listTokenGroups(true)
            .then(function(response) {
                if (response && response.status === 'ok') {
                    const allOtherTokens = processTokensList(response.other_tokens);
                    const devicesList = processDevicesList(response.device_tokens) || null;
                    const tokensPerDevice = res.locals.tokensPerDevice || [];

                    let yandexTokens = null;

                    let restTokens = null;

                    if (allOtherTokens) {
                        yandexTokens = allOtherTokens.filter(function(token) {
                            return token.client.isYandex === true;
                        });
                        restTokens = allOtherTokens.filter(function(token) {
                            return token.client.isYandex === false;
                        });
                    }

                    if (devicesList) {
                        devicesList.sort(function(a, b) {
                            if (a.deviceName < b.deviceName) {
                                return -1;
                            }
                            if (a.deviceName > b.deviceName) {
                                return 1;
                            }

                            return b.lastAuth - a.lastAuth;
                        });

                        const deviceNameMap = new Map();

                        for (let i = devicesList.length - 1; i >= 0; i--) {
                            const device = devicesList[i];
                            const doubleDevice = deviceNameMap.get(device.deviceName);

                            if (doubleDevice) {
                                doubleDevice.duble = device.deviceName;
                                doubleDevice.hasDubles = undefined;
                                device.hasDubles = true;
                            }
                            deviceNameMap.set(device.deviceName, device);
                        }
                    }

                    return res.json({
                        deviceTokens: devicesList,
                        otherTokens: restTokens,
                        isYandexTokens: yandexTokens,
                        tokensPerDevice
                    });
                }

                return next(new Error('Unknown error'));
            })
            .catch(function(err) {
                return next(err._response);
            });
    },
    // eslint-disable-next-line no-unused-vars
    function(err, req, res, next) {
        const logger = new PLog(req.logID, 'passport', 'get grouped tokens', 'getDevices');

        logger.info('Failed to get grouped tokens, error: %s', err);
        res.json();
    }
];

// revokeDeviceTokens
const revokeDeviceTokens = [
    apiSetup,
    validateCSRF,
    function(req, res, next) {
        const auth = req._controller.getAuth();
        const authRequest = auth.getLastRequestPromise() || auth.loggedIn();

        authRequest.then(function() {
            return next();
        });
    },
    function(req, res) {
        const controller = req._controller;
        const lang = req.body.lang;
        const deviceId = req.body.device_id;
        const oauthApi = new OauthApiLib(
            controller.getLogId(),
            controller.getHeaders(),
            lang,
            controller.getAuth().getUid()
        );

        return oauthApi
            .revokeDeviceTokens(deviceId)
            .then(function(response) {
                res.json(response);
            })
            .catch(function(err) {
                const logger = new PLog(req.logID, 'passport', 'revoke device tokens', 'revokeDeviceTokens');

                logger.info('Failed to revoke device tokens, error: %s', err);
                res.json(err._response);
            });
    }
];

function processDevicesList(devices) {
    if (!Array.isArray(devices)) {
        return [];
    }

    return devices.map(function(device, index) {
        return {
            key: index + 1,
            deviceName: device.device_name,
            deviceId: device.device_id,
            hasXtoken: device.has_xtoken,
            tokens: processTokensList(device.tokens).sort(function(a, b) {
                return b.issueTime - a.issueTime;
            })
        };
    });
}

function processTokensList(list) {
    if (!Array.isArray(list)) {
        return [];
    }

    return list.map(function(token, index) {
        return {
            key: index + 1,
            tokenId: token.id,
            tabOpened: false,
            // createTime: token.create_time,
            issueTime: token.issue_time,
            client: {
                id: token.client.id,
                title: token.client.title,
                iconUrl: token.client.icon_url,
                isYandex: token.client.is_yandex
            },
            scopes: getScopes(token.scopes)
        };
    });
}

function getScopes(scopes) {
    const scopesData = [];

    _.forIn(scopes, function(value, key) {
        const list = [];

        _.forIn(value, function(val) {
            list.push(val.title);
        });

        scopesData.push({scopeTitle: key, scopesList: list});
    });

    return scopesData;
}

function getHistoryTokens(req, res, next) {
    const controller = req._controller;
    const tokensPerDevice = {};
    const api = new Api(
        req.logID,
        controller.getHeaders({withPassportServiceTicket: true}),
        res.locals.language,
        controller.getAuth().getUid()
    );

    api.getLastAuth()
        .then(function(result) {
            result.tokens.concat(result.appPass).forEach(function(entry) {
                const deviceId = entry && entry._entry && entry._entry.oauth && entry._entry.oauth.device_id;
                const device = tokensPerDevice[deviceId] || (tokensPerDevice[deviceId] = []);

                device.push(new Date(entry.getTimestamp()).getTime() / 1000);
            });

            Object.keys(tokensPerDevice).forEach(function(deviceId) {
                tokensPerDevice[deviceId].sort((a, b) => b - a);
            });

            res.locals.tokensPerDevice = tokensPerDevice;

            next();
        })
        .catch(function() {
            next();
        });
}

exports.getAppPasswordsNumber = getAppPasswordsNumber;
exports.getTokensList = getTokensList;
exports.profileCreateApppassword = profileCreateApppassword;
exports.getTokenGroups = getTokenGroups;
exports.revokeToken = revokeToken;
exports.revokeTokens = revokeTokens;
exports.revokeDeviceTokens = revokeDeviceTokens;
