const PLog = require('plog');
const url = require('url');
const apiSetup = require('./common/apiSetup');
const BillingApi = require('../lib/api/billing');
const BillingInternalApi = require('../lib/api/billingInternal');
const PlusLandingApi = require('../lib/api/plusLanding');
const {SERVICE_ALIASES} = require('../lib/tvm');

const isVISA = (system = '') => system.toLowerCase().includes('visa');

const getBillingCardsInfo = (req) =>
    req.api.billingListPaymentMethods({track_id: req.body.track_id}).then((response) => {
        const responseBody = (response && response.body) || {};
        const billingResponse = responseBody.billing_response || {};
        const data = {
            cards: []
        };

        if (responseBody.status === 'error') {
            return responseBody;
        }

        data.status = billingResponse.status;

        if (billingResponse.status === 'error') {
            data.errors = [billingResponse.status_code];
        } else {
            const paymentMethods = billingResponse.payment_methods || {};

            Object.keys(paymentMethods).forEach((item) => {
                const method = paymentMethods[item];

                const isFamilyCard = Boolean(method.payer_info && method.payer_info.uid);
                const familyAdminUid = Number(method.payer_info && method.payer_info.uid);

                if (method.type === 'card') {
                    data.cards.push({
                        // потому что может быть visa electron ...
                        system: isVISA(method.system) ? 'VISA' : method.system,
                        number: method.number,
                        name: method.name,
                        id: method.id,
                        cardId: method.card_id,
                        proto: method.proto,
                        isFamilyCard,
                        familyAdminUid
                    });
                }
            });
        }

        return data;
    });

exports.billinCardsInfo = [
    (req, res) => {
        getBillingCardsInfo(req)
            .then((resp) => res.json(resp))
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.getCardsInfo')
                    .write(err);
                return {status: 'error'};
            });
    }
];

exports.billingCreateBinding = [
    (req, res) => {
        req.api
            .billingCreateBinding(req.body)
            .then((response) => {
                const responseBody = (response && response.body) || {};
                const billingResponse = responseBody.billing_response || {};
                const data = {
                    cards: []
                };

                if (responseBody.status === 'error') {
                    return res.json(responseBody);
                }

                data.status = billingResponse.status;

                if (billingResponse.status === 'error') {
                    data.errors = [billingResponse.status_code];
                } else {
                    data.token = billingResponse.purchase_token;
                }

                return res.json(data);
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.billingCreateBinding')
                    .write(err);
                return res.json({status: 'error'});
            });
    }
];

exports.billingDoBinding = [
    (req, res) => {
        req.api
            .billingDoBinding(req.body)
            .then((response) => {
                const data = {};
                const responseBody = (response && response.body) || {};
                const billingResponse = responseBody.billing_response || {};
                const billingFrom = billingResponse.binding_form || {};

                if (responseBody.status === 'error') {
                    return res.json(responseBody);
                }

                data.status = billingResponse.status;

                if (billingResponse.status === 'error') {
                    data.errors = [billingResponse.status_code];
                } else {
                    let target;

                    try {
                        target = url.parse(billingFrom._TARGET);
                    } catch (_) {
                        return res.json({status: 'error'});
                    }

                    delete billingFrom._TARGET;

                    data.origin = url.format({
                        protocol: target.protocol,
                        host: target.host
                    });
                    data.url = url.format(
                        Object.assign({}, target, {
                            query: billingFrom
                        })
                    );
                }

                return res.json(data);
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.billingDoBinding')
                    .write(err);
                return res.json({status: 'error'});
            });
    }
];

exports.billingUnbindCard = [
    (req, res) => {
        req.api
            .billingUnbindCard(req.body)
            .then((response) => {
                const data = {};
                const responseBody = (response && response.body) || {};
                const billingResponse = responseBody.billing_response || {};

                if (responseBody.status === 'error') {
                    return res.json(responseBody);
                }

                data.status = billingResponse.status;

                if (billingResponse.status === 'error') {
                    data.errors = [billingResponse.status_code];
                }

                return res.json(data);
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.billingUnbindCard')
                    .write(err);
                return res.json({status: 'error'});
            });
    }
];

exports.bilingGetNativeProducts = [
    apiSetup,
    (req, res) => {
        const controller = req._controller;
        const uid = controller.getAuth().getUid();
        const {target, language = 'ru'} = req.body;

        if (!uid) {
            return res.json({
                status: 'error',
                errors: 'no_auth'
            });
        }
        if (target == null) {
            return res.json({
                status: 'error',
                errors: 'no_target'
            });
        }

        return new BillingApi(req.logID)
            .getNativeProducts({uid, ip: req.ip, target, language})
            .then((response) => {
                const responseBody = (response && response.result) || {};
                const {nativeProducts} = responseBody;

                return res.json(nativeProducts.filter((product) => product.available !== false));
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.bilingGetNativeProducts')
                    .write(err);
                return res.json({status: 'error'});
            });
    }
];

exports.billingSubmitNativeOrder = [
    apiSetup,
    (req, res) => {
        const controller = req._controller;
        const uid = controller.getAuth().getUid();
        const {productId, source, from} = req.body;

        if (!uid) {
            return res.json({
                status: 'error',
                errors: 'no_auth'
            });
        }
        if (productId == null) {
            return {
                status: 'error',
                errors: 'no_productId'
            };
        }

        return Promise.all([getBillingCardsInfo(req), req._controller.getUatraits()])
            .then(([cardsInfo, uaTraits]) => {
                let paymentMethodId =
                    cardsInfo.cards == null || cardsInfo.cards[0] == null ? undefined : cardsInfo.cards[0].id;

                if (paymentMethodId && paymentMethodId.indexOf('card-') !== 0) {
                    paymentMethodId = `card-${paymentMethodId}`;
                }

                const isMobile = (uaTraits.isMobile || uaTraits.isTouch) && !uaTraits.isTablet;

                return new BillingApi(req.logID).submitNativeOrder({
                    productId,
                    ip: req.ip,
                    uid,
                    paymentMethodId,
                    source,
                    from,
                    templateTag: isMobile ? 'mobile/form' : 'desktop/form'
                });
            })
            .then((response) => res.send(response))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.billing.billingSubmitNativeOrder')
                    .write(errors);

                res.status(500);
                return res.send(errors);
            });
    }
];

exports.billingOrderInfo = [
    apiSetup,
    (req, res) => {
        const controller = req._controller;
        const uid = controller.getAuth().getUid();
        const {orderId} = req.body;

        if (!uid) {
            return res.json({
                status: 'error',
                errors: 'no_auth'
            });
        }

        return new BillingApi(req.logID)
            .orderInfo({uid, orderId})
            .then((response) => {
                return res.json(response);
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.billingOrderInfo')
                    .write(err);
                return res.json({status: 'error'});
            });
    }
];

exports.getSubscriptionsInfo = (req, res) => {
    const controller = req._controller;
    const uid = controller.getAuth().getUid();
    const tld = controller.getTld();
    const serviceTicket = req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.MEDIABILLING];

    if (!uid) {
        return res.json({
            status: 'error',
            errors: 'no_auth'
        });
    }

    return new BillingInternalApi(req.logID, serviceTicket)
        .getPassportInfo(uid, tld)
        .then((response) => res.send(response))
        .catch((errors) => {
            PLog.warn()
                .logId(req.logID)
                .type('profile.subscriptions.update')
                .write(errors);

            res.status(500);
            return res.send(errors);
        });
};

exports.getTransitions = (req, res) => {
    const controller = req._controller;
    const uid = controller.getAuth().getUid();
    const ip = req.ip;
    const serviceTicket = req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.MEDIABILLING];

    if (!uid) {
        return res.json({
            status: 'error',
            errors: 'no_auth'
        });
    }

    const {target = 'passport', language = 'ru', transitionTypes = 'DOWNGRADE'} = req.body;

    return new BillingInternalApi(req.logID, serviceTicket)
        .getTransitions({
            target,
            uid,
            ip,
            language,
            transitionTypes
        })
        .then((response) => {
            const transitions = response.transitions;

            if (!(transitions && transitions[0])) {
                res.send({status: 'error', data: 'no transitions'});
            } else {
                const billingApi = new BillingApi(req.logID);

                return Promise.all(
                    transitions.map((transition) => billingApi.productInfo({productId: transition.product.id}))
                ).then((products) => {
                    const enrichedTransitions = transitions.map((transition) =>
                        Object.assign({}, transition, {
                            generalizedProductInfo: products.find((p) => p.productId === transition.product.id)
                        })
                    );

                    res.send({
                        status: 'success',
                        transitions: enrichedTransitions
                    });
                });
            }
        })
        .catch((errors) => {
            PLog.warn()
                .logId(req.logID)
                .type('profile.subscriptions.getTransitions')
                .write(errors);

            res.status(500);
            return res.send(errors);
        });
};

exports.downgradeSubscription = (req, res) => {
    const controller = req._controller;
    const uid = controller.getAuth().getUid();
    const ip = req.ip;
    const serviceTicket = req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.MEDIABILLING];

    if (!uid) {
        return res.json({
            status: 'error',
            errors: 'no_auth'
        });
    }

    const {productIdFrom, productIdTo, source, origin, language} = req.body;

    return req._controller.getUatraits().then((uaTraits) => {
        const isMobile = (uaTraits.isMobile || uaTraits.isTouch) && !uaTraits.isTablet;

        const templateTag = isMobile ? 'mobile/form' : 'desktop/form';

        return new BillingInternalApi(req.logID, serviceTicket)
            .downgradeSubscription({uid, ip, productIdFrom, productIdTo, source, origin, language, templateTag})
            .then((response) => res.send(response))
            .catch((errors) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.subscriptions.downgradeSubscription')
                    .write(errors);

                res.status(500);
                return res.send(errors);
            });
    });
};

exports.yandexBalance = (req, res) => {
    const userTicket = req.userTicket;
    const serviceTicket = req.serviceTickets && req.serviceTickets[SERVICE_ALIASES.PLUS_LANDING];
    const userIp = req.ip;

    return new PlusLandingApi(req.logID, userTicket, serviceTicket, userIp)
        .yandexBalance()
        .then((response) => res.send(response))
        .catch((errors) => {
            PLog.warn()
                .logId(req.logID)
                .type('profile.subscriptions.yandexBalance')
                .write(errors);

            res.status(500);
            return res.send(errors);
        });
};
