const PLog = require('plog');
const dayjs = require('dayjs');
const express = require('express');
const tvm = require('../lib/tvm');
const config = require('../configs/current');
const apiSetup = require('./common/apiSetup');
const getTrack = require('./common/getTrack');
const langSetup = require('./common/langSetup');
const checkAuth = require('./common/checkAuth');
const isValid = require('./common/checkAuth/isValid');
const getDefaultUser = require('./common/checkAuth/getDefaultUser');
const getMetrics = require('./common/getMetrics');
const createState = require('./common/createState');
const writeStatbox = require('./common/writeStatbox');
const DeleteDataApi = require('../lib/api/deleteData');
const showPlus = require('./common/plusGuard').showPlus;
const getTakeoutInfo = require('./takeout/getTakeoutInfo');
const rumCounterSetup = require('./common/rumCounterSetup');
const userTicketsSetup = require('./common/userTicketsSetup');
const urlFormat = require('./common/urlFormat.js').urlFormat;
const profileStateSetup = require('./common/profileStateSetup');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const validateCSRF = require('./common/validateCSRF.js');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const cluster = require('cluster');

const INVALID_SESSION_JSON = {
    status: 'error',
    errors: ['session.invalid']
};
const IS_DEV = process.env.NODE_ENV === 'development';
const IS_TESTING = process.env.NODE_ENV === 'testing';
const IS_PROD = process.env.NODE_ENV === 'production';
const HALF_AN_HOUR = 30 * 60 * 1000;
const IS_SSR_DATA_DISABLED = true;

const router = express.Router();
const setup = [
    getYaExperimentsFlags,
    (req, res, next) => {
        if (IS_PROD && req._controller.hasExp('passport-gdpr-enabled')) {
            return next();
        }

        if (res.locals.isYandex) {
            return next();
        }

        return req.headers['x-requested-with'] === 'XMLHttpRequest' || req.method === 'POST'
            ? res.json(INVALID_SESSION_JSON)
            : req._controller.redirectToLocalUrl({pathname: '/profile'});
    },
    apiSetup,
    userTicketsSetup,
    langSetup
];
const shouldCheckSession = (req) => IS_PROD || ![req.query.skip_sess_check, req.body.skip_sess_check].includes('1');
const getSessionLifetime = (req) =>
    (!IS_PROD && Number(req.body['sess_lifetime'] || req.query['sess_lifetime'] || 0) * 60000) || HALF_AN_HOUR;

const ALL_SERVICES = [
    'adfox',
    'afishadata',
    'alisa',
    'appmetrika',
    'audience',
    'blogs',
    'browser',
    'carsharing',
    'commentator',
    'contest',
    'deli',
    'eats',
    'edadeal',
    'forms',
    'games',
    'geosmb',
    'kinopoisk',
    'lavka',
    'maps',
    'mapsbooking',
    'mapsconstructor',
    'mapshistory',
    'mapsmrc',
    'mapspano',
    'mapsphoto',
    'mapsroadevents',
    'mapsugc',
    'market',
    'messenger',
    'metrika',
    'music',
    'navigator',
    'nmaps',
    'overload',
    'plusstatistic',
    'portal',
    'praktikum',
    'profi',
    'q',
    'realty',
    'schoolbook',
    'sitesearch',
    'tanker',
    'taxi',
    'telemed',
    'translate',
    'travelorchestrator',
    'ugc',
    'uslugi',
    'weather',
    'webmaster',
    'xmlsearch',
    'zen'
];

const servicesDataFilter = {
    lavka: ['all_grocery'],
    taxi: ['taxi_history'],
    deli: ['all_grocery_deli'],
    eats: ['eda_history'],
    messenger: ['0'],
    commentator: ['1'],
    navigator: ['points-history', 'ynavisync', 'ynavicarinfo'],
    mapshistory: ['bookmarks', 'search-history', 'addresses', 'ridehistory']
};

if (!IS_PROD) {
    ALL_SERVICES.push('collections', 'crypta', 'surveys', 'toloka');
}

const getAllServices = (req) => {
    let services = ALL_SERVICES;

    if (!req._controller.hasExp('profile-data-lavka-exp')) {
        services = services.filter((i) => i !== 'lavka');
    }
    if (!req._controller.hasExp('profile-data-deli-exp')) {
        services = services.filter((i) => i !== 'deli');
    }

    if (!req._controller.hasExp('profile-data-taxi-exp')) {
        services = services.filter((i) => i !== 'taxi');
    }

    return services;
};

const checkConfirmationAge = (req, res, next) => {
    const controller = req._controller;
    const auth = controller.getAuth();
    const lifeTime = getSessionLifetime(req);

    if (!shouldCheckSession(req)) {
        return next();
    }

    return auth
        .sessionID({
            multisession: 'yes',
            getphones: 'bound',
            phone_attributes: '4,108'
        })
        .then((sessionInfo) => {
            const {auth = {}, phones = []} = getDefaultUser(sessionInfo.users, sessionInfo.default_uid);
            const passVerificationAge = (auth.password_verification_age || 0) * 1000;
            const phone = phones.find(({attributes = {}} = {}) => attributes[108] === '1');
            const hasPhone = phones.length > 0;
            const lastPhoneConfirmation = !phone ? -1 : Number(phone.attributes[4] || 0) * 1000;

            if (
                !isValid(sessionInfo) ||
                Date.now() - lastPhoneConfirmation > lifeTime ||
                (!hasPhone && passVerificationAge > lifeTime)
            ) {
                return res.json(INVALID_SESSION_JSON);
            }

            return next();
        })
        .catch(() => res.json(INVALID_SESSION_JSON));
};

const getServiceTicket = (req, serviceName) => req.serviceTickets[tvm.SERVICE_ALIASES[serviceName.toUpperCase()]];
const {baseUrls} = config.api.deleteData;

const sendEventToYasm = (event) => {
    if (cluster.isWorker) {
        cluster.worker.send(event);
    }
};

const getDate = (date, lang) =>
    !date
        ? null
        : dayjs(date)
              .locale(lang)
              .format(
                  `D${lang === 'en' ? ' [of] ' : ' '}MMMM${
                      new Date(date).getFullYear() !== new Date().getFullYear() ? ', YYYY' : ''
                  }`
              );
const getData = (req, res, services) =>
    new Promise((resolve, reject) =>
        Promise.all(
            (services || getAllServices(req))
                .filter((service) => getAllServices(req).includes(service))
                .map(
                    (service) =>
                        new Promise((resolve) => {
                            const logType = `takeout.status.${service}`;
                            const serviceTicket = getServiceTicket(req, service);

                            if (!serviceTicket || !baseUrls[service]) {
                                PLog.warn()
                                    .logId(req.logID)
                                    .type(logType)
                                    .write(!serviceTicket ? 'Service ticket not found' : 'Base url is empty');
                                return resolve({service, status: 'error'});
                            }

                            return new DeleteDataApi(
                                baseUrls[service],
                                req.logID,
                                req.userTicket,
                                serviceTicket,
                                service
                            )
                                .getStatus({request_id: req.logID})
                                .then((response = {}) => {
                                    sendEventToYasm(`deleteData.${service}.status.success_counter`);
                                    return resolve({
                                        ...response,
                                        service,
                                        data: (response.data || []).map((item = {}) => ({
                                            ...item,
                                            state: item.state || item.status,
                                            update_date: getDate(item.update_date, res.locals.language)
                                        }))
                                    });
                                })
                                .catch((error) => {
                                    sendEventToYasm(`deleteData.${service}.status.fail_counter`);
                                    PLog.warn()
                                        .logId(req.logID)
                                        .type(logType)
                                        .write(error);
                                    return resolve({service, status: 'error'});
                                });
                        })
                )
        )
            .then((r) =>
                resolve(
                    r.reduce(
                        (acc, {data, status, errors, service}) => {
                            if (status === 'error') {
                                acc.errors[service] = Array.isArray(errors) ? errors : [{code: 'internal'}];
                            } else if (Array.isArray(data) && data.length) {
                                acc.data[service] = data;
                            }

                            return acc;
                        },
                        {
                            data: {},
                            errors: {}
                        }
                    )
                )
            )
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('deleteData.getData')
                    .write(error);
                return reject(error);
            })
    );

router.use((req, res, next) => {
    if (![req.query.mimino, req.body.mimino].includes('0') && (IS_DEV || IS_TESTING)) {
        req._controller.setIsMimino(true);
    }

    return next();
});

router.post('/check-session', [
    setup,
    checkAuth,
    validateCSRF,
    checkConfirmationAge,
    (req, res) => res.json({status: 'ok'})
]);
router.post('/getdata', [
    setup,
    checkAuth,
    validateCSRF,
    checkConfirmationAge,
    (req, res) => {
        const onError = () => res.json({status: 'error'});
        const services = [].concat(req.body['services[]']).filter(Boolean);

        return !req.userTicket
            ? onError()
            : getData(req, res, services.length ? services : getAllServices(req))
                  .then((response) => {
                      Object.keys(servicesDataFilter).forEach((serviceName) => {
                          if (response.data[serviceName]) {
                              response.data[serviceName] = response.data[serviceName].filter(({id}) =>
                                  servicesDataFilter[serviceName].includes(id)
                              );
                          }
                      });

                      res.json({status: 'ok', ...response});
                  })
                  .catch(() => onError());
    }
]);
router.post('/delete', [
    setup,
    checkAuth,
    validateCSRF,
    checkConfirmationAge,
    (req, res) => {
        const service = req.body.service;
        const id = [].concat(req.body['id[]']).filter(Boolean);
        const serviceTicket = getServiceTicket(req, service);
        const logType = `takeout.delete.${service}`;
        const onError = (code) =>
            res.json({
                status: 'error',
                errors: [{code}]
            });

        if ((IS_DEV || IS_TESTING) && req.cookies['DEBUG_DELETE_ERROR_CODE']) {
            return onError(req.cookies['DEBUG_DELETE_ERROR_CODE']);
        }

        if (!serviceTicket) {
            PLog.warn()
                .logId(req.logID)
                .type(logType)
                .write('Service ticket not found');
            return onError('noticket');
        }

        if (!service || !getAllServices(req).includes(service)) {
            PLog.warn()
                .logId(req.logID)
                .type(logType)
                .write(`Unknown service: ${service || '-'}`);
            return onError('noservice');
        }

        if (!id.length) {
            PLog.warn()
                .logId(req.logID)
                .type(logType)
                .write('Wrong id');
            return onError('noid');
        }

        new DeleteDataApi(baseUrls[service], req.logID, req.userTicket, serviceTicket, service)
            .deleteItems({id, request_id: req.logID})
            .then((r) => {
                const isObject = Boolean(r) && typeof r === 'object';
                const isValidAnswer = isObject && ['ok', 'error'].includes(r.status);

                sendEventToYasm(`deleteData.${service}.delete.success_counter`);

                if (!isValidAnswer) {
                    PLog.warn()
                        .logId(req.logID)
                        .type(logType)
                        .write(`Wrong answer: ${isObject ? JSON.stringify(r) : r}`);
                }

                return res.json(isValidAnswer ? r : {status: 'ok'});
            })
            .catch((error) => {
                sendEventToYasm(`deleteData.${service}.delete.fail_counter`);
                PLog.warn()
                    .logId(req.logID)
                    .type(logType)
                    .write(error);
                return res.json({
                    status: 'error',
                    ...error
                });
            });
    }
]);
router.get('/', [
    setup,
    showPlus,
    (req, res, next) => {
        res.locals.blackboxPlus = res.locals.showPlus;

        return next();
    },
    multiAuthAccountsSetup,
    (req, res, next) => {
        const {lastPhoneConfirmation, hasPhone, status, passwordVerificationAge} =
            (res.locals.accounts || {}).defaultAccount || {};
        const checkSession = shouldCheckSession(req);
        const lifeTime = getSessionLifetime(req);

        if (status !== 'VALID') {
            return req._controller.redirectToAuth();
        }

        if (!IS_DEV && !IS_TESTING && !req._controller.hasExp('passport-gdpr-enabled')) {
            return req._controller.redirectToLocalUrl({pathname: '/profile'});
        }

        if (checkSession && Date.now() - lastPhoneConfirmation > lifeTime) {
            return req._controller.redirectToLocalUrl({
                pathname: '/phoneconfirm',
                query: {retpath: urlFormat(req._controller.getUrl()), noreturn: 1}
            });
        }

        if (checkSession && !hasPhone && passwordVerificationAge > lifeTime) {
            return req._controller.redirectToAuthVerify();
        }

        return next();
    },
    rumCounterSetup,
    profileStateSetup,
    getTrack({}, true),
    createState,
    (req, res, next) => {
        const {store, ua = {}} = res.locals;

        store.takeOutInfo = {
            status: 'isNotRequested'
        };
        store.settings.isMac = ua.OSFamily === 'MacOS';
        return next();
    },
    (req, res, next) => {
        const {store} = res.locals;
        const {default_uid: defaultUid, users} = req.blackbox;
        const defaultUser = getDefaultUser(users, defaultUid);
        const {attributes = {}} = defaultUser;

        store.deleteData = {
            error: null,
            errorServiceItems: {},
            loadingServiceItems: {},
            isTakeOutAvailable: Boolean(attributes && attributes['184'])
        };

        return IS_SSR_DATA_DISABLED
            ? next()
            : getData(req, res)
                  .then(({data}) => {
                      PLog.info()
                          .logId(req.logID)
                          .type('deleteData.render.getData')
                          .write('Data was received successfully');
                      store.deleteData.data = data;
                      store.deleteData.ready = true;
                      return next();
                  })
                  .catch((error) => {
                      PLog.warn()
                          .logId(req.logID)
                          .type('deleteData.render.getData')
                          .write(error);
                      return next();
                  });
    },
    getTakeoutInfo,
    writeStatbox({
        action: 'opened',
        mode: 'delete_data'
    }),
    getMetrics({
        header: 'Управление данными'
    }),
    (req, res) => res.render(`react.delete-data.${res.locals.language}.jsx`)
]);

exports.router = router;
exports.ALL_SERVICES = ALL_SERVICES;
