const url = require('url');
const _ = require('lodash');
const when = require('when');
const express = require('express');
const PLog = require('plog');
const checkAuth = require('./common/checkAuth');
const config = require('../configs/current');
const apiSetup = require('./common/apiSetup');
const langSetup = require('./common/langSetup');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const LangSwitcherView = require('../blocks/layout/LangSwitcherView');
const punycode = require('punycode');
const validateCSRF = require('./common/validateCSRF.js');
const showPlus = require('./common/plusGuard').showPlus;
const getMetrics = require('./common/getMetrics');
const rumCounterSetup = require('./common/rumCounterSetup');
const router = express.Router();
const getSidsInfo = require('../lib/service-sids/service-sids').getSidDescription;
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const yamoneyDefaultState = {
    sum: undefined,
    hasWallet: false,
    connectionError: true
};

const sendSms = [
    apiSetup,
    validateCSRF,
    checkCaptcha,
    function(req, res) {
        req.api
            .deleteAccountSendSms({track_id: req.body.track_id})
            .then(function(response) {
                const body = response && response.body;

                res.json({status: 'ok', timeout: body.resend_timeout});
            })
            .catch(function(error) {
                res.json({status: 'error', error: {field: 'phone', error: error[0]}});
            });
    }
];

const sendEmailCode = [
    apiSetup,
    validateCSRF,
    checkCaptcha,
    function(req, res) {
        req.api
            .deleteAccountSendEmail(req.body)
            .then(function() {
                res.json({status: 'ok'});
            })
            .catch(function(error) {
                res.json({status: 'error', code: error[0]});
            });
    }
];

const checkEmailCode = [
    apiSetup,
    validateCSRF,
    function(req, res) {
        req.api
            .deleteAccountConfirmEmail(req.body)
            .then(function() {
                res.json({status: 'ok'});
            })
            .catch(function(error) {
                res.json({status: 'error', code: error[0]});
            });
    }
];

const checkPhoneCode = [
    apiSetup,
    validateCSRF,
    function(req, res) {
        req.api
            .deleteAccountConfirmPhone(req.body)
            .then(function() {
                res.json({status: 'ok'});
            })
            .catch(function(error) {
                res.json({status: 'error', code: error[0]});
            });
    }
];

const checkQuestion = [
    apiSetup,
    validateCSRF,
    checkCaptcha,
    function(req, res) {
        req.api
            .deleteCheckQuestion(req.body)
            .then(function() {
                res.json({status: 'ok'});
            })
            .catch(function(error) {
                res.json({status: 'error', code: error[0]});
            });
    }
];

const deleteCommit = [
    apiSetup,
    validateCSRF,
    function(req, res) {
        const controller = req._controller;

        req.api
            .deleteAccountCommit(req.body)
            .then(function(result) {
                const body = result.body;

                controller.augmentResponse(body);

                res.json({status: 'ok'});
            })
            .catch(function(error) {
                res.json({status: 'error', errors: error});
            });
    }
];

const enter = [
    checkAuth,
    langSetup,
    apiSetup,
    getYaExperimentsFlags,
    function getDeleteTrack(req, res, next) {
        return req.api
            .getTrack({process: 'account_delete_v2_process'}, true)
            .then(function(response) {
                res.locals.track_id = (response.body && response.body.track_id) || '';
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.delete_account.getDeleteTrack')
                    .write(errors);

                return next(errors);
            });
    },
    getState,
    showPlus,
    function(req, res, next) {
        if (res.locals.showPlus) {
            res.locals.blackboxPlus = true;
        }

        return next();
    },
    multiAuthAccountsSetup,
    createStore,
    function deleteProcessSubmit(req, res, next) {
        const tld = req._controller.getTld();

        req.api
            .deleteAccountSubmit({track_id: res.locals.track_id})
            .then(function(response) {
                const accountInfo = processSubscribedServices(response.body, res.locals.language, tld);

                res.locals.store.deleteAccount = Object.assign({}, res.locals.store.deleteAccount, accountInfo);
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.delete.deleteProcessSubmit')
                    .write(errors);

                return next(errors);
            });
    },
    function handleError(errors, req, res, next) {
        const deleteAccountStoreObj = (res.locals.store || {}).deleteAccount;

        if (errors && Array.isArray(errors)) {
            const errorCode = errors[0];
            const isRedirectToAuthNeeded = (code) => {
                const codesToRedirect = ['account.invalid_type', 'account.disabled', 'track.invalid_state'];

                return codesToRedirect.includes(code);
            };
            const isPddAdmin = (code) =>
                ['unable_to_delete_pdd_admin', 'unable_to_delete_connect_admin'].includes(code);

            if (isRedirectToAuthNeeded(errorCode)) {
                return req._controller.redirectToAuth();
            }

            if (errorCode === 'unable_to_delete_pdd_account') {
                return req._controller.redirectToFrontpage();
            }

            if (typeof deleteAccountStoreObj === 'object') {
                if (errorCode === 'account.required_completion') {
                    deleteAccountStoreObj.isSocialchik = true;
                }

                if (isPddAdmin(errorCode)) {
                    deleteAccountStoreObj.isPddAdmin = true;
                }

                res.locals.store.deleteAccount = Object.assign({}, deleteAccountStoreObj, processSubscribedServices());

                return next();
            }

            return next(errors);
        }

        return next(errors);
    },
    getMetrics({header: 'Страница удаления аккаунта'}),
    rumCounterSetup,
    renderPage
];

function renderPage(req, res) {
    const lang = res.locals.language;

    req.api.statboxLogger({
        track_id: res.locals.track_id || null,
        action: 'opened',
        mode: 'delete.account',
        host: req.hostname,
        pathname: req.path,
        referer: req.headers.referer || null,
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        yandexuid: res._yandexuid || req.cookies.yandexuid,
        uid: (res.locals.store && res.locals.store.person && res.locals.store.person.uid) || null,
        origin: (req.query && req.query.origin) || null,
        experiments_flags: res.locals.experiments && res.locals.experiments.flagsString,
        experiment_boxes: res.locals.experiments && res.locals.experiments.boxes
    });

    res.render(`profile.delete.${lang}.jsx`);
}

function createStore(req, res, next) {
    // eslint-disable-line
    const langSwitcher = new LangSwitcherView(req._controller);
    const locals = res.locals;
    const account = locals.account;
    const tld = req._controller.getTld();
    const retpath = (req.query && (req.query.url || req.query.retpath)) || null;
    const handlers = [req._controller.getUatraits(), langSwitcher._compile()];
    const yandexuid = res._yandexuid && res._yandexuid.replace(/[^0-9]/g, '');
    const {userType = {}} = locals;
    const {version, paths = {}} = config;
    const {static: staticPath} = paths;
    const {metricsUserType} = userType;
    const {blackbox: {connection_id: ci = ''} = {}} = req;
    const isAppMode = req.query && req.query.mode && req.query.mode === 'app';

    if (retpath) {
        handlers.push(req.api.validateRetpath({retpath}));
    }

    locals.store = {
        settings: {
            version,
            metricsUserType,
            staticPath,
            host: url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname
            }),
            avatar: config.paths.avatar || {},
            embeddedauth: config.paths.embeddedauth || '',
            links:
                process.env.INTRANET === 'intranet'
                    ? config.links.intranet
                    : config.links[tld] || config.links.ru || {},
            help: config.paths.help || {},
            tld,
            language: locals.language,
            ua: {},
            env: {
                type: process.env.NODE_ENV,
                name: process.env.INTRANET
            },
            accountsUrl: config.paths.accountsUrl
        },
        common: {
            actionForRepeat: null,
            uid: account.uid,
            yandexuid,
            track_id: locals.track_id,
            retpath: url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: '/profile'
            }),
            experiments: res.locals.experiments,
            currentPage: req._controller.getUrl().href,
            defaultPage: url.format({
                pathname: '/profile/delete',
                query: req.query
            }),
            isPDD: account.domain !== undefined,
            isWSUser: locals.isWSUser,
            isAppMode
        },
        domik: {
            errors: {},
            requestPassword: false,
            requestCaptcha: false,
            passwordNotMatched: false,
            captchaNotMatched: false,
            loading: false
        },
        access: {
            is2faEnabled: account.is_2fa_enabled
        },
        deleteAccount: {
            isFetching: false,
            isModalOpened: false,
            isSocialchik: false,
            isPddAdmin: false
        },
        dashboard: {
            plus: Object.assign(
                {},
                {
                    country: locals.country,
                    allowed: locals.showPlus,
                    enabled: false
                },
                locals.accounts.defaultAccount.plus
            )
        },
        captcha: {
            loading: false,
            loadingAudio: false,
            playing: false,
            type: 'text',
            key: null,
            imageUrl: null,
            introSound: null,
            captchaSound: null
        },
        person: _.extend({}, account.person || {}, {
            uid: account.uid,
            login: account.login,
            displayLogin: account.display_login,
            avatarId: (account.display_name && account.display_name.default_avatar) || '',
            displayNames: account.display_names || {},
            displayName: (account.display_name && account.display_name.name) || '',
            escapedDisplayName: (account.display_name && _.escape(account.display_name.name)) || '',
            errors: {}
        }),
        yamoney: _.clone(yamoneyDefaultState),
        header: {
            accounts: (locals.accounts && locals.accounts.accounts) || [],
            defaultAccount: (locals.accounts && locals.accounts.defaultAccount) || {},
            canAddMore: locals.accounts && locals.accounts['can-add-more'],
            ci
        },
        footer: {
            langlist: []
        },
        metrics: {
            header: 'Страница удаления аккаунта'
        },
        monitoring: {
            page: 'profile.delete'
        }
    };

    when.all(handlers)
        .then(function(response) {
            const uatraits = response[0] || {};
            const langlist = (response[1] && response[1].langlist) || {};
            const validatedRetpath = response[2] || {};

            locals.store.settings.ua = uatraits;
            locals.store.footer.langlist = langlist;

            if (validatedRetpath.body && validatedRetpath.body.retpath) {
                locals.store.common.retpath = validatedRetpath.body.retpath;
            }

            if (!uatraits.isMobile && !uatraits.isTouch) {
                Object.keys(locals.store.person.displayNames).forEach(function(key) {
                    const value = locals.store.person.displayNames[key];

                    delete locals.store.person.displayNames[key];

                    locals.store.person.displayNames[_.escape(key)] = value;
                });
            }
        })
        .catch(function(error) {
            PLog.warn()
                .logId(req.logID)
                .type('profile.delete')
                .write(error);
        })
        .done(function() {
            delete locals.account;
            delete locals.accounts;
            return next();
        });
}

function getState(req, res, next) {
    return req.api
        .profileGetState()
        .then(function(response) {
            const profile = response.body;

            res.locals.account = profile.account || {};
            return next();
        })
        .catch(function(errors) {
            PLog.warn()
                .logId(req.logID)
                .type('profile.delete.getState')
                .write(errors);

            return next(errors);
        });
}

function processSubscribedServices(subscribtionInfo, lang, tld) {
    const phoneConfirmRequired = (subscribtionInfo && subscribtionInfo.requirements.phone_required) || null;
    const emailConfirmRequired = (subscribtionInfo && subscribtionInfo.requirements.email_required) || null;
    const questionConfirmRequired = (subscribtionInfo && subscribtionInfo.requirements.question_required) || null;
    const subscribedInfo = {
        subscribedTo: {
            mainList: [],
            secondaryList: []
        },
        hasCaptchaPassed: false,
        confirmation: {
            method: 'captcha',
            status: 'unconfirmed'
        },
        error: {
            field: '',
            message: ''
        },
        hasBlockingSids: false,
        form: {
            captcha: ''
        }
    };

    if (!subscribtionInfo) {
        return subscribedInfo;
    }

    if (emailConfirmRequired || phoneConfirmRequired) {
        subscribedInfo.form.code = '';
    }

    if (emailConfirmRequired) {
        subscribedInfo.confirmation.method = 'email';
        subscribedInfo.confirmation.emails = emailConfirmRequired.emails.map(function(item) {
            return {
                text: punycode.toUnicode(item.address),
                value: item.address
            };
        });
        subscribedInfo.form.email = subscribedInfo.confirmation.emails[0].value;
    }

    if (phoneConfirmRequired) {
        subscribedInfo.confirmation.method = 'phone';
        subscribedInfo.confirmation.phone = phoneConfirmRequired.phone.number.masked_international;
    }

    if (questionConfirmRequired) {
        subscribedInfo.form.answer = '';
        subscribedInfo.confirmation.method = 'question';
        subscribedInfo.confirmation.questionText = questionConfirmRequired.question.text;
    }

    const sidsData = subscribtionInfo.subscribed_to.map(function(item) {
        if (item.is_blocking === true) {
            subscribedInfo.hasBlockingSids = true;
            subscribedInfo.hasMoneySid = Boolean(item.sid === 20);
        }

        if (item.sid === 2) {
            subscribedInfo.hasMailSid = true;
        }

        return getSidsInfo(item, lang, tld);
    });

    subscribedInfo.subscribedTo.mainList = sidsData.filter(function(sid) {
        return sid.list === 'main';
    });
    subscribedInfo.subscribedTo.secondaryList = sidsData.filter(function(sid) {
        return sid.list === 'secondary';
    });

    return subscribedInfo;
}

function checkCaptcha(req, res, next) {
    const body = req.body;

    if (body.captcha === 'skip') {
        return next();
    }

    return req.api
        .captchaCheck({
            track_id: body.track_id,
            answer: body.captcha
        })
        .then(function(response) {
            const result = response && response.body;

            if (result.status === 'ok' && result.correct === true) {
                return next();
            }

            return res.json(response.body);
        })
        .catch(function(error) {
            res.json({status: 'error', error: {field: 'captcha', error: `captcha.${error[0].code}`}});
        });
}

router.get('/', enter);
router.post('/', enter);

exports.deleteCommit = deleteCommit;
exports.sendDeleteSms = sendSms;
exports.checkPhoneCode = checkPhoneCode;
exports.checkQuestion = checkQuestion;
exports.sendEmailCode = sendEmailCode;
exports.checkEmailCode = checkEmailCode;
exports.router = router;
