const PLog = require('plog');
const url = require('url');
const langSetup = require('./common/langSetup');
const apiSetup = require('./common/apiSetup');
const checkAuth = require('./common/checkAuth');
const getUatraitsData = require('./common/getUatraitsData');
const createState = require('./common/createState');
const getCountry = require('./common/getCountry');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const getMetrics = require('./common/getMetrics');
const showPlus = require('./common/plusGuard').showPlus;
const rumCounterSetup = require('./common/rumCounterSetup');
const validateRetpath = require('./common/validateRetpath');
const locs = require('../loc/locs.json');

const isNotProd = ['testing', 'development', 'rc'].includes(process.env.NODE_ENV);

const createCheckboxObj = (lang, labelKey, name, required) => ({
    label: locs[lang][labelKey],
    required: Boolean(required),
    name,
    checked: false
});

const setup = [
    apiSetup,
    checkAuth,
    getUatraitsData,
    langSetup,
    function getTrack(req, res, next) {
        return req.api
            .initTrack({
                type: 'authorize'
            })
            .then(function(response) {
                res.locals.track_id = (response.body && response.body.track_id) || '';
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.emailConfirm.getTrack')
                    .write(errors);

                return next();
            });
    },
    getCountry,
    showPlus,
    function(req, res, next) {
        if (res.locals.showPlus) {
            res.locals.blackboxPlus = true;
        }

        return next();
    },
    multiAuthAccountsSetup,
    (req, res, next) => {
        req.api
            .profileGetState()
            .then((response = {}) => {
                const {body = {}} = response;
                const {account = {}} = body;

                res.locals.account = account;

                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('email-confirm, getAccountInfo')
                    .write(error);
                return next();
            });
    },
    createState
];

const getCheckboxesByOrigin = (lang, origin) => {
    switch (origin) {
        case 'test':
            return isNotProd
                ? [
                      createCheckboxObj(lang, 'EmailConfirm.checkboxes.invest.spam', 'spam'),
                      createCheckboxObj(lang, 'EmailConfirm.checkboxes.invest.news', 'news', true)
                  ]
                : [];
        default:
            return [];
    }
};

const emailConfirmPage = [
    ...setup,
    getMetrics({header: 'Страница сбора email'}),
    validateRetpath,
    (req, res, next) => {
        const {query: {email} = {}} = req;
        const {store, track_id} = res.locals;
        const {common = {}, settings = {}} = store || {};

        if (!email) {
            return next();
        }

        req.api
            .sendConfirmationEmail({
                email,
                retpath: common.retpath,
                language: settings.language,
                uid: common.uid,
                track_id,
                validator_ui_url: common.emailValidatorUiUrl,
                is_safe: false,
                code_only: true
            })
            .catch((errors) => {
                res.locals.emailError = (errors && errors[0]) || 'internal';
            })
            .then(() => next());
    },
    (req, res, next) => {
        const {store, emailError, validRetpath} = res.locals;
        const {query: {origin, email} = {}} = req;

        store.emailConfirm = {
            step: email ? (emailError ? 'fail' : 'code') : 'add',
            email: email || '',
            code: '',
            error: emailError,
            origin,
            checkboxes: getCheckboxesByOrigin(res.locals.language, origin),
            isEmailExternal: Boolean(email),
            isEmailInProgress: Boolean(email),
            isLoading: false
        };

        if (validRetpath) {
            store.common.retpath = validRetpath;
        }

        next();
    },
    rumCounterSetup,
    (req, res) => res.render(`react.emailConfirm.${res.locals.language}.jsx`)
];

const emailConfirmExamplePage = [
    ...setup,
    getMetrics({header: 'Страница сбора email. Пример'}),
    (req, res, next) => {
        res.locals.store = res.locals.store || {};

        const {store} = res.locals;
        const {query: {origin, email} = {}, url: pathname, hostname, headers} = req;
        const protocol = headers['x-real-scheme'];

        const iframeSrc = url.format({
            protocol,
            hostname,
            pathname: '/profile/email/confirm',
            query: {
                origin,
                email,
                retpath: url.format({
                    protocol,
                    hostname,
                    pathname
                })
            }
        });

        store.emailConfirmExample = {
            iframeSrc
        };

        next();
    },
    rumCounterSetup,
    (req, res) => res.render(`react.emailConfirmExample.${res.locals.language}.jsx`)
];

exports.route = function(app) {
    app.get('/profile/email/confirm', emailConfirmPage);
    if (isNotProd) {
        app.get('/profile/email/confirm/example', emailConfirmExamplePage);
    }
};
