'use strict';

var url = require('url');
var express = require('express');
var router = express.Router();
var apiSetup = require('./common/apiSetup');
var validateCSRF = require('./common/validateCSRF.js');
var renderProfilePage = require('./profile.passport.v2').enter;

function apiRequest(apiMethod) {
    return function(req, res, next) {
        req.api[apiMethod](req.body)
            .then(function(result) {
                res.locals.result = result.body;
                next();
            })
            .catch(function(errors) {
                res.json({
                    status: 'error',
                    errors: errors
                });
            });
    };
}

function response(req, res) {
    res.json(res.locals.result);
}

function encodeEmailDomain(req, res, next) {
    if (req.body.email) {
        try {
            req.body.email = req._controller.encodeEmailToPunycode(req.body.email);
        } catch (e) {
            return res.json({
                status: 'error',
                errors: ['email.invalid']
            });
        }
    }

    next();
}

function getDecodedEmails(req, res, next) {
    var emails = req._controller.decodePunycodeEmails(res.locals.result.emails || {});

    res.locals.result.emails = emails;
    next();
}

router.use(apiSetup);
router.post('/list', validateCSRF, apiRequest('getEmails'), getDecodedEmails, response);
router.post('/delete', validateCSRF, encodeEmailDomain, apiRequest('deleteEmail'), response);
router.post(
    '/send-confirmation-email',
    validateCSRF,
    encodeEmailDomain,
    apiRequest('sendConfirmationEmail'),
    apiRequest('getEmails'),
    getDecodedEmails,
    response
);
router.post(
    '/confirm-by-code',
    validateCSRF,
    apiRequest('confirmEmailByCode'),
    apiRequest('getEmails'),
    getDecodedEmails,
    response
);
router.post(
    '/setup-confirmed',
    validateCSRF,
    encodeEmailDomain,
    apiRequest('setupConfirmedEmail'),
    apiRequest('getEmails'),
    getDecodedEmails,
    response
);
router.get(
    '/confirm-by-link',
    function(req, res, next) {
        var data = {
            key: req.query.key,
            uid: req.query.uid
        };
        var store = {};

        req.api
            .confirmEmailByLink(data)
            .then(function() {
                store.pageMessage = {
                    type: 'success',
                    message: 'emails.success'
                };

                res.locals.store = store;
                next();
            })
            .catch(function(errors) {
                const error = errors[0];

                if (error === 'sessionid.invalid') {
                    return req._controller.redirectToAuth();
                }

                if (error === 'sessionid.no_uid') {
                    return req._controller.redirectToLocalUrl({
                        pathname: '/auth',
                        query: {
                            uid: data.uid.replace(/[^0-9]/g, ''),
                            retpath: url.format(req._controller.getUrl())
                        }
                    });
                }

                if (error === 'email.already_confirmed') {
                    store.pageMessage = {
                        type: 'warning',
                        message: error
                    };
                } else {
                    store.pageMessage = {
                        type: 'error',
                        message: error
                    };
                }

                res.locals.store = store;
                next();
            });
    },
    (req, res, next) => {
        const store = res.locals.store;
        const retpath = decodeURIComponent(req.query.retpath);
        const defaultRetpath = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname,
            pathname: '/profile',
            query: req.query
        });

        req.api
            .validateRetpath({retpath: retpath})
            .then((values = {}) => {
                if (!values.body.retpath) {
                    store.common = {
                        retpath: defaultRetpath
                    };
                }

                store.common = {
                    retpath: values.body.retpath
                };
            })
            .catch(() => {
                store.common = {
                    retpath: defaultRetpath
                };
            })
            .finally(next);
    },
    renderProfilePage
);

exports.router = router;
