const url = require('url');
const config = require('../configs/current');
const PLog = require('plog');
const BillingApi = require('../lib/api/billing');
const FamilyPayApi = require('../lib/api/familypay');
const AntifraudApi = require('../lib/api/antifraud');
const tvm = require('../lib/tvm');
const isFamilyEnable = require('./common/isFamilyEnable');
const {urlFormat} = require('./common/urlFormat.js');
const {getFamilyOriginOptions} = require('./common/getFamilyOriginOptions');

const HAVE_PLUS_ATTRIBUTE = 1015;
const HAVE_FAMILY_ROLE_ATTRIBUTE = 1022;
const HAVE_SECURE_PHONE_ATTRIBUTE = 108;
const PAY_PENNY_COUNT = 100; // Потом когда будут сложные валюты, которые стоят не 100 аналога копеек, тут будет объект?
const PAY_NO_LIMIT_VALUE = 20 * 1000 * 1000;
const PAY_NO_LIMIT_VALUE_STARTS = PAY_NO_LIMIT_VALUE / 2;
const MAX_LIMIT = 100000;

const ENTRY_PAGES_MAP = {
    main: 'main',
    invite: 'invite.create',
    inviteRedesign: 'invite.redesign',
    inviteAccept: 'invite.accept',
    inviteWelcome: 'invite.welcome',
    limits: 'pay.limits',
    after3ds: 'pay.after3ds',
    limitsMember: 'pay.limitsMember',
    payCards: 'pay.cards',
    kiddish: 'kiddish',
    deleteMember: 'member.exclude',
    leaveMember: 'member.leave'
};

const PAY_SERVICES = ['dostavka', 'games', 'travel', 'market', 'drive', 'lavka', 'eats', 'taxi', 'kinopoisk'];

const KIDDISH_GENDER_MAP = {1: 'm', 2: 'f'};

const LIMIT_PAGE_ERRORS_WITH_STUB = {
    hasNotCard: 'hasNotCard',
    hasNotMembers: 'hasNotMembers'
};

const ORIGINS_WITHOUT_FAMILY_PAY_3DS_EXP = ['taxi_familycard'];

const errorHandler = (req, res) => (errors) =>
    res.json({status: 'error', code: Array.isArray(errors) ? errors[0] : 'family.error'});

const okHandler = (req, res, obj = {}) => () => res.json(Object.assign({status: 'ok'}, obj));

const createFamilyPayApi = (req, res) =>
    new FamilyPayApi(
        req.logID,
        res.locals.track_id,
        req.userTicket,
        req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.FAMILYPAY]
    );

const sortMembers = (members = []) => {
    const firstMember = members.find((member) => member.isYou);

    return firstMember ? [firstMember, ...members.filter((member) => !member.isYou)] : members;
};

const mapFamilyPayBalance = ({amount = 0} = {}, {value = 0} = {}) => value - amount;

const fetchFamilyInfo = async (req, res, {withUid = false, withAccount = false, withBBUsers = true} = {}) => {
    const controller = req._controller;
    const uid = Number(controller.getAuth().getUid());

    const {body = {}} = await req.api.getAccountWithFamilyInfo();

    const {
        account: {family_info: familyInfo},
        family_members: members = [],
        family_kids: kids = [],
        family_invites: invites = [],
        family_settings: {max_capacity: familyCapacity, max_kids_number: kiddishCapacity} = {}
    } = body;
    const adminUid = familyInfo && familyInfo.admin_uid;
    const familyId = familyInfo && familyInfo.family_id;

    const users = withBBUsers
        ? (
              await controller.getAuth().userInfo({
                  uid: members.map(({uid}) => uid).join(',') || uid,
                  attributes: [HAVE_FAMILY_ROLE_ATTRIBUTE],
                  returnAllUsers: true,
                  getphones: 'all',
                  phone_attributes: [HAVE_SECURE_PHONE_ATTRIBUTE]
              })
          ).map((i) => ({
              uid: Number(i.uid.value),
              attributes: i.attributes,
              phonesAttrs: i.phones.map((j) => j.attributes)
          }))
        : [];
    const hasFamilySubscription = users.some(({attributes = {}}) => Boolean(attributes[HAVE_FAMILY_ROLE_ATTRIBUTE]));
    const result = {
        members: sortMembers(
            members.map((member) => {
                const user = users.find((i) => i.uid === member.uid);

                const newMember = {
                    placeId: member.place_id,
                    name: member.display_name,
                    avatar: member.default_avatar,
                    isYou: member.uid === uid,
                    isAdmin: Boolean(adminUid) && member.uid === adminUid,
                    hasPlus: Boolean(member.has_plus)
                };

                if (withUid) {
                    newMember.uid = member.uid;
                }

                if (withBBUsers) {
                    newMember.isAbleToUsePay = user.phonesAttrs.some((attr) =>
                        Boolean(attr[HAVE_SECURE_PHONE_ATTRIBUTE])
                    );
                }

                return newMember;
            })
        ),
        kids: kids.map((kiddish) => ({
            uid: kiddish.uid,
            placeId: kiddish.place_id,
            name: kiddish.display_name && kiddish.display_name.name,
            avatar: kiddish.display_name && kiddish.display_name.default_avatar,
            gender: kiddish.person && KIDDISH_GENDER_MAP[kiddish.person.gender],
            birthday: kiddish.person && kiddish.person.birthday,
            rating: kiddish.content_rating_class,
            videoRating: kiddish.video_content_rating_class,
            musicRating: kiddish.music_content_rating_class,
            isYou: false,
            isAdmin: false,
            hasPlus: hasFamilySubscription
        })),
        invites: invites.map((invite) => ({
            inviteId: invite.invite_id,
            inviteMethod: invite.send_method,
            inviteContact: invite.contact
        })),
        familyId,
        hasFamilySubscription,
        kiddishCapacity,
        familyCapacity
    };

    if (familyId) {
        const api = createFamilyPayApi(req, res);
        const familyPayInfo = await api.fetchFamilyByIdSingleLimit(familyId).catch(() => ({error: true}));
        const {properties: {cardInfo = {}} = {}, userRestrictions = [], error, adminUid} = familyPayInfo;
        const adminInfo = userRestrictions.find((member) => member.uid === adminUid) || {};
        const {limit: {value: adminValue} = {}} = adminInfo;

        Object.assign(result, {
            pay: error
                ? {}
                : {
                      defaultValue: adminValue / PAY_PENNY_COUNT,
                      cardInfo,
                      usersInfo: userRestrictions
                          .map(
                              ({
                                  allowedServices,
                                  limit,
                                  limitCurrency,
                                  expenses,
                                  uid,
                                  allowAllServices,
                                  enabled
                              } = {}) => {
                                  const member = members.find((member) => member.uid === uid);

                                  if (!member) {
                                      // Участник удален из семьи но все еще не удален из familypay?
                                      return null;
                                  }

                                  const fullValue = limit.value / PAY_PENNY_COUNT;
                                  const limitMode =
                                      fullValue >= PAY_NO_LIMIT_VALUE_STARTS ? 'NOLIMIT' : limit.limitMode;
                                  const value = limitMode === 'NOLIMIT' ? 0 : fullValue;

                                  return {
                                      allowedServices,
                                      limit: {...limit, limitMode, value},
                                      currency: limitCurrency,
                                      allowAllServices: Boolean(allowAllServices),
                                      isEnabled: Boolean(enabled),
                                      balance: mapFamilyPayBalance(expenses, limit) / PAY_PENNY_COUNT,
                                      placeId: member.place_id
                                  };
                              }
                          )
                          .filter(Boolean)
                  }
        });
    }

    if (withAccount) {
        Object.assign(result, {account: body.account});
    }

    return result;
};

const checkFamilyExist = (req, res) =>
    fetchFamilyInfo(req, res, {withBBUsers: false}).then(({familyId}) => {
        if (!familyId) {
            return req.api.createFamily();
        }
    });

const getProductPrice = (product) => (product.introPrice || product.price).amount;

const getCheapestProduct = (products) => products.sort((i, j) => getProductPrice(i) - getProductPrice(j))[0];

const getCheapestIntroProduct = (products) => {
    const introProducts = products.filter((i) => i.introPeriodDuration);

    return getCheapestProduct(introProducts.length ? introProducts : products);
};

const getProduct = (products) => {
    const familyProducts = products.filter((i) => i.available && i.family);
    const trialProducts = familyProducts.filter((i) => i.trialDuration);

    return getCheapestIntroProduct(trialProducts.length ? trialProducts : familyProducts);
};

const getKiddishInfoFromBody = ({
    name,
    avatar,
    birthday,
    rating,
    musicRating,
    videoRating,
    firstname,
    gender,
    lastname,
    uid
}) => ({
    display_name: name,
    avatar_id: avatar,
    kiddish_uid: uid,
    content_rating_class: rating,
    music_content_rating_class: musicRating,
    video_content_rating_class: videoRating,
    birthday,
    firstname, // Не используется, но есть в доке
    lastname, // Не используется, но есть в доке
    gender
});

const postBindCard = async (req, res, trackId, cardId, retpath = '', origin = '') => {
    try {
        const controller = req._controller;
        const uid = Number(controller.getAuth().getUid());

        const {body: {billing_response: {payment_methods = {}} = {}} = {}} = await req.api.billingListPaymentMethods({
            track_id: trackId
        });
        const {number = '', system = '', currency = 'RUB'} =
            Object.values(payment_methods).find((i) => i.card_id === cardId) || {};

        if (!number || !system) {
            return {state: 'error', error: 'family.pay.no-card'};
        }

        if (
            req._controller.hasExp('profile-family-pay-3ds-exp') ||
            ORIGINS_WITHOUT_FAMILY_PAY_3DS_EXP.includes(origin)
        ) {
            const {account: {phones = {}} = {}} = await fetchFamilyInfo(req, res, {
                withAccount: true,
                withBBUsers: false
            });
            const {login_id: loginId} = await controller.getAuth().sessionID({get_login_id: 'yes'});

            const securePhoneId = Object.keys(phones).filter((phoneId) => phones[phoneId].bound)[0];
            const antifraudApi = new AntifraudApi(
                req.logID,
                req.serviceTickets && req.serviceTickets[tvm.SERVICE_ALIASES.ANTIFRAUD]
            );

            const {action, tags: [{url: redirectUrl} = {}] = []} = await antifraudApi.score({
                extId: trackId,
                uid,
                cardId,
                phone: phones[securePhoneId].number.original,
                loginId,
                yandexUid: req.cookies['yandexuid'],
                retpath: urlFormat({
                    protocol: 'https',
                    hostname: req.hostname,
                    pathname: '/profile/family/after3ds',
                    query: {cardId, retpath, origin}
                }),
                paymentType: 'family-card-binding',
                serviceId: 'family-card-binding'
            });

            // Нельзя ни под каким предлогом
            if (action === 'DENY') {
                return {state: 'error', error: 'family.pay.deny', retpath};
            }

            // Можно после челленджа
            if (action === 'ALLOW' && redirectUrl) {
                return {state: 'challenge', redirectUrl, retpath};
            }
        }

        // Можно
        await checkFamilyExist(req, res);
        const {familyId, pay: {cardInfo: {cardId: oldCardId, bound} = {}} = {}} = await fetchFamilyInfo(req, res, {
            withBBUsers: false
        });

        const familyPayApi = createFamilyPayApi(req, res);

        if (oldCardId) {
            const isOnlyBoundUpdate = !bound && oldCardId === cardId;

            await familyPayApi.updateCardSingleLimit(
                familyId,
                isOnlyBoundUpdate
                    ? {bound: true}
                    : {
                          cardId,
                          maskedNumber: number,
                          paySystem: system,
                          currency,
                          bound: true
                      }
            );
            return {state: 'bind', retpath};
        }

        await familyPayApi.bindCardSingleLimit(familyId, uid, cardId, number, system, currency);

        return {state: 'bind', retpath};
    } catch (err) {
        return {state: 'error', error: 'family.error', retpath};
    }
};

const fetchInviteInfo = (req, res, inviteId) =>
    req.api.getFamilyInviteInfo(inviteId).then(({body: {issuer_uid: uid, family_id: familyId}}) =>
        req._controller
            .getAuth()
            .userInfo({
                uid,
                regname: 'yes',
                attributes: [HAVE_PLUS_ATTRIBUTE]
            })
            .then(({display_name: {name, avatar: {default: avatar}}, attributes}) => ({
                admin: {
                    avatar,
                    name,
                    familyId,
                    isUser: true,
                    hasPlus: Boolean(attributes[HAVE_PLUS_ATTRIBUTE])
                }
            }))
    );

exports.getFamily = (req, res) =>
    fetchFamilyInfo(req, res, req.body)
        .then((result) => okHandler(req, res, result)())
        .catch(errorHandler(req, res));

exports.getInviteInfo = (req, res) =>
    fetchInviteInfo(req, res, req.body.inviteId)
        .then(({admin}) => okHandler(req, res, {admin})())
        .catch(errorHandler(req, res));

exports.confirmInvite = (req, res) =>
    req.api
        .confirmFamilyInvite(req.body.inviteId)
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));

exports.createInvite = (req, res) => {
    const {contact = ''} = req.body;
    const inviteRequest = {
        sms_phone_country: 'ru'
    };

    let inviteMethod = '';

    if (contact) {
        inviteMethod = contact.toString().includes('@') ? 'email' : 'sms_phone';
        inviteRequest[inviteMethod] = contact;
        if (inviteMethod === 'sms_phone') {
            inviteMethod = 'sms';
        }
    }

    checkFamilyExist(req, res)
        .then(() => req.api.createFamilyInvite(inviteRequest))
        .then(({body: {invite_id: inviteId}}) => {
            okHandler(req, res, {inviteId, inviteContact: contact, inviteMethod})();
        })
        .catch((errors) => {
            if (errors && errors[0] === 'email.invalid') {
                return errorHandler(req, res)(['sms_phone.invalid']);
            }
            return errorHandler(req, res)(errors);
        });
};

exports.deleteFamily = (req, res) =>
    req.api
        .deleteFamily()
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));

exports.cancelInvite = (req, res) =>
    req.api
        .cancelFamilyInvite(req.body.inviteId)
        .then(okHandler(req, res))
        .catch((errors) => {
            if (errors && errors[0] === 'family.invalid_invite') {
                return errorHandler(req, res)(['family.cant_remove_invite']);
            }
            return errorHandler(req, res)(errors);
        });

exports.excludeMember = (req, res) =>
    req.api
        .deleteFamilyMember(req.body.placeId)
        .then(okHandler(req, res))
        .catch((errors) => {
            if (errors && errors[0] === 'family.is_member_other') {
                return errorHandler(req, res)(['family.is_not_a_member']);
            }
            return errorHandler(req, res)(errors);
        });

exports.leaveFamily = (req, res) =>
    req.api
        .leaveFamily()
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));

exports.getSubscriptionOffer = async (req, res) => {
    const controller = req._controller;
    const uid = controller.getAuth().getUid();
    const {language} = req.body;

    if (!uid || !language) {
        return okHandler(req, res)();
    }

    try {
        const {result: {nativeProducts: products} = {}} = await new BillingApi(req.logID).getNativeProducts({
            uid,
            ip: req.ip,
            target: 'passport-family',
            language
        });

        const {buttonText: text, buttonAdditionalText: subText} = getProduct(products || []) || {};

        okHandler(req, res, {text, subText})();
    } catch {
        okHandler(req, res)();
    }
};

exports.createKiddish = (req, res) =>
    checkFamilyExist(req, res)
        .then(() => req.api.createKiddish(getKiddishInfoFromBody(req.body)))
        .then(okHandler(req, res))
        .catch((errors) => {
            if (errors && errors[0] === 'family.max_capacity') {
                return errorHandler(req, res)(['kiddish.max_capacity']);
            }
            return errorHandler(req, res)(errors);
        });

exports.editKiddish = (req, res) =>
    req.api
        .editKiddish(getKiddishInfoFromBody(req.body))
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));

exports.removeKiddish = (req, res) =>
    req.api
        .removeKiddish(getKiddishInfoFromBody(req.body))
        .then(okHandler(req, res))
        .catch(errorHandler(req, res));

const getPage = (entry, countryId) =>
    isFamilyEnable(countryId) ? ENTRY_PAGES_MAP[entry] || ENTRY_PAGES_MAP.main : 'main.stub';

const putFamilyInStoreCommon = async (req, res, entry, {withAccount = false, withBBUsers = true} = {}) => {
    const familyInfo = await fetchFamilyInfo(req, res, {withUid: true, withAccount, withBBUsers});
    const {
        members = [],
        kids = [],
        familyCapacity,
        kiddishCapacity,
        invites = [],
        pay = {},
        familyId,
        hasFamilySubscription,
        account = null
    } = familyInfo;

    if (withAccount && account) {
        const locals = res.locals;
        const tld = req._controller.getTld();
        const yandexuid = res._yandexuid && res._yandexuid.replace(/[^0-9]/g, '');
        const handlers = [req._controller.getUatraits()];
        const links = config.links[tld] || config.links.ru || {};
        const defaultAccount = (locals.accounts && locals.accounts.defaultAccount) || {};
        const plusInfo = defaultAccount.plus || {};
        const retpath = (req.query && (req.query.url || req.query.retpath)) || null;
        const embeddedauth = config.paths.embeddedauth && config.paths.embeddedauth.replace('%tld%', tld);
        const experiments = Object.assign(
            {},
            {
                flags: [],
                flagsString: '',
                boxes: '',
                encodedBoxes: ''
            },
            res.locals.experiments
        );

        handlers.push(req.api.validateRetpath({retpath}));

        locals.reactPage = 'profile.passportv2';

        const {paths = {}, version} = config;
        const {static: staticPath} = paths;
        const {userType = {}} = locals;
        const {metricsUserType} = userType;

        locals.store = {
            dashboard: {
                plus: Object.assign(
                    {},
                    {
                        country: locals.country,
                        allowed: locals.showPlus,
                        enabled: false,
                        enabledTime: 0,
                        trialUsedTime: 0,
                        subscriptionStoppedTime: 0,
                        subscriptionExpireTime: 0,
                        nextChargeTime: 0,
                        expireDate: ''
                    },
                    plusInfo
                ),
                errors: {},
                messages: {},
                isLoading: {},
                experiments: {}
            },
            settings: {
                isNewLayout: !locals.isLite,
                location: req.path,
                plusHost: config.paths && config.paths.plus && `//${config.paths.plus.replace('%tld%', tld)}`,
                host: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname
                }),
                avatar: config.paths.avatar || {},
                embeddedauth,
                links,
                help: config.paths.help || {},
                tld,
                language: locals.language,
                ua: {},
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                billingUpdateStatus: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: config.paths.billingUpdateStatusPath
                }),
                staticPath,
                accountsUrl: config.paths.accountsUrl,
                version,
                metricsUserType
            },
            common: {
                v2: !locals.isLite, // PASSP-32231
                origin: (req.query || {}).origin,
                process_uuid: res.locals.process_uuid,
                canChangePassword: account.can_change_password,
                edit: locals.edit || '',
                showRegPopup: false,
                hasComponentsForNav: true,
                actionForRepeat: null,
                uid: account.uid,
                yandexuid,
                // при загрузке срабатывает POP event и уменьшает историю на 1, а так тут должен быть 0
                historyOnPassport: 1,
                track_id: locals.track_id,
                retpath: null,
                backpath: null,
                defaultPage: url.format({
                    pathname: '/profile',
                    query: req.query
                }),
                isPDD: account.domain !== undefined,
                isWSUser: locals.isWSUser,
                dev: config.dev,
                experiments,
                isYandexoid: locals.isYandexoid,
                isYandex: locals.isYandex,
                locationCountryId: locals.countryId,
                isFamilyEnable: isFamilyEnable(locals.countryId),
                isSupportCenterEnable:
                    ((experiments.flags && experiments.flags.includes('support-window-enable')) ||
                        locals.isYandex ||
                        locals.isYandexoid) &&
                    locals.ua &&
                    locals.ua.BrowserName !== 'MSIE'
            },
            social: {
                brokerPath: config.paths.broker
            },
            person: Object.assign({}, account.person || {}, {
                avatarId: (account.display_name && account.display_name.default_avatar) || '',
                displayName: defaultAccount.displayName || ''
            }),
            phones: {
                restore: [],
                other: [],
                completeSms2fa: req.query.completeSms2fa && req.query.completeSms2fa === '1'
            },
            billing: {},
            form: {},
            metrics: {
                header: 'Webview семьи',
                experiments: experiments.encodedBoxes || ''
            },
            monitoring: {
                page: 'passport.v2'
            }
        };

        if (account.phones) {
            Object.keys(account.phones).forEach(function(code) {
                const phone = account.phones[code];

                if (phone.bound) {
                    locals.store.phones[phone.secured ? 'restore' : 'other'].push({
                        id: phone.id,
                        number: phone.number.masked_international,
                        isDefault: phone.is_default,
                        isAlias: phone.is_alias
                    });
                }
            });
        }

        await Promise.all(handlers)
            .then((response) => {
                const [uatraits = {}, validatedRetpath = {}] = response;

                locals.store.settings.ua = uatraits;
                locals.store.settings.isPhone = (uatraits.isMobile || uatraits.isTouch) && !uatraits.isTablet;
                locals.store.settings.isTouch = uatraits.isMobile || uatraits.isTouch;

                if (validatedRetpath.body && validatedRetpath.body.retpath) {
                    locals.store.common.retpath = validatedRetpath.body.retpath;
                }
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.v2')
                    .write(err);
            })
            .finally(() => {
                delete locals.account;
                delete locals.accounts;
            });
    }

    const store = res.locals.store || {};
    const {settings} = store;
    const isMobile = settings.ua && settings.ua.isMobile && !settings.ua.isTablet;

    res.locals.store.family = {
        hasFamilySubscription: false,
        currentSlot: {isEmpty: true},
        currentSlotUserIndex: 0,
        inviteTarget: isMobile ? 'link' : 'smsOrMail',
        familyId: null,
        hasFamily: false,
        loading: false,
        isOneEmptyCard: true,
        error: null,
        inviteId: null,
        inviteContact: null,
        inviteConfirmed: false,
        isAdminAccess: false,
        familyCapacity: 0,
        kiddishCapacity: 0,
        slots: [],
        memberSlots: [],
        membersAndEmptySlots: [],
        inviteSlots: [],
        kiddishSlots: [],
        isFamilyOfferLoading: false,
        pageInfo: {page: getPage(entry, res.locals.countryId)},
        notification: {
            isVisible: false,
            text: '',
            iconType: '',
            theme: 'default'
        }
    };

    // eslint-disable-next-line no-unused-vars
    const memberSlots = members.map(({uid, ...member}) => ({...member, isUser: true}));
    const kiddishSlots = kids.map((kid) => ({...kid, isKiddish: true}));
    const membersWithoutAdmin = memberSlots.filter((slot) => !slot.isAdmin);
    const inviteSlots = invites.map((invite) => ({...invite, isInvite: true}));
    const slots = memberSlots.concat(inviteSlots);
    const yourSlot = memberSlots.find((member) => member.isYou);
    const adminSlot = memberSlots.find((member) => member.isAdmin);
    const isAdminAccess = !familyId || Boolean(adminSlot.isYou);

    res.locals.store.family = {
        ...res.locals.store.family,
        hasFamilySubscription,
        memberSlots,
        membersAndEmptySlots: memberSlots.concat({isEmpty: true}),
        inviteSlots,
        slots: isAdminAccess && slots.length < familyCapacity ? slots.concat({isEmpty: true}) : slots,
        kiddishSlots: kiddishSlots.length < kiddishCapacity ? kiddishSlots.concat({isEmpty: true}) : kiddishSlots,
        yourSlot,
        currentSlot: {isEmpty: true},
        hasMembers: Boolean(membersWithoutAdmin.length),
        currentSlotLimitInfo: {},
        currentSlotUserIndex: 0,
        familyId,
        isAdminAccess,
        hasFamily: Boolean(familyId),
        familyCapacity,
        kiddishCapacity,
        slotsCountByGroup: {
            member: memberSlots.length,
            kiddish: kiddishSlots.length
        },
        membersWithoutAdminSlots: membersWithoutAdmin,
        membersWithoutAdminAndEmptySlots: membersWithoutAdmin.concat({isEmpty: true}),
        adminSlot
    };

    const {cardInfo: {bound} = {}} = pay;

    if (bound) {
        const usersInfo = (pay.usersInfo || []).map((userInfo) => ({
            ...userInfo,
            isCardActualyEnabled:
                userInfo.isEnabled &&
                (userInfo.allowAllServices ||
                    PAY_SERVICES.some((service) => userInfo.allowedServices.includes(service)))
        }));

        res.locals.store.family = {
            ...res.locals.store.family,
            yourSlotLimitInfo: usersInfo.find((userInfo) => userInfo.placeId === yourSlot.placeId),
            pay: {
                ...pay,
                usersInfoByPlaceId: usersInfo.reduce((acc, userInfo) => {
                    acc[userInfo.placeId] = userInfo;

                    return acc;
                }, {})
            },
            payLimitsForm: usersInfo,
            payFormLimitCurrency: pay.cardInfo && pay.cardInfo.currency
        };
    }

    return familyInfo;
};

exports.putFamilyInStore = (entry, options = {}) => async (req, res, next) => {
    const {withAccount = false, isAm = false, withBBUsers = true} = options;
    const origin = req.query && req.query.origin;
    const familyInfo = await putFamilyInStoreCommon(req, res, entry, {withAccount, withBBUsers});
    const {store = {}} = res.locals;
    const {isTouch = false} = store.settings;
    const originOptions = getFamilyOriginOptions(origin) || null;

    if (originOptions) {
        const {
            origin,
            withPayOrigin,
            isWebview,
            isLite,
            hideServices,
            hideFamilyInvite,
            hideWebviewHeader,
            hideKiddish,
            hideLimitsPageBackButton,
            hideSupport,
            hidePayCardsPageBackButton,
            hideInvitePageBackButton,
            hideMembersBackButton,
            hideInviteCancelBackButton,
            hideFormLimitsUsers,
            limitsPageBottomPadding,
            notRedirectWithoutRetpath,
            notRedirectAfterInvite,
            isCrossBackButton
        } = originOptions;

        store.settings.isLite = isLite;
        store.settings.isNewLayout = !isLite;

        store.family = {
            ...store.family,
            isWebview,
            origin,
            withPayOrigin,
            hideServices,
            hideFamilyInvite,
            hideWebviewHeader,
            hideKiddish,
            hideLimitsPageBackButton,
            hideSupport,
            hidePayCardsPageBackButton,
            hideInvitePageBackButton,
            hideMembersBackButton,
            hideInviteCancelBackButton,
            hideFormLimitsUsers,
            limitsPageBottomPadding,
            notRedirectWithoutRetpath,
            notRedirectAfterInvite,
            isCrossBackButton
        };
    }

    if (isAm) {
        store.family.withPayOrigin = true;
    }

    if (['limits', 'limitsMember'].includes(entry)) {
        const {family = {}} = store;
        const {members = []} = familyInfo;
        const {yourSlot = {}, membersWithoutAdminSlots = [], pay = {}} = family;
        const {cardInfo: {bound} = {}} = pay;
        const {isAbleToUsePay} = yourSlot;
        const showLimitsPageStub = originOptions && originOptions.showLimitsPageStub;
        const preFilledLimitValue = req.query && Number(req.query.member_add_limit);
        const preFilledLimitMode = req.query && String(req.query.member_limit_mode);
        const uidFromParams = req.query && Number(req.query.member_uid);
        const hasLimitPreFilled = preFilledLimitValue && preFilledLimitMode && uidFromParams;

        const needRedirectMemberToFamilyPush =
            entry === 'limitsMember' && isAm && (!isAbleToUsePay || !bound || !pay.usersInfo);
        const needRedirectAdminToFamilyPush =
            entry === 'limits' && isAm && (!isAbleToUsePay || !bound || !membersWithoutAdminSlots.length);

        if (needRedirectMemberToFamilyPush || needRedirectAdminToFamilyPush) {
            return req._controller.redirectToLocalUrl({
                pathname: '/am/push/family',
                query: req.query
            });
        }

        if (showLimitsPageStub) {
            let errorKey = null;

            if (!pay.usersInfo || !bound) {
                errorKey = 'hasNotCard';
            }
            if (!membersWithoutAdminSlots.length) {
                errorKey = 'hasNotMembers';
            }

            if (errorKey) {
                store.family.hasLimitsError = true;
                store.family.isLimitsPage = true;
                store.family.limitsError = LIMIT_PAGE_ERRORS_WITH_STUB[errorKey];

                return next();
            }
        }

        if (!pay.usersInfo || !membersWithoutAdminSlots.length) {
            return next('route');
        }

        let [currentMemberSlot, currentSlotUserIndex, hasPrefilledMember] = [null, 1, false];

        members.forEach((member, i) => {
            if (member.uid === uidFromParams) {
                currentMemberSlot = {...member, isUser: true};
                currentSlotUserIndex = i;
                hasPrefilledMember = true;
            }
        });

        const defaultAdminAccess = entry === 'limits';
        const yourSlotLimitInfo = pay.usersInfo.find((userInfo) => userInfo.placeId === yourSlot.placeId);
        const currentSlot = defaultAdminAccess ? currentMemberSlot || membersWithoutAdminSlots[0] : yourSlot;
        const currentSlotLimitInfo = pay.usersInfo.find((userInfo) => userInfo.placeId === currentSlot.placeId);

        let updatedCurrentSlotLimitInfo = null;

        if (hasLimitPreFilled) {
            const {limit: {value: oldValue} = {}} = currentSlotLimitInfo;
            const newValue =
                preFilledLimitMode === 'NOLIMIT' ? oldValue : Math.min(oldValue + preFilledLimitValue, MAX_LIMIT);

            updatedCurrentSlotLimitInfo = {
                ...currentSlotLimitInfo,
                limit: {...currentSlotLimitInfo.limit, value: newValue, limitMode: preFilledLimitMode}
            };
        }

        store.family = {
            ...store.family,
            currentSlot,
            currentSlotLimitInfo,
            currentSlotUserIndex,
            currentSlotFormLimitInfo: updatedCurrentSlotLimitInfo || currentSlotLimitInfo,
            isAdminAccess: defaultAdminAccess,
            isLimitsPage: true,
            yourSlotLimitInfo,
            hasLimitPreFilled,
            preFilledLimitValue,
            preFilledLimitMode,
            hasPrefilledMember
        };
    }

    if (entry === 'after3ds') {
        store.family.is3dsSuccess = req.query.status !== 'error';
        store.family.cardId3ds = req.query.cardId;
        store.family.pageInfo = isTouch
            ? {page: 'pay.after3ds', modal: null, isModalOpened: false}
            : {page: 'main', modal: 'pay.after3ds', isModalOpened: true};
    }

    if (entry === 'invite') {
        store.family.isAdminAccess = true;
        store.family.isInvitePage = true;
        store.family.pageInfo = {page: getPage('inviteRedesign', res.locals.countryId)};
    }

    if (entry === 'invite.confirm') {
        return fetchInviteInfo(req, res, req.params.inviteId)
            .then(({admin}) => {
                store.family = {
                    ...store.family,
                    pageInfo: {page: getPage('inviteAccept', res.locals.countryId)},
                    isInvitePage: true,
                    inviteId: req.params.inviteId,
                    currentSlot: admin
                };
                next();
            })
            .catch((errors = []) => {
                store.family = {
                    ...store.family,
                    isInvitePage: true,
                    error: errors[0],
                    pageInfo: {page: 'error'}
                };
                next();
            });
    }

    if (entry === 'inviteWelcome') {
        const {family: {isAdminAccess} = {}} = store;

        store.family.pageInfo = isAdminAccess
            ? {page: getPage('main', res.locals.countryId)}
            : {page: getPage(entry, res.locals.countryId)};
    }

    if (entry === 'payCards') {
        store.family.isPayCardsPage = true;
        store.family.pageInfo = isTouch
            ? {page: 'pay.cards', modal: null, isModalOpened: false}
            : {page: 'main', modal: 'pay.cards', isModalOpened: true};
    }

    if (['deleteMember', 'leaveMember'].includes(entry)) {
        const {family = {}} = store;
        const {members = []} = familyInfo;
        const {isAdminAccess} = family;
        const uidFromParams = req.query && Number(req.query.member_uid);
        const {blackbox: {default_uid: defaultUid} = {}} = req;

        if (entry === 'leaveMember' && Number(defaultUid) !== uidFromParams) {
            return next('route');
        }

        let currentMemberSlot = null;

        members.forEach((member) => {
            if (member.uid === uidFromParams) {
                currentMemberSlot = {...member, isUser: true};
            }
        });

        if (!currentMemberSlot) {
            return next('route');
        }

        const curPage =
            entry === 'leaveMember' || (isAdminAccess && currentMemberSlot.isAdmin) ? 'member.leave' : 'member.exclude';

        const pageInfo = isTouch
            ? {page: curPage, modal: null, isModalOpened: false}
            : {page: 'main', modal: curPage, isModalOpened: true};

        store.family = {
            ...store.family,
            currentSlot: currentMemberSlot,
            isMemberPage: true,
            pageInfo
        };
    }

    if (entry === 'inviteSlot') {
        const {family: {inviteSlots = {}} = {}} = store;
        const inviteIdFromParams = req.query && req.query.invite_id;

        const currentInviteSlot = inviteSlots.find((slot) => slot.inviteId === inviteIdFromParams);

        if (!currentInviteSlot) {
            return next('route');
        }

        const pageInfo = {page: 'slot.info', modal: null, isModalOpened: false};

        store.family = {
            ...store.family,
            currentSlot: currentInviteSlot,
            isMemberPage: true,
            isInviteCancelPage: true,
            pageInfo
        };
    }

    next();
};

exports.payUnbind = async (req, res) => {
    const {familyId} = await fetchFamilyInfo(req, res, {withBBUsers: false});

    createFamilyPayApi(req, res)
        .updateCardBound(familyId, false)
        .then(okHandler(req, res), errorHandler(req, res));
};

exports.payBind = async (req, res) => {
    const {cardId, track_id: trackId, retpath = '', origin = ''} = req.body;

    return postBindCard(req, res, trackId, cardId, retpath, origin).then(
        (result = {}) =>
            result.state === 'error' ? errorHandler(req, res)(result.error) : okHandler(req, res, result)(),
        errorHandler(req, res)
    );
};

exports.paySaveLimits = async (req, res) => {
    const {familyId = '', members = []} = await fetchFamilyInfo(req, res, {withUid: true});
    const {limits = {}} = req.body;
    const {users = []} = JSON.parse(limits);
    const newLimits = users.map(
        ({placeId, currency, limit: {limitMode, value} = {}, allowedServices = [], isEnabled, allowAllServices}) => {
            const newLimitMode = limitMode === 'NOLIMIT' ? 'DAY' : limitMode;
            const newValue = limitMode === 'NOLIMIT' ? PAY_NO_LIMIT_VALUE : Number(value);

            return {
                allowAllServices,
                uid: members.find((member) => member.placeId === placeId).uid,
                limitCurrency: currency,
                limit: {limitMode: newLimitMode, value: newValue * PAY_PENNY_COUNT},
                enabled: isEnabled,
                allowedServices
            };
        }
    );

    createFamilyPayApi(req, res)
        .updateUsersSingleLimit(familyId, newLimits)
        .then(okHandler(req, res), errorHandler(req, res));
};

exports.payResetUserExpenses = async (req, res) => {
    const {placeId = ''} = req.body;
    const {members = []} = await fetchFamilyInfo(req, res, {withUid: true});

    createFamilyPayApi(req, res)
        .resetUserExpenses(members.find((member) => member.placeId === placeId).uid)
        .then(okHandler(req, res), errorHandler(req, res));
};
