const url = require('url');
const apiSetup = require('./common/apiSetup');
const config = require('../configs/current');
const langs = config.langs;
const locs = require('../loc/auth.json');
const locsJournal = require('../loc/profilejournal.json');
const _ = require('lodash');
const PLog = require('plog');
const when = require('when');
const punycode = require('punycode');
const momentTimezone = require('moment-timezone');
const providers = require('../lib/providers/providers.json').providers;
const countries = require('../lib/geo/countries.json');
const timezones = require('../lib/geo/timezones.json');
const validateCSRF = require('./common/validateCSRF.js');
const dheaderSetup = require('./common/dheaderSetup');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const checkAuth = require('./common/checkAuth');
const OAuthAPI = require('../lib/api/oauth');
const periods = ['7', '30', '180'];
const lookup = require('../lib/getLookup.js');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');

const langSetup = [
    function(req, _res, next) {
        if (!req.blackbox && req.body.language) {
            req.blackbox = {
                dbfields: {
                    'userinfo.lang.uid': req.body.language
                }
            };
        }
        next();
    },
    function(req, res, next) {
        var controller = req._controller;

        controller
            .getLanguage()
            .then(function(lang) {
                res.locals.language = lang;
            })
            .catch(function(error) {
                res.locals.language = 'ru';

                PLog.warn()
                    .logId(req.logID)
                    .type('profile.journal', 'getLanguage')
                    .write(error);
            })
            .done(function() {
                return next();
            });
    },
    function(req, res, next) {
        const language = langs[langs.indexOf(res.locals.language)] || 'ru';

        res.loc = locs[language];

        if (req.api) {
            req.api.language(language);
        }
        next();
    }
];

const setup = [
    langSetup,
    function(req, res, next) {
        req._controller
            .getAuth()
            .sessionID({
                attributes: '1003' // password.2fa_on, see https://beta.wiki.yandex-team.ru/passport/dbmoving
            })
            .then(function(response) {
                if (!response) {
                    return next();
                }
                res.locals.is2FAOn = Boolean(req._controller.getAuth().getAttribute(1003));
                res.locals.yu = req.cookies.yandexuid;
                return next();
            })
            .catch((error) => {
                if (error && error.code === 'need_resign') {
                    return req._controller.getAuth().resign();
                }

                if (error) {
                    next(error);
                }
            });
    },
    apiSetup,
    function(req, res, next) {
        const yaMapLocales = {
            com: 'US',
            ru: 'RU',
            ua: 'UA',
            'com.tr': 'TR'
        };
        const tld = req._controller.getTld() || 'ru';
        const locale = `${res.locals.language}_${yaMapLocales[tld] || yaMapLocales.ru}`;

        res.locals.mapApiSrc = config.paths.mapsAPI.replace('%lang%', locale);
        return next();
    },
    returnUserTimezone
];

exports.routes = {};
exports.route = function(app) {
    const routes = this.routes;

    app.get('/profile/journal', routes.main);
    app.all('/profile/journal/pages', routes.journal);
    app.get('/profile/journal/actions', routes.actions);
};

exports.routes.main = [
    checkAuth,
    setup,
    getAllScopes,
    preloadAuthEvents,
    getAuthLog,
    getTokensInfo,
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    function addUserTypeInfo(req, res, next) {
        const {accounts = {}} = res.locals;
        const defaultAccount = accounts.defaultAccount || {};
        const {isNeoPhonish, isSocial, is_liteUser, havePassword} = defaultAccount;

        res.locals.userTypeInfo = {
            isSpecialTypeUser: isNeoPhonish || isSocial || (is_liteUser && !havePassword),
            isNeoPhonish,
            isSocial,
            isLite: is_liteUser && !havePassword
        };

        return next();
    },
    function getAccountInfo(req, res, next) {
        const {userTypeInfo = {}} = res.locals;

        if (!userTypeInfo.isSpecialTypeUser) {
            return next();
        }

        const params = {
            need_display_name_variants: false,
            need_phones: true,
            need_emails: true,
            need_social_profiles: true,
            need_question: false,
            need_additional_account_data: false
        };

        req.api
            .profileGetState(params)
            .then((response = {}) => {
                const account = (response.body && response.body.account) || {};
                const {isNeoPhonish, isSocial, isLite} = userTypeInfo;

                if (isNeoPhonish && account.phones) {
                    const securedPhone = Object.values(account.phones).find((item) => item.secured);

                    userTypeInfo.phone =
                        securedPhone && securedPhone.number && securedPhone.number.masked_international;
                }

                if (isSocial && account.profiles) {
                    const authProfile = account.profiles.find((profile) => profile.allow_auth);

                    userTypeInfo.provider = authProfile.provider_code;
                }

                if (isLite && account.emails) {
                    userTypeInfo.email = account.emails.default;
                }

                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile_journal, getAccountInfo')
                    .write(error);
                return next();
            });
    },
    function(req, res, next) {
        const allData = res.locals.listData.itemsToRender;
        const nextPage = res.locals.history.nextPage;

        res.locals.listData = {
            itemsToRender: groupByDates(allData),
            next: Boolean(nextPage)
        };

        if (req.body.csrf_token) {
            const pageData = {
                map: res.locals.map,
                journal: allData,
                is2faOn: res.locals.is2FAOn,
                nextPage
            };

            return res.json(pageData);
        }

        if (allData && (allData.length > 5 || nextPage)) {
            res.locals.listData.allHistory = {
                data: allData
            };
            const itemsList = [].concat(allData).splice(0, 5);

            res.locals.listData.itemsToRender = groupByDates(itemsList);
            res.locals.next = true;
        }
        return next();
    },
    function(req, res, next) {
        const retpath = req.body.retpath || req.query.retpath;

        res.locals.passportHost = url.format({
            protocol: req.headers['x-real-scheme'],
            hostname: req.hostname
        });

        if (retpath) {
            return req.api
                .validateRetpath({retpath})
                .then(({body = {}}) => {
                    if (body.retpath) {
                        res.locals.retpath = body.retpath;
                    }
                })
                .catch((error) => {
                    PLog.warn()
                        .logId(req.logID)
                        .type('profile.journal', 'validateRetpath')
                        .write(error);
                })
                .done(next);
        }

        return next();
    },
    function(req, res) {
        const isNotMeExperiment =
            res.locals.experiments && res.locals.experiments.flags.includes('passport-itsnotme-exp');

        if (isNotMeExperiment) {
            res.locals.notMeLink = url.format({
                protocol: req.headers['x-real-scheme'],
                hostname: req.hostname,
                pathname: '/profile/security/notme'
            });
        }

        res.locals.historyType = 'auth';
        return res.render(`profile.journal.${res.locals.language}.js`);
    }
];

exports.routes.journal = [
    setup,
    getAllScopes,
    validateCSRF,
    preloadAuthEvents,
    getYaExperimentsFlags,
    function(req, res, next) {
        const DEFAULT_DAYS_LIMIT = 30;
        const period = req.body.hours_limit || DEFAULT_DAYS_LIMIT;
        const eventsLimit = period * 24;
        const data = {
            password_auths: true,
            hours_limit: eventsLimit
        };

        if (req.body.from_auth_row) {
            data.from_auth_row = req.body.from_auth_row;
        }

        if (res.locals.tempResult.body.next_auth_row) {
            data.from_auth_row = res.locals.tempResult.body.next_auth_row;
        }

        if (data.from_auth_row) {
            req.api
                .getAccountHistory(data)
                .then(function(result) {
                    result.body.events = res.locals.tempResult.body.events.concat(result.body.events);
                    filterDataByPeriod(result.body, period);
                    const events = result.body.events;

                    res.locals.period = period;
                    res.locals.history = processAccountHistory(result, req, res);
                    res.locals.map = processHistoryDataForMap(events, res);
                    next();
                })
                .catch(function(error) {
                    PLog.warn()
                        .logId(req.logID)
                        .type('profile journal', 'getAuthLog')
                        .write(error);
                    dheaderSetup(req, res, () => processError(error, req, res));
                });
        } else {
            const result = res.locals.tempResult;
            const events = result.body.events;

            res.locals.period = period;
            res.locals.history = processAccountHistory(result, req, res);
            res.locals.map = processHistoryDataForMap(events, res);
            next();
        }
    },
    getTokensInfo,
    function(req, res) {
        const allData = res.locals.listData.itemsToRender;
        const nextPage = res.locals.history.nextPage;

        res.locals.listData = {
            itemsToRender: groupByDates(allData),
            next: Boolean(nextPage)
        };

        if (req.body.csrf_token) {
            const pageData = {
                map: res.locals.map,
                journal: allData,
                nextPage
            };

            return res.json(pageData);
        }

        if ((allData && allData.length > 5) || nextPage) {
            res.locals.listData.allHistory = {
                data: allData
            };

            const itemsList = [].concat(allData).splice(0, 5);

            res.locals.listData.itemsToRender = groupByDates(itemsList);
            res.locals.next = true;
        }
    }
];

exports.routes.actions = [
    checkAuth,
    setup,
    getHistoryActions,
    multiAuthAccountsSetup,
    getYaExperimentsFlags,
    dheaderSetup,
    function(_req, res) {
        res.locals.historyType = 'actions';
        return res.render(`profile.journal.${res.locals.language}.js`);
    }
];

function processError(error, req, res) {
    const lang = res.locals.language;

    if (Array.isArray(error)) {
        const errorCode = error[0];

        if (errorCode === 'request.uid_or_session_expected' || errorCode === 'sessionid.invalid') {
            return req._controller.redirectToAuth();
        }

        if (errorCode === 'exception.unhandled' || errorCode === 'backend.historydb_api_failed') {
            res.locals.error = locs[lang].Errors['ErrorsTexts.subscr_err'];
            res.render(`profile.journal.${lang}.js`);
        } else {
            return req._controller.redirectToAuth();
        }
    } else {
        res.render(`profile.journal.${lang}.js`);
    }
}

function getAuthLog(req, res, next) {
    const DEFAULT_DAYS_LIMIT = 30;
    const period = periods[periods.indexOf(req.query.period)] || DEFAULT_DAYS_LIMIT;
    const eventsLimit = period * 24;

    const data = {
        hours_limit: eventsLimit,
        password_auths: true
    };

    if (!res.locals.tempResult) {
        return next();
    }

    if (res.locals.tempResult.body.next_auth_row) {
        data.from_auth_row = res.locals.tempResult.body.next_auth_row;

        req.api
            .getAccountHistory(data)
            .then(function(result) {
                result.body.events = res.locals.tempResult.body.events.concat(result.body.events);
                filterDataByPeriod(result.body, period);
                const events = result.body.events;

                res.locals.period = period;
                res.locals.history = processAccountHistory(result, req, res);
                res.locals.map = processHistoryDataForMap(events, res);
                next();
            })
            .catch(function(error) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile journal', 'getAuthLog')
                    .write(error);
                processError(error, req, res);
            });
    } else {
        const result = res.locals.tempResult;
        const events = result.body.events;

        res.locals.period = period;
        res.locals.history = processAccountHistory(result, req, res);
        res.locals.map = processHistoryDataForMap(events, res);
        next();
    }
}

function getTokensInfo(req, res, next) {
    const {history: {events = []} = {}} = res.locals;
    const eventsLocalCopy = [].concat(events);

    if (eventsLocalCopy.length) {
        const oauthApiClient = new OAuthAPI(
            req.logID,
            req._controller.getHeaders(),
            res.locals.language,
            req._controller.getAuth().getUid()
        );
        const handles = [];
        const itemsWithTokens = [];
        const isValidClientId = (function() {
            const clientIdRe = /^[0-9a-f]{32}$/;

            return function(clientId) {
                return clientIdRe.test(clientId);
            };
        })();

        const handlesCache = {};

        eventsLocalCopy.forEach(function(event, _idx, list) {
            if (typeof event === 'undefined') {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.journal', 'getTokensInfo')
                    .write(list);

                return;
            }
            if (event.token && event.token.clientId && isValidClientId(event.token.clientId)) {
                const token = event.token;
                const handle =
                    handlesCache[token.clientId] ||
                    (handlesCache[token.clientId] = oauthApiClient.clientInfoVer3(token.clientId));

                handles.push(handle);
                itemsWithTokens.push(event);
            } else {
                return;
            }
        });

        return when
            .all(handles)
            .then(function(result) {
                for (let d = 0; d < itemsWithTokens.length; d++) {
                    if (itemsWithTokens[d].token) {
                        itemsWithTokens[d].token.icon = result[d]._client.icon_url;
                        itemsWithTokens[d].token.isYandex = result[d]._client.is_yandex;
                        itemsWithTokens[d].token.title = result[d]._client.title;
                    }
                }

                res.locals.listData = {
                    itemsToRender: eventsLocalCopy,
                    next: false
                };

                return next();
            })
            .catch(function(error) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile journal', 'getTokensInfo')
                    .write(error);
                res.locals.listData = {
                    itemsToRender: eventsLocalCopy,
                    next: false
                };
                return next();
            });
    }

    res.locals.listData = {
        itemsToRender: null,
        next: false
    };

    return next();
}

function processEventsDataForMap(events, res) {
    const lang = res.locals.language;

    let map = {
        points: [],
        pointInfo: []
    };

    events.forEach(function(item) {
        const geoId = item.ip.geoid;
        const coords = [];
        const browser = item.browser || {};
        const os = item.os || {};
        const info = {};

        if (geoId) {
            info.actionName = returnEventType(lang, item.event_type);

            if (browser.version || browser.name) {
                info.browser = {
                    version: browser.version,
                    name: browser.name
                };
            }

            if (os.version || os.name) {
                info.os = {
                    version: os.version,
                    name: os.name
                };
            }

            if (lookup && geoId) {
                const regionData = lookup.getRegionById(geoId);

                coords.push(regionData.latitude, regionData.longitude);
                // в геобазе зум для Мск = 10
                info.zoom = geoId === 213 ? 8 : regionData.zoom;
            }
            map.pointInfo.push(info);
            map.points.push(coords);
        }
    });
    if (map.points.length === 0) {
        map = null;
    }
    return map;
}

function processHistoryDataForMap(events, res) {
    const lang = res.locals.language;

    let map = {
        points: [],
        pointInfo: []
    };

    if (events && events.length) {
        events.forEach(function(item) {
            const eventAuth = Boolean(returnAuthType(lang, item.auth.authtype));

            if (eventAuth) {
                const authData = item.auth;
                const geoId = authData.ip.geoid;
                const info = {
                    authType: returnAuthType(lang, authData.authtype),
                    count: item.count
                };
                const coords = [];
                const eventInfo = item.authentications;

                eventInfo.forEach(function(n) {
                    if (n.latitude && n.longitude) {
                        coords.push(n.latitude, n.longitude);
                    }
                });

                if (authData && authData.browser) {
                    info.browser = {
                        version: authData.browser.version,
                        name: authData.browser.name
                    };
                }

                if (authData && authData.os) {
                    info.os = {
                        version: authData.os.version,
                        name: authData.os.name
                    };
                }

                if (lookup && geoId) {
                    const regionData = lookup.getRegionById(geoId);

                    if (coords.length === 0) {
                        coords.push(regionData.latitude, regionData.longitude);
                    }
                    // в геобазе зум для Мск = 10
                    info.zoom = geoId === 213 ? 8 : regionData.zoom;
                }

                if (authData.token && authData.token.deviceName) {
                    info.deviceName = authData.token.deviceName;
                }
                map.pointInfo.push(info);
                map.points.push(coords);
            }
        });
    }
    if (map.points.length === 0) {
        map = null;
    }

    return map;
}

function processCommonAccountInfo(data, res) {
    const lang = res.locals.language;
    const geoId = data.ip.geoid;
    const regionLocales = geoId ? lookup.getLinguistics(geoId, lang) : '';
    const provider = providers[`AS${data.ip.AS}`] || {};
    const commonInfo = {
        ip: data.ip.ip,
        provider: provider[lang] || provider.en
    };

    if (geoId) {
        if (regionLocales) {
            commonInfo.locRegion = regionLocales.nominative;
        } else {
            commonInfo.locRegion = lookup.getRegionById(geoId) ? lookup.getRegionById(geoId).name : '';
        }
    }

    if (data.browser && data.browser.version) {
        commonInfo.browser = {
            version: data.browser.version,
            name: data.browser.name
        };
    }

    if (data.os && data.os.version) {
        commonInfo.os = {
            version: data.os.version,
            name: data.os.name
        };
    }

    return commonInfo;
}

function processAccountHistory(data, req, res) {
    const body = data.body;
    const lang = res.locals.language;
    const history = {
        nextPage: null
    };
    const browsers = {
        Firefox: 'firefox',
        MobileFirefox: 'firefox',
        MobileSafari: 'msafari',
        Opera: 'opera',
        OperaMobile: 'opera',
        AndroidBrowser: 'android',
        YandexBrowser: 'yabrowser',
        Safari: 'safari',
        MSIE: 'ie',
        Edge: 'edge',
        Chrome: 'chrome',
        ChromeMobile: 'chrome',
        Chromium: 'chromium',
        'Samsung Internet': 'samsung-internet',
        YandexSearch: 'yasearch',
        YandexBrowserLite: 'yalite'
    };

    if (body && body.events && body.events.length) {
        // sort items to get the proper order by time
        body.events.sort((a, b) => {
            const authenticationsIsArray =
                a.authentications && Array.isArray(a.authentications) && a.authentications.length > 0;

            if (authenticationsIsArray) {
                return b.authentications[0].timestamp - a.authentications[0].timestamp;
            }
        });

        const journal = body.events.reduce(function(result, item) {
            const eventAuth = Boolean(returnAuthType(lang, item.auth.authtype));

            if (eventAuth) {
                const authData = item.auth;
                const tld = req._controller.getTld() || 'ru';
                const geoId = authData.ip.geoid;
                const regionLocales = geoId ? lookup.getLinguistics(geoId, lang) : '';
                const provider = providers[`AS${authData.ip.AS}`] || {};
                const processedDate = processTimeDataForAuthEvents(item.authentications, res.locals.timezone);
                const event = {
                    count: item.count,
                    timestampOriginal: item.authentications[0] && item.authentications[0].timestamp,
                    timestamp: processedDate.date.day.toString() + processedDate.date.month + processedDate.date.year,
                    titleTime: processedDate.date,
                    time: processedDate.time,
                    authType: returnAuthType(lang, authData.authtype),
                    ip: authData.ip.ip,
                    provider: provider[lang] || provider.en
                };

                if (geoId) {
                    if (regionLocales && regionLocales.nominative) {
                        event.locRegion = regionLocales.nominative;
                    } else {
                        const locName = () => {
                            const geoInfoById = lookup.getRegionById(geoId) || {};
                            const isCom = ['com', 'com.tr'].includes(tld);

                            return isCom ? geoInfoById.en_name : geoInfoById.name;
                        };

                        event.locRegion = locName();
                    }
                }

                if (authData.browser) {
                    event.browser = {
                        version: authData.browser.version,
                        name: authData.browser.name
                    };
                    event.cssClass = browsers[authData.browser.name] || 'default';
                } else {
                    event.cssClass = authData.authtype;
                }

                if (authData.os) {
                    event.os = {
                        version: authData.os.version,
                        name: authData.os.name
                    };
                }

                if (item.auth.token) {
                    const token = item.auth.token;
                    const tokenScopes = token.scopes.split(',');
                    const allScopeList = res.locals.scopesList;

                    event.token = {
                        scopes: [],
                        clientId: token.clientId,
                        grant_xtoken: false
                    };

                    if (token.deviceName) {
                        event.token.deviceName = token.deviceName.replace(/\+/g, ' ');
                    }

                    _.forIn(allScopeList, function(el) {
                        const allScopeListItem = el.items;
                        const tokenScope = {
                            scopeTitle: '',
                            extendedScopes: []
                        };

                        for (let i = 0, l1 = tokenScopes.length; i < l1; i++) {
                            const t = tokenScopes[i];

                            if (t === 'oauth:grant_xtoken') {
                                event.token.grant_xtoken = true;
                            }
                            for (let k = 0, l2 = allScopeListItem.length; k < l2; k++) {
                                if (t === allScopeListItem[k].id) {
                                    tokenScope.scopeTitle = el.section;
                                    tokenScope.extendedScopes.push(allScopeListItem[k].title);
                                }
                            }
                        }
                        if (tokenScope.scopeTitle || tokenScope.extendedScopes.length) {
                            event.token.scopes.push(tokenScope);
                        }
                    });
                }

                if (authData.authtype !== 'web' && !item.auth.token) {
                    event.protocol = authData.authtype;
                }

                result.push(event);
            }

            return result;
        }, []);

        history.nextPage = body.next_auth_row;
        history.events = journal;
    }
    return history;
}

/*
 * PASSP-15093
 * Using for pagination instead of `from_auth_row` param in /2/bundle/account/history/
 */
function filterDataByPeriod(data, period) {
    const DAY_MILLISECONDS = 86400000;
    const currentDate = new Date();
    const minTimestamp = currentDate.getTime() - period * DAY_MILLISECONDS;
    const filteredAuthentications = [];

    data.events = data.events
        .map(function(event) {
            event.authentications = event.authentications.filter(function(authentication) {
                if (authentication.timestamp * 1000 < minTimestamp) {
                    filteredAuthentications.push(authentication);
                }

                return authentication.timestamp * 1000 >= minTimestamp;
            });

            return event;
        })
        .filter(function(event) {
            return event.authentications.length;
        });

    if (filteredAuthentications.length) {
        data.next_auth_row = null;
    }
}

function groupByDates(data) {
    let current;

    let currentDate;
    const dates = [];

    let events = [];

    if (data && data.length) {
        current = data[0].timestamp;
        currentDate = data[0].titleTime;
    } else {
        return [];
    }

    for (let i = 0; i < data.length; i++) {
        const loopItem = data[i];

        if (current !== loopItem.timestamp) {
            // save remembered events in new group
            dates.push({
                data: currentDate,
                items: events
            });

            currentDate = loopItem.titleTime;
            current = loopItem.timestamp;
            events = [];
        }
        // add event to group
        events.push(loopItem);
    }

    if (events.length) {
        dates.push({
            data: currentDate,
            items: events
        });
    }
    return dates;
}

function getAllScopes(req, res, next) {
    const oauthApiClient = new OAuthAPI(
        req.logID,
        req._controller.getHeaders(),
        res.locals.language,
        req._controller.getAuth().getUid()
    );

    if (res.locals.scopesList) {
        return next();
    }

    oauthApiClient
        .allScopes()
        .then(function(response) {
            const result = response.groupBy(function(scope) {
                return scope.getSectionTitle();
            });
            const scopesList = [];

            _.forEach(result, function(scopes, section) {
                const items = _.map(scopes, function(scope) {
                    return {
                        id: scope._id,
                        title: scope.getTitle()
                    };
                });

                scopesList.push({
                    section,
                    items
                });
            });
            res.locals.scopesList = scopesList;
            return next();
        })
        .catch(function(error) {
            PLog.warn()
                .logId(req.logID)
                .type('profile journal', 'getAllScopes')
                .write(error);
            return next();
        });
}

/*
 * PASSP-15093
 * Using for pagination instead of `from_auth_row` param in /2/bundle/account/history/
 */
function preloadAuthEvents(req, res, next) {
    const DEFAULT_DAYS_LIMIT = 30;
    const period = periods[periods.indexOf(req.query.period)] || req.body.hours_limit || DEFAULT_DAYS_LIMIT;
    const eventsLimit = period * 24;
    const data = {
        hours_limit: eventsLimit,
        password_auths: true
    };

    if (req.body.from_auth_row) {
        data.from_auth_row = req.body.from_auth_row;
    }

    req.api
        .getAccountHistory(data)
        .then(function(result) {
            res.locals.tempResult = result;
            next();
        })
        .catch(function(error) {
            PLog.warn()
                .logId(req.logID)
                .type('profile', 'journal', 'preloadAuthEvents')
                .write(error);
            processError(error, req, res);
        });
}

function getHistoryActions(req, res, next) {
    const DEFAULT_DAYS_LIMIT = 30;
    const DEFAULT_HOURS_LIMIT = 720; // 30 дней

    const period = periods[periods.indexOf(req.query.period)];

    let eventsLimit = Number(req.body.hours_limit) * 24 || DEFAULT_HOURS_LIMIT;

    if (period) {
        eventsLimit = period * 24;
    }

    const lang = res.locals.language;
    const data = {
        hours_limit: eventsLimit
    };

    if (req.body.from_timestamp) {
        data.from_timestamp = req.body.from_timestamp;
    }

    req.api
        .getAccountHistoryEvents(data)
        .then(function(result) {
            const events = result.body.events;

            let historyEvent = [];

            events.forEach(function(item) {
                const eventType = item.event_type;
                const action = {
                    time: processTimeDataForActions(item.timestamp, res.locals.timezone),
                    eventName: returnEventType(lang, item.event_type),
                    eventAction: getEventActions(item.actions, eventType, lang)
                };

                if (eventType === 'personal_data') {
                    const changePersonalInfoActions = item.actions.filter((actionItem) => actionItem.changed_fields);

                    action.actionsDescr = handlePersonalDataAction(changePersonalInfoActions, lang) || [];
                }
                historyEvent.push(_.assign(action, processCommonAccountInfo(item, res)));
            });

            if (historyEvent.length === 0) {
                historyEvent = null;
            }

            if (req.body.csrf_token) {
                const pageData = {
                    map: processEventsDataForMap(result.body.events, res),
                    journal: historyEvent,
                    nextPage: result.body.next_from_timestamp
                };

                return res.json(pageData);
            }

            res.locals.period = periods[periods.indexOf(req.query.period)] || DEFAULT_DAYS_LIMIT;
            res.locals.listData = {
                itemsToRender: historyEvent
            };
            res.locals.history = {
                nextPage: result.body.next_from_timestamp
            };

            if (historyEvent && historyEvent.length > 5) {
                res.locals.listData.itemsToRender = [].concat(historyEvent).splice(0, 5);
                res.locals.listData.allActions = {
                    data: historyEvent
                };
            }
            res.locals.map = processEventsDataForMap(result.body.events, res);
            return next();
        })
        .catch(function(error) {
            PLog.warn()
                .logId(req.logID)
                .type('profile journal', 'getHistoryActions')
                .write(error);
            processError(error, req, res);
        });
}

function getCountryName(isoCode, lang) {
    const countryNameFromCountriesList = countries[lang].find((item) => item.iso === isoCode.toLowerCase()).name;

    return countryNameFromCountriesList ? countryNameFromCountriesList : isoCode;
}

function getTimezoneText(timezone, lang) {
    const tzFromTimezonesList = timezones[lang].find((tzone) => tzone.val === timezone).text;

    return tzFromTimezonesList ? tzFromTimezonesList : timezone;
}

function processEmail(email) {
    const splited = email.split('@');

    return _.escape([splited[0], punycode.toUnicode(splited[1])].join('@'));
}

function returnUserTimezone(req, res, next) {
    req.api
        .suggestTimezone()
        .then(function(result) {
            res.locals.timezone = result.body.timezone[0] || 'Europe/Moscow';
            return next();
        })
        .catch(function(error) {
            PLog.warn()
                .logId(req.logID)
                .type('profile journal', 'returnUserTimezone')
                .write(error);
            return next();
        });
}

function returnAuthType(lang, authType) {
    const types = {
        imap: locsJournal[lang].ProfileJournal['profile.journal.enter-imap'],
        smtp: locsJournal[lang].ProfileJournal['profile.journal.enter-imap'],
        pop3: locsJournal[lang].ProfileJournal['profile.journal.enter-imap'],
        xmpp: locsJournal[lang].ProfileJournal['profile.journal.enter-xmpp'],
        webdav: locsJournal[lang].ProfileJournal['profile.journal.enter-webdav'],
        calendar: locsJournal[lang].ProfileJournal['profile.journal.enter-calendar'],
        web: locsJournal[lang].ProfileJournal['profile.journal.enter-browser'],
        'password-oauth': locsJournal[lang].ProfileJournal['profile.journal.enter-mobile']
    };

    return types[authType];
}

function returnEventType(lang, eventType) {
    const events = {
        forced_password_change: locsJournal[lang].ProfileJournal['profile.journal.force-pass-change'],
        password: locsJournal[lang].ProfileJournal['profile.journal.pass-change'],
        personal_data: locsJournal[lang].ProfileJournal['profile.journal.change-personal-data'],
        phone_bind: locsJournal[lang].ProfileJournal['profile.journal.bind-phone'],
        phone_unbind: locsJournal[lang].ProfileJournal['profile.journal.unbind-phone'],
        phone_add: locsJournal[lang].ProfileJournal['profile.journal.add-secure-phone'],
        phone_remove: locsJournal[lang].ProfileJournal['profile.journal-delete-secure-phone'],
        phone_replace: locsJournal[lang].ProfileJournal['profile.journal.replace-phone'],
        email_add: locsJournal[lang].ProfileJournal['profile.journal-add-email'],
        email_remove: locsJournal[lang].ProfileJournal['profile.journal.remove-email'],
        totp_enabled: locsJournal[lang].ProfileJournal['profile.journal.2fa-enable'],
        totp_migrated: locsJournal[lang].ProfileJournal['profile.journal.2fa-migrate'],
        totp_disabled: locsJournal[lang].ProfileJournal['profile.journal.2fa-disable'],
        app_passwords_enabled: locsJournal[lang].ProfileJournal['profile.journal.appwd-enable'],
        app_passwords_disabled: locsJournal[lang].ProfileJournal['profile.journal.appwd-disable'],
        restore: locsJournal[lang].ProfileJournal['profile.journal.restoration'],
        restore_entities_flushed: locsJournal[lang].ProfileJournal['profile.journal.restore-entities-flushed'],
        global_logout: locsJournal[lang].ProfileJournal['profile.journal.glogout'],
        questions: locsJournal[lang].ProfileJournal['profile.journal.change-kq'],
        secure_phone_set: locsJournal[lang].ProfileJournal['profile.journal.secure-phone-bound'],
        secure_phone_replace: locsJournal[lang].ProfileJournal['profile.journal.secure-phone-changed'],
        secure_phone_unset: locsJournal[lang].ProfileJournal['profile.journal.secure-phone-deleted'],
        hint: locsJournal[lang].ProfileJournal['profile.journal.personal-hint']
    };

    return events[eventType];
}

function processTimeDataForActions(secs, localsTimezone) {
    const date = new Date(secs * 1000); // Sec to ms
    const timezone = localsTimezone || 'Europe/Moscow';
    const momentTZ = momentTimezone.tz(date, timezone);
    const momentTimezoneObj = momentTimezone(momentTZ);
    const minutes = momentTimezone(date).minutes();

    return {
        date: {
            year: momentTimezoneObj.year(),
            month: momentTimezoneObj.month() + 1,
            day: momentTimezoneObj.date()
        },
        time: `${momentTimezone(date).hour()}:${minutes >= 10 ? minutes : `0${minutes}`}`
    };
}

function processTimeDataForAuthEvents(infoArray, localsTimezone) {
    const timestamps = infoArray.map((item) => item.timestamp);
    const date = new Date(timestamps[0] * 1000); // sec to ms
    const authTimes = timestamps.slice(0, 5);
    const timezone = localsTimezone || 'Europe/Moscow';
    const momentTZ = momentTimezone.tz(date, timezone);
    const momentTimezoneObj = momentTimezone(momentTZ);
    const result = {
        date: {
            year: momentTimezoneObj.year(),
            month: momentTimezoneObj.month() + 1,
            day: momentTimezoneObj.date()
        },
        time: []
    };

    authTimes.forEach((n) => {
        const tDate = new Date(n * 1000);
        const t = momentTimezone.tz(tDate, timezone);
        const minutes = momentTimezone(t).minute();
        const times = `${momentTimezone(t).hour()}:${minutes >= 10 ? minutes : `0${minutes}`}`;

        result.time.push(times);
    });

    return result;
}

function handlePersonalDataAction(actions, lang = 'ru') {
    const journalLocs = locsJournal[lang].ProfileJournal;
    const personal = {
        city: journalLocs['profile.journal.city'],
        country: journalLocs['profile.journal.country'],
        tz: journalLocs['profile.journal.timezone'],
        firstname: journalLocs['profile.journal.name'],
        lastname: journalLocs['profile.journal.surname'],
        birthday: journalLocs['profile.journal.birthday'],
        sex: journalLocs['profile.journal.sex'],
        display_name: journalLocs['profile.journal.displayname']
    };
    const actionDescriptions = [];

    actions.forEach((actionItem) => {
        const getGenderText = (val) => {
            if (val === '2') {
                return locsJournal[lang].ProfileJournal['profile.journal.female'];
            }
            return locsJournal[lang].ProfileJournal['profile.journal.male'];
        };
        const isTemlateDisplayName = actionItem.display_name_format === 'template';

        _.forIn(actionItem, function(val, key) {
            if (personal[key]) {
                const changedPersonalData = {
                    actionName: personal[key],
                    actionValue: actionItem[key]
                };

                if (key === 'sex') {
                    changedPersonalData.actionValue = getGenderText(val);
                }

                if (key === 'country') {
                    changedPersonalData.actionValue = getCountryName(val, lang);
                }

                if (key === 'tz') {
                    changedPersonalData.actionValue = getTimezoneText(val, lang);
                }

                if (isTemlateDisplayName) {
                    changedPersonalData.actionValue = '';
                }
                actionDescriptions.push(changedPersonalData);
            }
        });
    });
    return actionDescriptions;
}

function getEventActions(actions, eventType, lang = 'ru') {
    const journalLocs = locsJournal[lang].ProfileJournal;
    const eventActions = [];
    const actionsTexts = {
        email_add: journalLocs['profile.journal.email-added'],
        email_remove: journalLocs['profile.journal.email-removed'],
        email_remove_all: journalLocs['profile.journal.email-remove-all'],
        global_logout: journalLocs['profile.journal.glogout'],
        questions_change: journalLocs['profile.journal.kq-changed'],
        questions_remove: journalLocs['profile.journal.kq-removed'],
        password_change: journalLocs['profile.journal.password-changed'],
        password_remove: journalLocs['profile.journal.password-removed'],
        web_sessions_revoked: journalLocs['profile.journal.web-sessions-revoked'],
        tokens_revoked: journalLocs['profile.journal.tokens-revoked'],
        app_passwords_revoked: journalLocs['profile.journal.app-passwords-revoked'],
        personal_data: journalLocs['profile.journal.change-personal-data'],
        secure_phone_set: journalLocs['profile.journal.secure-phone-bound'],
        secure_phone_replace: journalLocs['profile.journal.secure-phone-changed'],
        secure_phone_unset: journalLocs['profile.journal.secure-phone-deleted'],
        semi_auto: journalLocs['profile.journal.semi_auto'],
        link: journalLocs['profile.journal.restore-link'],
        restore: journalLocs['profile.journal.restore-by'],
        phone_and_2fa_factor: journalLocs['profile.journal.phone-and-2fa_factor'],
        totp_disabled: journalLocs['profile.journal.2fa-disable'],
        email: journalLocs['profile.journal.personal-email'],
        phone: journalLocs['profile.journal.personal-phone'],
        hint: journalLocs['profile.journal.personal-hint'],
        phone_set: journalLocs['profile.journal.phone-set'],
        phone_bind: journalLocs['profile.journal.phone-set'],
        phone_unset: journalLocs['profile.journal.phone-unset'],
        phone_unbind: journalLocs['profile.journal.phone-unset']
    };

    _.forEach(actions, function(val) {
        if (eventType === val.type) {
            if (val.email_bind) {
                eventActions.push(`${actionsTexts[val.type]} <br /> ${processEmail(val.email_bind)}`);
            }

            if (val.email_unbind) {
                eventActions.push(`${actionsTexts[val.type]} <br /> ${processEmail(val.email_unbind)}`);
            }

            if (val.phone_set) {
                eventActions.push(`${actionsTexts.phone_set} <br /> ${val.phone_set}`);
            }

            if (val.phone_bind) {
                eventActions.push(`${actionsTexts.phone_bind} <br /> ${val.phone_bind}`);
            }

            if (val.phone_unbind) {
                eventActions.push(`${actionsTexts.phone_unbind} <br /> ${val.phone_unbind}`);
            }

            if (val.phone_unset) {
                eventActions.push(`${actionsTexts.phone_unset} <br /> ${val.phone_unset}`);
            }

            if (val.restore_by) {
                const method = val.restore_by;

                let actionText;

                if (val[method]) {
                    actionText = `${actionsTexts[val.type]}: ${val[method]}`;
                } else {
                    actionText = `${actionsTexts[val.type]}: ${actionsTexts[method]}`;
                }

                eventActions.push(actionText);
            }
        } else {
            eventActions.push(actionsTexts[val.type]);
        }
    });
    return eventActions;
}

exports.getAllScopes = getAllScopes;
exports.preloadAuthEvents = preloadAuthEvents;
exports.getAuthLog = getAuthLog;
exports.getTokensInfo = getTokensInfo;
exports.groupByDates = groupByDates;
exports.processTimeDataForActions = processTimeDataForActions;
module.exports.journal = exports.routes.journal;
module.exports.actions = exports.routes.actions;
