const express = require('express');
const router = express.Router();
const PLog = require('plog');
const langSetup = require('./common/langSetup');
const lookup = require('../lib/getLookup.js');
const multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
const apiSetup = require('./common/apiSetup');
const getUatraitsData = require('./common/getUatraitsData');
const getCountry = require('./common/getCountry');
const getTrack = require('./common/getTrack');
const createSecurityState = require('./security/createState');
const getAccountInfo = require('./security/getAccountInfo');
const showPlus = require('./common/plusGuard').showPlus;
const rumCounterSetup = require('./common/rumCounterSetup');
const createState = require('./common/createState');
const getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
const getMetrics = require('./common/getMetrics');
const processTimeDataForActions = require('./profile.journal').processTimeDataForActions;

const enter = [
    apiSetup,
    getYaExperimentsFlags,
    getUatraitsData,
    langSetup,
    getTrack(),
    getCountry,
    function checkEnv(req, res, next) {
        if (process.env.NODE_ENV === 'development' || process.env.NODE_ENV === 'testing') {
            return next();
        }

        return req._controller.redirectToFrontpage();
    },
    showPlus,
    function(req, res, next) {
        if (res.locals.showPlus) {
            res.locals.blackboxPlus = true;
        }

        return next();
    },
    multiAuthAccountsSetup,
    createState,
    createSecurityState,
    getAccountInfo,
    function getTimezone(req, res, next) {
        if (req.query && req.query.ts) {
            return req.api
                .suggestTimezone()
                .then(function(result) {
                    res.locals.timezone = result.body.timezone[0] || 'Europe/Moscow';
                    return next();
                })
                .catch(function(error) {
                    PLog.warn()
                        .logId(req.logID)
                        .type('profile notme', 'getTimezone')
                        .write(error);
                });
        }

        return next();
    },
    getDataFromTimestamp,
    function createNotMeState(req, res, next) {
        const {authInfo, store = {}} = res.locals;

        if (store.securityInfo) {
            store.securityInfo.step = 'passwordForm';
            store.securityInfo.isNotMePage = true;
            store.securityInfo.screenType = 'password';

            if (req.query && req.query.ts && authInfo) {
                store.securityInfo.step = 'authInfo';
                store.securityInfo.authInfo = authInfo;
            }
        }

        return next();
    },
    getMetrics({
        header: 'Страница "Это не я"'
    }),
    rumCounterSetup,
    function(req, res) {
        const lang = res.locals.language;

        res.render(`react.notme.${lang}.jsx`);
    }
];

function getDataFromTimestamp(req, res, next) {
    const ts = req.query && req.query.ts;

    if (!ts) {
        return next();
    }

    if (req.query && req.query.origin === 'journal') {
        req.api
            .getAccountHistory({
                password_auths: true
            })
            .then((result = {}) => {
                const {body = {}} = result;
                const {events = {}} = body;
                const tld = req._controller.getTld() || 'ru';
                const authFromtimestamp = Object.values(events).find(
                    (item) => item.authentications && item.authentications[0].timestamp.toString() === ts
                );

                res.locals.authInfo = formatAuthData(authFromtimestamp, tld, res.locals.language);
                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile notme', 'getDataFromTimestamp')
                    .write(error);
                return next();
            });
    } else {
        req.api
            .getChallengeInfo({
                challenge_time: ts
            })
            .then((result = {}) => {
                const tld = req._controller.getTld() || 'ru';
                const {body = {}} = result;
                const {challenge = {}} = body;

                if (challenge.timestamp) {
                    const dateObj = processTimeDataForActions(challenge.timestamp);

                    res.locals.authInfo = {
                        date: `${dateObj.date.day}.${dateObj.date.month}.${dateObj.date.year}`,
                        time: dateObj.time,
                        place: getRegionName(parseInt(challenge['ip.geoid'], 10), tld, res.locals.language),
                        system: `${challenge['os.name']}, ${challenge['os.version']}`,
                        browser: ` ${challenge['browser.name']}, ${challenge['browser.version']}`
                    };
                }

                return next();
            })
            .catch((error) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile notme', 'getDataFromTimestamp')
                    .write(error);

                return next();
            });
    }
}

function formatAuthData(data = {}, tld = 'ru', lang) {
    const {ip = {}, os = {}, browser = {}} = data.auth;
    const dateObj = processTimeDataForActions(data.authentications[0].timestamp);

    return {
        date: `${dateObj.date.day}.${dateObj.date.month}.${dateObj.date.year}`,
        time: dateObj.time,
        place: getRegionName(ip.geoid, tld, lang),
        system: `${os.name}, ${os.version}, ${browser.name}, ${browser.version}`
    };
}

function getRegionName(geoId, tld = 'ru', lang) {
    if (!geoId || typeof geoId !== 'number') {
        return;
    }

    const regionLocales = geoId ? lookup.getLinguistics(geoId, lang) : '';

    let locRegion = '';

    if (regionLocales && regionLocales.nominative) {
        locRegion = regionLocales.nominative;
    } else {
        const locName = () => {
            const geoInfoById = lookup.getRegionById(geoId) || {};
            const isCom = ['com', 'com.tr'].includes(tld);

            return isCom ? geoInfoById.en_name : geoInfoById.name;
        };

        locRegion = locName();
    }

    return locRegion;
}

router.get('/', enter);
router.post('/', enter);

exports.router = router;
