'use strict';

var url = require('url');
var _ = require('lodash');
var when = require('when');
var PLog = require('plog');
var geo = require('yandex-geobase')();
var config = require('../configs/current');
var apiSetup = require('./common/apiSetup');
var getYaExperimentsFlags = require('./common/getYaExperimentsFlags');
var processSocialProfiles = require('./profile.social').processProfiles;
var checkUidFromQuery = require('./avatars').checkUidFromQuery;
var removePhonishes = require('./profile.social').removePhonishes;
var socialProviders = require('./common/socialSetup').socialProviders;
var multiAuthAccountsSetup = require('./common/multiAuthAccountsSetup').getAccounts;
var LangSwitcherView = require('../blocks/layout/LangSwitcherView');
var xmlParser = require('xml2json-light');
var moment = require('moment');

var brokerPath = config.paths.broker;
var yamoneyDefaultState = {
    sum: undefined,
    hasWallet: false,
    connectionError: true
};
var redirectMap = {
    passport: '/profile',
    changereg: '/profile/personal-info',
    changehint: '/profile/change-hint',
    changedisplayname: '/profile/display-name',
    changeemails: '/profile/emails/list',
    changeavatar: '/profile/avatars'
};
var setup;

function checkSocialtschick(req, res, next) {
    var account = res.locals.account;
    var currentUrl;

    if (account.password_info.strength === -1 && !account.is_2fa_enabled) {
        currentUrl = req._controller.getUrl();

        return res.redirect(
            url.format(
                _.extend({}, currentUrl, {
                    pathname: '/profile/upgrade',
                    query: {
                        origin: 'passport_profile',
                        retpath: url.format(currentUrl)
                    }
                })
            )
        );
    }

    return next();
}

function checkStrongPolicy(req, res, next) {
    var account = res.locals.account;

    if (account.password_info.strong_policy_on || (account.is_2fa_enabled && !account.question)) {
        return res.redirect(
            url.format({
                pathname: '/profile',
                query: req.query
            })
        );
    }

    return next();
}

function getLanguage(req, res, next) {
    req._controller
        .getLanguage()
        .then(function(lang) {
            res.locals.language = lang;
        })
        .catch(function(err) {
            res.locals.language = 'ru';

            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(err);
        })
        .done(function() {
            next();
        });
}

function renderPage(req, res) {
    const experimentFlags = res.locals.experiments && res.locals.experiments.flagsString;
    const experimentBoxes = res.locals.experiments && res.locals.experiments.boxes;
    var lang = res.locals.language;

    req.api.statboxLogger({
        track_id: res.locals.track_id || null,
        action: 'opened',
        mode: 'morda',
        host: req.hostname,
        pathname: req.path,
        referer: req.headers.referer || null,
        ip: req.headers['x-real-ip'],
        user_agent: req.headers['user-agent'],
        yandexuid: req.cookies.yandexuid,
        uid: (res.locals.store && res.locals.store.person && res.locals.store.person.uid) || null,
        origin: (req.query && req.query.origin) || null,
        experiment_flags: experimentFlags,
        experiment_boxes: experimentBoxes
    });

    res.render(`profile.passport.${lang}.jsx`);
}

function createStore(req, res, next) {
    // eslint-disable-line
    var locals = res.locals;
    var langSwitcher = new LangSwitcherView(req._controller, locals.track_id);
    var account = locals.account;
    var tld = req._controller.getTld();
    var retpath = (req.query && (req.query.url || req.query.retpath)) || null;
    var yandexuid = res._yandexuid && res._yandexuid.replace(/[^0-9]/g, '');
    var handlers = [req._controller.getUatraits(), langSwitcher._compile()];
    var geoId;
    var regionLocales;
    var region;

    if (retpath) {
        handlers.push(req.api.validateRetpath({retpath}));
    } else {
        retpath = url.format(
            _.extend({}, req._controller.getUrl(), {
                pathname: 'profile',
                search: null,
                query: null
            })
        );
    }

    locals.store = _.merge(
        {},
        {
            settings: {
                location: req.path,
                host: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname
                }),
                avatar: config.paths.avatar || {},
                embeddedauth: config.paths.embeddedauth || '',
                links:
                    process.env.INTRANET === 'intranet'
                        ? config.links.intranet
                        : config.links[tld] || config.links.com || {},
                help: config.paths.help || {},
                tld,
                isYandex: locals.isYandex,
                language: locals.language,
                ua: {},
                env: {
                    type: process.env.NODE_ENV,
                    name: process.env.INTRANET
                },
                billingUpdateStatus: url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: config.paths.billingUpdateStatusPath
                }),
                yamoneyCards: config.paths.yamoneyCards,
                tunePlaces: config.paths.tunePlaces,
                marketAddresses: config.paths.marketAddresses,
                staticMaps: config.paths.staticMaps,
                accountsUrl: config.paths.accountsUrl
            },
            common: {
                actionForRepeat: null,
                hasComponentsForNav: true,
                uid: account.uid,
                showRegPopup: false,
                yandexuid,
                track_id: locals.track_id,
                edit: locals.edit || '',
                canChangePassword: account.can_change_password,
                retpath: null,
                // при загрузке срабатывает POP event и уменьшает историю на 1, а так тут должен быть 0
                historyOnPassport: 1,
                currentPage: req._controller.getUrl().href,
                defaultPage: url.format({
                    pathname: '/profile',
                    query: req.query
                }),
                isPDD: account.domain !== undefined,
                experiments: res.locals.experiments || {
                    flags: [],
                    flagsString: '',
                    boxes: '',
                    encodedBoxes: ''
                },
                isWSUser: locals.isWSUser,
                dev: config.dev
            },
            captcha: {
                loading: false,
                loadingAudio: false,
                playing: false,
                type: 'text',
                key: null,
                imageUrl: null,
                introSound: null,
                captchaSound: null
            },
            person: _.extend({}, account.person || {}, {
                city: account.person.city || '',
                uid: account.uid,
                login: account.login,
                displayLogin: account.display_login,
                birthday: account.person.birthday || '',
                avatarId: (account.display_name && account.display_name.default_avatar) || '',
                displayNames: account.display_names || {},
                displayName: (account.display_name && account.display_name.name) || '',
                escapedDisplayName: (account.display_name && _.escape(account.display_name.name)) || '',
                errors: {}
            }),
            rfcTotp: {
                qrSVG: null,
                secret: ''
            },
            access: {
                is2faEnabled: account.is_2fa_enabled,
                isAppPasswordsEnabled: account.app_passwords_enabled,
                isSocialchik: _.get(account, 'password_info.strength', null) === -1 && !account.is_2fa_enabled,
                passwordInfo: account.password_info || {},
                featureHint: ''
            },
            appPasswords: {
                createAppPassword: {
                    configClients: config.appPasswordsClientIdMapping,
                    clients: [],
                    clientId: '',
                    deviceName: '',
                    clientName: '',
                    passwordCreated: ''
                },
                tokens: {
                    appPasswordsCount: 0,
                    list: [],
                    showDisableScreen: false
                }
            },
            security: {
                securityLevel: config.securityLevels[account.security_level] || 'undefined',
                lastAuth: account.lastauth || {},
                controlQuestion: {
                    errors: {},
                    available: [],
                    visible: false,
                    loading: false,
                    current: account.question,
                    selected: '0'
                }
            },
            social: {
                providers: socialProviders[tld].providers.slice().filter((i) => i.enabled),
                profiles:
                    (account.profiles && removePhonishes(processSocialProfiles(account.profiles, locals.language))) ||
                    [],
                brokerPath,
                showAllSettings: false,
                allowAuthMethodError: {
                    profileId: 0,
                    error: ''
                },
                phonishes: res.locals.phonishes
            },
            emails: {
                loading: null,
                error: null,
                states: [],
                addedEmail: null,
                openAliasesList: false,
                emails: locals.emails
            },
            domik: {
                errors: {},
                requestPassword: false,
                requestCaptcha: false,
                passwordNotMatched: false,
                captchaNotMatched: false,
                loading: false
            },
            phones: {
                restore: [],
                other: []
            },
            yamoney: _.clone(yamoneyDefaultState),
            billing: {},
            header: {
                accounts: (locals.accounts && locals.accounts.accounts) || [],
                defaultAccount: (locals.accounts && locals.accounts.defaultAccount) || {},
                canAddMore: locals.accounts && locals.accounts['can-add-more']
            },
            footer: {
                langlist: []
            },
            metrics: {
                header: 'Страница профиля',
                experiments: res.locals.experiments ? res.locals.experiments.encodedBoxes : ''
            },
            monitoring: {
                page: 'profile'
            },
            pageMessage: null,
            changeAvatar: {
                track_id: '',
                queryUid: res.locals.queryUid || '',
                url: null,
                defaultUrl: config.paths.avatar.default_300,
                backupAvatar: {
                    id: 0,
                    url: ''
                },
                hasChanged: false,
                id: 0,
                isByUrl: false,
                status: 'normal',
                error: '',
                clear: false
            }
        },
        res.locals.store
    );

    if (account.phones) {
        Object.keys(account.phones).forEach(function(code) {
            var phone = account.phones[code];
            var _phone;

            if (phone.bound) {
                _phone = {
                    id: phone.id,
                    number: phone.number.masked_international,
                    isDefault: phone.is_default,
                    isAlias: phone.is_alias
                };

                locals.store.phones[phone.secured ? 'restore' : 'other'].push(_phone);
            }
        });
    }

    if (locals.store.security.lastAuth) {
        geoId = (locals.store.security.lastAuth.ip && locals.store.security.lastAuth.ip.geoid) || '';

        if (geoId) {
            regionLocales = geoId ? geo.linguistics(geoId, locals.language) : '';

            if (regionLocales) {
                locals.store.security.lastAuth.region = regionLocales.nominative || '';
            } else {
                region = geo.regionById(geoId);
                locals.store.security.lastAuth.region = region.name || '';
            }
        }
    }

    when.all(handlers)
        .then(function(response) {
            var uatraits = response[0] || {};
            var langlist = (response[1] && response[1].langlist) || {};
            var validatedRetpath = response[3] || {};

            locals.store.settings.ua = uatraits;
            locals.store.footer.langlist = langlist;

            if (validatedRetpath.body && validatedRetpath.body.retpath) {
                locals.store.common.retpath = validatedRetpath.body.retpath;
            }

            if (!uatraits.isMobile && !uatraits.isTouch) {
                Object.keys(locals.store.person.displayNames).forEach(function(key) {
                    var value = locals.store.person.displayNames[key];

                    delete locals.store.person.displayNames[key];

                    locals.store.person.displayNames[_.escape(key)] = value;
                });
            }
        })
        .catch(function(err) {
            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(err);
        })
        .done(function() {
            delete locals.account;
            delete locals.accounts;

            return next();
        });
}

function getState(req, res, next) {
    when.all([req.api.profileGetState(), req.api.getEmails()])
        .then(function(response) {
            var profile = response[0].body;
            var emails = response[1].body;

            res.locals.emails = req._controller.decodePunycodeEmails(emails.emails || {});
            res.locals.account = profile.account || {};

            return next();
        })
        .catch(function(errors) {
            var retpath;
            var query;

            PLog.warn()
                .logId(req.logID)
                .type('profile.passport')
                .write(errors);

            if (Array.isArray(errors)) {
                retpath = url.format(req._controller.getUrl());

                if (errors.indexOf('sessionid.invalid') !== -1 || errors.indexOf('account.disabled') !== -1) {
                    return res.redirect(
                        url.format(
                            _.extend({}, req._controller.getAuthUrl(), {
                                query: {
                                    retpath
                                }
                            })
                        )
                    );
                }

                if (errors[0].code && errors[0].code === 'unknowntrack') {
                    query = _.clone(req.query);

                    delete query.track_id;

                    return res.redirect(
                        url.format({
                            protocol: req.headers['x-real-scheme'],
                            hostname: req.hostname,
                            pathname: req.path,
                            query
                        })
                    );
                }
            }

            return next(errors);
        });
}

setup = [
    multiAuthAccountsSetup,
    function(req, res, next) {
        var retpath;
        var controller = req._controller;

        if (!req.blackbox || !req.blackbox.users) {
            retpath = url.format(controller.getUrl());

            return res.redirect(
                url.format(
                    _.extend({}, controller.getAuthUrl(), {
                        query: {
                            retpath
                        }
                    })
                )
            );
        }

        return next();
    },
    function(req, res, next) {
        if (res.locals.isWSUser && req.path === redirectMap.changereg) {
            return res.redirect(url.format(_.extend({}, req._controller.getModePassportUrl(), {query: req.query})));
        }

        return next();
    },
    getLanguage,
    getYaExperimentsFlags,
    apiSetup,
    function(req, res, next) {
        return req.api
            .initTrack({
                type: 'authorize'
            })
            .then(function(response) {
                res.locals.track_id = (response.body && response.body.track_id) || '';
                return next();
            })
            .catch(function(errors) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport')
                    .write(errors);

                if (Array.isArray(errors) && errors[0] === 'account.disabled_on_deletion') {
                    return res.redirect(
                        url.format(
                            Object.assign({}, req._controller.getAuthUrl(), {
                                query: {
                                    retpath: url.format(req._controller.getUrl())
                                }
                            })
                        )
                    );
                }

                return next();
            });
    },
    function(req, res, next) {
        res.locals.phonishes = [];

        if (!res.locals.track_id || process.env.INTRANET === 'intranet') {
            return next();
        }

        return req.api
            .getPhonishProfiles(res.locals.track_id)
            .then((data) => {
                if (data.hasOwnProperty('body') && data.body.hasOwnProperty('profiles')) {
                    for (let i = 0; i < data.body.profiles.length; i++) {
                        const profile = data.body.profiles[i];

                        if (profile.provider_code === 'ya' && profile.hasOwnProperty('phonish')) {
                            res.locals.phonishes.push(
                                Object.assign({}, profile, {
                                    provider: {
                                        code: profile.provider_code
                                    },
                                    profileId: profile.profile_id,
                                    registrationDate: profile.phonish.registration_timestamp
                                        ? moment(profile.phonish.registration_timestamp * 1000)
                                              .locale(res.locals.language)
                                              .format('D MMMM YYYY')
                                        : ''
                                })
                            );
                        }
                    }
                }

                return next();
            })
            .catch((err) => {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport')
                    .write(err);

                return next();
            });
    }
];

function transformDate(day, month, year) {
    if (day.length !== 2 && parseInt(day, 10) < 10) {
        day = `0${day}`;
    }

    return [year, month, day].join('-');
}

exports.route = function(app) {
    // REDIRECT FROM OLD PATHS
    app.get('/passport', function(req, res, next) {
        var mode = req.body.mode || req.query.mode;
        var query = _.clone(req.query);

        delete query.mode;

        if ((mode && mode in redirectMap) || !mode) {
            PLog.info()
                .logId(req.logID)
                .type('profile.passport')
                .write(
                    'Redirect from',
                    mode ? `/passport?mode=${mode}` : '/passport',
                    'to',
                    redirectMap[mode ? mode : 'passport']
                );

            return res.redirect(
                301,
                url.format({
                    protocol: req.headers['x-real-scheme'],
                    hostname: req.hostname,
                    pathname: redirectMap[mode ? mode : 'passport'],
                    query
                })
            );
        }

        return next('route');
    });

    app.get('/profile', this.enter)
        .get('/profile/personal-info', this.enter)
        .get('/profile/display-name', this.enter)
        .get('/profile/change-hint', this.kvkoEnter)
        .get('/profile/emails', this.emailsEnter)
        .get('/profile/emails/*', this.emailsEnter)
        .get('/profile/social', this.enter)
        .get('/profile/social/:profileId', this.enter)
        .get('/profile/avatars', this.avatarEnter)
        .get('/profile/cards', this.enter)
        .post('/profile/personal-info', this.submit)
        .post('/profile/display-name', this.submit)
        .get('/profile/generate-apppassword', this.emailsEnter)
        .get('/profile/apppasswords-list', this.emailsEnter)
        .get('/profile/address', [
            function(req, res, next) {
                var tld = req._controller.getTld();

                if (tld === 'com') {
                    return next('route');
                }

                return next();
            },
            this.enter
        ])
        .get('/profile/devices', this.emailsEnter);
};

exports.kvkoEnter = [setup, getState, checkSocialtschick, checkStrongPolicy, createStore, renderPage];

exports.emailsEnter = [setup, getState, createStore, renderPage];

exports.enter = [setup, getState, createStore, renderPage];

exports.avatarEnter = [setup, getState, checkUidFromQuery, createStore, renderPage];

exports.submit = [
    setup,
    function(req, res, next) {
        const data = _.extend({}, req.body);

        if (data.hasOwnProperty('birthday-day')) {
            data.birthday = transformDate(
                req.body['birthday-day'],
                req.body['birthday-month'],
                req.body['birthday-year']
            );

            ['birthday-day', 'birthday-month', 'birthday-year'].forEach(function(key) {
                delete data[key];
            });
        }

        req.api.profileUpdate(data).then(
            function() {
                return next();
            },
            function() {
                return next();
            }
        );
    },
    getState,
    createStore,
    renderPage
];

exports.getBundle = [
    setup,
    getState,
    createStore,
    function(req, res) {
        return res.json(res.locals.store);
    }
];

exports.getYamoneyInfo = [
    apiSetup,
    function(req, res) {
        var time = Date.now() / 1000;
        var store = _.clone(yamoneyDefaultState);

        if (!config.paths.yamoney) {
            return res.json(store);
        }

        return req._controller
            .getAuth()
            .sessionID()
            .then(function(response) {
                var uid = (response && response.uid && response.uid.value) || '';

                if (!uid) {
                    return when.reject('no uid');
                }

                return req.api.getYaMoneyAccountInfoShort({
                    hash: req._controller.getYaMoneyRequestHash(uid, time),
                    uid,
                    time
                });
            })
            .then(function(yamoney) {
                var yamoneyJSON = xmlParser.xml2json((yamoney && yamoney.body) || '') || {};
                var yamoneyResponse = {
                    status: (yamoneyJSON.yamoney && yamoneyJSON.yamoney.status) || 'error',
                    body: (yamoneyJSON.yamoney && yamoneyJSON.yamoney.response) || {}
                };

                if (yamoneyResponse.status === 'OK') {
                    store.hasWallet = yamoneyResponse.body.error !== 'ACCOUNT_NOT_FOUND';
                    store.connectionError = false;

                    if (yamoneyResponse.body.sum) {
                        store.sum = yamoneyResponse.body.sum['_@ttribute'];
                    }
                }
            })
            .catch(function(err) {
                PLog.warn()
                    .logId(req.logID)
                    .type('profile.passport.yamoney')
                    .write(err);
            })
            .done(function() {
                return res.json(store);
            });
    }
];
